#!/bin/sh

# mkfntview.sh fontname.tt[fc]
# generate a POV-Ray scene file which displays one "page",
# of four, of 256 glyphs from a given font.
# laid out for 'verdana.ttf', other fonts will need tweaking
# the camera settings and or table's position (and or scale).

# version 202106.2

if [ 1 -ne $# ]; then
  echo usage: $(basename $0) 'name.tt[fc]'
  exit 1
fi

set -eu

# extension required, and scene must not exist.
declare -r extn=${1:${#1}-3:3}
if [[ ! ("ttf" == $extn || "ttc" == $extn) ]]; then
  echo oops, expected a TrueType font '(or collection)' name.
  exit 1
fi

declare -r fname=view_$(sed -e 's#\.#_#g' <<< $1).pov
if [ -f $fname ]; then
  echo oops, "file '$fname' already" exists.
  exit 1
fi

# the scene.
cat > $fname <<EOHERE
// $fname
// generated $(date +'%d.%m.%Y %H%Mh'), by '$(basename $0)'.
// +w600 +h800 +a0.1

#version 3.8;

global_settings {assumed_gamma 1}

light_source {<-1,1,-1> * 10e3 color srgb 1 parallel}
light_source {<1,1,-1> * 10e3 color srgb 1 parallel}

camera {
  location <2.5,1,-15.25>
  right x * (3/4)
  up y
  angle 60
  look_at <2.5,1,0>
}

#declare fonts_ = array [3] {"timrom.ttf","${1}","timrom.ttf"};

#include "tabulated.inc"

/* many fonts have 4 * 256 glyphs, or more.
 * use page to select from (first) four.
 * modify the '#elseif' to access higher pages.
 */
#ifndef (global.page)
  #declare pg_ = 0;
#elseif (0 > page | 3 < page | page != int(page))
  #error "oops, 'page' must be integer in range 0,..,3."
#else
  #declare pg_ = page;
#end

#declare n_    = 16;
#declare offs_ = n_ * n_ * pg_;
#declare cs_   = concat("font: ", fonts_[1],
        "  (",str(offs_,0,0),",..,",str(255 + offs_,0,0),")");

/* make the various arrays.
 * '.DataColumns', '.ColAlign', '.ColLabels', '.RowLabels',
 * and '.DataTable'.
 * when first page overwrite asc(0) with space.
 */
#declare dddc_ = array [n_];
#declare ddca_ = array [n_];
#declare clbl_ = array [n_];
#declare rlbl_ = array [n_];
#declare data_ = array [n_][n_];
#for (i_, 0, n_-1)
  #local dddc_[i_] = "S";
  #local ddca_[i_] = 0;
  #local clbl_[i_] = str(i_,0,0);
  #local rlbl_[i_] = str(offs_ + n_ * i_,0,0);
  #for (j_, 0, n_-1)
    #local data_[i_][j_] = chr(offs_ + n_ * i_ + j_);
  #end
#end
#if (!offs_)
  #local data_[0][0] = chr(32);
#end

#declare dict_glyphs_ = dictionary {
  .DataColumns: dddc_,
  .DataTable  : data_,
  .ColAlign   : ddca_,
  .ColLabels  : clbl_,
  .RowLabels  : rlbl_,
  .Fonts      : fonts_,
  .Caption    : array mixed [4] {cs_,0,true,7}
};
object {Tabulated(dict_glyphs_) scale .5 translate <-4.80,10,0>}

/* the "sampler" */
#declare dict_phrase_ = dictionary {
  .DataColumns: array [1] {"S"},
  .DataTable  : array [3]
          {"The quick brown fox","jumped","over the lazy dog."},
  .ColAlign   : array [1] {0},
  .Padding    : <.4, .3>,
  .Fonts      : fonts_,
  .Borders    : off
};
object {Tabulated(dict_phrase_) scale .75 translate <-1.5,-4.5,0>}

EOHERE

# feedback.
echo created file "'${fname}'."

exit 0

# --------------------------------------------------- #
# the content above is covered by the GPLv3+ license. #
# copyright (c) 2021 jr <creature.eternal@gmail.com>. #
# all rights reserved.                                #
# --------------------------------------------------- #
