/*
 ||
 |  TiledCeiling.inc  |
 | 2011  Samuel Benge |
 |____________________|
  
  Creates a tiled ceiling (or floor) using multiple copies of
  a height_field given by a user-supplied PNG image.
  
  Each tile is flipped, so for symmetrical tiles you can get
  away with using only 1/4 of the original image. You can turn
  off this behavior by typing the following before invoking
  the Tiled_Ceiling macro:

    #declare Tiled_Ceiling_FlipTiles = false;
    
  
  Tiled_Ceiling(ImgString, BBoxMin, BBoxMax, DivXZ, Height)
    *ImgString  = input PNG file name
    *BBoxMin    = one corner of the ceiling, a 2D vector
    *BBoxMax    = the other corner of the ceiling, a 2D vector
    *DivXZ      = tile divisions per direction, a 2D vector
    *Height     = the height, or depth of the tiles
  
  Example:
  
    Tiled_Ceiling(
     "MyHeightMap.png",
     <-100,-100>,
     <100,100>,
     <10,10>,
     1
    )
  ____________________________________________________________

*/

#declare Tiled_Ceiling_FlipTiles = true;

#macro Tiled_Ceiling(ImgString, BBoxMin, BBoxMax, DivXZ, Height)

  #local Tile =
    height_field{
      png ImgString
      smooth
      scale<1,Height,1>
      rotate x*180
      translate z
    }
  
  #local TileStrip =
    union{
      #declare X=0;
      #while(X<DivXZ.x)
        object{
         Tile
         #if(Tiled_Ceiling_FlipTiles)
           scale<1-2*Tiled_Ceiling_Odd(X),1,1>
           translate x*Tiled_Ceiling_Odd(X)
         #end
         translate x*X
        }
        #declare X=X+1;
      #end
    }
  
  #local XExtent = (BBoxMax.x-BBoxMin.x);
  #local ZExtent = (BBoxMax.y-BBoxMin.y);
    
  union{
    #local Z=0;
    #while(Z<DivXZ.y)
      object{
        TileStrip
        #if(Tiled_Ceiling_FlipTiles)
          scale<1,1,1-2*Tiled_Ceiling_Odd(Z)>
          translate z*Tiled_Ceiling_Odd(Z)
        #end
        translate z*Z
      }
      #declare Z=Z+1;
    #end
    scale<XExtent/DivXZ.x,Height,ZExtent/DivXZ.y>
    translate <BBoxMin.x,0,BBoxMin.y>
  }

#end

#declare Tiled_Ceiling_Odd  = function(x) {select(mod(x, 2), 1, 0, 1)}