
# POV‑Ray Parser/Scanner Walkthrough — v2 (adds `#declare` variant)

This version adds a `#declare` example and explains **how the parser builds the AST/scene**.

## New files

- `sphere_with_declare.pov` — uses `#declare R = 1;` then references `R` as the sphere radius.
- `outputs/*_declare.json` — the five pipeline stages for the declare variant:
  1. `1_scanner_lexemes_declare.json`
  2. `2_raw_tokens_declare.json`
  3. `3_language_tokens_declare.json`
  4. `4_ast_declare.json`
  5. `5_symbol_table_declare.json`

---

## How POV‑Ray’s parser builds the AST / scene

> Strictly speaking POV‑Ray does **not** keep a single monolithic compiler‑style AST; instead the **recursive‑descent parser** consumes the token stream and **constructs scene data structures directly** (objects, textures, lights, etc.) while evaluating expressions and maintaining a symbol table. You can think of it as building a *scene graph* incrementally rather than producing a separate AST to be lowered later. The implementation lives across `source/parser/*.cpp`.  
**Refs:** core parser and architecture pages. [parser.cpp](https://github.com/POV-Ray/povray/blob/master/source/parser/parser.cpp) · [Parser System overview](https://deepwiki.com/POV-Ray/povray/3-parser-system)

### 1) Token lookahead & dispatch
- The tokenizer has already mapped keywords to language tokens (e.g., `SPHERE_TOKEN`, `TEXTURE_TOKEN`). The parser peeks at the next token and **dispatches** to a handler: e.g., *Parse_Object()*, *Parse_Texture()*, *Parse_Camera()*.  
**Refs:** tokenization and parser entry points. [parser_tokenizer.cpp](https://github.com/POV-Ray/povray/blob/master/source/parser/parser_tokenizer.cpp) · [parser.cpp](https://github.com/POV-Ray/povray/blob/master/source/parser/parser.cpp)

### 2) Recursive‑descent productions
- Each construct has a function that mirrors the SDL grammar. For a `sphere { <x,y,z>, r [MODIFIERS] }` statement, the sphere parser:
  1. Expects `LBRACE`, then a vector (`<, FLOAT, ',', FLOAT, ',', FLOAT, '>'`),
  2. Expects a comma and a radius expression,
  3. Consumes an arbitrary sequence of **modifiers** (e.g., `texture { … }`, `translate`, `rotate`, `scale`, `interior { … }`),
  4. Expects the closing `RBRACE`.
  As it recognizes each part, it **instantiates engine objects** (sphere primitive, texture structures) rather than pushing generic AST nodes.  
**Refs:** object/material parsers. [parser.cpp](https://github.com/POV-Ray/povray/blob/master/source/parser/parser.cpp) · [parser_materials.cpp](https://github.com/POV-Ray/povray/blob/master/source/parser/parser_materials.cpp)

### 3) Expression evaluation & symbol resolution
- Numeric/vector expressions (e.g., the radius or color components) are parsed by the expression subsystem and **evaluated immediately**. Identifiers are resolved via the **symbol table stack** (global → local/macro). In the `#declare` variant, `R` is looked up, yielding `1.0`, which the parser stores in the constructed sphere.  
**Refs:** expressions & symbols. [parser_expressions.cpp](https://github.com/POV-Ray/povray/blob/master/source/parser/parser_expressions.cpp) · [symboltable.cpp](https://github.com/POV-Ray/povray/blob/master/source/parser/symboltable.cpp)

### 4) Scope management (directives)
- Directives beginning with `#` (e.g., `#declare`, `#local`, `#macro`, `#end`) drive the **symbol table** and scope stack. A `#declare` inserts or updates a name in the current table; `#local` creates names in a nested table; entering/leaving a macro or block pushes/pops scopes.  
**Refs:** directives and symbol handling. [parser.cpp](https://github.com/POV-Ray/povray/blob/master/source/parser/parser.cpp) · [symboltable.h/.cpp](https://github.com/POV-Ray/povray/blob/master/source/parser/symboltable.cpp)

### 5) Building textures and materials on the fly
- For `texture { pigment { rgb <…> } finish { specular … roughness … } }` the parser calls material helper routines to **assemble a `Texture` object** with nested `Pigment` and `Finish` components and attaches it to the sphere instance.  
**Refs:** materials parsing. [parser_materials.cpp](https://github.com/POV-Ray/povray/blob/master/source/parser/parser_materials.cpp)

### 6) Transformations & modifiers
- Modifiers in the object block (e.g., `translate`, `rotate`, `scale`, `matrix`) create transform nodes which the parser applies to the object instance before it’s committed to the scene. Bounding flags and clipping are also incorporated during this stage.  
**Refs:** object parsing and transform handling. [parser.cpp](https://github.com/POV-Ray/povray/blob/master/source/parser/parser.cpp)

### 7) Committing to scene data
- When a construct is complete, the parser **inserts the resulting object into the scene’s data structures** (camera list, lights, object collection, etc.). These are the structures consumed by the rendering engine (view/tile scheduler and `TracePixel`).  
**Refs:** tying parser to engine. [parser.cpp](https://github.com/POV-Ray/povray/blob/master/source/parser/parser.cpp) · [Rendering Engine overview](https://deepwiki.com/POV-Ray/povray/5-rendering-engine)

---

## What the five stages look like for the `#declare` variant

1. **Scanner**: produces lexemes including the `#` and `declare` word, then the `sphere` block (see `1_scanner_lexemes_declare.json`).  
2. **Raw tokenizer**: preliminary token kinds (`HASH`, `IDENT 'declare'`, `IDENT 'R'`, `EQUALS`, `NUMBER '1'`, …), then the `sphere` sequence (see `2_raw_tokens_declare.json`).  
3. **Tokenizer**: maps to language tokens (`DIRECTIVE_DECLARE`, `SPHERE_TOKEN`, `TEXTURE_TOKEN`, `RGB_TOKEN`, …). Identifiers not reserved remain `IDENTIFIER` (see `3_language_tokens_declare.json`).  
4. **Parser**: builds a `Sphere` instance. The `radius` is resolved from the symbol `R = 1.0`; the texture is attached (see `4_ast_declare.json`).  
5. **Symbol table**: contains `R : FLOAT = 1.0` in the global table (see `5_symbol_table_declare.json`).  
**Refs:** phases and files. [scanner.h](https://github.com/POV-Ray/povray/blob/master/source/parser/scanner.h) · [parser_tokenizer.cpp](https://github.com/POV-Ray/povray/blob/master/source/parser/parser_tokenizer.cpp) · [reservedwords.h](https://github.com/POV-Ray/povray/blob/master/source/parser/reservedwords.h) · [parser.cpp](https://github.com/POV-Ray/povray/blob/master/source/parser/parser.cpp) · [symboltable.cpp](https://github.com/POV-Ray/povray/blob/master/source/parser/symboltable.cpp)

---

## Scenes

### `sphere.pov`
```pov
#version 3.8;
camera { location <0,1.5,-5> look_at <0,1,0> }
light_source { <5,6,-5>, rgb 1 }
sphere { <0,1,0>, 1 texture { pigment { rgb <0.8,0.2,0.2> } finish { specular 0.6 roughness 0.02 } } }
// ***end of code***
```

### `sphere_with_declare.pov`
```pov
#version 3.8;
#declare R = 1;
camera { location <0,1.5,-5> look_at <0,1,0> }
light_source { <5,6,-5>, rgb 1 }
sphere { <0,1,0>, R texture { pigment { rgb <0.8,0.2,0.2> } finish { specular 0.6 roughness 0.02 } } }
// ***end of code***
```
