
# POV‑Ray Parser/Scanner Walkthrough (Sphere Example)

This package explains the **scanner → raw tokenizer → tokenizer → parser → symbol table** pipeline in POV‑Ray and shows **illustrative outputs** for a tiny scene with a single sphere. The outputs are **didactic examples** based on the official docs and source layout; they are *not* emitted by the POV‑Ray binary verbatim (token names and structures are simplified for clarity).

## Contents

- `sphere.pov` – a compact scene (camera, light, sphere)
- `outputs/1_scanner_lexemes.json` – lexemes with categories (scanner stage)
- `outputs/2_raw_tokens.json` – preliminary tokens (raw tokenizer)
- `outputs/3_language_tokens.json` – language tokens mapped to reserved words (tokenizer)
- `outputs/4_ast.json` – simplified AST / scene graph fragment
- `outputs/5_symbol_table.json` – user symbol table snapshot

---

## 1) Scanner (lexical analysis)

**What it does.** Reads characters, handles encoding and comments, and emits **lexemes** classified as `kWord`, `kFloatLiteral`, `kStringLiteral`, or `kOther` (punctuation/operators). See `source/parser/scanner.h/.cpp` for the `Lexeme` struct and the scanner class. [GitHub: scanner.h][s_scanner_h] · [Deep overview][dw_scanner]

**Output here.** See `outputs/1_scanner_lexemes.json`.

**References**: official headers/impl and architecture pages. [scanner.h][s_scanner_h], [parser_tokenizer.cpp][s_par_tok], [Parser System][dw_parser]

---

## 2) Raw tokenizer

**What it does.** Consumes lexemes and assigns **preliminary token kinds** (IDENT, NUMBER, LBRACE, …) without yet mapping words to language keywords. This logic resides in `parser/parser_tokenizer.cpp`. [Tokenization source][s_par_tok] · [Parser System][dw_parser]

**Output here.** See `outputs/2_raw_tokens.json`.

---

## 3) Tokenizer (language tokens)

**What it does.** Maps identifiers that match POV‑Ray keywords to **language tokens** defined via the reserved‑words tables (`reservedwords.h/.cpp`), e.g. `sphere → SPHERE_TOKEN`, `pigment → PIGMENT_TOKEN`, `rgb → RGB_TOKEN`. Literals remain numeric/string tokens. [reservedwords.h][s_reserved_h] · [reservedwords.cpp][s_reserved_cpp]

**Output here.** See `outputs/3_language_tokens.json`.

---

## 4) Parser (syntax + semantics)

**What it does.** The recursive‑descent **Parser** (`parser/parser.cpp`) recognizes statements/blocks and **builds the in‑memory scene**: objects, CSG, textures, camera, lights, media, and render settings. Expression evaluation, directives (`#declare`, `#macro`, `#if/#while`), and version behaviors live in the parser and helper files (`parser_expressions.cpp`, `parser_functions*.cpp`, `parser_materials.cpp`, `parser_strings.cpp`). [parser.cpp][s_parser_cpp] · [Parser System][dw_parser]

**Output here.** See `outputs/4_ast.json` for a simplified sphere node with center/radius and a texture (pigment/finish).

---

## 5) Symbol table (identifiers & scope)

**What it does.** Tracks **user‑defined identifiers** declared via `#declare`/`#local`/`#macro`, with proper scoping rules (global vs. macro/local). Implemented in `symboltable.h/.cpp`; the parser pushes/pops tables as scopes change. In this tiny scene we do not declare any identifiers, so the table is empty. For comparison, if you added `#declare R = 1; sphere { <0,1,0>, R }` you would see a global symbol `R : FLOAT`. [symboltable.cpp][s_symtab_cpp] · [Symbol Table overview][dw_sym]

**Output here.** See `outputs/5_symbol_table.json`.

---

## The scene (`sphere.pov`)

```pov
#version 3.8;
camera { location <0,1.5,-5> look_at <0,1,0> }
light_source { <5,6,-5>, rgb 1 }
sphere { <0,1,0>, 1 texture { pigment { rgb <0.8,0.2,0.2> } finish { specular 0.6 roughness 0.02 } } }
// ***end of code***
```

---

## Caveats

- The JSON files illustrate **shape and intent** of internal data, not exact structures/IDs used by a particular POV‑Ray build.
- Names like `SPHERE_TOKEN`, `RGB_TOKEN` mirror the reserved‑word mapping in source but may differ in exact spelling in future versions.

---

## Pointers for deeper reading

- **Scanner / Tokenizer / Parser**: `source/parser/scanner.h`, `parser/parser_tokenizer.cpp`, `parser/parser.cpp` ([links below]).
- **Reserved Words**: `source/parser/reservedwords.h/.cpp`.
- **Symbol Table**: `source/parser/symboltable.h/.cpp`.
- **Rendering Engine** (beyond parsing): `core/render/tracepixel.cpp`, `core/render/trace.cpp`, `backend/scene/view.cpp` (not used in this demo but helpful context).

---

## References (source & docs)

- POV‑Ray GitHub, `scanner.h` – Lexeme and Scanner definitions.  
  <https://github.com/POV-Ray/povray/blob/master/source/parser/scanner.h>
- POV‑Ray GitHub, `parser_tokenizer.cpp` – tokenization logic.  
  <https://github.com/POV-Ray/povray/blob/master/source/parser/parser_tokenizer.cpp>
- POV‑Ray GitHub, `parser.cpp` – core parser implementation.  
  <https://github.com/POV-Ray/povray/blob/master/source/parser/parser.cpp>
- POV‑Ray GitHub, `reservedwords.h/.cpp` – keyword mapping.  
  <https://github.com/POV-Ray/povray/blob/master/source/parser/reservedwords.h>  
  <https://github.com/POV-Ray/povray/blob/master/source/parser/reservedwords.cpp>
- POV‑Ray GitHub, `symboltable.h/.cpp` – symbol table implementation.  
  <https://github.com/POV-Ray/povray/blob/master/source/parser/symboltable.h>  
  <https://github.com/POV-Ray/povray/blob/master/source/parser/symboltable.cpp>
- Deep overviews: Parser System / Scanner and Tokenizer / Rendering Engine.  
  <https://deepwiki.com/POV-Ray/povray/3-parser-system>  
  <https://deepwiki.com/POV-Ray/povray/3.1-scanner-and-tokenizer>  
  <https://deepwiki.com/POV-Ray/povray/5-rendering-engine>

[s_scanner_h]: https://github.com/POV-Ray/povray/blob/master/source/parser/scanner.h
[s_par_tok]: https://github.com/POV-Ray/povray/blob/master/source/parser/parser_tokenizer.cpp
[s_parser_cpp]: https://github.com/POV-Ray/povray/blob/master/source/parser/parser.cpp
[s_reserved_h]: https://github.com/POV-Ray/povray/blob/master/source/parser/reservedwords.h
[s_reserved_cpp]: https://github.com/POV-Ray/povray/blob/master/source/parser/reservedwords.cpp
[s_symtab_cpp]: https://github.com/POV-Ray/povray/blob/master/source/parser/symboltable.cpp
[dw_parser]: https://deepwiki.com/POV-Ray/povray/3-parser-system
[dw_scanner]: https://deepwiki.com/POV-Ray/povray/3.1-scanner-and-tokenizer
[dw_sym]: https://deepwiki.com/POV-Ray/povray/3.2-symbol-table-and-expression-handling
