#!/usr/bin/env python3
import sys, re
if len(sys.argv) < 3:
    print("Usage: povsrc_to_inc.py <input.pov> <output.inc>")
    sys.exit(1)
src_path, out_path = sys.argv[1], sys.argv[2]
code = open(src_path, 'r', encoding='utf-8', errors='replace').read()
KW = r'\b(camera|light_source|sky_sphere|background|global_settings|#(declare|local|macro|include|if|else|end|while|read|write|fopen|fclose|warning|version|debug))\b'
RE_STRING   = re.compile(r'"([^"\\]|\\.)*"')
RE_LINECMT  = re.compile(r'//.*')
RE_NUMBER   = re.compile(r'(?<![A-Za-z_])[-+]?\d+(\.\d+)?([eE][-+]?\d+)?')
RE_DIRECT   = re.compile(r'#\w+')
RE_KEYWORD  = re.compile(KW)
COLORS = {
  'comment': (0.35,0.50,0.35),
  'string' : (0.80,0.20,0.30),
  'direct' : (0.55,0.30,0.85),
  'keyword': (0.15,0.35,0.85),
  'number' : (0.80,0.50,0.10),
  'plain'  : (0.85,0.85,0.85),
}

def esc(s):
    return s.replace('\\','\\\\').replace('"','\\"')

def colorize_line(line: str):
    spans = []
    L = len(line)
    mline = RE_LINECMT.search(line)
    tail_idx = mline.start() if mline else L
    work = line[:tail_idx]
    tokens = []
    for pat, kind in [(RE_STRING,'string'),(RE_DIRECT,'direct'),(RE_KEYWORD,'keyword'),(RE_NUMBER,'number')]:
        for m in pat.finditer(work):
            tokens.append((m.start(), m.end(), kind))
    tokens.sort(key=lambda t:(t[0], -(t[1]-t[0])))
    painted = [False]*L
    for s,e,k in tokens:
        for j in range(s,e):
            if j < tail_idx and not painted[j]:
                painted[j] = k
    j=0
    while j<tail_idx:
        k = painted[j] if painted[j] else 'plain'
        start=j
        while j<tail_idx and (painted[j] if painted[j] else 'plain')==k:
            j+=1
        spans.append((line[start:j], k))
    if mline:
        spans.append((line[tail_idx:], 'comment'))
    return spans

lines = code.splitlines(True)
all_lines = [ colorize_line(ln.rstrip('\n')) for ln in lines ]

with open(out_path,'w',encoding='utf-8') as w:
    w.write("// AUTO-GENERATED by povsrc_to_inc.py — do not edit.\n")
    w.write("#declare A1D_LINES = %d;\n" % len(all_lines))
    w.write("#declare A2D_CODE = array[A1D_LINES]{\n")
    for L in all_lines:
        w.write("  array[%d]{\n" % len(L))
        for (txt, kind) in L:
            r,g,b = COLORS.get(kind,COLORS['plain'])
            w.write('    "%s", <%0.3f,%0.3f,%0.3f>,\n' % (esc(txt), r,g,b))
        w.write("  },\n")
    w.write("}\n")
