#include <iostream>
#include <cmath>
#include "geometry.h"
using namespace geometry;

// default constructor.  Creates the zero-vector;
vector3d::vector3d() :
    point3d(0.0, 0.0, 0.0),
    mag(0.0),
    phi(0.0),
    theta(0.0)
{}

// constructs a vector through the origin from x, y and z coordinates.
vector3d::vector3d(double x, double y, double z) :
    point3d(x,y,z)
{}

vector3d::vector3d(const point3d &lhs, const point3d &rhs) :
    point3d(lhs[X] - rhs[X], lhs[Y] - rhs[Y], lhs[Z] - rhs[Z])
{}

bool vector3d::updated()
{
    return (*dynamic_cast<point3d *>(this) == this->reference);
}

vector3d vector3d::normalize() const
{
    if(mag != 0.0 && mag != 1.0)
        return vector3d((*this)[X]/mag, (*this)[Y]/mag, (*this)[Z]/mag);

    return *this;
}

double vector3d::dot(const vector3d &v) const
{
    return (*this)[X] * v[X] + (*this)[Y] * v[Y] + (*this)[Z] * v[Z];
}

// no need to compute spherical inverse, it will be handled on next request.
vector3d vector3d::cross(const vector3d &v) const
{
    vector3d cProd;

    cProd[X] = (*this)[Y]*v[Z] - (*this)[Z]*v[Y];   // u2*v3 - u3*v2
    cProd[Y] = (*this)[Z]*v[X] - (*this)[X]*v[Z];   // u3*v1 - u1*v3
    cProd[Z] = (*this)[X]*v[Y] - (*this)[Y]*v[X];   // u1*v2 - u2*v1

    return cProd;
}

double vector3d::getMagnitude()
{
    if(*dynamic_cast<point3d *>(this) != reference)
        this->recompute_spherical();

    return this->mag;
}

double vector3d::getTheta()
{
    // If the alert boolean is set in the base object, then recompute.
    if(*dynamic_cast<point3d *>(this) != reference)
        this->recompute_spherical();

    return this->theta;
}

double vector3d::getPhi()
{
    // If the alert boolean is set in the base object, then recompute.
    if(*dynamic_cast<point3d *>(this) != reference)
        this->recompute_spherical();

    return this->phi;
}

void vector3d::recompute_spherical()
{
    double x = (*this)[X];
    double y = (*this)[Y];
    double z = (*this)[Z];

    // if the current object is not equal to the Zero Vector, proceed.
    // otherwise, zero out the spherical components.
    if(*this != ZERO_VECTOR)
    {
        this->mag = sqrt(abs(x*x + y*y + z*z));
        this->theta = (x == 0.0 ? (z < 0 ? -M_PI/2 : M_PI/2) : atan( z / x ));
        this->phi = acos(y / this->mag);
    }
    else
    {
        this->mag = 0.0;
        this->theta = 0.0;
        this->phi = 0.0;
    }
    reference = *dynamic_cast<point3d *>(this);     // reset the reference copy.
}

// Should allow constructs like: P1.getIntercept() = P2
point3d &vector3d::getIntercept()
{
    return *dynamic_cast<point3d *>(this);
}

// returns a copy of the intercept if magnitude = 1
// else, calculates the normal vector and stores it to temp, then returns temp.
point3d vector3d::getUnitIntercept() const
{
    vector3d temp;

    if(mag == 1.0)
        return (*this);
    temp = this->normalize();

    return temp.getIntercept();
}

vector3d &vector3d::operator =(const vector3d &rhs)
{
    (*this)[X] = rhs[X];
    (*this)[Y] = rhs[Y];
    (*this)[Z] = rhs[Z];

    // if the referent copy of the base object is different, copy the spherical
    // components and reset the alert.  Otherwise, it will be recomputed next
    // time one of the components is requested.
    if(*dynamic_cast<point3d *>(this) == reference)
    {
        mag = rhs.mag;
        phi = rhs.phi;
        theta = rhs.theta;
    }

    return *this;
}

// check to make certain that intercept alert is not changed by this.
vector3d vector3d::operator +(const vector3d &rhs)
{
    double rX, rY, rZ;

    rX = (*this)[X] + rhs[X];
    rY = (*this)[Y] + rhs[Y];
    rZ = (*this)[Z] + rhs[Z];

    return vector3d(rX, rY, rZ);
}

vector3d vector3d::operator -(const vector3d &rhs)
{
    double rX, rY, rZ;

    rX = (*this)[X] - rhs[X];
    rY = (*this)[Y] - rhs[Y];
    rZ = (*this)[Z] - rhs[Z];

    return vector3d(rX, rY, rZ);
}

// No need to bother recomputing spherical components for the inverse
// it will be computed the next time it is requested because the assignment
// operator will have marked the intercept as altered.
vector3d vector3d::operator -()
{
    vector3d result;

    result[X] = -result[X];
    result[Y] = -result[Y];
    result[Z] = -result[Z];

    return result;
}

vector3d vector3d::operator *(const double &scalar)
{
    double rX, rY, rZ;

    rX = (*this)[X] * scalar;
    rY = (*this)[Y] * scalar;
    rZ = (*this)[Z] * scalar;

    return vector3d(rX, rY, rZ);
}


vector3d vector3d::operator +=(const vector3d &rhs)
{
    (*this) = (*this) + rhs;

    return (*this);
}

vector3d vector3d::operator -=(const vector3d &rhs)
{
    (*this) = (*this) - rhs;

    return (*this);
}

bool vector3d::operator ==(const vector3d &rhs)
{
    return (mag == rhs.mag && phi == rhs.phi && theta == rhs.theta);
}

bool vector3d::operator !=(const vector3d &rhs)
{
    return !(*this == rhs);
}

