// DO_Plants.inc
// Some tropical plants created with constructive solid geometry,
// and some pots to hold them
// These are complicated geometrical objects and may take a long time to render.
// Use anti aliasing to make the edges of the leaves look good.

// The following objects are defined in this file:
//    DO_Plant1  - green leafy plant with wide leaves
//    DO_Plant1c - green leafy plant with wide leaves with some colored stripes
//    DO_Plant2  - green leafy plant with narrow leaves
//    DO_Plant3  - palm plant with sparse foliage
//    DO_Plant4  - palm plant with more dense foliage
//    DO_Plant5  - large leafed green plant - looks like a peace lilly
//    DO_Pot1    - cylindrical pot, gray stone
//    DO_Pot2    - roughly spherical pot, gray stone

#ifndef (DO_PLANTS_INC)
#declare DO_PLANTS_INC = version
#version 3

#include "colors.inc"
#include "textures.inc"
#include "stones.inc"


// ============================= //
// ========== plant 1 ========== //
// ============================= //

// A green, leafy, tropical plant.
// The leaves are wide and have a "V" cross section.
// This plant is basically round.
// Diameter: 1 at widest point
// Height: approx. 0.75 at highest point
// bottom center of plant is at <0,0,0>

#declare R1 = seed(0)

#declare LeafTexture1 = texture {
   pigment { color rgb <0, 0.5, 0> }
   finish { specular .2 }
}

#declare Leaf1Part1 =
difference {
   cylinder { <-2,-1,0>, <2,1,0>, 1 }
   box { <-5,-5,-5> <0, 5, 5> }
   box { <0, -5, -5> <5, 5, 5> rotate -12*z }
}

#declare Leaf1Part2 =
difference {
   object { Leaf1Part1 }
   cylinder { <-2,-1,0>, <2,1,0>, 1 translate <0,-.001,0> }
   texture { LeafTexture1 }
}

// a wide tropical leaf with a shallow "v" cross section
#declare Leaf1 =
union {
   object { Leaf1Part2 }
   object { Leaf1Part2 rotate <0,180,0> }
   translate z*1.0
   rotate x*-rand(R1)*5
}

#declare RowOfLeaf1 =
union {
#declare Loop = 0
#while (Loop < 360)
   object { Leaf1
      scale .86+rand(R1)*.14
      rotate y*Loop + rand(R1)*10
      translate y*(rand(R1)*.2-.1)
   }
   #declare Loop = Loop + 30
#end
}

#declare DO_Plant1 =
union {
#declare Loop = 0
#while (Loop < 1.25)
   object { RowOfLeaf1
      translate y*Loop
      rotate y*360*rand(R1)
      }
   #declare Loop = Loop + .4
#end
// the trunk or stalk
cone { <0,0,0>, 0.02, <0,.5,0>, 0
   texture { pigment { color rgb <0,.5,0> } }
}
// make the whole plant 1m in diameter
scale 0.25
}



// ============================== //
// ========== plant 1c ========== //
// ============================== //

// A leafy, tropical plant with some coloration in the leaves.
// The leaves are wide and have a "V" cross section.
// This plant is basically round.
// Diameter: 1 at widest point
// Height: approx. 0.75 at highest point
// bottom center of plant is at <0,0,0>

#declare R1 = seed(0)

#declare LeafTexture2 = texture {
   pigment {
      onion
      color_map {
         [0.04  color rgb <0,.5,0> ]
         [0.08  color rgb <0,.4,0> ]
         [0.16  color rgb <0,.5,0> ]
         [0.24  color rgb <0,.3,0> ]
         [0.32  color rgb <0,.4,0> ]
         [0.40  color rgb <0,.5,0> ]
         [0.48  color rgb <0,.3,0> ]
         [0.56  color rgb <0,.4,0> ]
         [0.64  color rgb <.5,0,0> ]
         [0.72  color rgb <0,.5,0> ]
         [0.80  color rgb <.9,1,0> ]
         [0.88  color rgb <0,.5,0> ]
         [0.96  color rgb <0,.4,0> ]
      }
   }
   finish { specular .2 }
   normal { bumps .3 scale <.01,.2,.2> }
   scale .249
   translate <-15,3.5,0>
}

#declare Leaf1Part1 =
difference {
   cylinder { <-2,-1,0>, <2,1,0>, 1 }
   box { <-5,-5,-5> <0, 5, 5> }
   box { <0, -5, -5> <5, 5, 5> rotate -12*z }
}

#declare Leaf1Part2 =
difference {
   object { Leaf1Part1 }
   cylinder { <-2,-1,0>, <2,1,0>, 1 translate <0,-.001,0> }
   texture { LeafTexture2 }
}

// a wide tropical leaf with a shallow "v" cross section
#declare Leaf1 =
union {
   object { Leaf1Part2 }
   object { Leaf1Part2 rotate <0,180,0> }
   translate z*1.0
   rotate x*-rand(R1)*5
}

#declare RowOfLeaf1 =
union {
#declare Loop = 0
#while (Loop < 360)
   object { Leaf1
      scale .86+rand(R1)*.14
      rotate y*Loop + rand(R1)*10
      translate y*(rand(R1)*.2-.1)
   }
   #declare Loop = Loop + 30
#end
}

#declare DO_Plant1c =
union {
#declare Loop = 0
#while (Loop < 1.25)
   object { RowOfLeaf1
      translate y*Loop
      rotate y*360*rand(R1)
      }
   #declare Loop = Loop + .4
#end
// the trunk or stalk
cone { <0,0,0>, 0.02, <0,.5,0>, 0
   texture { pigment { color rgb <0,.5,0> } }
}
// make the whole plant 1m in diameter
scale 0.25
}



// ============================= //
// ========== plant 2 ========== //
// ============================= //

// A green, leafy, tropical plant.
// The leaves are narrow and have a "V" cross section.
// This plant is basically round.
// Diameter: 1 at widest point
// Height: approx. 0.55 at highest point
// bottom center of plant is at <0,0,0>

#declare R1 = seed(0)

#declare Leaf2Part1 =
difference {
  cylinder { <-2,-1,0>, <2,1,0>, 1 }
  box { <-5,-5,-5> <0, 5, 5> }
  box { <0, -5, -5> <5, 5, 5> rotate -2.5*z }
  }

#declare Leaf2Part2 =
difference {
  object { Leaf2Part1 }
  cylinder { <-2,-1,0>, <2,1,0>, 1 translate <0,-.0001,0> }
  }

// a narrow tropical leaf with a shallow "v" cross section
#declare Leaf2 =
union {
  object { Leaf2Part2 }
  object { Leaf2Part2 rotate <0,180,0> }
  texture {
     pigment { color rgb <.1,.4,.1> }
     finish { specular .1 }
     }
  translate z*1.0
  }

#declare RowOfLeaf2 =
union {
#declare Loop = 0
#while (Loop < 360)
  object { Leaf2
    scale .8+rand(R1)*.2
    rotate y*(Loop + rand(R1)*20)
    translate y*(rand(R1)*.5-.25)
    }
  #declare Loop = Loop + 15
#end
}

#declare DO_Plant2 =
union {
#declare Loop = 0
#while (Loop < 1.21)
  object { RowOfLeaf2
    translate y*Loop
    rotate y*360*rand(R1)
    }
  #declare Loop = Loop + .6
#end
// the trunk or stalk
cone { <0,0,0>, 0.03, <0,1,0>, 0.01
  texture { pigment { color rgb <0,.4,0> } }
  }
scale 0.25
}



// ============================= //
// ========== Plant 3 ========== //
// ============================= //

// A nice looking palm plant.
// The number of leaves and their spacing can easily be varied.
// height: approximately 1.5
// diameter: approx. 1
// bottom center of plant is at <0,0,0>

#declare R1 = seed(0)

#declare Leaf3Pt1 =
difference {
   cylinder { <-.3,0,0>, <.3,0,0>, 1 open translate <0,-.866,0> }
   box { <-2,-2,-2> <0,2,2> rotate z*20 }
   box { < 0,-2,-2> <2,2,2> rotate z*-20 }
   texture {
      pigment { color rgb <0,.5,0> }
      finish { specular .1 }
      }
   }

#declare Leaf3 =
difference {
   object { Leaf3Pt1 }
   object { Leaf3Pt1 scale <2,1,1> translate <0, -0.001, 0> }
   translate <0,0,.5>
   }

#declare Stem3 =
difference {
   cylinder { <0,0,-.01>, <0,0,.01>, 1 translate x }
   cylinder { <0,0,-.03>, <0,0,.03>, 0.99 translate x }
   box { <.5,-3,-3> <3,3,3> }
   box { <-3,-3,-3> <3,0,3> }
   box { <-3,-3,-3> <3,3,0> rotate x*.66  translate z*-.01 }
   box { <-3,-3, 0> <3,3,3> rotate x*-.66 translate z*.01  }
   texture { pigment { color rgb <0,.9,0> } }
   }

#declare Angle = 20
#declare Delta = 2
#declare NumLeaves = (60-Angle)/Delta
#declare LeafSize = .2  // largest leaf spans this distance

// Stalk3.  Looks like a palm frond.

#declare Stalk3 =
union {
object { Stem3 }
#while (Angle < 60)
   #declare Size = LeafSize*(sin( ((((Angle-20)/40))) *3.14159 ) )+(LeafSize*.1)
   object { Leaf3 scale Size
      rotate z*90  rotate x*(-30)
      translate <-1,0,0> rotate z*(-Angle) translate <1,0,0> }
   object { Leaf3 scale Size
      rotate y*180 rotate z*90 rotate x*30
      translate <-1,0,0> rotate z*(-Angle) translate <1,0,0> }
   #declare Angle = Angle + Delta
#end
rotate <0,0,-90>
}

#declare Rp = seed (10)

// this is the minimum height (approximately) of the smallest stalk
#declare P3minscale = .5
// this is the max height (approximately) of the largest stalk,
// and therefore of the plant
#declare P3maxscale = 1.5
#declare P3scale = P3minscale
// the approximate minimum angle (relative to horizon)
// that one of the stalks will lean
#declare P3minangle = 60
#declare P3maxangle = 95
#declare P3angle = P3minangle
#declare P3rot = 0
// the number of stalks in the whole plant;
// determines how full or sparse the plant is
#declare Numstalks = 8

#declare N = 0
#declare DO_Plant3 = union {
#while (N<Numstalks)
   object { Stalk3 scale P3scale
      rotate z*P3angle+rand(R1)*16-8  // make it lean a little at random
      rotate y*P3rot }
   #declare P3rot = P3rot + rand(R1)*30-15 + 105  // make the rotation a bit random
   #declare P3angle = P3angle + (P3maxangle-P3minangle)/(Numstalks-1)
   #declare P3scale = P3scale + (P3maxscale-P3minscale)/(Numstalks-1)
   #declare N = N + 1
#end
}



// ============================= //
// ========== Plant4 ========== //
// ============================= //

// A nice looking palm plant, 1.5m tall
// based on Plant3, but with denser foliage
// height: approximately 1.5
// diameter: approx. 1
// bottom center of plant is at <0,0,0>

// Note: This plant contains 1696 CSG objects.
// It requires approximately 14.5MB to parse and render.
// It may take a long time to render.  Use Plant3
// for a similar plant that renders more quickly.

#declare R1 = seed(0)

#declare Leaf3Pt1 =
difference {
   cylinder { <-.3,0,0>, <.3,0,0>, 1 open translate <0,-.866,0> }
   box { <-2,-2,-2> <0,2,2> rotate z*20 }
   box { < 0,-2,-2> <2,2,2> rotate z*-20 }
   texture {
      pigment { color rgb <0,.5,0> }
      finish { specular .1 }
      }
   }

#declare Leaf3 =
difference {
   object { Leaf3Pt1 }
   object { Leaf3Pt1 scale <2,1,1> translate <0, -0.001, 0> }
   translate <0,0,.5>
   }

#declare Stem3 =
difference {
   cylinder { <0,0,-.01>, <0,0,.01>, 1 translate x }
   cylinder { <0,0,-.03>, <0,0,.03>, 0.99 translate x }
   box { <.5,-3,-3> <3,3,3> }
   box { <-3,-3,-3> <3,0,3> }
   box { <-3,-3,-3> <3,3,0> rotate x*.66  translate z*-.01 }
   box { <-3,-3, 0> <3,3,3> rotate x*-.66 translate z*.01  }
   texture { pigment { color rgb <0,.9,0> } }
   }

#declare Angle = 20
#declare Delta = 1.5
#declare NumLeaves = (60-Angle)/Delta
#declare LeafSize = .2  // largest leaf spans this distance

// Stalk3.  Looks like a palm frond.

#declare Stalk3 =
union {
object { Stem3 }
#while (Angle < 60)
   #declare Size = LeafSize*(sin( ((((Angle-20)/40))) *3.14159 ) )+(LeafSize*.1)
   object { Leaf3 scale Size
      rotate z*90  rotate x*(-30)
      translate <-1,0,0> rotate z*(-Angle) translate <1,0,0> }
   object { Leaf3 scale Size
      rotate y*180 rotate z*90 rotate x*30
      translate <-1,0,0> rotate z*(-Angle) translate <1,0,0> }
   #declare Angle = Angle + Delta
#end
rotate <0,0,-90>
}

#declare Rp = seed (10)
// this is the minimum height (approximately) of the smallest stalk
#declare P3minscale = .5
// this is the max height (approximately) of the largest stalk,
// and therefore of the plant
#declare P3maxscale = 1.5
#declare P3scale = P3minscale
// the approximate minimum angle (relative to horizon)
// that one of the stalks will lean
#declare P3minangle = 45
#declare P3maxangle = 90
#declare P3angle = P3minangle
#declare P3rot = 0
// the number of stalks in the whole plant;
// determines how full or sparse the plant is
#declare Numstalks = 32

#declare N = 0
#declare DO_Plant4 = union {
#while (N<Numstalks)
   object { Stalk3 scale P3scale
      rotate z*P3angle+rand(R1)*16-8  // make it lean a little at random
      rotate y*P3rot }
   #declare P3rot = P3rot + rand(R1)*30-15 + 70  // make the rotation a bit random
   #declare P3angle = P3angle + (P3maxangle-P3minangle)/(Numstalks-1)
   #declare P3scale = P3scale + (P3maxscale-P3minscale)/(Numstalks-1)
   #declare N = N + 1
#end
}



// ============================= //
// ========== Plant 5 ========== //
// ============================= //

// A large-leafed green plant - look like a peace lilly.
// This is the fastest of the plants to render.
// Bottom center of plant is at <0,0,0>

#declare R1 = seed(0)

#declare LeafTexture = texture {
   pigment { color rgb <.3, .4, .1> }
   finish { specular .2 }
   }

#declare Radius = 0.02

#declare Leaf5Pt1 = difference {
   torus { .8,.2 rotate x*90 translate x }
   sphere { <0,0,0>, 1.2 translate x*1.2 rotate z*-10 }
   }

#declare Leaf5Pt2 = difference {
   object { Leaf5Pt1 }
   object { Leaf5Pt1 translate <.001,-.001,0> }
   }

#declare RibPt1 = difference {
   torus { 1, Radius rotate x*90 translate x }
   box { <-.1,-1.1,-.1>, <2.1,0,.1> }
   box { <.5,-1.1,-.1>, <2.1,1.1,.1> }
   }

#declare RibPt2 = intersection {
   object { RibPt1 }
   sphere { <0,0,0>, 1 scale <1+Radius,1,1> translate x }
   }

#declare RibPt3 = difference {
   object { RibPt2 }
   sphere { <0,0,0> 1 scale <1-Radius,1,1> translate x }
   }

#declare Leaf5 = union {
   object { Leaf5Pt2 }
   object { RibPt3 }
   texture { LeafTexture }
   }

#declare Stalk5 = union {
   cylinder { <0,0,0>, <0,1,0>, Radius }
   object { Leaf5 translate y }
   texture { LeafTexture }
   }

#declare Angle = 0
#declare Height = .3
#declare Num = 20
#declare R = .2
#declare RInc = R/Num
#declare YAngle = 20
#declare YAngleInc = .5*YAngle / Num
#declare HeightInc =  (1-Height)/Num
#declare DO_Plant5 = union {
   #while (Height < 1)
      object { Stalk5
         scale <Height*(1+rand(R1)*.2-.1),Height*(1+rand(R1)*.2-.1),Height*(1+rand(R1)*.2-.1)>
         rotate z*(-YAngle*(1+rand(R1)*.2-.1)) translate x*R rotate y*Angle
         }
      #declare Height = Height + HeightInc + rand(R1)*.08-.04
      #declare Angle = Angle + 90 + rand(R1)*30
      #declare R = R - RInc
      #declare YAngle = YAngle - YAngleInc
   #end
   scale .5
   }



// ============================== //
// ========== the pots ========== //
// ============================== //


// ========== Pot1 ========== //
// A cylindrical, shiny, gray stone pot with black dirt.
// height: .3,  diameter: .3
// bottom center is at <0,0,0>

#declare DO_Pot1 =
union {
   difference {
      cylinder { <0, 0.0, 0>, <0, .95, 0>, 0.5 }
      cylinder { <0, 0.9, 0>, <0, 1.1, 0>, 0.4 }
      }
   // the dirt
   cylinder {
      <0, 0.9, 0>, <0, .98, 0>, 0.4
      texture {
         pigment { color rgb <0.02,0.02,0.02> }
         finish { specular .2 }
         normal { bumps .3 scale .02 }
         }
      }
   torus { .45, .05 translate y*.95 }
   texture { T_Stone13 }
   scale .3
   }



// ========== Pot2 ========== //

// A spherical, shiny, gray stone pot with black dirt.
// height: .3
// diameter: approximately .42
// bottom center is at <0,0,0>

#declare DO_Pot2 =
union {
  difference {
    sphere { <0,0,0> 1 }
    box { <-1.1, .707, -1.1> <1.1,   1.1, 1.1> }
    box { <-1.1, -1.1, -1.1> <1.1, -.707, 1.1> }
    cylinder { <0, .6, 0>, <0, 1, 0>, 0.6 }
    }
  cylinder {
    <0, .6, 0>, <0, .7, 0>, 0.65
    texture {
       pigment { color rgb <0.02,0.02,0.02> }
       finish { specular .2 }
       normal { bumps .3 scale .02 }
       }
    }
  texture { T_Stone13 }
  translate y*0.707
  scale .21216
  }



// ======================================== //
// ========== The plants in pots ========== //
// ======================================== //

// Uncomment one of the sections below to render an image.
// Make sure the appropriate plant and pot is uncommented above.

/*
// Plant1 in Pot2
object { DO_Pot2 }
object { DO_Plant1 scale .7 translate y*.26 }
plane {  y,  0 texture { pigment { color <.7,.7,.7> } } }
plane {  x, -5 texture { pigment { color <.7,.7,.7> } } rotate y*35 }
plane { -z, -5 texture { pigment { color <.7,.7,.7> } } rotate y*35 }
light_source { <-20,50,-20> color rgb <1,1,1> }
light_source { <10,12,-20> color rgb <1,1,1> }
camera { location <0,.4,-1.1> look_at <0,.35,0> }
*/

/*
// Plant1c in Pot2
object { DO_Pot2 }
object { DO_Plant1c scale .7 translate y*.26 }
plane {  y,  0 texture { pigment { color <.7,.7,.7> } } }
plane {  x, -5 texture { pigment { color <.7,.7,.7> } } rotate y*35 }
plane { -z, -5 texture { pigment { color <.7,.7,.7> } } rotate y*35 }
light_source { <-20,50,-20> color rgb <1,1,1> }
light_source { <10,12,-20> color rgb <1,1,1> }
camera { location <0,.4,-1.1> look_at <0,.35,0> }
*/

/*
// Plant2 in Pot2
object { DO_Pot2 }
object { DO_Plant2 scale .8 translate y*.26 }
plane {  y,  0 texture { pigment { color <.7,.7,.7> } } }
plane {  x, -5 texture { pigment { color <.7,.7,.7> } } rotate y*35 }
plane { -z, -5 texture { pigment { color <.7,.7,.7> } } rotate y*35 }
light_source { <-20,50,-20> color rgb <1,1,1> }
light_source { <10,12,-20> color rgb <1,1,1> }
camera { location <0,.4,-1.1> look_at <0,.4,0> }
*/

/*
// Plant3 in Pot2
object { DO_Pot2 }
object { DO_Plant3 scale 1 translate y*.26 }
plane {  y,  0 texture { pigment { color <.7,.7,.7> } } }
plane {  x, -5 texture { pigment { color <.7,.7,.7> } } rotate y*35 }
plane { -z, -5 texture { pigment { color <.7,.7,.7> } } rotate y*35 }
light_source { <-20,50,-20> color rgb <1,1,1> }
light_source { <10,12,-20> color rgb <1,1,1> }
camera { location <0,.8,-2.2> look_at <0,.8,0> }
*/

/*
// Plant4 in Pot2
object { DO_Pot2 }
object { DO_Plant4 scale 1 translate y*.26 }
plane {  y,  0 texture { pigment { color <.7,.7,.7> } } }
plane {  x, -5 texture { pigment { color <.7,.7,.7> } } rotate y*35 }
plane { -z, -5 texture { pigment { color <.7,.7,.7> } } rotate y*35 }
light_source { <-20,50,-20> color rgb <1,1,1> }
light_source { <10,12,-20> color rgb <1,1,1> }
camera { location <0,.9,-2.2> look_at <0,.9,0> }
*/

/*
// Plant5 in Pot2
object { DO_Pot2 }
object { DO_Plant5 scale 1 translate y*.3 }
plane {  y,  0 texture { pigment { color <.7,.7,.7> } } }
plane {  x, -5 texture { pigment { color <.7,.7,.7> } } rotate y*35 }
plane { -z, -5 texture { pigment { color <.7,.7,.7> } } rotate y*35 }
light_source { <-20,50,-20> color rgb <1,1,1> }
light_source { <10,12,-20> color rgb <1,1,1> }
camera { location <0,.7,-2> look_at <0,.7,0> }
*/

/*
// Several of the plants
camera { location <0,1,-3.5> look_at <0,1,0> }
light_source { <-20,50,-20> color rgb <1,1,1> }
light_source { <10,12,-20> color rgb <1,1,1> }

plane {  y,  0 texture { pigment { color <.7,.7,.7> } } }
plane {  x, -5 texture { pigment { color <.7,.7,.7> } } rotate y*35 }
plane { -z, -5 texture { pigment { color <.7,.7,.7> } } rotate y*35 }

#declare X = -1.5 #declare Z = 0 #declare S = 1
object { DO_Pot2 scale S translate <X,0,Z> }
object { DO_Plant1 scale S translate <X,.3*S,Z> }

#declare X = -1.6 #declare Z = -.6 #declare S = .5
object { DO_Pot2 scale S translate <X,0,Z> }
object { DO_Plant1c scale S translate <X,.3*S,Z> }

#declare X = -1.0 #declare Z = -.6 #declare S = .5
object { DO_Pot2 scale S translate <X,0,Z> }
object { DO_Plant1c scale S translate <X,.3*S,Z> }

#declare X = -.8 #declare Z = .3 #declare S = 1
object { DO_Pot2 scale S translate <X,0,Z> }
object { DO_Plant3 scale S translate <X,.3*S,Z> }

#declare X = -.3 #declare Z = -.4 #declare S = 1
object { DO_Pot2 scale S translate <X,0,Z> }
object { DO_Plant5 scale S translate <X,.3*S,Z> }

#declare X = .5 #declare Z = -.5 #declare S = 0.6
object { DO_Pot2 scale S translate <X,0,Z> }
object { DO_Plant2 scale S*2.5 translate <X,.3*S,Z> }

#declare X = 1.5 #declare Z = .3 #declare S = 1
object { DO_Pot2 scale S translate <X,0,Z> }
object { DO_Plant4 scale S translate <X,.3*S,Z> }
*/

#version DO_PLANTS_INC
#end
