#local StrisciaTmpVersion = version;
#version 3.1;

/******************************************************************************
**                         Mostra i credits iniziali                         **
******************************************************************************/
#render "\n\"Striscia\" v. 1.01\n"
#render " a totally free utility for POV-Ray v. 3.1\n"
#render " written by Daniele Varrazzo\n"
#render " (w) 1997-1999 by PiroSoftware c.n.f.\n"
#render " please report suggests & bug reports to piro@officine.it\n"

#include "Spline.inc"
#include "SplLib.inc"

// Costanti
// Tipi di trasformazione
#declare HORZ_SCALE = 0;
#declare VERT_SCALE = 1;
#declare PATH_SCALE = 2;
#declare UNIF_SCALE = 3;
#declare HORZ_TRANSLATE = 4;
#declare VERT_TRANSLATE = 5;
#declare PATH_TRANSLATE = 6;
#declare PATH_ROTATE = 7;

// Oggetto creato
#declare TRIANGLES = 0;
#declare UNION = 1;
#declare MESH = 2;
#declare POLYGON = 3;

// Quando disegno un segmento di striscia, devo usare i calcoli fatti
// Precedentemente?
#declare USE_BUFFER = 1;
#declare DONT_USE_BUFFER = 0;

// Parametri dell'utente
#ifndef (gPathDiv)        #declare gPathDiv        = 4;    #end
#ifndef (gPathDepth)      #declare gPathDepth      = 0;    #end
#ifndef (gPathErr)        #declare gPathErr        = 1E-2; #end
#ifndef (gShapeDiv)       #declare gShapeDiv       = 4;    #end
#ifndef (gShapeDepth)     #declare gShapeDepth     = 0;    #end
#ifndef (gShapeErr)       #declare gShapeErr       = 1E-2; #end
#ifndef (gCorrArea)       #declare gCorrArea       = 2;    #end
#ifndef (gvSky)           #declare gvSky           = y;    #end
#ifndef (gEpsilon)        #declare gEpsilon        = 1E-3; #end
#ifndef (msMarker)        #declare msMarker        = ".\n" #end

// Variabili locali
#declare mNumStrisce = 0;
#declare mNumDiaframmi = 0;

#macro ResetParameters()	
	#ifdef(miUserTransformTypes) #undef miUserTransformTypes #end
	#ifdef(mslpUserTransform)    #undef mslpUserTransform    #end
	#ifdef(giTransformTypes)     #undef giTransformTypes     #end
	#ifdef(gslpTransform)        #undef gslpTransform        #end
	#ifdef(msplUserPath)         #undef msplUserPath         #end
	#ifdef(mslpUserMorph)        #undef mslpUserMorph        #end
	#ifdef(gsplShape)            #undef gsplShape            #end
	#ifdef(gsllShapes)           #undef gsllShapes           #end
	#ifdef(gsBufferName)         #undef gsBufferName         #end
	#ifdef(gtxtWrapped)          #undef gtxtWrapped          #end
	#ifdef(fInternalVars)      #undef fInternalVars      #end
#end

// aggiunge una trasformazione all'elenco delle trasformazioni 
// sulla sezione dipendenti dal tempo.
#macro AddShapeTransform (Type, Slope)
	#ifndef (miUserTransformTypes)
		#declare miUserTransformTypes = array[1] {Type}
		#declare mslpUserTransform = array[1] {Slope}
	#else
		#local Position = dimension_size(miUserTransformTypes, 1);
		#declare miUserTransformTypes = aInsertElement(miUserTransformTypes, Type, Position)
		#declare mslpUserTransform = InsertArrayElement(mslpUserTransform, Slope, Position)
	#end
#end

// Imposta la slope per il morph
#macro SetMorphSlope(Slope)
	#declare mslpUserMorph = Slope
#end

// imposta la spline del percorso
#macro SetPathSpline(Spline)
	#declare msplUserPath = Spline
	#declare gsltPath = sltOptimizeSpline(Spline)
	#declare gPathLength = CalcSplineLength(gsltPath);
#end

// imposta la sezione della superficie. La sezione sara' costante.
#macro SetShapeSpline(Spline)
	#declare gsplShape = Spline
	#declare gNumShapeSegments = GetSegmentsNumber(Spline);
	#ifdef(gsllShapes) #undef gsllShapes #end
#end

// imposta una delle sezioni della superficie.
// si considera che la superficie avra' una sezione variabile, e devono essere
// specificate almeno due sezioni.
#macro AddShapeSpline(Spl, Pos)
	#ifndef(gsllShapes)
		#declare gsllShapes = sllCreateSplineList(Spl, Pos)
		#declare gNumShapeSegments = GetSegmentsNumber(Spl);
	#else
		#if (GetSegmentsNumber(Spl) != gNumShapeSegments)
			#error "Inconsistence! All the shapes must have the same number of segments!\n"
		#end
		#declare gsllShapes = sllAddSpline(gsllShapes, Spl, Pos)
	#end
	#ifdef(gsplShape) #undef gsplShape #end
#end

// imposta la texture da mappare sulla superficie
#macro SetWrappedTexture(Texture)
	#declare gtxtWrapped = Texture
#end

// Costruisce una slope per il morph se l'utente non ne ha impostata una.
#macro BuildDefaultMorphSlope()
	#local A = array[GetSplinesNumber(gsllShapes)*2 - 2]
	#local I=0;
	#while (I < GetSplinesNumber(gsllShapes)-1)
		#local P1 = GetSplinePosition(gsllShapes, I);
		#local P2 = GetSplinePosition(gsllShapes, I+1);
		#local A[I*2] = <P1, 0, 1/(P2-P1)>;
		#local A[I*2+1] = <P2, 1, 1/(P2-P1)>;
		#local I=I+1;
	#end
	#declare gslpMorph = slpSlope(A)
#end

// Legge il valore del morph in un punto
#macro GetMorph(X)
	EvalSlope(gslpMorph, X)
#end

// costruisce una matrice di trasformazione lineare dipendente dalle slope.
#macro bGetTransformMatrix(Type, Slope, X)
	#local Val = EvalSlope(Slope, X);
	#switch(Type)
		#case (HORZ_SCALE)     array[4] {<Val, 0, 0>, <0, 1, 0>, <0, 0, 1>, <0, 0, 0>}     #break
		#case (VERT_SCALE)     array[4] {<1, 0, 0>, <0, Val, 0>, <0, 0, 1>, <0, 0, 0>}     #break
		#case (PATH_SCALE)     array[4] {<1, 0, 0>, <0, 1, 0>, <0, 0, Val>, <0, 0, 0>}     #break
		#case (UNIF_SCALE)     array[4] {<Val, 0, 0>, <0, Val, 0>, <0, 0, Val>, <0, 0, 0>} #break
		#case (HORZ_TRANSLATE) array[4] {<1, 0, 0>, <0, 1, 0>, <0, 0, 1>, <Val, 0, 0>}     #break
		#case (VERT_TRANSLATE) array[4] {<1, 0, 0>, <0, 1, 0>, <0, 0, 1>, <0, Val, 0>}     #break
		#case (PATH_TRANSLATE) array[4] {<1, 0, 0>, <0, 1, 0>, <0, 0, 1>, <0, 0, Val>}     #break
		#case (PATH_ROTATE)
			#local Val1 = radians(Val);
			array[4] {<cos(Val1), sin(Val1), 0>, <-sin(Val1), cos(Val1), 0>, 
			          <0, 0, 1>, <0, 0, 0>}
		#break
	#end
#end

// Applica una trasformazione ad una base (moltiplica a destra la base)
#macro ApplyTransform(Type, Val, B)
	#switch(Type)
		#case (HORZ_TRANSLATE) #local B[3] = B[3] + Val * B[0]; #break
		#case (VERT_TRANSLATE) #local B[3] = B[3] + Val * B[1]; #break
		#case (PATH_TRANSLATE) #local B[3] = B[3] + Val * B[2]; #break
		#case (HORZ_SCALE) #local B[0] = Val * B[0]; #break
		#case (VERT_SCALE) #local B[1] = Val * B[1]; #break
		#case (PATH_SCALE) #local B[2] = Val * B[2]; #break
		#case (UNIF_SCALE) 
			#local B[0] = Val * B[0];
			#local B[1] = Val * B[1];
			#local B[2] = Val * B[2];
		#break
		#case (PATH_ROTATE)
			#local CosVal = cos(radians(Val));
			#local SinVal = sin(radians(Val));
			#local vTmpB0 = CosVal*B[0] + SinVal*B[1];
			#local B[1] = CosVal*B[1] - SinVal*B[0];
			#local B[0] = vTmpB0;
		#break
	#end
#end

// Costruisce una base ortonormale partendo dal versore che sara' il terzo asse.
#macro bBuildUnitBase(A3)
	#local B = array[4]
	#local B[2] = A3;
	#local B[1] = vnormalize(gvSky - A3 * vdot(A3, gvSky));
	#local B[0] = vcross(B[1], A3);
	B
#end

// Applica ad una base tutte le trasformazioni necessarie
#macro ApplyAllTransforms(B, T, X)
	#ifdef (giTransformTypes)
		#local I=dimension_size(giTransformTypes,1)-1;
		#while (I>=0)
			ApplyTransform(giTransformTypes[I], EvalSlope(gslpTransform[I], X), B)
			//#local BT = bGetTransformMatrix(giTransformTypes[I], gslpTransform[I], X)
			//#local B = bTransformMatrix(B, BT)
			#local I=I-1;
		#end
	#end
#end

// Costruisce la base non ortogonale all'inizio di un tratto
#macro BuildStartBase(T)
	#declare gCornerAtStart = IsCuspSpline(msplUserPath ,T);
	#if(gCornerAtStart)
		#local bL = bBuildUnitBase(vnormalize(EvalLeftDerSpline(msplUserPath ,T)))
		#local bR = bBuildUnitBase(vnormalize(EvalRightDerSpline(msplUserPath ,T)))
		#local vAvg0 = vnormalize(bR[0] + bL[0]);
		#local vAvg1 = vnormalize(bR[1] + bL[1]);
		#declare bStart = array[4] {
			vAvg0 / vdot(vAvg0, bR[0]),
			vAvg1 / vdot(vAvg1, bR[1]),
			bR[2], <0, 0, 0> }
	#end
#end

// Costruisce la base non ortogonale alla fine di un tratto
#macro BuildEndBase(T)
	#declare gCornerAtEnd = IsCuspSpline(msplUserPath ,T);
	#if(gCornerAtEnd)
		#local bL = bBuildUnitBase(vnormalize(EvalLeftDerSpline(msplUserPath ,T)))
		#local bR = bBuildUnitBase(vnormalize(EvalRightDerSpline(msplUserPath ,T)))
		#local vAvg0 = vnormalize(bR[0] + bL[0]);
		#local vAvg1 = vnormalize(bR[1] + bL[1]);
		#declare bEnd = array[4] {
			vAvg0 / vdot(vAvg0, bR[0]),
			vAvg1 / vdot(vAvg1, bR[1]),
			bL[2], <0, 0, 0> }
	#end
#end

// Costruisce una base di trasformazione per una sezione.
#macro bBuildBase(T, X)
	#local vDer = vnormalize(EvalDerSpline(gsplPath, T));
	#if (gCornerAtStart & gCornerAtEnd & (gCropEndPos-gCropStartPos < 2*gCorrArea))
		#local Alfa = (X-gCropStartPos)/(gCropEndPos-gCropStartPos);
		#local B = array[4] { Alfa * (bEnd[0]-bStart[0]) + bStart[0],
		                      Alfa * (bEnd[1]-bStart[1]) + bStart[1],
		                      vDer, (EvalSpline(gsplPath, T)) }
	#else
		#local B = bBuildUnitBase(vDer)
		#local B[3] = EvalSpline(gsplPath, T);
		#if(gCornerAtStart)
			#if (X - gCropStartPos <= gCorrArea)
				#local Alfa = (X - gCropStartPos) / gCorrArea;
				#local B[0] = Alfa*(B[0]-bStart[0]) + bStart[0];
				#local B[1] = Alfa*(B[1]-bStart[1]) + bStart[1];
			#end
		#end
		#if(gCornerAtEnd)
			#if (gCropEndPos - X <= gCorrArea)
				#local Alfa = (gCropEndPos - X) / gCorrArea;
				#local B[0] = Alfa*(B[0]-bEnd[0]) + bEnd[0];
				#local B[1] = Alfa*(B[1]-bEnd[1]) + bEnd[1];
			#end
		#end
	#end
	ApplyAllTransforms(B, T, X)
	B
#end

// Costruisce una base di trasformazione per una sezione.
// versione con il controllo del parallelismo col cielo e velocita' 0
#macro bBuildSafeBase(T, X)
	#local vDer = EvalDerSpline(gsplPath, T);
	#if (vlength(vDer) = 0)
		#error concat("The path has 0 speed (at ",
		              str(X, 0, 3),
		              " from the start).\n")
	#else
		#local vDer = vnormalize(vDer);
	#end
	#if (vlength(vcross(vDer,gvSky)) = 0)
		#error concat("The path moves in the sky direction! (at ",
		              str(X, 0, 3),
		              " from the start.\nPlease change the gvSky value!\n")
	#end
	bBuildBase(T, X)
#end

// Funzione di valutazione dell'andamento della superficie
// lungo il percorso.
// Tiene in considerazione l'andamento del percorso e le slope di
// modifica ma non della forma o della dimensione della sezione.
// E' ottimizzata per sezioni dell'ordine dell'unita'.
#macro FEval(T)
	#local X = CalcChordAbscissa(gsltPath,T);
	#local Base = bBuildSafeBase(T,X)
	(Base[0]+Base[1]+Base[3])
#end

// Costruisce un vettore in cui sono indicati tutti i punti in cui
// va forzato un campionamento della striscia e vanno resettati i 
// buffer. Sono tutti i punti in cui c'e' una discontinuita' di una 
// slope o del percorso o delle loro derivate prime.
#macro aGetBreaks()
	#ifdef(mslpUserTransform)
		#local nSlopes = dimension_size(mslpUserTransform, 1);
		#local I = 0;
		#local Breaks = array[1] {0}
		#while (I < nSlopes)
			#local SlopeBreaks = array[1] {0}
			#local nSlopeSgm = GetSlopeSegmentsNumber(mslpUserTransform[I]);
			#local J = 1;
			#while (J < nSlopeSgm)
				#if (IsDiscontinualSlope(mslpUserTransform[I], GetSlopeSegmentStart(mslpUserTransform[I][J])))
					#local SlopeBreaks = aInsertElement(SlopeBreaks, GetSlopeSegmentStart(mslpUserTransform[I][J]), 
					                     dimension_size(SlopeBreaks,1))
				#end
				#local J = J + 1;
			#end
			#local Breaks = aMergeSorted(Breaks, SlopeBreaks)
			#local I = I + 1;
		#end
		#local Breaks = aMergeSorted(Breaks, array[2] {gStartPos, gEndPos})
	#else
		#local Breaks = array[2] {gStartPos, gEndPos}
	#end
	
	// Converte le ascisse curvilinee in ascisse temporali
	#local Dim = dimension_size(Breaks,1);
	#local tBreaks = array[Dim]
	#local I = 0;
	#while (I < Dim)
		#local tBreaks[I] = CalcTimeAbscissa(gsltPath, Breaks[I]);
		#local I = I + 1;
	#end
	
	// aggiunge le interruzioni del percorso
	#local PathBreaks = array[1] {0}
	#local nPathSgm = GetSegmentsNumber(msplUserPath);
	#local J=1;
	#while (J < nPathSgm)
		#if (IsCuspSpline(msplUserPath,J) | IsBrokenSpline(msplUserPath, J))
			#local PathBreaks = aInsertElement(PathBreaks, J, dimension_size(PathBreaks,1))
		#end
		#local J=J+1;
	#end
	#local tBreaks = aMergeSorted(tBreaks, PathBreaks)
	tBreaks
#end

// passo adattivo: divide l'intervallo [T0, T1] in due
// finche' consente la profondita' e finche' l'errore e'
// troppo alto.
#macro SplitPath(T0, F0, T1, F1, Depth)
	#local Tmid = (T0+T1)/2;
	#local Fmid = FEval(Tmid);
	#local Fapp = (F0+F1)/2;
	#local Err = vlength(Fmid - Fapp);
	#if(Depth)
		#if (Err < gPathErr)
			#declare Vect1[Pos] = T1;
			#declare Pos = Pos + 1;
		#else
			SplitPath(T0, F0, Tmid, Fmid, Depth-1)
			SplitPath(Tmid, Fmid, T1, F1, Depth-1)
		#end
	#else
		#if (Err > gMaxPathErr) #declare gMaxPathErr = Err; #end
		#declare Vect1[Pos] = T1;
		#declare Pos = Pos + 1;
	#end
#end

// Genera il vettore dei tempi di campionamento
// tra T0 e T1 non c'e' alcuna discontinuita' nelle T-derivate.
// La prima parte evita di creare passi piu' corti del minimo 
// raggiungibile dalla routine di adattamento.
#macro aBuildSamplingTimes(T0, T1)
	// Strato non adattivo.
	// Prepara i passi minimi da rispettare
	#local Vect = array[(int(T1) - int(T0) + 2) * gPathDiv]
	#local Vect[0] = T0;
	#local J = 1;
	#local MinStep = 1 / (gPathDiv * pow(2, gPathDepth));
	#if (int(T0) != int(T1))
		#if(Float(T0))
			#local Steps = int((1-Float(T0))*gPathDiv) + 1;
			#local I = 1;
			#while (I <= Steps)
				#local Vect[J] = T0 + (I/Steps) * (1-Float(T0));
				#local J = J + 1;
				#local I = I + 1;
			#end
			#local CT0 = int(T0) + 1;
		#else
			#local CT0 = T0;
		#end
		#local Steps = (int(T1)-CT0) * gPathDiv;
		#if ((CT0 > T0) & (CT0-T0 < MinStep)) #local J = J - 1; #end
		#local I = 1;
		#while (I <= Steps)
			#local Vect[J] = CT0 + (I/Steps) * (int(T1)-CT0);
			#local J = J + 1;
			#local I = I + 1;
		#end
		#if(Float(T1))
			#if(Float(T1) < MinStep & J > 1) #local J = J - 1; #end
			#local Steps = int(Float(T1)*gPathDiv) + 1;
			#local I = 1;
			#while (I <= Steps)
				#local Vect[J] = int(T1) + (I/Steps) * Float(T1);
				#local J = J + 1;
				#local I = I + 1;
			#end
		#end
	#else
		#local Steps = int((T1-T0) * gPathDiv) + 1;
		#local I = 1;
		#while (I <= Steps)
			#local Vect[J] = T0 + (I/Steps) * (T1-T0);
			#local J = J + 1;
			#local I = I + 1;
		#end
	#end

	// Strato adattivo
	#local Vect1 = array[J * pow(2, gPathDepth)+1]
	#local Vect1[0] = Vect[0];
	#local I = 0;
	#local Pos = 1;
	#local OldVal = FEval(T0);
	#while(Vect[I+1] < T1)
		#local NewVal = FEval(Vect[I+1]);
		SplitPath(Vect[I], OldVal, Vect[I+1], NewVal, gPathDepth)
		#local I = I + 1;
		#local OldVal = NewVal;
	#end
	#local NewVal = FEval(Vect[I+1]);
	SplitPath(Vect[I], OldVal, Vect[I+1], NewVal, gPathDepth)
	#local Vect1[Pos] = T1;
	aCrop(Vect1,Pos)
	
#end

// Genera i parametri globali tagliando il percorso e le slopes
// in modo che siano definite le derivate negli estremi.
#macro CropUserParams(T0, T1)
	#declare gsplPath = splCropSpline(msplUserPath, T0, T1)
	#declare gCropStartPos = CalcChordAbscissa(gsltPath,T0);
	#declare gCropEndPos   = CalcChordAbscissa(gsltPath,T1);
   
   #ifdef(miUserTransformTypes)
	   #local Dim = dimension_size(miUserTransformTypes, 1);
	   #declare gslpTransform = array[Dim]
	   #local I = 0;
	   #while (I < Dim)
	   	#declare gslpTransform[I] = slpCropSlope(mslpUserTransform[I],
	   	                                         gCropStartPos,gCropEndPos)
	   	#local I = I + 1;
	   #end
	#end
	BuildStartBase(T0)
	BuildEndBase(T1)
#end

// Aggiunge la funzione di inversione della superficie in caso di spigoli di 180 gradi
// alle trasformazioni immesse dall'utente.
#macro AddCornerTransform()
	#local aScale = array[1] {<0, 1, 0>}
	#local Sign = 1;
	#local nSeg = GetSegmentsNumber(msplUserPath);
	#local I = 1;
	#while (I < nSeg)
		#if (Is180CuspSpline(msplUserPath,I))
			#local X = CalcChordAbscissa(gsltPath, I);
			#local aScale = aInsertElement(aScale, <X,Sign,0>, dimension_size(aScale,1))
			#local Sign = -Sign;
			#local aScale = aInsertElement(aScale, <X,Sign,0>, dimension_size(aScale,1))
		#end
		#local I=I+1;
	#end
	#local aScale = aInsertElement(aScale, <gPathLength,Sign,0>, dimension_size(aScale,1))
	#if (dimension_size(aScale,1) > 2)
		#local slpScale = slpSlope(aScale)
		AddShapeTransform(HORZ_SCALE, slpScale)
	#end
#end

// Apre il buffer di salvataggio, se l'utente ne ha definito uno.
#macro OpenBuffer()
	#ifdef(gsBufferName) #fopen fbfBufferStriscia concat(gsBufferName, ".inc") write #end
#end

// Chiude il buffer di salvataggio, se l'utente ne ha definito uno.
#macro CloseBuffer()
	#ifdef(gsBufferName) #fclose fbfBufferStriscia #end
#end

// Controlla se esiste il buffer richiesto
#macro BufferExists()
	(file_exists(concat(gsBufferName, ".inc")) | file_exists(concat(gsBufferName, ".pcm")))
#end

// Include il buffer nella scena
#macro ReadBuffer()
	#if(file_exists(concat(gsBufferName, ".pcm")))
		#ifndef (fPcmIncluded)
			#declare fPcmIncluded = 1;
			#include "pcm.mcr"
		#end
		read_mesh(concat(gsBufferName, ".pcm"))
	#else
		#include concat(gsBufferName, ".inc")
	#end
#end

// Inizializza le variabili interne. Va fatto una volta sola, prima di
// utilizzare le funzioni per accedere alla superficie
#macro InitInternalVars()

	#ifndef(fInternalVars)

		// Controlla che tutti i parametri necessari siano stati impostati.
		#ifndef (msplUserPath)
			#error "You have to specify a path (use the \"SetPathSpline\" macro)\n"
		#end
		#ifndef (gsplShape) #ifndef (gsllShapes)
			#error "You have to specify a section (use the \"SetShapeSpline\" or \"AddShapeSpline\" macros)\n"
		#end #end
	   #ifdef(gtxtWrapped) #if (gObjectType = MESH)
	   	#error "You can't create a mesh with a wrapped texture. Use an union instead."
	   #end #end
		// copia delle variabili utente nelle variabili di lavoro
		#declare fInternalVars = 1;
		#ifdef(gsllShapes)
		   #declare gNumShapeSegments = GetSegmentsNumber(splGetSpline(gsllShapes, 0));
			#declare gStartPos = GetSplinePosition(gsllShapes, 0);
			#declare gStartTime = CalcTimeAbscissa(gsltPath, gStartPos);
			#declare gEndPos = GetSplinePosition(gsllShapes, GetSplinesNumber(gsllShapes)-1);
			#declare gEndTime = CalcTimeAbscissa(gsltPath, gEndPos);
			#ifdef (mslpUserMorph)
				#declare gslpMorph = mslpUserMorph
			#else
				BuildDefaultMorphSlope()
			#end
		#else
			#declare gStartPos = 0;
			#declare gStartTime = 0;
			#declare gEndPos = gPathLength;
			#declare gEndTime = GetSegmentsNumber(msplUserPath);
		#end
		AddCornerTransform()
		#ifdef (miUserTransformTypes)
			#declare giTransformTypes = miUserTransformTypes
			#declare gslpTransform = mslpUserTransform
		#end
	   
	   // calcolo delle variabili supplementari
		#declare gMaxPathErr = 0;
		#declare gMaxShapeErr = 0;
		#declare gaBreaks = aGetBreaks()
		#declare gvSky = vnormalize(gvSky);
      
	   // generazione delle funzioni variabili
	   #include "StrVarFs.inc"
	#end	   
#end

// Disegna tutta la superficie
#macro DrawSurface()

   // Inizio dell'oggetto
   BeginSurface()
   
   // Ciclo delle sezioni a t-derivate costanti.
   #local TCI = 0;
   #while (gaBreaks[TCI] < gStartTime) #local TCI = TCI + 1; #end
   #while (TCI < dimension_size(gaBreaks,1)-1 | gaBreaks[TCI] > gEndTime)
	   // Prepara come variabili di lavoro le varianti croppate 
	   // delle scelte dell'utente
	   #debug "Crop\n"
	   CropUserParams(gaBreaks[TCI], gaBreaks[TCI+1])

		// Costruisce un vettore di campionamento in una sezione a 
		// T-Der. costante.
	   #debug "Sampling\n"
		#local aSamplingTimes = aBuildSamplingTimes(gaBreaks[TCI], gaBreaks[TCI+1])
	   
		// Disegno delle sezioni
	   #debug "Draw\n"
		DrawSection(aSamplingTimes[0], aSamplingTimes[1], DONT_USE_BUFFER)
		#debug msMarker

		#local I=2;
		#while (I<dimension_size(aSamplingTimes, 1))
			DrawSection(aSamplingTimes[I-1], aSamplingTimes[I], USE_BUFFER)
			#debug msMarker
			#local I=I+1;
		#end

   	#local TCI = TCI + 1;
   #end

   // Fine dell'oggetto
   EndSurface()
   
	#if (gMaxShapeErr > gShapeErr)
		#statistics concat("Maximum error along the sections: ", 
		                str(gMaxShapeErr ,0,-1), "\n")
	#end
	#if (gMaxPathErr > gPathErr)
		#statistics concat("Maximum error along the path: ", 
		                str(gMaxPathErr ,0,-1), "\n")
	#end
	#debug "\n"
#end

#macro Striscia(ObjectType)
   #declare gObjectType = ObjectType;
	#declare mNumStrisce = mNumStrisce + 1;
	#debug concat("Striscia #", str(mNumStrisce,1,0), "\n")
   
   // Tenta di usare il buffer.
   #ifdef(gsBufferName)
   	// Tenta di aprire il buffer salvato
      #if (BufferExists())
      	ReadBuffer()
      #else
		   // Genera la scena e il buffer di salvataggio
		   InitInternalVars()
		   OpenBuffer()
   		DrawSurface()
   		CloseBuffer()
   		ReadBuffer()
      #end
	#else
		InitInternalVars()
		DrawSurface()	
	#end
#end

// Disegna un diaframma
#macro Diaframma(X, Type)

	#declare mNumDiaframmi = mNumDiaframmi + 1;
	#debug concat("Diaframma #", str(mNumDiaframmi,1,0), "\n")

	// Sistemazione delle variabili interessanti
   InitInternalVars()
	#local T = CalcTimeAbscissa(gsltPath, X);
	#local I=0;
	#while ((gaBreaks[I+1] < T) & (I+1 < dimension_size(gaBreaks,1))) #local I=I+1; #end
	#ifndef(OldSectionD) #declare OldSectionD = -1; #end
	#if (I != OldSectionD) 
		#declare OldSectionD = I;
		CropUserParams(gaBreaks[I], gaBreaks[I+1])
	#end
	
	// Creazione della sezione
	#local Sect = splBuildSection(T)
	
	// Inizio dell'oggetto
	#switch (Type)
		#case (POLYGON) polygon { #break
		#case (UNION) union { #break
		#case (MESH) mesh { #break
   #end
   
	#local aSmpPos = array[1]
	#local aSmpDer = array[1]
	DiafSamples()
	#local Dim = dimension_size(aSmpPos,1);
	
	#if (Type = POLYGON)
		Dim
		#local I=0;
		#while (I < Dim)
			aSmpPos[I]
			#local I=I+1;
		#end
	#else
		// Identificazione del centro
		#ifndef (gvShapeCenter) #declare gvShapeCenter = <0,0,0>; #end
		#local B = bBuildBase(T, X)
		#local vCenter = vMatrix(B, gvShapeCenter) + B[3];
		#local nCenter = EvalDerSpline(gsplPath, T);
		#if(vdot(nCenter, vcross(aSmpPos[0] - vCenter, aSmpDer[0])) < 0)
			#local nCenter = -nCenter 
		#end
		#local I=1;
		#while (I < Dim)
			smooth_triangle { vCenter, nCenter,
			                  aSmpPos[I-1], vcross(aSmpPos[I-1] - vCenter, aSmpDer[I-1])
			                  aSmpPos[I], vcross(aSmpPos[I] - vCenter, aSmpDer[I]) }
			#local I=I+1;
		#end
	#end
	
	// Fine dell'oggetto
	#switch (Type)
		#case (POLYGON)
		#case (UNION) 
		#case (MESH) } #break
   #end

#end

// Crea una base che metta un oggetto sulla superficie della striscia.
// la x va orientata lungo la sezione, 
// la y va perpendicolare alla superficie  
// la z va orientata lungo il percorso.
// l'oggetto e' sull'esterno della superficie se essa gira in senso antiorario 
// lungo il percorso.
#macro bBaseOnSurface(X, TS)

   // Sistemazione delle variabili
   InitInternalVars()

	// Sistemazione delle variabili interessanti
	#local TP = CalcTimeAbscissa(gsltPath, X);
	#ifndef(XBOS) #declare XBOS = -1e7; #declare SBOS = -1; #end
	#if(X != XBOS)
		#declare XBOS = X;
		#local I=0;
		#while ((gaBreaks[I+1] < TP) & (I+1 < dimension_size(gaBreaks,1))) #local I=I+1; #end
		#if (I != SBOS) 
			#declare SBOS = I;
			CropUserParams(gaBreaks[I], gaBreaks[I+1])
		#end
		#declare splSectBOS = splBuildSection(TP)
	#end
	
   // Creazione della base
   #local nSegm = GetSegmentsNumber(splSectBOS);
   #local bBase = array[4]
   #local bBase[0] = -vnormalize(EvalDerSpline(splSectBOS, nSegm*TS));
   #local bBase[1] = vnormalize(vcross(EvalDerSpline(gsplPath, TP), bBase[0]));
   #local bBase[2] = vcross(bBase[0], bBase[1]);
   #local bBase[3] = EvalSpline(splSectBOS, nSegm*TS);
   
   bBase
#end

#macro PutOnSurface(X, T)
	#local bBase = bBaseOnSurface(X, T)
	matrix <bBase[0].x, bBase[0].y, bBase[0].z,
	        bBase[1].x, bBase[1].y, bBase[1].z,
	        bBase[2].x, bBase[2].y, bBase[2].z,
	        bBase[3].x, bBase[3].y, bBase[3].z>
#end

#macro PutUnderSurface(X, T)
	#local bBase = bBaseOnSurface(X, T)
	matrix <-bBase[0].x, -bBase[0].y, -bBase[0].z,
	        -bBase[1].x, -bBase[1].y, -bBase[1].z,
	        -bBase[2].x, -bBase[2].y, -bBase[2].z,
	         bBase[3].x,  bBase[3].y,  bBase[3].z>
#end

/******************************************************************************
**                           Spline/slope workshop                           **
******************************************************************************/
#macro PlotSpline(Spline, vPos)

	#local NUM_COLORS = 6;
	#local aColors = array[NUM_COLORS] { 
		<1, 0, 0>, <0, 1, 0>, <0, 0, 1>, 
		<1, 1, 0>, <1, 0, 1>, <0, 1, 1> }

	#ifndef(gvMaxSlope)
		// e' la prima scena
		#include "SplLib.inc"
		#declare gvMaxSlope = <0,0,0>;
		#declare gvMinSlope = <0,0,0>;
		GetSplineBox(Spline, gvMinSlope, gvMaxSlope)
		#local vSplineCenter = 0.5 * (gvMaxSlope + gvMinSlope);
		#declare Size = vlength(gvMinSlope-gvMaxSlope) / 75;
		#declare gPlottedSplines = 0;
		
		// Disegna gli assi
		#local objXAxis = union {
			cylinder { 0, 10*x, Size }
			cone { 10*x, Size * 1.5, 10.5*x, 0 }
			pigment { 
				gradient x 
				color_map {
					[0.0 color rgb 1]
					[0.5 color rgb 1]
					[0.5 color red 1]
					[1.0 color red 1]
				}
				scale 2
			}
			finish {
				phong 1
				diffuse 0.8
			}
		}
		object { objXAxis }
		object { objXAxis rotate 90 * z }
		object { objXAxis rotate -90 * y }
		#local B = bBuildUnitBase(vnormalize(vPos - vSplineCenter))
		light_source {B[0] * vlength(gvMinSlope-gvMaxSlope) + vPos color rgb 1 }
		light_source {-B[0] * vlength(gvMinSlope-gvMaxSlope) + vPos color rgb 1 }
	#else
		#local vMaxSlope = <0,0,0>;
		#local vMinSlope = <0,0,0>;
		GetSplineBox(Spline, vMinSlope, vMaxSlope)
		#declare gvMaxSlope = vMax(gvMaxSlope, vMaxSlope);
		#declare gvMinSlope = vMin(gvMinSlope, vMinSlope);
	#end
	#local vSplineCenter = 0.5 * (gvMaxSlope + gvMinSlope);
	
	ResetParameters()
	#local splShape = splCircle(4)
	#local splShape = splScaleSpline(splShape, <Size, Size, Size>)
	SetShapeSpline(splShape)
	SetPathSpline(Spline)
	#declare gShapeDiv = 2;
	object {
		Striscia(UNION)
		pigment { color rgb aColors[mod(gPlottedSplines,NUM_COLORS)] }
		finish { phong 1 diffuse 0.8 }
	}
	
	camera {
		location vPos
		look_at vSplineCenter
	}
	#declare gPlottedSplines = gPlottedSplines + 1;
#end

#macro PlotSlope(Slope)
	#local NUM_COLORS = 6;
	#local aColors = array[NUM_COLORS] { 
		<1, 0, 0>, <0, 1, 0>, <0, 0, 1>, 
		<1, 1, 0>, <1, 0, 1>, <0, 1, 1> }

	#local NSeg = GetSlopeSegmentsNumber(Slope);
	#ifndef(gMaxX)
		#declare gvSky = z;
		#declare gPlottedSlopes = 0;
		#declare gMinX = GetSlopeStart(Slope);
		#declare gMaxX = GetSlopeEnd(Slope);

		#declare gMaxY = EvalSlope(Slope,gMinX);
		#declare gMinY = gMaxY;
		#local I=0;
		#while (I < NSeg)
			#local Start = GetSlopeSegmentStart(Slope[I]);
			#local End = GetSlopeSegmentEnd(Slope[I]);
			#declare gMaxY = max(gMaxY, EvalSlopeSegment(Slope[I],Start));
			#declare gMaxY = max(gMaxY, EvalSlopeSegment(Slope[I],End));
			#declare gMinY = min(gMinY, EvalSlopeSegment(Slope[I],Start));
			#declare gMinY = min(gMinY, EvalSlopeSegment(Slope[I],End));
			#local I=I+1;
		#end
		#local Size = max(gMaxX-gMinX, 4/3*(gMaxY-gMinY));
		#local Thickness = Size/100;
		#declare gCorrArea = Thickness * 4;
		#local objXAxis = box {<0,-Thickness,-Thickness>,<10,Thickness,Thickness> translate z
			pigment { gradient x 
				color_map {
					[0.0 color rgb 1]
					[0.5 color rgb 1]
					[0.5 color red 1]
					[1.0 color red 1]
				} scale 2
			}
			finish { ambient 1 diffuse 0 }
		}
		object { objXAxis }
		object { objXAxis rotate z * 90 }

	#else
		#declare gMinX = min(gMinX, GetSlopeStart(Slope));
		#declare gMaxX = max(gMaxX, GetSlopeEnd(Slope));
		#local I=0;
		#while (I < NSeg)
			#local Start = GetSlopeSegmentStart(Slope[I]);
			#local End = GetSlopeSegmentEnd(Slope[I]);
			#declare gMaxY = max(gMaxY, EvalSlopeSegment(Slope[I],Start));
			#declare gMaxY = max(gMaxY, EvalSlopeSegment(Slope[I],End));
			#declare gMinY = min(gMinY, EvalSlopeSegment(Slope[I],Start));
			#declare gMinY = min(gMinY, EvalSlopeSegment(Slope[I],End));
			#local I=I+1;
		#end
		#local Size = max(gMaxX-gMinX, 4/3*(gMaxY-gMinY));
		#local Thickness = Size/100;
	#end

	camera { orthographic
		location <(gMaxX+gMinX)/2,(gMaxY+gMinY)/2, -1000>
		look_at <(gMaxX+gMinX)/2,(gMaxY+gMinY)/2, 0>
		right x * Size * 1.05
		up y * Size * 3/4 * 1.05
	}
	
	ResetParameters()
	#declare gShapeDiv = 1;
	#declare gShapeDepth = 0;
	#declare splShape = splLinearSpline(array[2] {<-Thickness, 0, 0>, <Thickness, 0,0>})
	SetShapeSpline(splShape)
	
	#local DataPath = array[NSeg * 4]
	#local I=0;
	#while (I < NSeg)
		#local Start = GetSlopeSegmentStart(Slope[I]);
		#local End = GetSlopeSegmentEnd(Slope[I]);
		cylinder { 0, x * Size / 20, Thickness * 0.33
			pigment {color rgbf <1,1,1,0.25> } finish {ambient 1 diffuse 0 }
			scale <sqrt(1+pow(EvalSlopeSegmentDer(Slope[I],Start),2)),1,1>
			rotate z * degrees(atan2(EvalSlopeSegmentDer(Slope[I],Start),1))
			translate <Start,EvalSlopeSegment(Slope[I],Start),-gPlottedSlopes>
		}
		cylinder { 0, -x * Size / 20, Thickness * 0.33
			pigment {color rgbf <1,1,1,0.25>} finish {ambient 1 diffuse 0 }
			scale <sqrt(1+pow(EvalSlopeSegmentDer(Slope[I],End),2)),1,1>
			rotate z * degrees(atan2(EvalSlopeSegmentDer(Slope[I],End),1))
			translate <End,EvalSlopeSegment(Slope[I],End),-gPlottedSlopes>
		}
		sphere {<Start,EvalSlopeSegment(Slope[I],Start),-gPlottedSlopes>, Thickness scale <1,1,1.5>
			pigment { color rgb 0 }
		}
		#local DataPath[I*4] = <Start,EvalSlopeSegment(Slope[I],Start),-gPlottedSlopes>;
		#local DataPath[I*4+3] = <End,EvalSlopeSegment(Slope[I],End),-gPlottedSlopes>;
		#local DataPath[I*4+1] = DataPath[I*4] + 
			<GetSlopeSegmentLength(Slope[I])/3, EvalSlopeSegmentDer(Slope[I],Start)*GetSlopeSegmentLength(Slope[I])/3, 0>;
		#local DataPath[I*4+2] = DataPath[I*4+3] - 
			<GetSlopeSegmentLength(Slope[I])/3, EvalSlopeSegmentDer(Slope[I],End)*GetSlopeSegmentLength(Slope[I])/3, 0>;
		#local I=I+1;
	#end

	#declare splPath = splBezierSpline(DataPath)
	SetPathSpline(splPath) 

	object {
		Striscia(UNION)
		pigment { color rgb aColors[mod(gPlottedSlopes,NUM_COLORS)] }
		finish { ambient 1 diffuse 0 }
	}

	#declare gPlottedSlopes = gPlottedSlopes + 1;
#end

#macro PlotXYSpline(Spline)

	#local NUM_COLORS = 6;
	#local aColors = array[NUM_COLORS] { 
		<1, 0, 0>, <0, 1, 0>, <0, 0, 1>, 
		<1, 1, 0>, <1, 0, 1>, <0, 1, 1> }

	#ifndef(gvMaxSpline)
		// e' la prima scena
		#include "SplLib.inc"
		#declare gvMaxSpline = <0,0,0>;
		#declare gvMinSpline = <0,0,0>;
		GetSplineBox(Spline, gvMinSpline, gvMaxSpline)
		#local vSplineCenter = 0.5 * (gvMaxSpline + gvMinSpline);
		#declare Size = max(gvMaxSpline.x-gvMinSpline.x, 4/3*(gvMaxSpline.y-gvMinSpline.y));

		#declare gPlottedSplines = 0;
		
		// Disegna gli assi
		#local Thickness = Size/100;
		#local objXAxis = box {<0,-Thickness,-Thickness>,<10,Thickness,Thickness> translate z
			pigment { gradient x 
				color_map {
					[0.0 color rgb 1]
					[0.5 color rgb 1]
					[0.5 color red 1]
					[1.0 color red 1]
				} scale 2
			}
			finish { ambient 1 diffuse 0 }
		}
		object { objXAxis }
		object { objXAxis rotate z * 90 }
	#else
		#local vMaxSpline = <0,0,0>;
		#local vMinSpline = <0,0,0>;
		GetSplineBox(Spline, vMinSpline, vMaxSpline)
		#declare gvMaxSpline = vMax(gvMaxSpline, vMaxSpline);
		#declare gvMinSpline = vMin(gvMinSpline, vMinSpline);
		#declare Size = max(gvMaxSpline.x-gvMinSpline.x, 4/3*(gvMaxSpline.y-gvMinSpline.y));
	#end
	#local vSplineCenter = 0.5 * (gvMaxSpline + gvMinSpline);
	
	ResetParameters()
	#declare gvSky = -z;
	#local Thickness = Size/100;
	#local splShape = splLinearSpline(array[2] {<-Thickness, 0, 0>, <Thickness, 0, 0>})
	SetShapeSpline(splShape)
	SetPathSpline(Spline)
	#declare gShapeDiv = 1;
	#declare gShapeDepth = 0;
	object {
		Striscia(UNION)
		translate -z * gPlottedSplines
		pigment { color rgb aColors[mod(gPlottedSplines,NUM_COLORS)] }
		finish { ambient 1 diffuse 0 }
	}
	
	camera {
		orthographic
		up y * Size * 3/4 * 1.05
		right x * Size * 1.05
		location 0.5*(gvMaxSpline+gvMinSpline) - z*1000
		look_at 0.5*(gvMaxSpline+gvMinSpline)
	}
	#declare gPlottedSplines = gPlottedSplines + 1;
#end

#version StrisciaTmpVersion;
