//////////
//
//	File:		VRSound.h
//
//	Contains:	Sound support for QuickTime VR movies.
//
//	Written by:	Tim Monroe
//
//	Copyright:	 1996-1997 by Apple Computer, Inc., all rights reserved.
//
//	Change History (most recent first):
//
//	   <2>	 	08/11/97	rtm		added support for chunk-based sound files (AIFF and WAVE)
//	   <1>	 	12/09/96	rtm		first file
//	   
//////////

#pragma once

// header files
#include <math.h>
#include <Endian.h>

#include "ComApplication.h"

#if TARGET_OS_MAC
#include "MacFramework.h"
#endif

#if TARGET_OS_WIN32
#include "WinFramework.h"
#endif

#include "VRScript.h"

// for WAVE files, we'll use the Windows data types
#if TARGET_OS_MAC
#define WORD					SInt16
#define DWORD					UInt32
#define FOURCC					OSType
#endif

//typedef DWORD					*WordPtr;

// constants

// values that we pass in param1 of a callBackCmd
enum {
	kVRSound_Complete	= 1				// the sound has finished playing
};

#define kVRSound_NumFadeSteps	8		// number of steps while fading a sound to silence
#define kVRSound_FadeStepWait	5		// number of ticks to wait on each fade step
#define kVRSound_NumCmdsInQueue	4		// number of commands in a sound channel queue
#define kVRSound_SpeakerAngle	15.0	// angular distance (in degrees) of left or right speaker from viewer vector

// ***start of support for WAVE files

// RIFF forms
#define kRIFFType_WAVE			'WAVE'		// WAVE form: contains digitized sound

// chunk types
#define kChunkType_RIFF			'RIFF'		// RIFF chunk (the main container)
#define kChunkType_List			'LIST'		// list chunk
#define kChunkType_Format		'fmt '		// format chunk
#define kChunkType_Data			'data'		// wave-data chunk

// format specifiers
#if TARGET_OS_MAC
#define WAVE_FORMAT_PCM			0x0001		// Microsoft Pulse Code Modulation (PCM) format
#define IBM_FORMAT_MULAW		0x0101		// IBM mu-law format
#define IBM_FORMAT_ALAW			0x0102		// IBM a-law format
#define IBM_FORMAT_ADPCM		0x0103		// IBM AVC Adaptive Differential Pulse Code Modulation format
#endif

// compression types
#define kCompressType_None		'NONE'
#define kCompressType_SOWT		'sowt'


// WAVE files are a kind of Resource Interchange File Format (RIFF) file;
// every RIFF file contains some number of chunks, and every chunk begins
// with a chunk header.

typedef struct ChunkHeader {
	FOURCC			fChunkType;				// the type of this chunk
	DWORD			fChunkSize;				// the size (in bytes) of the chunk data
											// the chunk data follows here
} ChunkHeader, *ChunkHeaderPtr;

// RIFF chunk
// always the first chunk in a file
typedef struct RIFFChunk {
	ChunkHeader		fChunkHeader;			// the chunk header; fChunkType == 'RIFF'
	FOURCC			fFormType;				// the data format; 'WAVE' for waveform files
											// additional chunk data follows here
} RIFFChunk, *RIFFChunkPtr;

// List chunk
// 
typedef struct ListChunk {
	ChunkHeader		fChunkHeader;			// the chunk header; fChunkType == 'LIST'
	FOURCC			fListType;				// the type of the contents of the list
											// additional chunk data follows here
} ListChunk, *ListChunkPtr;

// Format chunk
// specifies the format of the waveform data (which is in a wave-data chunk)
typedef struct FormatChunk {
	ChunkHeader		fChunkHeader;			// the chunk header; fChunkType == 'fmt '
	WORD			fFormatType;			// format type
	WORD			fNumChannels;			// number of channels
	DWORD			fSamplesPerSec;			// sample rate
	DWORD			fAvgBytesPerSec;		// byte rate (for buffer estimation)
	WORD			fBlockAlign;			// data block size
											// additional type-specific chunk data might follow here
	WORD			fAdditionalData;
} FormatChunk, *FormatChunkPtr;

// PCM type-specific data for Format chunk
// 
typedef struct PCMFormatChunk {
	WORD			fBitsPerSample;			// number of bits in each sample
} PCMFormatChunk, *PCMFormatChunkPtr;

// Wave-data chunk
// 
typedef struct DataChunk {
	ChunkHeader		fChunkHeader;			// the chunk header; fChunkType == 'data'
											// data follows here
} DataChunk, *DataChunkPtr;

// ***end of support for WAVE files

// function prototypes
Boolean						VRSound_CheckVersionNumber (const NumVersion *theVersion, UInt8 theMajor, UInt8 theMinor, UInt8 theBug);
SoundHeaderPtr				VRSound_GetSoundHeader (Handle theSndHandle);
long						VRSound_GetSndBaseFrequency (Handle theSndHandle);
OSErr						VRSound_GetVolume (SndChannelPtr theChannel, unsigned short *theLeftVol, unsigned short *theRightVol);
OSErr						VRSound_SetVolume (SndChannelPtr theChannel, unsigned short theLeftVol, unsigned short theRightVol);
SndChannelPtr				VRSound_CreateSoundChannel (Boolean theSoundIsLocalized);
SSpSourceReference			VRSound_CreateLocalizedSource (void);
void						VRSound_SetLocation (WindowObject theWindowObject, UInt32 theEntryID, float theX, float theY, float theZ, UInt32 theOptions);
short						VRSound_GetSndResourceID (short theRefNum);
OSErr						VRSound_GetChunkType (short theFile, OSType theType, long theStartPos, long *theChunkPos);
ChunkHeaderPtr				VRSound_GetChunkData (short theFile, OSType theType, long theStartPos);
void						VRSound_Update3DSoundEnv (WindowObject theWindowObject);
void						VRSound_SetBalanceAndVolume (WindowObject theWindowObject, float thePan, float theTilt);
void						VRSound_Init (void);
void						VRSound_Stop (void);
void						VRSound_InitWindowData (WindowObject theWindowObject);
void						VRSound_DumpWindowData (WindowObject theWindowObject);
void						VRSound_FadeNodeSounds (WindowObject theWindowObject);
void						VRSound_DumpNodeSounds (WindowObject theWindowObject);
void						VRSound_DumpSceneSounds (WindowObject theWindowObject);
void						VRSound_DumpSelectedSounds (WindowObject theWindowObject, UInt32 theOptions);
void						VRSound_FadeSilence (WindowObject theWindowObject, SndChannelPtr theChannel);
void						VRSound_PlaySilence (WindowObject theWindowObject, SndChannelPtr theChannel);
void						VRSound_PlayResource (WindowObject theWindowObject, SndChannelPtr theChannel, SndListHandle theResHandle, UInt32 theOptions);
void						VRSound_PlaySound (WindowObject theWindowObject, UInt32 theNodeID, UInt32 theResID, UInt32 theEntryID, float theX, float theY, float theZ, float theProjAngle, UInt32 theSourceMode, UInt32 theMode, UInt32 theFade, UInt32 theOptions);
void						VRSound_InstallCallbackMessage (VRScriptSoundPtr theEntry, short theMessage);
VRScriptSoundPtr			VRSound_GetFinishedSound (WindowObject theWindowObject);
PASCAL_RTN void				VRSound_CallbackProc (SndChannelPtr theChannel, SndCommand *theCommand);
void						VRSound_CheckForCompletedSounds (WindowObject theWindowObject);
void						VRSound_DumpEntryMem (VRScriptSoundPtr theEntry);

