//////////
//
//	File:		VR3DTexture.c
//
//	Contains:	Support for adding a QuickTime movie or a PICT as a texture on a QD3D object.
//
//	Written by:	Tim Monroe
//				Parts modeled on BoxMoov code by Rick Evans and Robert Dierkes.
//
//	Copyright:	 1996-1998 by Apple Computer, Inc., all rights reserved.
//
//	Change History (most recent first):
//
//	   <6>	 	01/28/98	rtm		changed NewGWorld calls to QTNewGWorld
//	   <5>	 	01/27/98	rtm		made minor changes to VR3DTexture_Delete
//	   <4>	 	05/02/97	rtm		added fMediaHandler field to textures, for movies with sound
//	   <3>	 	01/14/97	rtm		added support for PICT textures
//	   <2>	 	12/17/96	rtm		modified VR3DTexture_AddToGroup to replace a texture shader
//	   <1>	 	12/16/96	rtm		first file; revised to personal coding style
//	   
//////////

// header files
#include <QuickDraw.h>
#include <Resources.h>

#include <QD3D.h>
#include <QD3DGroup.h>
#include <QD3DShader.h>
#include <QD3DSet.h>

#include "VR3DTexture.h"
#include "VR3DObjects.h"
#include "QTUtilities.h"
#include "QTVRUtilities.h"


//////////
//
// VR3DTexture_New
// Create a new texture from a QuickTime movie or PICT file.
//
//////////

TextureHdl VR3DTexture_New (char *thePathName, Boolean isTextureMovie)
{
	FSSpec				myFSSpec;
	short				myRefNum;
	unsigned long		myPictMapAddr;
	unsigned long 		myPictRowBytes;
	GWorldPtr 			myGWorld;
	PixMapHandle 		myPixMap;
	GDHandle			myOldGD;
	GWorldPtr			myOldGW;
	Boolean				mySuccess = false;
	Rect				myBounds;
	TQ3StoragePixmap	*myStrgPMapPtr;
	TextureHdl			myTexture = NULL;
	QDErr				myErr = noErr;

	myTexture = (TextureHdl)NewHandleClear(sizeof(Texture));
	if (myTexture == NULL)
		return(myTexture);
		
	HLock((Handle)myTexture);

	// save the current port
	GetGWorld(&myOldGW, &myOldGD);
		
	FSMakeFSSpec(0, 0L, c2pstr(thePathName), &myFSSpec);

	if (isTextureMovie) {
		// open the movie and store its address in our data structure
		mySuccess = VR3DObjects_LoadEmbeddedMovie(&myFSSpec, &(**myTexture).fMovie);
		if (!mySuccess)
			goto bail;

		// get the sound media handler for movies with sound
		if (QTUtils_IsMediaTypeInMovie((**myTexture).fMovie, SoundMediaType) || QTUtils_IsMediaTypeInMovie((**myTexture).fMovie, MusicMediaType))
			(**myTexture).fMediaHandler = QTUtils_GetSoundMediaHandler((**myTexture).fMovie);
		
		// get the size of the movie
		GetMovieBox((**myTexture).fMovie, &myBounds);
		
		// create a new offscreen graphics world (into which we will draw the movie)
		myErr = QTNewGWorld(&myGWorld, kOffscreenPixelType, &myBounds, NULL, NULL, 0L);
		if (myErr != noErr) {
			StopMovie((**myTexture).fMovie);
			DisposeMovie((**myTexture).fMovie);
			(**myTexture).fMovie = NULL;
			mySuccess = false;
			goto bail;
		}
	} else {
		// open the PICT and store its address in our data structure
		myErr = FSpOpenDF(&myFSSpec, fsRdPerm, &myRefNum);
		if (myErr != noErr) {
			mySuccess = false;
			goto bail;
		}
			
		(**myTexture).fPicture = VR3DObjects_GetEmbeddedPicture(myRefNum);		
		if ((**myTexture).fPicture == NULL) {
			mySuccess = false;
			goto bail;
		}

		// get the size of the picture;
		// NOTE: the earlier call to VR3DObjects_GetEmbeddedPicture
		// has already resolved any endian issues with the picture data
		myBounds = (**(**myTexture).fPicture).picFrame;
		
		// create a new offscreen graphics world (into which we will draw the picture)
		myErr = QTNewGWorld(&myGWorld, kOffscreenPixelType, &myBounds, NULL, NULL, 0L);
		if (myErr != noErr) {
			KillPicture((**myTexture).fPicture);
			(**myTexture).fPicture = NULL;
			mySuccess = false;
			goto bail;
		}
	}

	mySuccess = true;

	myPixMap = GetGWorldPixMap(myGWorld);
	LockPixels(myPixMap);
	myPictMapAddr = (unsigned long)GetPixBaseAddr(myPixMap);
	
	// get the offset, in bytes, from one row of pixels to the next;
	// according to IM: Imaging With QuickDraw (p. 4-47), the two high-order bits are used as flags,
	// so we need to mask them off (0x3fff == 0b0011 1111 1111 1111)
	myPictRowBytes = (unsigned long)((**myPixMap).rowBytes & 0x3fff);

	SetGWorld(myGWorld, NULL);

	// create a storage object associated with the new offscreen graphics world
	myStrgPMapPtr = &(**myTexture).fStoragePixmap;
	myStrgPMapPtr->image = Q3MemoryStorage_NewBuffer((void *)myPictMapAddr,
													myPictRowBytes * myBounds.bottom,
													myPictRowBytes * myBounds.bottom);
	myStrgPMapPtr->width		= myBounds.right;
	myStrgPMapPtr->height		= myBounds.bottom;
	myStrgPMapPtr->rowBytes		= myPictRowBytes;
	myStrgPMapPtr->pixelSize	= 32;
	myStrgPMapPtr->pixelType	= kQ3PixelTypeRGB32;
	myStrgPMapPtr->bitOrder		= kQ3EndianBig;
	myStrgPMapPtr->byteOrder	= kQ3EndianBig;

	if (isTextureMovie) {
		// start playing the movie in a loop
		VR3DObjects_LoopEmbeddedMovie((**myTexture).fMovie, myGWorld);
	} else {
		// draw the picture into the offscreen graphics world
		EraseRect(&myBounds);
		DrawPicture((**myTexture).fPicture, &myBounds);
	}

	(**myTexture).fpGWorld = myGWorld;

bail:
	SetGWorld(myOldGW, myOldGD);
	HUnlock((Handle)myTexture);

	if (mySuccess)
		return(myTexture);
	else {
		if (myTexture != NULL)
			DisposeHandle((Handle)myTexture);
		return(NULL);
	}
}


//////////
//
// VR3DTexture_AddToGroup
// Create a new texture shader based on the pixmap storage data and add it to the group.
//
//////////

TQ3Status VR3DTexture_AddToGroup (TextureHdl theTexture, TQ3GroupObject theGroup)
{
	TQ3Status			myStatus = kQ3Failure;
	TQ3StoragePixmap	myStoragePixmap;
	TQ3TextureObject	myTextureObject;
	
	if (theTexture == NULL || theGroup == NULL)
		return(myStatus);
	
	myStoragePixmap = (**theTexture).fStoragePixmap;
	myTextureObject = Q3PixmapTexture_New(&myStoragePixmap);
	if (myTextureObject != NULL) {

		TQ3ShaderObject		myTextureShader;
		
		myTextureShader = Q3TextureShader_New(myTextureObject);
		Q3Object_Dispose(myTextureObject);
		
		if (myTextureShader != NULL) {
			TQ3GroupPosition	myPosition = NULL;
			
			// look for an existing texture shader in group
			Q3Group_GetFirstPositionOfType(theGroup, kQ3SurfaceShaderTypeTexture, &myPosition);
			if (myPosition != NULL) {
			
				// there is an existing texture shader; just replace it
				Q3Group_SetPositionObject(theGroup, myPosition, myTextureShader);
			} else {
			
				// there is no existing texture shader; add one to the group
				Q3Group_GetFirstPosition(theGroup, &myPosition);
				Q3Group_AddObjectBefore(theGroup, myPosition, myTextureShader);
			}
			
			Q3Object_Dispose(myTextureShader);
			myStatus = kQ3Success;
		}
	}

	return(myStatus);
}


//////////
//
// VR3DTexture_Delete
// Deallocate the texture.
//
//////////

Boolean	VR3DTexture_Delete (TextureHdl theTexture)
{
	PixMapHandle		myPixMap;

	if (theTexture == NULL)
		return(false);

	HLock((Handle)theTexture);

	if ((**theTexture).fMovie != NULL) {
		StopMovie((**theTexture).fMovie);
		DisposeMovie((**theTexture).fMovie);
		(**theTexture).fMovie = NULL;
	}

	if ((**theTexture).fPicture != NULL) {
		KillPicture((**theTexture).fPicture);
		(**theTexture).fPicture = NULL;
	}

	if ((**theTexture).fpGWorld == NULL) {
		myPixMap = GetGWorldPixMap((**theTexture).fpGWorld);
		if (myPixMap != NULL)
			UnlockPixels(myPixMap);
	
		DisposeGWorld((**theTexture).fpGWorld);
		(**theTexture).fpGWorld = NULL;
	}

	if ((**theTexture).fStoragePixmap.image != NULL) {
		Q3Object_Dispose((**theTexture).fStoragePixmap.image);
		(**theTexture).fStoragePixmap.image = NULL;
	}

	HUnlock((Handle)theTexture);
	DisposeHandle((Handle)theTexture);
	
	return(true);
}


//////////
//
// VR3DTexture_NextFrame
// Advance the texture's movie to the next frame.
//
//////////

Boolean VR3DTexture_NextFrame (TextureHdl theTexture)
{
	TQ3StoragePixmap	*myStrgPMapPtr;
	long				mySize;
	TQ3Status			myStatus;
	unsigned long		myPictMapAddr;

	// if fpGWorld is non-NULL then the fMovie is non-NULL and the movie needs updating
	if ((**theTexture).fpGWorld == NULL)
		return(false);

	HLock((Handle)theTexture);

	if ((**theTexture).fMovie)
		MoviesTask((**theTexture).fMovie, 0);	// draw the next movie frame

	myStrgPMapPtr = &(**theTexture).fStoragePixmap;
	mySize = myStrgPMapPtr->height * myStrgPMapPtr->rowBytes;
	myPictMapAddr = (unsigned long)GetPixBaseAddr((**theTexture).fpGWorld->portPixMap);

	// tell QD3D the buffer changed
	myStatus = Q3MemoryStorage_SetBuffer(myStrgPMapPtr->image, (void *)myPictMapAddr, mySize, mySize);

	HUnlock((Handle)theTexture);
	
	if (myStatus == kQ3Success)
		return(true);
	else
		return(false);
}
