//////////
//
//	File:		VRScript.c
//
//	Contains:	Functions for script file processing.
//
//	Written by:	Tim Monroe
//
//	Copyright:	 1997 by Apple Computer, Inc., all rights reserved.
//
//	Change History (most recent first):
//
//	   <37>	 	02/25/98		rtm		revised VRScript_EnteringNodeProc to call MoviesTask instead of QTVRUpdate
//	   <36>	 	12/12/97		rtm		fixed VRScript_FindAndOpenQTVRMovieFile (was opening the movie file
//										instead of allowing DoCreateMovieWindow to do so)
//	   <35>	 	10/29/97		rtm		finished adding support for QuickTime video effects
//	   <34>	 	10/27/97		rtm		began adding support for QuickTime video effects
//	   <33>	 	10/06/97		rtm		fixed bug in replace cursor commands (wouldn't restore original
//										cursors if a click on the hot spot caused the node to be exited)
//	   <32>	 	09/10/97		rtm		made changes necessary to compile for Windows execution
//	   <31>	 	07/21/97		rtm		added VRScript_DumpUnexpiredCommands
//	   <30>	 	07/17/97		rtm		revised all list-walking code to avoid dangling pointers
//	   <29>	 	07/15/97		rtm		reworked VRScript_MouseOverHotSpotProc to use a single while loop
//	   <28>	 	07/14/97		rtm		fixed SetHotSpotTypeCursors to use OSTypes
//	   <27>	 	07/12/97		rtm		fixed a bug in VRScript_FindAndOpenQTVRMovieFile
//	   <26>	 	07/11/97		rtm		added support for encoded commands
//	   <25>	 	07/08/97		rtm		added SetHotSpotTypeCursors command
//	   <24>	 	07/07/97		rtm		added ReplaceResource and SetHotSpotIDCursors commands
//	   <23>	 	06/19/97		rtm		added PlaySceneSound and PlaySceneQTMidi, for persistent sounds
//	   <22>	 	06/13/97		rtm		added code to make sure QD3D present before executing 3D commands
//	   <21>	 	06/12/97		rtm		added VRScript_FindAndOpenQTVRMovieFile
//	   <20>	 	06/05/97		rtm		added VRScript_SetControllerButtonState;
//										changed semantics of SetVariable and If commands
//	   <19>	 	06/04/97		rtm		added Mode parameter to SetPanTiltZoom;
//	   <18>	 	05/30/97		rtm		added script-defined variables (SetVariable and If commands)
//	   <17>	 	05/22/97		rtm		added ability to cancel node exit to VRScript_LeavingNodeProc
//	   <16>	 	05/01/97		rtm		code clean-up: put list heads into an array
//	   <15>	 	04/28/97		rtm		added support for QuickTime MIDI files
//	   <14>	 	04/17/97		rtm		added VRScript_DelistEntry; added fMaxExecutions to some commands
//	   <13>	 	04/10/97		rtm		added 3D object and 3D sound "Set" calls
//	   <12>	 	04/07/97		rtm		added SetResizeState
//	   <11>	 	04/02/97		rtm		added support for 3DMF files
//	   <10>	 	04/01/97		rtm		added VRScript_CheckForAngleCommands
//	   <9>	 	03/31/97		rtm		added node-entry and node-exit command support;
//										added support for additional imaging and interaction properties
//	   <8>	 	03/21/97		rtm		added VRScript_SetHotSpotState
//	   <7>	 	03/17/97		rtm		localized embedded QuickTime movie sounds
//	   <6>	 	03/13/97		rtm		reworked hot spot sounds; fixed code for 680x0 compilation;
//										added support for angle commands
//	   <5>	 	03/12/97		rtm		added support for mouse-over hot spot commands, external resource files,
//										hot spot click commands, and QT movies;
//	   <4>	 	03/11/97		rtm		added support for timed commands (cool!)
//	   <3>	 	03/10/97		rtm		added support for controller bar, localized sounds, and overlay pictures
//	   <2>	 	03/07/97		rtm		got hot spot sounds working (except for any options)
//	   <1>	 	03/06/97		rtm		first file
//
//
//	This file contain functions that support an external text script file for driving QuickTime VR movies.
//  The script file contains one command per line; a command line is a command word followed by one or more
//	command parameters. The number and meaning of the parameters depends on the command word. See the file 
//	"Script Syntax" for a complete description of the scripting language.
//
//////////

// 	TO DO:
//	+ finish hide region code
//	+ change AtClickHS command to AtClickHSID and AtClickHSType commands

// header files
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ComApplication.h"

// header files for special features
#include "VRSound.h"
#include "VRPicture.h"
#include "VRMovies.h"
#include "VR3DObjects.h"
#include "VR3DTexture.h"
#include "VREffects.h"

#include "VRScript.h"

// global variables
Boolean					gReadingScriptFile;						// are we reading a script file?
char					gScriptFileName[kMaxFileNameLength];	// the name of the script file

extern Boolean			gHasSoundSprockets;						// is SoundSprockets available?
extern Boolean			gHasSoundManager30;						// is Sound Manager version 3.0 (or greater) available?
extern Boolean			gHasQuickDraw3D;						// is QuickDraw 3D available?
extern Boolean			gHasQuickDraw3D15;						// is QuickDraw 3D version 1.5 (or greater) available?
extern Boolean			gHasQTVideoEffects;						// are the QuickTime video effects available?

extern long				gAbsoluteElapsedTime;
extern long				gNodeStartingTime;
extern long				gNodeElapsedTime;

extern Rect				gLimitRect;								// max size for any window


//////////
//
// VRScript_OpenScriptFile
// Open the external script file and process the commands (one command per line).
//
//////////

void VRScript_OpenScriptFile (WindowObject theWindowObject, char *theFileName)
{
	FILE		*myFile = NULL;
	char		myString[kMaxCmdLineLength];
	
	if (theFileName == NULL)
		return;
		
	myFile = fopen(theFileName, "r");
	if (myFile == NULL)
		return;

	gReadingScriptFile = true;

	while (fgets(myString, sizeof(myString), myFile) != NULL)
		VRScript_ProcessScriptCommandLine(theWindowObject, myString);

	gReadingScriptFile = false;

	fclose(myFile);
}


//////////
//
// VRScript_FindAndOpenQTVRMovieFile
// Open the specified script file and look for the first OpenQTVRMovieFile command in it;
// open the specified movie file if one is found.
//
//////////

void VRScript_FindAndOpenQTVRMovieFile (FSSpec *theFSSpecPtr)
{
	FILE		*myFile = NULL;
	char		myString[kMaxCmdLineLength];
	char		myCommand[kMaxCmdWordLength];
	short		myLength;
	
	// a bit of terminology: on MacOS, the directory that is searched when only a filename is given
	// is called the "default directory"; on Windows, this is called the "current directory".
	
#if TARGET_OS_MAC
	// set the default directory and volume to be those of the script file, not the application
	HSetVol(NULL, theFSSpecPtr->vRefNum, theFSSpecPtr->parID);
#elif TARGET_OS_WIN32

#endif
	
	// convert the filename to a C string
	myLength = theFSSpecPtr->name[0];
	memcpy(gScriptFileName, &theFSSpecPtr->name[1], myLength);
	gScriptFileName[myLength] = '\0';
	
	// open the script file			
	myFile = fopen(gScriptFileName, "r");
	if (myFile == NULL)
		return;

	// search through the script file for a line beginning "OpenQTVRMovieFile"
	while (fgets(myString, sizeof(myString), myFile) != NULL) {
	
		// get the command word
		sscanf(myString, "%s ", myCommand);
		if (strlen(myCommand) == 0)
			break;
	
	 	// open the specified VR movie file in a movie window
		if (strcmp(myCommand, "OpenQTVRMovieFile") == 0) {
			char		myPathName[kMaxFileNameLength];
			UInt32		myOptions;
			FSSpec		myFSSpec;

			sscanf(myString, "%*s %d %s", &myOptions, myPathName);
			VRScript_UnpackString(myPathName);
			
			// create an FSSpec that picks out the QTVR movie file
			FSMakeFSSpec(theFSSpecPtr->vRefNum, theFSSpecPtr->parID, c2pstr(myPathName), &myFSSpec);
						
			// now act as if "Open" were chosen from the File menu....
			DoCreateMovieWindow(NULL, &myFSSpec);
			
			// break out of the while loop
			break;
		}
	}
	
	fclose(myFile);
}


//////////
//
// VRScript_ProcessScriptCommandLine
// Process a script command line.
//
//////////

//////////
// WARNING: This is NOT a good example of how to write a parser. The most that can be said
// for this code is that (1) it works, and (2) it's easy to modify and expand. This is the
// one routine in this application that should be rewritten from the ground up, along with
// a corresponding reworking of the scripting language. Remember that this application is
// intended to illustrate how to integrate media with QuickTime VR movies; it is not meant 
// to provide a commercial-quality scripting language and parser. You've been warned!
//////////

void VRScript_ProcessScriptCommandLine (WindowObject theWindowObject, char *theCommandLine)
{
	UInt32		myResID;
	UInt32		myNodeID;
	UInt32		myOptions;
	float		myPanAngle;
	float		myTiltAngle;
	float		myFOVAngle;
	float		myTempAngle;
	char		myCommand[kMaxCmdWordLength];
	
	// decode the string, if necessary;
	// any line that begins with '@' is assumed to be encoded using a simple rotate-11 scheme
	if (theCommandLine[0] == '@') {
		theCommandLine++;							// get rid of the '@'
		VRScript_DecodeString(theCommandLine);		// decode the rest of the command
	}
#if ONLY_ENCODED_SCRIPTS
	else {
		if (gReadingScriptFile)
			return;									// we allow only encoded scripts
	}
#endif

	// strip off any leading white space
	while (isspace(theCommandLine[0]))
		theCommandLine++;
		
	// any line that begins with '#' is a comment; ignore it
	if (theCommandLine[0] == '#')
		return;
		
	// get the command word
	sscanf(theCommandLine, "%s ", myCommand);
	if (strlen(myCommand) == 0)
		return;
		
	// ***now look for the command word and process it***
	
 	// ***SetBarState***
 	// show or hide the controller bar
	if (strcmp(myCommand, "SetBarState") == 0) {
		UInt32		myState;

		sscanf(theCommandLine, "%*s %d %d", &myState, &myOptions);
		VRScript_SetControllerBarState(theWindowObject, (Boolean)myState, myOptions);
		return;
	}
	
 	// ***SetButtonState***
 	// show or hide a button in the controller bar; or, enable or disable the display text in the bar
	if (strcmp(myCommand, "SetButtonState") == 0) {
		UInt32		myState;
		UInt32		myButton;
		
		sscanf(theCommandLine, "%*s %d %d %d", &myButton, &myState, &myOptions);
		VRScript_SetControllerButtonState(theWindowObject, myButton, (Boolean)myState, myOptions);
		return;
	}
	
 	// ***SetResizeState***
 	// enable or disable window resizing
	if (strcmp(myCommand, "SetResizeState") == 0) {
		UInt32		myState;

		sscanf(theCommandLine, "%*s %d %d", &myState, &myOptions);
		VRScript_SetResizeState(theWindowObject, (Boolean)myState, myOptions);
		return;
	}
	
 	// ***SetWindowSize***
 	// set the current size of a movie window
	if (strcmp(myCommand, "SetWindowSize") == 0) {
		UInt32				myHeight;
		UInt32				myWidth;
		Rect				myRect;
		MovieController 	myMC;
		
		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d %d", &myHeight, &myWidth, &myOptions);
		myMC = (**theWindowObject).fController;
		if (QTVRUtils_IsControllerBarVisible(myMC)) {
			MCGetControllerBoundsRect(myMC, &myRect);
		 } else {
			Movie	myMovie;
			
			myMovie = MCGetMovie(myMC);
			GetMovieBox(myMovie, &myRect);
		}

		myRect.right = (short)myWidth;
		myRect.bottom = (short)myHeight;
		MCSetControllerBoundsRect(myMC, &myRect);
		return;
	}
	
 	// ***SetMaxWindowSize***
 	// set the maximum size of a movie window
	if (strcmp(myCommand, "SetMaxWindowSize") == 0) {
		UInt32				myHeight;
		UInt32				myWidth;
		
		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d %d", &myHeight, &myWidth, &myOptions);
		gLimitRect.right = (short)myWidth;
		gLimitRect.bottom = (short)myHeight;
		VRScript_SetResizeState(theWindowObject, (**theWindowObject).fCanResizeWindow, myOptions);
		return;
	}
	
 	// ***ReplaceCursor***
 	// replace one cursor by another, or restore the original QTVR cursor
	if (strcmp(myCommand, "ReplaceCursor") == 0) {
		SInt32				myPrevID;
		SInt32				myNewID;
		QTVRCursorRecord	myCursorRec;
		
		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d %d", &myPrevID, &myNewID, &myOptions);

		myCursorRec.theType = (UInt16)myOptions;		// the type of cursor to replace
		myCursorRec.rsrcID = (SInt16)myPrevID;			// the resource ID of cursor to replace
		if (myOptions == kQTVRUseDefaultCursor) {
			myCursorRec.handle = NULL;
		} else {
			myCursorRec.handle = (Handle)MacGetCursor((short)myNewID);
			DetachResource(myCursorRec.handle);
		}
		
		QTVRReplaceCursor((**theWindowObject).fInstance, &myCursorRec);
		
		// QTVRReplaceCursor makes a copy of the handle we pass it, so we can dispose of our handle;
		// make sure not to dispose the handle if QTVR loaded it, however
		if ((myCursorRec.handle != NULL) && (myOptions != kQTVRUseDefaultCursor))
			DisposeHandle(myCursorRec.handle);
		
		return;
	}
	
 	// ***SetHotSpotIDCursors***
 	// replace the triad of cursors for a hot spot specified by its ID;
 	// currently this works only for *undefined* hot spots
	if (strcmp(myCommand, "SetHotSpotIDCursors") == 0) {
		UInt32		myHotSpotID;
		SInt32		myCurs1ID, myCurs2ID, myCurs3ID;	// resource IDs of the three replacement cursors
		char		myCmdLine[kMaxCmdLineLength];

		if (theWindowObject == NULL)
			return;

		// read the command paramters	
		sscanf(theCommandLine, "%*s %d %d %d %d %d %d", &myNodeID, &myHotSpotID, &myCurs1ID, &myCurs2ID, &myCurs3ID, &myOptions);

		// enlist three ReplaceCursor calls for entering the hot spot: install new cursor triad
		sprintf(myCmdLine, "ReplaceCursor %d %d %d", kCursID_MouseUpOnUndefHS, myCurs3ID, kQTVRStdCursorType);
		VRScript_EnlistMouseOverHSCommand(theWindowObject, myNodeID, myHotSpotID, 0L, kVRDoIt_Forever, kQTVRHotSpotEnter, myCmdLine);
		sprintf(myCmdLine, "ReplaceCursor %d %d %d", kCursID_MouseDownOnUndefHS, myCurs2ID, kQTVRStdCursorType);
		VRScript_EnlistMouseOverHSCommand(theWindowObject, myNodeID, myHotSpotID, 0L, kVRDoIt_Forever, kQTVRHotSpotEnter, myCmdLine);
		sprintf(myCmdLine, "ReplaceCursor %d %d %d", kCursID_MouseOverUndefHS, myCurs1ID, kQTVRStdCursorType);
		VRScript_EnlistMouseOverHSCommand(theWindowObject, myNodeID, myHotSpotID, 0L, kVRDoIt_Forever, kQTVRHotSpotEnter, myCmdLine);

		// enlist three ReplaceCursor calls for leaving the hot spot: reinstall original cursor triad
		// NOTE: this prevents the new cursor from appearing on other hot spots of the same type, and
		// also works around a bug in QTVR 2.0.0 and 2.0.1: replacing a cursor by itself disposes of 
		// the cursor and eventually might lead to a crash.
		sprintf(myCmdLine, "ReplaceCursor %d %d %d", kCursID_MouseUpOnUndefHS, 0, kQTVRUseDefaultCursor);
		VRScript_EnlistMouseOverHSCommand(theWindowObject, myNodeID, myHotSpotID, 0L, kVRDoIt_Forever, kQTVRHotSpotLeave, myCmdLine);
		sprintf(myCmdLine, "ReplaceCursor %d %d %d", kCursID_MouseDownOnUndefHS, 0, kQTVRUseDefaultCursor);
		VRScript_EnlistMouseOverHSCommand(theWindowObject, myNodeID, myHotSpotID, 0L, kVRDoIt_Forever, kQTVRHotSpotLeave, myCmdLine);
		sprintf(myCmdLine, "ReplaceCursor %d %d %d", kCursID_MouseOverUndefHS, 0, kQTVRUseDefaultCursor);
		VRScript_EnlistMouseOverHSCommand(theWindowObject, myNodeID, myHotSpotID, 0L, kVRDoIt_Forever, kQTVRHotSpotLeave, myCmdLine);

		// enlist three ReplaceCursor calls for leaving the node: reinstall original cursor triad
		sprintf(myCmdLine, "ReplaceCursor %d %d %d", kCursID_MouseUpOnUndefHS, 0, kQTVRUseDefaultCursor);
		VRScript_EnlistNodeExitCommand(theWindowObject, myNodeID, kVRAnyNode, kVRDoIt_Forever, (UInt32)0, myCmdLine);
		sprintf(myCmdLine, "ReplaceCursor %d %d %d", kCursID_MouseDownOnUndefHS, 0, kQTVRUseDefaultCursor);
		VRScript_EnlistNodeExitCommand(theWindowObject, myNodeID, kVRAnyNode, kVRDoIt_Forever, (UInt32)0, myCmdLine);
		sprintf(myCmdLine, "ReplaceCursor %d %d %d", kCursID_MouseOverUndefHS, 0, kQTVRUseDefaultCursor);
		VRScript_EnlistNodeExitCommand(theWindowObject, myNodeID, kVRAnyNode, kVRDoIt_Forever, (UInt32)0, myCmdLine);

		return;
	}
	
 	// ***SetHotSpotTypeCursors***
 	// replace the triad of cursors for a hot spot specified by its type
	if (strcmp(myCommand, "SetHotSpotTypeCursors") == 0) {
		char		myHotSpotType[kMaxOSTypeLength];
		SInt32		myCurs1ID, myCurs2ID, myCurs3ID;	// resource IDs of the three replacement cursors
		char		myCmdLine[kMaxCmdLineLength];
		OSType		myType;

		if (theWindowObject == NULL)
			return;

		// read the command paramters	
		sscanf(theCommandLine, "%*s %d %s %d %d %d %d", &myNodeID, myHotSpotType, &myCurs1ID, &myCurs2ID, &myCurs3ID, &myOptions);
		
		// convert the string to an OSType
		myType = VRScript_StringToOSType(myHotSpotType);
		
		// enlist three ReplaceCursor calls for entering the hot spot: install new cursor triad
		sprintf(myCmdLine, "ReplaceCursor %d %d %d", kCursID_MouseUpOnUndefHS, myCurs3ID, kQTVRStdCursorType);
		VRScript_EnlistMouseOverHSCommand(theWindowObject, myNodeID, (UInt32)0, myType, kVRDoIt_Forever, kQTVRHotSpotEnter, myCmdLine);
		sprintf(myCmdLine, "ReplaceCursor %d %d %d", kCursID_MouseDownOnUndefHS, myCurs2ID, kQTVRStdCursorType);
		VRScript_EnlistMouseOverHSCommand(theWindowObject, myNodeID, (UInt32)0, myType, kVRDoIt_Forever, kQTVRHotSpotEnter, myCmdLine);
		sprintf(myCmdLine, "ReplaceCursor %d %d %d", kCursID_MouseOverUndefHS, myCurs1ID, kQTVRStdCursorType);
		VRScript_EnlistMouseOverHSCommand(theWindowObject, myNodeID, (UInt32)0, myType, kVRDoIt_Forever, kQTVRHotSpotEnter, myCmdLine);

		// enlist three ReplaceCursor calls for leaving the hot spot: reinstall original cursor triad
		sprintf(myCmdLine, "ReplaceCursor %d %d %d", kCursID_MouseUpOnUndefHS, 0, kQTVRUseDefaultCursor);
		VRScript_EnlistMouseOverHSCommand(theWindowObject, myNodeID, (UInt32)0, myType, kVRDoIt_Forever, kQTVRHotSpotLeave, myCmdLine);
		sprintf(myCmdLine, "ReplaceCursor %d %d %d", kCursID_MouseDownOnUndefHS, 0, kQTVRUseDefaultCursor);
		VRScript_EnlistMouseOverHSCommand(theWindowObject, myNodeID, (UInt32)0, myType, kVRDoIt_Forever, kQTVRHotSpotLeave, myCmdLine);
		sprintf(myCmdLine, "ReplaceCursor %d %d %d", kCursID_MouseOverUndefHS, 0, kQTVRUseDefaultCursor);
		VRScript_EnlistMouseOverHSCommand(theWindowObject, myNodeID, (UInt32)0, myType, kVRDoIt_Forever, kQTVRHotSpotLeave, myCmdLine);

		// enlist three ReplaceCursor calls for leaving the node: reinstall original cursor triad
		sprintf(myCmdLine, "ReplaceCursor %d %d %d", kCursID_MouseUpOnUndefHS, 0, kQTVRUseDefaultCursor);
		VRScript_EnlistNodeExitCommand(theWindowObject, myNodeID, kVRAnyNode, kVRDoIt_Forever, (UInt32)0, myCmdLine);
		sprintf(myCmdLine, "ReplaceCursor %d %d %d", kCursID_MouseDownOnUndefHS, 0, kQTVRUseDefaultCursor);
		VRScript_EnlistNodeExitCommand(theWindowObject, myNodeID, kVRAnyNode, kVRDoIt_Forever, (UInt32)0, myCmdLine);
		sprintf(myCmdLine, "ReplaceCursor %d %d %d", kCursID_MouseOverUndefHS, 0, kQTVRUseDefaultCursor);
		VRScript_EnlistNodeExitCommand(theWindowObject, myNodeID, kVRAnyNode, kVRDoIt_Forever, (UInt32)0, myCmdLine);

		return;
	}
	
 	// ***GoToNodeID***
 	// go to a node
	if (strcmp(myCommand, "GoToNodeID") == 0) {
	
		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d", &myNodeID, &myOptions);
		QTVRGoToNodeID((**theWindowObject).fInstance, myNodeID);
		QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);
		return;
	}
	
 	// ***ShowDefaultView***
 	// display the default view of the current node
	if (strcmp(myCommand, "ShowDefaultView") == 0) {
	
		if (theWindowObject == NULL)
			return;
			
		QTVRShowDefaultView((**theWindowObject).fInstance);
		QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);
		return;
	}
	
 	// ***OpenResourceFile***
 	// open the specified resource file
	if (strcmp(myCommand, "OpenResourceFile") == 0) {
		char		myPathName[kMaxFileNameLength];

		sscanf(theCommandLine, "%*s %d %s", &myOptions, myPathName);
		VRScript_UnpackString(myPathName);
		VRScript_OpenResourceFile(theWindowObject, myOptions, myPathName);
		return;
	}
	
 	// ***SetCorrection***
 	// set the imaging correction mode
	if (strcmp(myCommand, "SetCorrection") == 0) {
		UInt32		myCorrection;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d", &myCorrection, &myOptions);
		QTVRSetImagingProperty((**theWindowObject).fInstance, (QTVRImagingMode)myOptions, kQTVRImagingCorrection, myCorrection);
		QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);
		return;
	}
	
 	// ***SetQuality***
 	// set the image quality
	if (strcmp(myCommand, "SetQuality") == 0) {
		UInt32		myQuality;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d", &myQuality, &myOptions);
		QTVRSetImagingProperty((**theWindowObject).fInstance, (QTVRImagingMode)myOptions, kQTVRImagingQuality, myQuality);
		QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);
		return;
	}
	
 	// ***SetSwingSpeed***
 	// set the speed of swing transitions
	if (strcmp(myCommand, "SetSwingSpeed") == 0) {
		UInt32		mySpeed;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d", &mySpeed, &myOptions);
		QTVRSetTransitionProperty((**theWindowObject).fInstance, kQTVRTransitionSwing, kQTVRTransitionSpeed, mySpeed);
		return;
	}
	
 	// ***SetSwingDirection***
 	// set the direction of swing transitions
	if (strcmp(myCommand, "SetSwingDirection") == 0) {
		UInt32		myDirection;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d", &myDirection, &myOptions);
		QTVRSetTransitionProperty((**theWindowObject).fInstance, kQTVRTransitionSwing, kQTVRTransitionDirection, myDirection);
		return;
	}
	
 	// ***SetSwingState***
 	// enable or disable swing transitions
	if (strcmp(myCommand, "SetSwingState") == 0) {
		UInt32		myState;
		
		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d", &myState, &myOptions);
		QTVREnableTransition((**theWindowObject).fInstance, kQTVRTransitionSwing, (Boolean)myState);
		return;
	}
	
 	// ***SetPanAngle***
 	// set the pan angle
	if (strcmp(myCommand, "SetPanAngle") == 0) {

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %f %d", &myPanAngle, &myOptions);
		myPanAngle = QTVRUtils_DegreesToRadians(myPanAngle);
		
		if (myOptions == kVRValue_Relative) {
			myTempAngle = QTVRGetPanAngle((**theWindowObject).fInstance);
			myPanAngle += myTempAngle;
		}
		
		QTVRSetPanAngle((**theWindowObject).fInstance, myPanAngle);
		QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);
		return;
	}
	
 	// ***SetTiltAngle***
 	// set the tilt angle
	if (strcmp(myCommand, "SetTiltAngle") == 0) {

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %f %d", &myTiltAngle, &myOptions);
		myTiltAngle = QTVRUtils_DegreesToRadians(myTiltAngle);
		
		if (myOptions == kVRValue_Relative) {
			myTempAngle = QTVRGetTiltAngle((**theWindowObject).fInstance);
			myTiltAngle += myTempAngle;
		}
		
		QTVRSetTiltAngle((**theWindowObject).fInstance, myTiltAngle);
		QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);
		return;
	}
	
 	// ***SetPanTiltZoom***
 	// set the pan, tilt, and zoom angles
	if (strcmp(myCommand, "SetPanTiltZoom") == 0) {
		UInt32		myMode;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %f %f %f %d %d", &myPanAngle, &myTiltAngle, &myFOVAngle, &myMode, &myOptions);
		myPanAngle = QTVRUtils_DegreesToRadians(myPanAngle);
		myTiltAngle = QTVRUtils_DegreesToRadians(myTiltAngle);
		myFOVAngle = QTVRUtils_DegreesToRadians(myFOVAngle);
		
		if (myOptions == kVRValue_Relative) {
			myTempAngle = QTVRGetPanAngle((**theWindowObject).fInstance);
			myPanAngle += myTempAngle;
			myTempAngle = QTVRGetTiltAngle((**theWindowObject).fInstance);
			myTiltAngle += myTempAngle;
			myTempAngle = QTVRGetFieldOfView((**theWindowObject).fInstance);
			myFOVAngle += myTempAngle;
		}
		
		// enable swing transitions, if requested by myMode parameter
		if ((myMode == kVRTransition_Swing) || (myMode == kVRTransition_SwingWait))
			QTVREnableTransition((**theWindowObject).fInstance, kQTVRTransitionSwing, true);
		
		// set the desired FOV, pan, and tilt angles
		QTVRSetFieldOfView((**theWindowObject).fInstance, myFOVAngle);
		QTVRSetPanAngle((**theWindowObject).fInstance, myPanAngle);
		QTVRSetTiltAngle((**theWindowObject).fInstance, myTiltAngle);

		// update the screen
		QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);

		// if a blocking swing is requested, "spin our wheels" until we get to the destination angles
		if (myMode == kVRTransition_SwingWait)
			while ((myPanAngle != QTVRGetPanAngle((**theWindowObject).fInstance)) ||
				   (myTiltAngle != QTVRGetTiltAngle((**theWindowObject).fInstance)) ||
				   (myFOVAngle != QTVRGetFieldOfView((**theWindowObject).fInstance))) {
				DoIdle((**theWindowObject).fWindow); 
				MCIdle((**theWindowObject).fController); 
			}

		// disable swing transitions, if previously enabled
		if ((myMode == kVRTransition_Swing) || (myMode == kVRTransition_SwingWait))
			QTVREnableTransition((**theWindowObject).fInstance, kQTVRTransitionSwing, false);
			
		return;
	}
	
 	// ***SetFieldOfView***
 	// set the field of view
	if (strcmp(myCommand, "SetFieldOfView") == 0) {

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %f %d", &myFOVAngle, &myOptions);
		myFOVAngle = QTVRUtils_DegreesToRadians(myFOVAngle);
		
		if (myOptions == kVRValue_Relative) {
			myTempAngle = QTVRGetFieldOfView((**theWindowObject).fInstance);
			myFOVAngle += myTempAngle;
		}
		
		QTVRSetFieldOfView((**theWindowObject).fInstance, myFOVAngle);
		QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);
		return;
	}
	
 	// ***SetViewCenter***
 	// set the view center of an object node
	if (strcmp(myCommand, "SetViewCenter") == 0) {
		QTVRFloatPoint		myPoint;
		
		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %f %f %d", &myPoint.x, &myPoint.y, &myOptions);
		QTVRSetViewCenter((**theWindowObject).fInstance, &myPoint);
		QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);
		return;
	}
	
 	// ***SetPanLimits***
 	// set the current pan angle constraints
	if (strcmp(myCommand, "SetPanLimits") == 0) {
		float		myMinAngle;
		float		myMaxAngle;

		sscanf(theCommandLine, "%*s %f %f %d", &myMinAngle, &myMaxAngle, &myOptions);
		VRScript_SetAngleConstraints(theWindowObject, kQTVRPan, myMinAngle, myMaxAngle, myOptions);
		return;
	}
	
 	// ***SetTiltLimits***
 	// set the current tilt angle constraints
	if (strcmp(myCommand, "SetTiltLimits") == 0) {
		float		myMinAngle;
		float		myMaxAngle;

		sscanf(theCommandLine, "%*s %f %f %d", &myMinAngle, &myMaxAngle, &myOptions);
		VRScript_SetAngleConstraints(theWindowObject, kQTVRTilt, myMinAngle, myMaxAngle, myOptions);
		return;
	}
	
 	// ***SetZoomLimits***
 	// set the current zoom angle constraints
	if (strcmp(myCommand, "SetZoomLimits") == 0) {
		float		myMinAngle;
		float		myMaxAngle;

		sscanf(theCommandLine, "%*s %f %f %d", &myMinAngle, &myMaxAngle, &myOptions);
		VRScript_SetAngleConstraints(theWindowObject, kQTVRFieldOfView, myMinAngle, myMaxAngle, myOptions);
		return;
	}
	
 	// ***SetHotSpotState***
 	// enable or disable a hot spot
	if (strcmp(myCommand, "SetHotSpotState") == 0) {
		UInt32		myHotSpotID;
		UInt32		myState;

		sscanf(theCommandLine, "%*s %d %d %d", &myHotSpotID, &myState, &myOptions);
		VRScript_SetHotSpotState(theWindowObject, myHotSpotID, myState, myOptions);
		return;
	}
	
 	// ***SetTranslateState***
 	// enable or disable object translation
	if (strcmp(myCommand, "SetTranslateState") == 0) {
		UInt32		myState;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d", &myState, &myOptions);
		QTVRSetInteractionProperty((**theWindowObject).fInstance, kQTVRInteractionTranslateOnMouseDown, (void *)myState);
		return;
	}
	
	// ***SetClickRadius***
 	// set the radius within which clicks occur
	if (strcmp(myCommand, "SetClickRadius") == 0) {
		UInt16		myRadius;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d", &myRadius, &myOptions);
		QTVRSetInteractionProperty((**theWindowObject).fInstance, kQTVRInteractionMouseClickHysteresis, (void *)myRadius);
		return;
	}
	
	// ***SetClickTimeout***
 	// set the timeout for clicks
	if (strcmp(myCommand, "SetClickTimeout") == 0) {
		UInt32		myTicks;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d", &myTicks, &myOptions);
		QTVRSetInteractionProperty((**theWindowObject).fInstance, kQTVRInteractionMouseClickTimeout, (void *)myTicks);
		return;
	}
	
	// ***SetPanTiltSpeed***
 	// set the pan and tilt speed
	if (strcmp(myCommand, "SetPanTiltSpeed") == 0) {
		UInt32		mySpeed;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d", &mySpeed, &myOptions);
		QTVRSetInteractionProperty((**theWindowObject).fInstance, kQTVRInteractionPanTiltSpeed, (void *)mySpeed);
		return;
	}
	
	// ***SetZoomSpeed***
 	// set the zoom speed
	if (strcmp(myCommand, "SetZoomSpeed") == 0) {
		UInt32		mySpeed;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d", &mySpeed, &myOptions);
		QTVRSetInteractionProperty((**theWindowObject).fInstance, kQTVRInteractionZoomSpeed, (void *)mySpeed);
		return;
	}
	
	// ***SetMouseScale***
 	// set the mouse-motion scale
	if (strcmp(myCommand, "SetMouseScale") == 0) {
		float		myScale;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %f %d", &myScale, &myOptions);
		myScale = QTVRUtils_DegreesToRadians(myScale);
		QTVRSetInteractionProperty((**theWindowObject).fInstance, kQTVRInteractionMouseMotionScale, &myScale);
		return;
	}
	
 	// ***SetFrameRate***
 	// set the frame rate of an object node
	if (strcmp(myCommand, "SetFrameRate") == 0) {
		float		myRate;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %f %d", &myRate, &myOptions);

		if (myOptions == kVRValue_Relative)
			myRate += QTVRGetFrameRate((**theWindowObject).fInstance);
		
		QTVRSetFrameRate((**theWindowObject).fInstance, myRate);
		return;
	}
	
 	// ***SetViewRate***
 	// set the view rate of an object node
	if (strcmp(myCommand, "SetViewRate") == 0) {
		float		myRate;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %f %d", &myRate, &myOptions);
		
		if (myOptions == kVRValue_Relative)
			myRate += QTVRGetViewRate((**theWindowObject).fInstance);
		
		QTVRSetViewRate((**theWindowObject).fInstance, myRate);
		return;
	}
	
 	// ***SetViewTime***
 	// set the current view time of an object node
	if (strcmp(myCommand, "SetViewTime") == 0) {
		TimeValue	myTime;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %f %d", &myTime, &myOptions);
		
		if (myOptions == kVRValue_Relative)
			myTime += QTVRGetViewCurrentTime((**theWindowObject).fInstance);
		
		QTVRSetViewCurrentTime((**theWindowObject).fInstance, myTime);
		return;
	}
	
 	// ***SetViewState***
 	// set the current view state of an object node
	if (strcmp(myCommand, "SetViewState") == 0) {
		UInt32		myType;
		UInt32		myState;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d %d", &myType, &myState, &myOptions);
		QTVRSetViewState((**theWindowObject).fInstance, (QTVRViewStateType)myType, (UInt16)myState);
		return;
	}
	
 	// ***SetAnimationState***
 	// set the animation state of an object node
	if (strcmp(myCommand, "SetAnimationState") == 0) {
		UInt32		mySetting;
		UInt32		myState;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d %d", &mySetting, &myState, &myOptions);
		QTVRSetAnimationSetting((**theWindowObject).fInstance, (QTVRObjectAnimationSetting)mySetting, (Boolean)myState);
		return;
	}
	
 	// ***SetControlState***
 	// set the control state of an object node
	if (strcmp(myCommand, "SetControlState") == 0) {
		UInt32		mySetting;
		UInt32		myState;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d %d", &mySetting, &myState, &myOptions);
		QTVRSetControlSetting((**theWindowObject).fInstance, (QTVRControlSetting)mySetting, (Boolean)myState);
		return;
	}
	
 	// ***SetFrameAnimState***
 	// enable or disable frame animation in an object node
	if (strcmp(myCommand, "SetFrameAnimState") == 0) {
		UInt32		myState;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d", &myState, &myOptions);
		QTVREnableFrameAnimation((**theWindowObject).fInstance, (Boolean)myState);
		return;
	}
	
 	// ***SetViewAnimState***
 	// enable or disable view animation in an object node
	if (strcmp(myCommand, "SetViewAnimState") == 0) {
		UInt32		myState;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d", &myState, &myOptions);
		QTVREnableViewAnimation((**theWindowObject).fInstance, (Boolean)myState);
		return;
	}
	
 	// ***SetQTVRVisState***
 	// enable or disable QTVR movie visibility
	if (strcmp(myCommand, "SetQTVRVisState") == 0) {
		UInt32		myState;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d", &myState, &myOptions);
		QTVRSetVisible((**theWindowObject).fInstance, (Boolean)myState);
		return;
	}
	
	// ***SetCachePrefs***
 	// set the back buffer resolution, depth, and size
	if (strcmp(myCommand, "SetCachePrefs") == 0) {
		SInt32		myResolution;
		SInt32		myDepth;
		SInt32		mySize;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d %d %d", &myResolution, &myDepth, &mySize, &myOptions);
		QTVRSetBackBufferPrefs((**theWindowObject).fInstance, kQTVRUseMovieGeometry, (UInt16)myResolution, (SInt16)myDepth, (SInt16)mySize);
		return;
	}
	
 	// ***SetMovieVolume***
 	// set the volume of a QTVR sound track
	if (strcmp(myCommand, "SetMovieVolume") == 0) {
		UInt32		myVolume;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d", &myVolume, &myOptions);
		SetMovieVolume((**theWindowObject).fMovie, (short)myVolume);
		return;
	}
	
 	// ***SetSoundVolume***
 	// set the volume of a sound
	if (strcmp(myCommand, "SetSoundVolume") == 0) {
		UInt32				myEntryID;
		UInt32				myVolume;
		VRScriptSoundPtr	mySoundPtr;
		VRScriptMoviePtr	myMoviePtr;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d %d", &myEntryID, &myVolume, &myOptions);
		
		// is it a sound resource?
		mySoundPtr = (VRScriptSoundPtr)VRScript_GetObjectByEntryID(theWindowObject, kVREntry_Sound, myEntryID);
		if (mySoundPtr != NULL) {
			VRSound_SetVolume(mySoundPtr->fChannel, (unsigned short)myVolume, (unsigned short)myVolume);
			return;
		}
		
		// is it a movie sound track?
		myMoviePtr = (VRScriptMoviePtr)VRScript_GetObjectByEntryID(theWindowObject, kVREntry_QTMovie, myEntryID);
		if (myMoviePtr != NULL) {
			SetMovieVolume(myMoviePtr->fMovie, (short)myVolume);
		}

		return;
	}
	
 	// ***SetSoundBalance***
 	// set the balance of a sound
	if (strcmp(myCommand, "SetSoundBalance") == 0) {
		UInt32				myEntryID;
		UInt32				myLeftPct;
		UInt32				myRightPct;
		VRScriptSoundPtr	mySoundPtr;
		VRScriptMoviePtr	myMoviePtr;
		unsigned short		myLeftVol, myRightVol;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d %d %d", &myEntryID, &myLeftPct, &myRightPct, &myOptions);
		
		// is it a sound resource?
		mySoundPtr = (VRScriptSoundPtr)VRScript_GetObjectByEntryID(theWindowObject, kVREntry_Sound, myEntryID);
		if (mySoundPtr != NULL) {
			VRSound_GetVolume(mySoundPtr->fChannel, &myLeftVol, &myRightVol);
			VRSound_SetVolume(mySoundPtr->fChannel, myLeftVol * (myLeftPct / 100), myRightVol * (myRightPct / 100));
			return;
		}
		
		// is it a movie sound track?
		myMoviePtr = (VRScriptMoviePtr)VRScript_GetObjectByEntryID(theWindowObject, kVREntry_QTMovie, myEntryID);
		if (myMoviePtr != NULL) {
			short			myValue;
			
			myValue = (((float)myRightPct / 100) * kQTMaxSoundVolume) - (((float)myLeftPct / 100) * kQTMaxSoundVolume);
			MediaSetSoundBalance(myMoviePtr->fMediaHandler, myValue);
		}
		
		return;
	}
	
 	// ***PlaySceneSound***
	// play a movie-wide ambient sound asynchronously
	if (strcmp(myCommand, "PlaySceneSound") == 0) {
		UInt32		myEntryID;
 		UInt32		myMode;
 		UInt32		myFade;

		sscanf(theCommandLine, "%*s %d %d %d %d %d", &myResID, &myEntryID, &myMode, &myFade, &myOptions);
		VRSound_PlaySound(theWindowObject, kVRAnyNode, myResID, myEntryID, 1.0, 1.0, 1.0, 1.0, kSSpSourceMode_Ambient, myMode, myFade, myOptions);
		return;
	}
	
 	// ***PlaySceneQTMidi***
	// play a movie-wide ambient QuickTime sound-only file asynchronously
	if (strcmp(myCommand, "PlaySceneQTMidi") == 0) {
		UInt32		myEntryID;
		UInt32		myMode;
 		UInt32		myFade;
 		UInt32		myIsLocal;
		float		myX, myY, myZ;
		float		myProjAngle;
		char		myPathName[kMaxFileNameLength];
	
		sscanf(theCommandLine, "%*s %d %d %f %f %f %f %d %d %d %s", &myEntryID, &myIsLocal, &myX, &myY, &myZ, &myProjAngle, &myMode, &myFade, &myOptions, myPathName);
		myProjAngle = QTVRUtils_DegreesToRadians(myProjAngle);
		VRMoov_PlayMovie(theWindowObject, kVRAnyNode, myEntryID, QTVRUtils_Point3DToPanAngle(myX, myY, myZ), QTVRUtils_Point3DToTiltAngle(myX, myY, myZ), 0.0, 0.0, 0.0, 0.0, 0.0, false, false, false, false, myIsLocal, myIsLocal, myProjAngle, myMode, myOptions, myPathName);
		return;
	}
	
 	// ***PlayNodeQTMidi***
 	// play a QuickTime MIDI file in a node
 	// [just add a PlayQTMidi command to the node-entering command list]
	if (strcmp(myCommand, "PlayNodeQTMidi") == 0) {
		UInt32		myEntryID;
		UInt32		myMode;
		UInt32		myFade;
		SInt32		myMaxTimes;
 		UInt32		myIsLocal;
		float		myX, myY, myZ;
		float		myProjAngle;
		char		myPathName[kMaxFileNameLength];
		char		myCmdLine[kMaxCmdLineLength];
		
		sscanf(theCommandLine, "%*s %d %d %d %d %f %f %f %f %d %d %d %s", &myNodeID, &myEntryID, &myMaxTimes, &myIsLocal, &myX, &myY, &myZ, &myProjAngle, &myMode, &myFade, &myOptions, myPathName);
		sprintf(myCmdLine, "PlayQTMidi %d %d %f %f %f %f %d %d %d %s", myEntryID, myIsLocal, myX, myY, myZ, myProjAngle, myMode, myFade, myOptions, myPathName);
		VRScript_EnlistNodeEntryCommand(theWindowObject, myNodeID, myMaxTimes, 0, myCmdLine);
		return;
	}
	
	// ***PlayNodeSound***
 	// play a sound in a node
 	// [just add a PlaySndResource command to the node-entering command list]
	if (strcmp(myCommand, "PlayNodeSound") == 0) {
		UInt32		myEntryID;
		UInt32		myMode;
		UInt32		myFade;
		SInt32		myMaxTimes;
		char		myCmdLine[kMaxCmdLineLength];

		sscanf(theCommandLine, "%*s %d %d %d %d %d %d", &myResID, &myNodeID, &myEntryID, &myMaxTimes, &myMode, &myFade, &myOptions);
		sprintf(myCmdLine, "PlaySndResource %d %d %d %d %d", myResID, myEntryID, myMode, myFade, myOptions);
		VRScript_EnlistNodeEntryCommand(theWindowObject, myNodeID, myMaxTimes, 0, myCmdLine);
		return;
	}
	
 	// ***PlayNode3DSound***
 	// play a 3D sound in a node
 	// [just add a Play3DSndResource command to the node-entering command list]
	if (strcmp(myCommand, "PlayNode3DSound") == 0) {
		UInt32		myEntryID;
		float		myX, myY, myZ;
		float		myProjAngle;
		UInt32		mySourceMode;
		UInt32		myMode;
		UInt32		myFade;
		SInt32		myMaxTimes;
		char		myCmdLine[kMaxCmdLineLength];

		sscanf(theCommandLine, "%*s %d %d %d %f %f %f %f %d %d %d %d %d", &myResID, &myNodeID, &myEntryID, &myX, &myY, &myZ, &myProjAngle, &mySourceMode, &myMaxTimes, &myMode, &myFade, &myOptions);
		sprintf(myCmdLine, "Play3DSndResource %d %d %f %f %f %f %d %d %d %d", myResID, myEntryID, myX, myY, myZ, myProjAngle, mySourceMode, myMode, myFade, myOptions);
		VRScript_EnlistNodeEntryCommand(theWindowObject, myNodeID, myMaxTimes, 0, myCmdLine);
		return;
	}
	
 	// ***HotSpotQTMidi***
 	// play a QuickTime MIDI file when a hot spot is clicked
 	// [just add a PlayQTMidi command to the hot spot command list]
	if (strcmp(myCommand, "HotSpotQTMidi") == 0) {
		UInt32		myHotSpotID;
		UInt32		myEntryID;
		UInt32		myMode;
		UInt32		myFade;
		SInt32		myMaxTimes;
 		UInt32		myIsLocal;
		float		myX, myY, myZ;
		float		myProjAngle;
		char		myPathName[kMaxFileNameLength];
		char		myCmdLine[kMaxCmdLineLength];
		
	
		sscanf(theCommandLine, "%*s %d %d %d %d %d %f %f %f %f %d %d %d %s", &myNodeID, &myHotSpotID, &myEntryID, &myMaxTimes, &myIsLocal, &myX, &myY, &myZ, &myProjAngle, &myMode, &myFade, &myOptions, myPathName);
		sprintf(myCmdLine, "PlayQTMidi %d %d %f %f %f %f %d %d %d %s", myEntryID, myIsLocal, myX, myY, myZ, myProjAngle, myMode, myFade, myOptions, myPathName);
		VRScript_EnlistClickHSCommand(theWindowObject, myNodeID, myHotSpotID, myMaxTimes, 0, myCmdLine);
		return;
	}
	
 	// ***HotSpotSound***
 	// play a sound when a hot spot is clicked
 	// [just add a PlaySndResource command to the hot spot command list]
	if (strcmp(myCommand, "HotSpotSound") == 0) {
		UInt32		myHotSpotID;
		UInt32		myEntryID;
		UInt32		myMode;
		UInt32		myFade;
		SInt32		myMaxTimes;
		char		myCmdLine[kMaxCmdLineLength];

		sscanf(theCommandLine, "%*s %d %d %d %d %d %d %d %d", &myResID, &myNodeID, &myHotSpotID, &myEntryID, &myMaxTimes, &myMode, &myFade, &myOptions);
		sprintf(myCmdLine, "PlaySndResource %d %d %d %d %d", myResID, myEntryID, myMode, myFade, myOptions);
		VRScript_EnlistClickHSCommand(theWindowObject, myNodeID, myHotSpotID, myMaxTimes, 0, myCmdLine);
		return;
	}
	
 	// ***HotSpot3DSound***
 	// play a 3D sound when a hot spot is clicked
 	// [just add a Play3DSndResource command to the hot spot command list]
	if (strcmp(myCommand, "HotSpot3DSound") == 0) {
		UInt32		myHotSpotID;
		UInt32		myEntryID;
		float		myX, myY, myZ;
		float		myProjAngle;
		UInt32		mySourceMode;
		UInt32		myMode;
		UInt32		myFade;
		SInt32		myMaxTimes;
		char		myCmdLine[kMaxCmdLineLength];

		sscanf(theCommandLine, "%*s %d %d %d %d %f %f %f %f %d %d %d %d %d", &myResID, &myNodeID, &myHotSpotID, &myEntryID, &myX, &myY, &myZ, &myProjAngle, &mySourceMode, &myMaxTimes, &myMode, &myFade, &myOptions);
		sprintf(myCmdLine, "Play3DSndResource %d %d %f %f %f %f %d %d %d %d", myResID, myEntryID, myX, myY, myZ, myProjAngle, mySourceMode, myMode, myFade, myOptions);
		VRScript_EnlistClickHSCommand(theWindowObject, myNodeID, myHotSpotID, myMaxTimes, 0, myCmdLine);
		return;
	}
	
 	// ***HotSpotMovie***
 	// play a movie when a hot spot is clicked
 	// [just add a PlayMovie command to the hot spot command list]
	if (strcmp(myCommand, "HotSpotMovie") == 0) {
		UInt32		myHotSpotID;
		UInt32		myEntryID;
		char		myCmdLine[kMaxCmdLineLength];
 		float		myScale;
 		float		myWidth;
 		UInt32		myKeyRed;
 		UInt32		myKeyGreen;
 		UInt32		myKeyBlue;
		UInt32		myUseBuffer;
		UInt32		myUseCenter;
		UInt32		myUseKey;
		UInt32		myUseHide;
		UInt32		myUseDir;
		UInt32		myUseVol;
 		float		myVolAngle;
		SInt32		myMaxTimes;
		UInt32		myMode;
		char		myPathName[kMaxFileNameLength];

		sscanf(theCommandLine, "%*s %d %d %d %f %f %f %f %d %d %d %d %d %d %d %d %d %f %d %d %d %s", 
				&myEntryID, 
				&myNodeID, 
				&myHotSpotID, 
				&myPanAngle,
				&myTiltAngle,
				&myScale, 
				&myWidth, 
				&myKeyRed, 
				&myKeyGreen, 
				&myKeyBlue, 
				&myUseBuffer,
				&myUseCenter,
				&myUseKey,
				&myUseHide,
				&myUseDir,
				&myUseVol,
				&myVolAngle,
				&myMaxTimes,
				&myMode,
				&myOptions,
				myPathName);
			
		sprintf(myCmdLine, "PlayMovie %d %f %f %f %f %d %d %d %d %d %d %d %d %d %f %d %d %s",
				myEntryID,
				myPanAngle,
				myTiltAngle,
				myScale, 
				myWidth, 
				myKeyRed, 
				myKeyGreen, 
				myKeyBlue, 
				(Boolean)myUseBuffer,
				(Boolean)myUseCenter,
				(Boolean)myUseKey,
				(Boolean)myUseHide,
				(Boolean)myUseDir,
				(Boolean)myUseVol,
				myVolAngle,
				myMode,
				myOptions,
				myPathName);

		VRScript_EnlistClickHSCommand(theWindowObject, myNodeID, myHotSpotID, myMaxTimes, 0, myCmdLine);
		return;
	}
	
 	// ***TriggerHotSpot***
 	// trigger a particular hot spot
	if (strcmp(myCommand, "TriggerHotSpot") == 0) {
		UInt32		myHotSpotID;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d", &myHotSpotID, &myOptions);
		QTVRTriggerHotSpot((**theWindowObject).fInstance, myHotSpotID, 0, 0);
		return;
	}
	
 	// ***PlayQTMidi***
	// play a QuickTime sound-only file asynchronously, or stop a file from playing
	if (strcmp(myCommand, "PlayQTMidi") == 0) {
		UInt32		myEntryID;
		UInt32		myMode;
 		UInt32		myFade;
 		UInt32		myIsLocal;
		float		myX, myY, myZ;
		float		myProjAngle;
		char		myPathName[kMaxFileNameLength];
	
		if (theWindowObject == NULL)
			return;
			
		myNodeID = QTVRGetCurrentNodeID((**theWindowObject).fInstance);

		sscanf(theCommandLine, "%*s %d %d %f %f %f %f %d %d %d %s", &myEntryID, &myIsLocal, &myX, &myY, &myZ, &myProjAngle, &myMode, &myFade, &myOptions, myPathName);
		myProjAngle = QTVRUtils_DegreesToRadians(myProjAngle);
		VRMoov_PlayMovie(theWindowObject, myNodeID, myEntryID, QTVRUtils_Point3DToPanAngle(myX, myY, myZ), QTVRUtils_Point3DToTiltAngle(myX, myY, myZ), 0.0, 0.0, 0.0, 0.0, 0.0, false, false, false, false, myIsLocal, myIsLocal, myProjAngle, myMode, myOptions, myPathName);
		return;
	}
	
 	// ***PlaySndResource***
	// play a sound resource ambiently, or stop an ambient sound resource from playing
	if ((strcmp(myCommand, "PlaySndResource") == 0) || (strcmp(myCommand, "PlaySound") == 0)) {
		UInt32		myEntryID;
 		UInt32		myMode;
 		UInt32		myFade;

		if (theWindowObject == NULL)
			return;
		
		myNodeID = QTVRGetCurrentNodeID((**theWindowObject).fInstance);
		
		sscanf(theCommandLine, "%*s %d %d %d %d %d", &myResID, &myEntryID, &myMode, &myFade, &myOptions);
		VRSound_PlaySound(theWindowObject, myNodeID, myResID, myEntryID, 1.0, 1.0, 1.0, 1.0, kSSpSourceMode_Ambient, myMode, myFade, myOptions);
		return;
	}
	
 	// ***PlaySoundFile***
	// play a sound file ambiently, or stop an ambient sound file from playing
	if (strcmp(myCommand, "PlaySoundFile") == 0) {
		UInt32		myEntryID;
 		UInt32		myMode;
 		UInt32		myFade;
		short		myRefNum;
		short		myResID;
		char		myPathName[kMaxFileNameLength];

		if (theWindowObject == NULL)
			return;
		
		myNodeID = QTVRGetCurrentNodeID((**theWindowObject).fInstance);
		
		sscanf(theCommandLine, "%*s %d %d %d %d %s", &myEntryID, &myMode, &myFade, &myOptions, myPathName);
		
		// a sound file (of type 'sfil') is a resource file that contains one 'snd ' resource,
		// so open the resource file and get the resource ID; then call VRSound_PlaySound.
		myRefNum = VRScript_OpenResourceFile(theWindowObject, 0, myPathName);
		myResID = VRSound_GetSndResourceID(myRefNum);
		VRSound_PlaySound(theWindowObject, myNodeID, myResID, myEntryID, 1.0, 1.0, 1.0, 1.0, kSSpSourceMode_Ambient, myMode, myFade, myOptions);
		return;
	}
	
 	// ***Play3DSndResource***
	// play a sound file localized, or stop an ambient sound file from playing
	if (strcmp(myCommand, "Play3DSndResource") == 0) {
		float		myX, myY, myZ;
		float		myProjAngle;
		UInt32		mySourceMode;
		UInt32		myEntryID;
 		UInt32		myMode;
		UInt32		myFade;

		if (theWindowObject == NULL)
			return;

		myNodeID = QTVRGetCurrentNodeID((**theWindowObject).fInstance);
		
		sscanf(theCommandLine, "%*s %d %d %f %f %f %f %d %d %d %d", &myResID, &myEntryID, &myX, &myY, &myZ, &myProjAngle, &mySourceMode, &myMode, &myFade, &myOptions);
		myProjAngle = QTVRUtils_DegreesToRadians(myProjAngle);
		VRSound_PlaySound(theWindowObject, myNodeID, myResID, myEntryID, myX, myY, myZ, myProjAngle, mySourceMode, myMode, myFade, myOptions);
		return;
	}
	
 	// ***Play3DSndResourceAngle***
 	// play a localized sound, specified using angles
	if (strcmp(myCommand, "Play3DSndResourceAngle") == 0) {
		TQ3Point3D	myPoint;
		float		myDistance;
		float		myProjAngle;
		UInt32		mySourceMode;
		UInt32		myEntryID;
		UInt32		myMode;
		UInt32		myFade;
	
		if (theWindowObject == NULL)
			return;

		myNodeID = QTVRGetCurrentNodeID((**theWindowObject).fInstance);
		
		sscanf(theCommandLine, "%*s %d %d %f %f %f %f %d %d %d", &myResID, &myEntryID, &myPanAngle, &myTiltAngle, &myDistance, &myProjAngle, &mySourceMode, &myMode, &myFade, &myOptions);
		myPanAngle = QTVRUtils_DegreesToRadians(myPanAngle);
		myTiltAngle = QTVRUtils_DegreesToRadians(myTiltAngle);
		myProjAngle = QTVRUtils_DegreesToRadians(myProjAngle);

		myPoint.x = -sin(myPanAngle) * cos(myTiltAngle) * (myDistance);
		myPoint.y = sin(myTiltAngle) * (myDistance);
		myPoint.z = -cos(myPanAngle) * cos(myTiltAngle) * (myDistance);

		VRSound_PlaySound(theWindowObject, myNodeID, myResID, myEntryID, myPoint.x, myPoint.y, myPoint.z, myProjAngle, mySourceMode, myMode, myFade, myOptions);
		return;
	}
	
 	// ***ShowPicture***
 	// overlay a picture (in the front buffer)
	if (strcmp(myCommand, "ShowPicture") == 0) {
		UInt32		myEntryID;
		UInt32		myHeight;
		UInt32		myWidth;
		UInt32		myPegSides;
		UInt32		myOffset;

		sscanf(theCommandLine, "%*s %d %d %d %d %d %d %d", &myResID, &myEntryID, &myHeight, &myWidth, &myPegSides, &myOffset, &myOptions);
		VRPicture_ShowPicture(theWindowObject, myResID, myEntryID, myHeight, myWidth, myPegSides, myOffset, myOptions);
		QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);
		return;
	}
	
 	// ***ShowNodePicture***
 	// overlay a picture (in the front buffer) in a particular node
	if (strcmp(myCommand, "ShowNodePicture") == 0) {
		UInt32		myEntryID;
		UInt32		myHeight;
		UInt32		myWidth;
		UInt32		myPegSides;
		UInt32		myOffset;
		char		myCmdLine[kMaxCmdLineLength];

		sscanf(theCommandLine, "%*s %d %d %d %d %d %d %d %d", &myResID, &myEntryID, &myNodeID, &myHeight, &myWidth, &myPegSides, &myOffset, &myOptions);
		sprintf(myCmdLine, "ShowPicture %d %d %d %d %d %d %d", myResID, myEntryID, myHeight, myWidth, myPegSides, myOffset, myOptions);
		VRScript_EnlistNodeEntryCommand(theWindowObject, myNodeID, kVRDoIt_Forever, 0, myCmdLine);
		return;
	}
	
 	// ***AtTime***
 	// execute a command at a specific time
 	if (strcmp(myCommand, "AtTime") == 0) {
		UInt32		myTicks;
		UInt32		myMode;
		UInt32		myRepeat;
		UInt32		myPeriod;
		SInt32		myMaxTimes;
		char		myCmdLine[kMaxCmdLineLength];
		
		sscanf(theCommandLine, "%*s %d %d %d %d %d %d %d %s", &myTicks, &myMode, &myNodeID, &myRepeat, &myPeriod, &myMaxTimes, &myOptions, myCmdLine);
		VRScript_UnpackString(myCmdLine);
		VRScript_EnlistTimedCommand(theWindowObject, myTicks, myMode, myNodeID, myRepeat, myPeriod, myMaxTimes, myOptions, myCmdLine);
		return;
	}
	
 	// ***AtAppLaunch***
 	// execute a command when the application is launched
 	// (this is essentially trivial...)
 	if (strcmp(myCommand, "AtAppLaunch") == 0) {
		char		myCmdLine[kMaxCmdLineLength];
		
		sscanf(theCommandLine, "%*s %d %s", &myOptions, myCmdLine);
		VRScript_UnpackString(myCmdLine);
		VRScript_ProcessScriptCommandLine(theWindowObject, myCmdLine);
		return;
	}
	
 	// ***AtAppQuit***
 	// execute a command when the application is quit
 	if (strcmp(myCommand, "AtAppQuit") == 0) {
		char		myCmdLine[kMaxCmdLineLength];
		
		sscanf(theCommandLine, "%*s %d %s", &myOptions, myCmdLine);
		VRScript_UnpackString(myCmdLine);
		VRScript_EnlistQuitCommand(theWindowObject, myOptions, myCmdLine);
		return;
	}
	
 	// ***AtMouseOverHSID***
 	// execute a command when the mouse is over a hot spot, targeted by ID
 	if (strcmp(myCommand, "AtMouseOverHSID") == 0) {
		UInt32		myHotSpotID;
		SInt32		myMaxTimes;
		char		myCmdLine[kMaxCmdLineLength];
		
		sscanf(theCommandLine, "%*s %d %d %d %d %s", &myNodeID, &myHotSpotID, &myMaxTimes, &myOptions, myCmdLine);
		VRScript_UnpackString(myCmdLine);
		VRScript_EnlistMouseOverHSCommand(theWindowObject, myNodeID, myHotSpotID, (OSType)0, myMaxTimes, myOptions, myCmdLine);
		return;
	}
	
 	// ***AtMouseOverHSType***
 	// execute a command when the mouse is over a hot spot, targeted by type
 	if (strcmp(myCommand, "AtMouseOverHSType") == 0) {
		SInt32		myMaxTimes;
		char		myCmdLine[kMaxCmdLineLength];
		OSType		myType;
		char		myHotSpotType[kMaxOSTypeLength];
				
		if (theWindowObject == NULL)
			return;

		sscanf(theCommandLine, "%*s %d %s %d %d %s", &myNodeID, &myHotSpotType, &myMaxTimes, &myOptions, myCmdLine);
		myType = VRScript_StringToOSType(myHotSpotType);
		
		VRScript_UnpackString(myCmdLine);
		VRScript_EnlistMouseOverHSCommand(theWindowObject, myNodeID, (UInt32)0, myType, myMaxTimes, myOptions, myCmdLine);
		return;
	}
	
 	// ***AtClickHS***
 	// execute a command when the mouse is clicked on a hot spot
 	if (strcmp(myCommand, "AtClickHS") == 0) {
		UInt32		myHotSpotID;
		SInt32		myMaxTimes;
		char		myCmdLine[kMaxCmdLineLength];
		
		sscanf(theCommandLine, "%*s %d %d %d %d %s", &myNodeID, &myHotSpotID, &myMaxTimes, &myOptions, myCmdLine);
		VRScript_UnpackString(myCmdLine);
		VRScript_EnlistClickHSCommand(theWindowObject, myNodeID, myHotSpotID, myMaxTimes, myOptions, myCmdLine);
		return;
	}
	
 	// ***AtNodeEntry***
 	// execute a command when the specified node is entered
 	if (strcmp(myCommand, "AtNodeEntry") == 0) {
		SInt32		myMaxTimes;
		char		myCmdLine[kMaxCmdLineLength];
		
		sscanf(theCommandLine, "%*s %d %d %d %s", &myNodeID, &myMaxTimes, &myOptions, myCmdLine);
		VRScript_UnpackString(myCmdLine);
		VRScript_EnlistNodeEntryCommand(theWindowObject, myNodeID, myMaxTimes, myOptions, myCmdLine);
		return;
	}
	
 	// ***AtNodeExit***
 	// execute a command when the specified node is exited
 	if (strcmp(myCommand, "AtNodeExit") == 0) {
		UInt32		myFromNodeID;
		UInt32		myToNodeID;
		SInt32		myMaxTimes;
		char		myCmdLine[kMaxCmdLineLength];
		
		sscanf(theCommandLine, "%*s %d %d %d %d %s", &myFromNodeID, &myToNodeID, &myMaxTimes, &myOptions, myCmdLine);
		VRScript_UnpackString(myCmdLine);
		VRScript_EnlistNodeExitCommand(theWindowObject, myFromNodeID, myToNodeID, myMaxTimes, myOptions, myCmdLine);
		return;
	}
	
 	// ***AtPanAngle***
 	// execute a command at the specified pan angle
 	if (strcmp(myCommand, "AtPanAngle") == 0) {
		float		myMinAngle;
		float		myMaxAngle;
		SInt32		myMaxTimes;
		char		myCmdLine[kMaxCmdLineLength];

		sscanf(theCommandLine, "%*s %d %f %f %d %d %s", &myNodeID, &myMinAngle, &myMaxAngle, &myMaxTimes, &myOptions, myCmdLine);
		VRScript_EnlistAngleCommand(theWindowObject, kVREntry_PanAngleCmd, myNodeID, myMinAngle, myMaxAngle, myMaxTimes, myOptions, myCmdLine);
		return;
	}
	
 	// ***AtTiltAngle***
 	// execute a command at the specified tilt angle
 	if (strcmp(myCommand, "AtTiltAngle") == 0) {
		float		myMinAngle;
		float		myMaxAngle;
		SInt32		myMaxTimes;
		char		myCmdLine[kMaxCmdLineLength];

		sscanf(theCommandLine, "%*s %d %f %f %d %d %s", &myNodeID, &myMinAngle, &myMaxAngle, &myMaxTimes, &myOptions, myCmdLine);
		VRScript_EnlistAngleCommand(theWindowObject, kVREntry_TiltAngleCmd, myNodeID, myMinAngle, myMaxAngle, myMaxTimes, myOptions, myCmdLine);
		return;
	}
	
 	// ***AtZoomAngle***
 	// execute a command at the specified zoom angle
 	if (strcmp(myCommand, "AtZoomAngle") == 0) {
		float		myMinAngle;
		float		myMaxAngle;
		SInt32		myMaxTimes;
		char		myCmdLine[kMaxCmdLineLength];

		sscanf(theCommandLine, "%*s %d %f %f %d %d %s", &myNodeID, &myMinAngle, &myMaxAngle, &myMaxTimes, &myOptions, myCmdLine);
		VRScript_EnlistAngleCommand(theWindowObject, kVREntry_FOVAngleCmd, myNodeID, myMinAngle, myMaxAngle, myMaxTimes, myOptions, myCmdLine);
		return;
	}
	
 	// ***DoBoth***
 	// execute both of the specified commands
 	if (strcmp(myCommand, "DoBoth") == 0) {
		char		myCmdLine1[kMaxCmdLineLength];
		char		myCmdLine2[kMaxCmdLineLength];

		sscanf(theCommandLine, "%*s %d %s %s", &myOptions, myCmdLine1, myCmdLine2);
		VRScript_UnpackString(myCmdLine1);
		VRScript_UnpackString(myCmdLine2);
		VRScript_ProcessScriptCommandLine(theWindowObject, myCmdLine1);
		VRScript_ProcessScriptCommandLine(theWindowObject, myCmdLine2);
		return;
	}
	
 	// ***DoNothing***
 	// don't do anything; this can be useful to trigger some side effects of an "At" command
 	if (strcmp(myCommand, "DoNothing") == 0) {
		return;
	}
	
 	// ***PlayMovie***
 	// play a QuickTime movie
 	if (strcmp(myCommand, "PlayMovie") == 0) {
 		float		myScale;
 		float		myWidth;
		UInt32		myEntryID;
 		UInt32		myKeyRed;
 		UInt32		myKeyGreen;
 		UInt32		myKeyBlue;
		UInt32		myUseBuffer;
		UInt32		myUseCenter;
		UInt32		myUseKey;
		UInt32		myUseHide;
		UInt32		myUseDir;
		UInt32		myUseVol;
		UInt32		myMode;
 		float		myVolAngle;
		char		myPathName[kMaxFileNameLength];
		
		sscanf(theCommandLine, "%*s %d %f %f %f %f %d %d %d %d %d %d %d %d %d %f %d %d %s", 
				&myEntryID,
				&myPanAngle,
				&myTiltAngle,
				&myScale, 
				&myWidth, 
				&myKeyRed, 
				&myKeyGreen, 
				&myKeyBlue, 
				&myUseBuffer,
				&myUseCenter,
				&myUseKey,
				&myUseHide,
				&myUseDir,
				&myUseVol,
				&myVolAngle,
				&myMode,
				&myOptions,
				myPathName);
			
		myPanAngle = QTVRUtils_DegreesToRadians(myPanAngle);
		myTiltAngle = QTVRUtils_DegreesToRadians(myTiltAngle);
		myVolAngle = QTVRUtils_DegreesToRadians(myVolAngle);
		myWidth = QTVRUtils_DegreesToRadians(myWidth);

		if (theWindowObject == NULL)
			return;

		myNodeID = QTVRGetCurrentNodeID((**theWindowObject).fInstance);

		VRScript_UnpackString(myPathName);
		VRMoov_PlayMovie(theWindowObject,
				myNodeID,
				myEntryID,
				myPanAngle,
				myTiltAngle,
				myScale, 
				myWidth, 
				myKeyRed, 
				myKeyGreen, 
				myKeyBlue, 
				(Boolean)myUseBuffer,
				(Boolean)myUseCenter,
				(Boolean)myUseKey,
				(Boolean)myUseHide,
				(Boolean)myUseDir,
				(Boolean)myUseVol,
				myVolAngle,
				myMode,
				myOptions,
				myPathName);
		return;
	}
	
 	// ***PlayTransMovie***
 	// play a QuickTime movie as a transition between two nodes
 	if (strcmp(myCommand, "PlayTransMovie") == 0) {
		char			myPathName[kMaxFileNameLength];
		
		sscanf(theCommandLine, "%*s %d %s", &myOptions, myPathName);
		VRMoov_PlayTransitionMovie(theWindowObject, myOptions, myPathName);
		return;
	}
	
 	// ***PlayTransEffect***
 	// play a QuickTime video effect as a transition between two nodes
 	if (strcmp(myCommand, "PlayTransEffect") == 0) {
		char		myEffectType[kMaxOSTypeLength];
		UInt32		myFromNodeID;
		UInt32		myToNodeID;
		SInt32		myMaxTimes;
		long		myEffectNum;
		OSType		myType;
		
		sscanf(theCommandLine, "%*s %d %d %d %s %d %d", &myFromNodeID, &myToNodeID, &myMaxTimes, myEffectType, &myEffectNum, &myOptions);

		// convert the string to an OSType
		myType = VRScript_StringToOSType(myEffectType);

		VRScript_EnlistTransitionEffect(theWindowObject, myFromNodeID, myToNodeID, myMaxTimes, myType, myEffectNum, myOptions);
		return;
	}
	
 	// ***MoveScreen***
	 // shift the movie screen center
	if (strcmp(myCommand, "MoveScreen") == 0) {
		QTVRFloatPoint	myCenter;
		float			myHoriz;
		float			myVert;

		sscanf(theCommandLine, "%*s %f %f %d", &myHoriz, &myVert, &myOptions);
		myHoriz = QTVRUtils_DegreesToRadians(myHoriz);
		myVert = QTVRUtils_DegreesToRadians(myVert);
		VRMoov_GetEmbeddedMovieCenter(theWindowObject, &myCenter);
		myCenter.x += myHoriz;
		myCenter.y += myVert;
		VRMoov_SetEmbeddedMovieCenter(theWindowObject, &myCenter);
		return;
	}
	
 	// ***Beep***
	// play the system beep
	if (strcmp(myCommand, "Beep") == 0) {
		DoBeep();
		return;
	}
	
 	// ***ProcessScript***
	// open and process a script file (synchronously!)
	if (strcmp(myCommand, "ProcessScript") == 0) {
		char		myPathName[kMaxFileNameLength];
	
		sscanf(theCommandLine, "%*s %d %s", &myOptions, myPathName);
		VRScript_OpenScriptFile(theWindowObject, myPathName);
		return;
	}
	
#if QD3D_AVAIL
 	// ***CreateBox***
 	// create a box
	if (strcmp(myCommand, "CreateBox") == 0) {
		float		myX, myY, myZ;
		float		myXSize, myYSize, myZSize;
		UInt32		myEntryID;

		if (!gHasQuickDraw3D)	// ignore command if QD3D not available
			return;
			
		sscanf(theCommandLine, "%*s %d %f %f %f %f %f %f %d", &myEntryID, &myX, &myY, &myZ, &myXSize, &myYSize, &myZSize, &myOptions);
		VR3DObjects_EnlistBox(theWindowObject, myEntryID, myX, myY, myZ, myXSize, myYSize, myZSize, myOptions);
		return;
	}

 	// ***CreateCone***
 	// create a cone
	if (strcmp(myCommand, "CreateCone") == 0) {
		float		myX, myY, myZ;
		float		myMajRad, myMinRad, myHeight;
		UInt32		myEntryID;

		if (!gHasQuickDraw3D15)	// ignore command if QD3D 1.5 not available
			return;
			
		sscanf(theCommandLine, "%*s %d %f %f %f %f %f %f %d", &myEntryID, &myX, &myY, &myZ, &myMajRad, &myMinRad, &myHeight, &myOptions);
		VR3DObjects_EnlistCone(theWindowObject, myEntryID, myX, myY, myZ, myMajRad, myMinRad, myHeight, myOptions);
		return;
	}

 	// ***CreateCylinder***
 	// create a cylinder
	if (strcmp(myCommand, "CreateCylinder") == 0) {
		float		myX, myY, myZ;
		float		myMajRad, myMinRad, myHeight;
		UInt32		myEntryID;

		if (!gHasQuickDraw3D15)	// ignore command if QD3D 1.5 not available
			return;
			
		sscanf(theCommandLine, "%*s %d %f %f %f %f %f %f %d", &myEntryID, &myX, &myY, &myZ, &myMajRad, &myMinRad, &myHeight, &myOptions);
		VR3DObjects_EnlistCylinder(theWindowObject, myEntryID, myX, myY, myZ, myMajRad, myMinRad, myHeight, myOptions);
		return;
	}

 	// ***CreateEllipsoid***
 	// create an ellipsoid
	if (strcmp(myCommand, "CreateEllipsoid") == 0) {
		float		myX, myY, myZ;
		float		myMajRad, myMinRad, myHeight;
		UInt32		myEntryID;

		if (!gHasQuickDraw3D15)	// ignore command if QD3D 1.5 not available
			return;
			
		sscanf(theCommandLine, "%*s %d %f %f %f %f %f %f %d", &myEntryID, &myX, &myY, &myZ, &myMajRad, &myMinRad, &myHeight, &myOptions);
		VR3DObjects_EnlistEllipsoid(theWindowObject, myEntryID, myX, myY, myZ, myMajRad, myMinRad, myHeight, myOptions);
		return;
	}

 	// ***CreateTorus***
 	// create a torus
	if (strcmp(myCommand, "CreateTorus") == 0) {
		float		myX, myY, myZ;
		float		myMajRad, myMinRad, myHeight, myRatio;
		UInt32		myEntryID;

		if (!gHasQuickDraw3D15)	// ignore command if QD3D 1.5 not available
			return;
			
		sscanf(theCommandLine, "%*s %d %f %f %f %f %f %f %f %d", &myEntryID, &myX, &myY, &myZ, &myMajRad, &myMinRad, &myHeight, &myRatio, &myOptions);
		VR3DObjects_EnlistTorus(theWindowObject, myEntryID, myX, myY, myZ, myMajRad, myMinRad, myHeight, myRatio, myOptions);
		return;
	}

 	// ***CreateRectangle***
 	// create a rectangle
	if (strcmp(myCommand, "CreateRectangle") == 0) {
		float		myX, myY, myZ;
		float		myX1, myY1, myZ1, myX2, myY2, myZ2, myX3, myY3, myZ3, myX4, myY4, myZ4;
		UInt32		myEntryID;

		if (!gHasQuickDraw3D15)	// ignore command if QD3D 1.5 not available
			return;
			
		sscanf(theCommandLine, "%*s %d %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %d", &myEntryID, &myX, &myY, &myZ, &myX1, &myY1, &myZ1, &myX2, &myY2, &myZ2, &myX3, &myY3, &myZ3, &myX4, &myY4, &myZ4, &myOptions);
		VR3DObjects_EnlistRectangle(theWindowObject, myEntryID, myX, myY, myZ, myX1, myY1, myZ1, myX2, myY2, myZ2, myX3, myY3, myZ3, myX4, myY4, myZ4, myOptions);
		return;
	}

 	// ***Open3DMFFile***
 	// load a 3D object contained in a 3DMF file
	if (strcmp(myCommand, "Open3DMFFile") == 0) {
		float		myX, myY, myZ;
		UInt32		myEntryID;
		char		myPathName[kMaxFileNameLength];

		if (!gHasQuickDraw3D)	// ignore command if QD3D not available
			return;
			
		sscanf(theCommandLine, "%*s %d %f %f %f %d %s", &myEntryID, &myX, &myY, &myZ, &myOptions, myPathName);
		VR3DObjects_Enlist3DMFFile(theWindowObject, myEntryID, myX, myY, myZ, myOptions, myPathName);
		return;
	}

 	// ***Set3DObjLocation***
	// set the location of a 3D object
	if (strcmp(myCommand, "Set3DObjLocation") == 0) {
		float		myX, myY, myZ;
		UInt32		myEntryID;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %f %f %f %d", &myEntryID, &myX, &myY, &myZ, &myOptions);
		VR3DObjects_SetLocation(theWindowObject, myEntryID, myX, myY, myZ, myOptions);
		QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);
		return;
	}
	
 	// ***Set3DObjColor***
	// set the color of a 3D object
	if (strcmp(myCommand, "Set3DObjColor") == 0) {
		float		myRed, myGreen, myBlue;
		UInt32		myEntryID;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %f %f %f %d", &myEntryID, &myRed, &myGreen, &myBlue, &myOptions);
		VR3DObjects_SetColor(theWindowObject, myEntryID, myRed, myGreen, myBlue, myOptions);
		QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);
		return;
	}
	
 	// ***Set3DObjTransp***
	// set the transparency level of a 3D object
	if (strcmp(myCommand, "Set3DObjTransp") == 0) {
		float		myRed, myGreen, myBlue;
		UInt32		myEntryID;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %f %f %f %d", &myEntryID, &myRed, &myGreen, &myBlue, &myOptions);
		VR3DObjects_SetTransparency(theWindowObject, myEntryID, myRed, myGreen, myBlue, myOptions);
		QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);
		return;
	}
	
 	// ***Set3DObjInterp***
	// set the interpolation style of a 3D object
	if (strcmp(myCommand, "Set3DObjInterp") == 0) {
		UInt32		myStyle;
		UInt32		myEntryID;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d %d", &myEntryID, &myStyle, &myOptions);
		VR3DObjects_SetInterpolation(theWindowObject, myEntryID, myStyle, myOptions);
		QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);
		return;
	}
	
 	// ***Set3DObjBackface***
	// set the backfacing style of a 3D object
	if (strcmp(myCommand, "Set3DObjBackface") == 0) {
		UInt32		myStyle;
		UInt32		myEntryID;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d %d", &myEntryID, &myStyle, &myOptions);
		VR3DObjects_SetBackfacing(theWindowObject, myEntryID, myStyle, myOptions);
		QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);
		return;
	}
	
 	// ***Set3DObjFill***
	// set the fill style of a 3D object
	if (strcmp(myCommand, "Set3DObjFill") == 0) {
		UInt32		myStyle;
		UInt32		myEntryID;

		if (theWindowObject == NULL)
			return;

		sscanf(theCommandLine, "%*s %d %d %d", &myEntryID, &myStyle, &myOptions);
		VR3DObjects_SetFill(theWindowObject, myEntryID, myStyle, myOptions);
		QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);
		return;
	}
	
	// ***Set3DObjRotation***
	// set the rotation factors of a 3D object
	if (strcmp(myCommand, "Set3DObjRotation") == 0) {
		float		myX, myY, myZ;
		UInt32		myEntryID;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %f %f %f %d", &myEntryID, &myX, &myY, &myZ, &myOptions);
		myX = QTVRUtils_DegreesToRadians(myX);
		myY = QTVRUtils_DegreesToRadians(myY);
		myZ = QTVRUtils_DegreesToRadians(myZ);
		VR3DObjects_SetRotation(theWindowObject, myEntryID, myX, myY, myZ, myOptions);
		QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);
		return;
	}
	
 	// ***Set3DObjRotState***
	// set the rotation state of a 3D object
	if (strcmp(myCommand, "Set3DObjRotState") == 0) {
		UInt32		myState;
		UInt32		myEntryID;

		sscanf(theCommandLine, "%*s %d %d %d", &myEntryID, &myState, &myOptions);
		VR3DObjects_SetRotationState(theWindowObject, myEntryID, myState, myOptions);
		return;
	}
	
 	// ***Set3DObjVisState***
	// set the visibility state of a 3D object
	if (strcmp(myCommand, "Set3DObjVisState") == 0) {
		UInt32		myState;
		UInt32		myEntryID;

		if (theWindowObject == NULL)
			return;
			
		sscanf(theCommandLine, "%*s %d %d %d", &myEntryID, &myState, &myOptions);
		VR3DObjects_SetVisibleState(theWindowObject, myEntryID, myState, myOptions);
		QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);
		return;
	}
	
 	// ***Set3DObjTexture***
	// set the texture of a 3D object
	if (strcmp(myCommand, "Set3DObjTexture") == 0) {
		UInt32		myEntryID;
		UInt32		isMovie;
		char		myPathName[kMaxFileNameLength];

		sscanf(theCommandLine, "%*s %d %d %d %s", &myEntryID, &isMovie, &myOptions, myPathName);
		VR3DObjects_SetTexture(theWindowObject, myEntryID, (Boolean)isMovie, myOptions, myPathName);
		return;
	}
#endif // QD3D_AVAIL
	
 	// ***Set3DSndLocation***
	// set the location of a localized sound
	if (strcmp(myCommand, "Set3DSndLocation") == 0) {
		float		myX, myY, myZ;
		UInt32		myEntryID;

		sscanf(theCommandLine, "%*s %d %f %f %f %d", &myEntryID, &myX, &myY, &myZ, &myOptions);
		VRSound_SetLocation(theWindowObject, myEntryID, myX, myY, myZ, myOptions);
		VRSound_Update3DSoundEnv(theWindowObject);
		return;
	}

 	// ***SetVariable***
	// set the value of a variable; variables are used only in If commands
	if (strcmp(myCommand, "SetVariable") == 0) {
		char					myVarName[kMaxVarNameLength];
		SInt32					myVarValue;
		VRScriptVariablePtr		myPointer;
		
		sscanf(theCommandLine, "%*s %s %d %d", myVarName, &myVarValue, &myOptions);
		myPointer = VRScript_GetVariableEntry(theWindowObject, myVarName);
		
		if (myPointer == NULL) {
			VRScript_EnlistVariable(theWindowObject, myVarName, myVarValue);
		} else {
			if (myOptions == kVRValue_Absolute)
				myPointer->fValue = myVarValue;
			if (myOptions == kVRValue_Relative)
				myPointer->fValue += myVarValue;
		}
		
		return;
	}

 	// ***If***
	// evaluate the expression "var op value"; execute a command if true
	if (strcmp(myCommand, "If") == 0) {
		char					myVarName[kMaxVarNameLength];
		char					myOperation[kMaxVarOpLength];
		SInt32					myVarValue;
		VRScriptVariablePtr		myPointer;
		char					myCmdLine[kMaxCmdLineLength];
		
		sscanf(theCommandLine, "%*s %s %s %d %d %s", myVarName, myOperation, &myVarValue, &myOptions, myCmdLine);
		myPointer = VRScript_GetVariableEntry(theWindowObject, myVarName);
		
		if (myPointer != NULL) {
			Boolean				myRunCommand = false;

			// find the appropriate operation and test for its satisfaction
			if ((strcmp(myOperation, "=") == 0) || (strcmp(myOperation, "==") == 0)) {
				if (myPointer->fValue == myVarValue)
					myRunCommand = true;
			} else if (strcmp(myOperation, "!=") == 0) {
				if (myPointer->fValue != myVarValue)
					myRunCommand = true;
			} else if (strcmp(myOperation, "<") == 0) {
				if (myPointer->fValue < myVarValue)
					myRunCommand = true;
			} else if (strcmp(myOperation, "<=") == 0) {
				if (myPointer->fValue <= myVarValue)
					myRunCommand = true;
			} else if (strcmp(myOperation, ">") == 0) {
				if (myPointer->fValue > myVarValue)
					myRunCommand = true;
			} else if (strcmp(myOperation, ">=") == 0) {
				if (myPointer->fValue >= myVarValue)
					myRunCommand = true;
			}		
				
			if (myRunCommand) {
				VRScript_UnpackString(myCmdLine);
				VRScript_ProcessScriptCommandLine(theWindowObject, myCmdLine);
			}
		}
		
		return;
	}

	// if we got here, we encountered an unrecognized command;
	// just silently ignore it
	
	return;
}


//////////
//
// VRScript_SetControllerBarState
// Set the state of the controller bar.
//
//////////

void VRScript_SetControllerBarState (WindowObject theWindowObject, Boolean theState, UInt32 theOptions)
{
#pragma unused(theOptions)

	if (theWindowObject == NULL)
		return;

	if (theState == kVRState_Toggle)
		theState = !MCGetVisible((**theWindowObject).fController);
	
	MCSetVisible((**theWindowObject).fController, theState);
}


//////////
//
// VRScript_SetControllerButtonState
// Set the state of the specified button (or text display) in the controller bar.
//
//////////

void VRScript_SetControllerButtonState (WindowObject theWindowObject, UInt32 theButton, Boolean theState, UInt32 theOptions)
{
#pragma unused(theOptions)
	
	long	myButton;
	
	if (theWindowObject == NULL)
		return;
		
	if ((**theWindowObject).fController == NULL)
		return;

	// convert VRScript's enum into the actual value expected by the movie controller
	switch (theButton) {
		case kVRButton_Step:			myButton = mcFlagSuppressStepButtons;			break;
		case kVRButton_Speaker:			myButton = mcFlagSuppressSpeakerButton;			break;
		case kVRButton_GoBack:			myButton = mcFlagQTVRSuppressBackBtn;			break;
		case kVRButton_ZoomInOut:		myButton = mcFlagQTVRSuppressZoomBtns;			break;
		case kVRButton_ShowHotSpots:	myButton = mcFlagQTVRSuppressHotSpotBtn;		break;
		case kVRButton_Translate:		myButton = mcFlagQTVRSuppressTranslateBtn;		break;
		case kVRButton_HelpText:		myButton = mcFlagQTVRSuppressHelpText;			break;
		case kVRButton_HotSpotNames:	myButton = mcFlagQTVRSuppressHotSpotNames;		break;
		default:
			break;
	}

	// set the correct state of the button		
	switch (theState) {
		case kVRState_Hide:
			QTVRUtils_HideControllerButton((**theWindowObject).fController, myButton);
			break;
		case kVRState_Show:
			QTVRUtils_ShowControllerButton((**theWindowObject).fController, myButton);
			break;
		case kVRState_Toggle:
			QTVRUtils_ToggleControllerButton((**theWindowObject).fController, myButton);
			break;
		default:
			break;
	}
}


//////////
//
// VRScript_SetResizeState
// Set the state of window resizing.
//
//////////

void VRScript_SetResizeState (WindowObject theWindowObject, Boolean theState, UInt32 theOptions)
{
#pragma unused(theOptions)

	Rect			myRect;

	if (theWindowObject == NULL)
		return;

	if (theState == kVRState_Toggle)
		theState = !(**theWindowObject).fCanResizeWindow;

	(**theWindowObject).fCanResizeWindow = theState;
	
	// set size of growable window
	if ((**theWindowObject).fCanResizeWindow)
		MacSetRect(&myRect, gLimitRect.left, gLimitRect.top, gLimitRect.right, gLimitRect.bottom);
	else
		MacSetRect(&myRect, 0, 0, 0, 0);
	
	MCDoAction((**theWindowObject).fController, mcActionSetGrowBoxBounds, &myRect);
	QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);
}


//////////
//
// VRScript_SetAngleConstraints
// Set the pan, tilt, or zoom constraints.
// Note: this function expects theMinAngle and theMaxAngle to be in *degrees*.
//
//////////

void VRScript_SetAngleConstraints (WindowObject theWindowObject, UInt16 theKind, float theMinAngle, float theMaxAngle, UInt32 theOptions)
{
	float myMinAngle;
	float myMaxAngle;

	if (theWindowObject == NULL)
		return;
	
	myMinAngle = QTVRUtils_DegreesToRadians(theMinAngle);
	myMaxAngle = QTVRUtils_DegreesToRadians(theMaxAngle);
	
	if (theOptions == kVRValue_Relative) {
		float	myCurrMin;
		float	myCurrMax;
		OSErr	myErr;

		myErr = QTVRGetConstraints((**theWindowObject).fInstance, theKind, &myCurrMin, &myCurrMax);
		if (myErr == noErr) {
			myMinAngle += myCurrMin;
			myMaxAngle += myCurrMax;
		}
	}
				
	QTVRSetConstraints((**theWindowObject).fInstance, theKind, myMinAngle, myMaxAngle);
	QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);
}


//////////
//
// VRScript_SetHotSpotState
// Set the state of a hot spot.
// (There's no easy way to get the current state of a hot spot, so we don't support toggling.)
//
//////////

void VRScript_SetHotSpotState (WindowObject theWindowObject, UInt32 theHotSpotID, UInt32 theState, UInt32 theOptions)
{
	if (theWindowObject == NULL)
		return;

	QTVREnableHotSpot((**theWindowObject).fInstance, theOptions, theHotSpotID, (Boolean)theState);
	
	// we need to update, because the hot spot regions might currently be showing
	QTVRUpdate((**theWindowObject).fInstance, kQTVRCurrentMode);
}


//////////
//
// VRScript_OpenResourceFile
// Open the specified resource file and add it to the application's resource chain.
// Returns the file reference number, if successful, or -1 otherwise.
//
//////////

short VRScript_OpenResourceFile (WindowObject theWindowObject, UInt32 theOptions, char *thePathName)
{
#pragma unused(theWindowObject, theOptions)

	FSSpec		myFSSpec;
	short		myRefNum = -1;

	FSMakeFSSpec(0, 0L, c2pstr(thePathName), &myFSSpec);
	
	myRefNum = FSpOpenResFile(&myFSSpec, fsRdPerm);
	if (myRefNum == -1)
		;					// no error handling yet....
		
	return(myRefNum);
}


//////////
//
// VRScript_EnlistSound
// Install an ambient sound.
//
//////////

VRScriptSoundPtr VRScript_EnlistSound (WindowObject theWindowObject, UInt32 theNodeID, UInt32 theResID, UInt32 theEntryID, UInt32 theMode, UInt32 theFade, UInt32 theOptions, SndChannelPtr theChannel, SndListHandle theResource)
{
	VRScriptSoundPtr		myPointer;
	ApplicationDataHdl		myAppData;
	
	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return(NULL);
	
	myPointer = (VRScriptSoundPtr)NewPtrClear(sizeof(VRScriptSound));
	if (myPointer != NULL) {
		myPointer->fEntryType = kVREntry_Sound;
		myPointer->fEntryID = theEntryID;
		myPointer->fNodeID = theNodeID;
		myPointer->fOptions = theOptions;
		myPointer->fMode = theMode;
		myPointer->fSoundContainer = kVRSound_SoundResource;
		myPointer->fResID = theResID;
		myPointer->fResourceData = theResource;
		myPointer->fChannel = theChannel;
		myPointer->fFadeWhenStopping = (Boolean)theFade;
		myPointer->fSoundIsLocalized = false;
		myPointer->fSource = 0;
		myPointer->fLocation.x = 0.0;
		myPointer->fLocation.y = 0.0;
		myPointer->fLocation.z = 0.0;
		myPointer->fProjAngle = 0.0;
		myPointer->fNextEntry = (VRScriptSoundPtr)(**myAppData).fListArray[kVREntry_Sound];
		(**myAppData).fListArray[kVREntry_Sound] = (VRScriptGenericPtr)myPointer;
	}
	
	return(myPointer);
}


//////////
//
// VRScript_EnlistLocalizedSound
// Install a localized sound.
//
//////////

VRScriptSoundPtr VRScript_EnlistLocalizedSound (WindowObject theWindowObject, UInt32 theNodeID, UInt32 theResID, UInt32 theEntryID, float theX, float theY, float theZ, float theProjAngle, UInt32 theMode, UInt32 theFade, UInt32 theOptions, SndChannelPtr theChannel, SndListHandle theResource, SSpSourceReference theSource)
{
	VRScriptSoundPtr		myPointer;
	ApplicationDataHdl		myAppData;
	
	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return(NULL);

	myPointer = (VRScriptSoundPtr)NewPtrClear(sizeof(VRScriptSound));
	if (myPointer != NULL) {
		myPointer->fEntryType = kVREntry_Sound;
		myPointer->fEntryID = theEntryID;
		myPointer->fNodeID = theNodeID;
		myPointer->fOptions = theOptions;
		myPointer->fMode = theMode;
		myPointer->fSoundContainer = kVRSound_SoundResource;
		myPointer->fResID = theResID;
		myPointer->fResourceData = theResource;
		myPointer->fChannel = theChannel;
		myPointer->fSource = theSource;
		myPointer->fLocation.x = theX;
		myPointer->fLocation.y = theY;
		myPointer->fLocation.z = theZ;
		myPointer->fProjAngle = theProjAngle;
		myPointer->fFadeWhenStopping = (Boolean)theFade;
		myPointer->fSoundIsLocalized = true;
		myPointer->fNextEntry = (VRScriptSoundPtr)(**myAppData).fListArray[kVREntry_Sound];
		(**myAppData).fListArray[kVREntry_Sound] = (VRScriptGenericPtr)myPointer;
	}
	
	return(myPointer);
}


//////////
//
// VRScript_EnlistMovie
// Install a QuickTime movie by adding the given information to our linked list.
//
//////////

VRScriptMoviePtr VRScript_EnlistMovie (
				WindowObject theWindowObject,
				UInt32 theNodeID,
				UInt32 theEntryID,
				float thePanAngle,
				float theTiltAngle,
				float theScale, 
				float theWidth, 
				UInt32 theKeyRed, 
				UInt32 theKeyGreen, 
				UInt32 theKeyBlue, 
				Boolean theUseBuffer,
				Boolean theUseCenter,
				Boolean theUseKey,
				Boolean theUseHide,
				Boolean theUseDir,
				Boolean theUseVol,
				float theVolAngle,
				UInt32 theMode,
				UInt32 theOptions,
				char *thePathName)
{
	VRScriptMoviePtr		myPointer;
	ApplicationDataHdl		myAppData;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return(NULL);

	myPointer = (VRScriptMoviePtr)NewPtrClear(sizeof(VRScriptMovie));
	if (myPointer != NULL) {
		myPointer->fEntryType = kVREntry_QTMovie;
		myPointer->fEntryID = theEntryID;
		myPointer->fNodeID = theNodeID;
		myPointer->fOptions = theOptions;
		myPointer->fMode = theMode;
		myPointer->fMovie = NULL;
		myPointer->fMovieGWorld = NULL;
		myPointer->fMoviePixMap = NULL;
		myPointer->fMovieCenter.x = thePanAngle;
		myPointer->fMovieCenter.y = theTiltAngle;
		myPointer->fMovieScale = theScale;
		myPointer->fMovieWidth = theWidth;
		myPointer->fUseMovieGWorld = theUseBuffer;
		myPointer->fUseMovieCenter = theUseCenter;
		myPointer->fQTMovieHasSound = false;		// we don't really know until we open the file....
		myPointer->fQTMovieHasVideo = false;		// we don't really know until we open the file....
		myPointer->fCompositeMovie = theUseKey;
		myPointer->fUseHideRegion = theUseHide;
		myPointer->fChromaColor.red = (unsigned short)theKeyRed;
		myPointer->fChromaColor.green = (unsigned short)theKeyGreen;
		myPointer->fChromaColor.blue = (unsigned short)theKeyBlue;
		myPointer->fHideRegion = NULL;
		myPointer->fSoundIsLocalized = theUseDir;
		myPointer->fIsAttenuated = theUseVol;
		myPointer->fVolAngle = theVolAngle;
		myPointer->fPathname = malloc(strlen(thePathName) + 1);
		strncpy(myPointer->fPathname, thePathName, strlen(thePathName) + 1);
		myPointer->fNextEntry = (VRScriptMoviePtr)(**myAppData).fListArray[kVREntry_QTMovie];
		(**myAppData).fListArray[kVREntry_QTMovie] = (VRScriptGenericPtr)myPointer;
	}
	
	return(myPointer);
}


//////////
//
// VRScript_Enlist3DObject
// Install a 3D object.
//
//////////

#if QD3D_AVAIL
void VRScript_Enlist3DObject (WindowObject theWindowObject, TQ3GroupObject theGroup, UInt32 theEntryID, float theX, float theY, float theZ, UInt32 theOptions)
{
	VRScript3DObjPtr		myPointer;
	ApplicationDataHdl		myAppData;
	TQ3Point3D				myPoint;
	
	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;
	
	myPointer = (VRScript3DObjPtr)NewPtrClear(sizeof(VRScript3DObj));
	if (myPointer != NULL) {
		myPointer->fEntryType = kVREntry_QD3DObject;
		myPointer->fEntryID = theEntryID;
		myPointer->fOptions = theOptions;
		myPointer->fModel = theGroup;

		myPointer->fInterpolation = Q3InterpolationStyle_New((TQ3InterpolationStyle)kDefaultInterpolation);
		myPointer->fBackFacing = Q3BackfacingStyle_New((TQ3BackfacingStyle)kDefaultBackfacing);
		myPointer->fFillStyle = Q3FillStyle_New((TQ3FillStyle)kDefaultFill);
		
		Q3Matrix4x4_SetIdentity(&(myPointer->fRotation));	
		myPointer->fRotateFactors.x = kDefaultRotateRadians;		// default rotation factors
		myPointer->fRotateFactors.y = kDefaultRotateRadians;
		myPointer->fRotateFactors.y = kDefaultRotateRadians;
			
		Q3Point3D_Set(&myPoint, -1.0 * theX, theY, -1.0 * theZ);
		myPointer->fGroupCenter = myPoint;
		myPointer->fGroupScale = kDefaultScale;
		myPointer->fTexture = NULL;
		myPointer->fTextureIsMovie = false;
		myPointer->fModelIsVisible = true;
		myPointer->fModelIsAnimated = false;
		myPointer->fNextEntry = (VRScript3DObjPtr)(**myAppData).fListArray[kVREntry_QD3DObject];
		(**myAppData).fListArray[kVREntry_QD3DObject] = (VRScriptGenericPtr)myPointer;
	}
}
#endif


//////////
//
// VRScript_EnlistOverlayPicture
// Install an overlay picture by adding the given information to our linked list.
//
//////////

VRScriptPicturePtr VRScript_EnlistOverlayPicture (WindowObject theWindowObject, UInt32 theResID, UInt32 theEntryID, UInt32 theHeight, UInt32 theWidth, UInt32 thePegSides, UInt32 theOffset, PicHandle theResource, UInt32 theOptions)
{
	VRScriptPicturePtr		myPointer;
	ApplicationDataHdl		myAppData;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return(NULL);

	myPointer = (VRScriptPicturePtr)NewPtrClear(sizeof(VRScriptPicture));
	if (myPointer != NULL) {
		myPointer->fEntryType = kVREntry_OverlayPicture;
		myPointer->fEntryID = theEntryID;
		myPointer->fNodeID = 0;
		myPointer->fOptions = theOptions;
		myPointer->fResID = theResID;
		myPointer->fBoxHeight = theHeight;
		myPointer->fBoxWidth = theWidth;
		myPointer->fPegSides = thePegSides;
		myPointer->fOffset = theOffset;
		myPointer->fResourceData = theResource;
		myPointer->fNextEntry = (VRScriptPicturePtr)(**myAppData).fListArray[kVREntry_OverlayPicture];
		(**myAppData).fListArray[kVREntry_OverlayPicture] = (VRScriptGenericPtr)myPointer;
	}
	
	return(myPointer);
}


//////////
//
// VRScript_EnlistTimedCommand
// Install a timed-activated command by adding the given information to our linked list.
//
//////////

void VRScript_EnlistTimedCommand (WindowObject theWindowObject, UInt32 theTicks, UInt32 theMode, UInt32 theNodeID, UInt32 theRepeat, UInt32 thePeriod, SInt32 theMaxTimes, UInt32 theOptions, char *theCmdLine)
{
	VRScriptAtTimePtr		myPointer;
	ApplicationDataHdl		myAppData;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;

	myPointer = (VRScriptAtTimePtr)NewPtrClear(sizeof(VRScriptAtTime));
	if (myPointer != NULL) {
		myPointer->fEntryType = kVREntry_TimedCommand;
		myPointer->fNodeID = theNodeID;
		myPointer->fOptions = theOptions;
		myPointer->fTime = theTicks;
		myPointer->fMode = theMode;
		myPointer->fTimeInstalled = gAbsoluteElapsedTime;
		myPointer->fCommandLine = malloc(strlen(theCmdLine) + 1);
		strncpy(myPointer->fCommandLine, theCmdLine, strlen(theCmdLine) + 1);
		myPointer->fRepeat = (Boolean)theRepeat;
		myPointer->fPeriod = thePeriod;
		myPointer->fMaxExecutions = theMaxTimes;
		myPointer->fNextEntry = (VRScriptAtTimePtr)(**myAppData).fListArray[kVREntry_TimedCommand];
		(**myAppData).fListArray[kVREntry_TimedCommand] = (VRScriptGenericPtr)myPointer;
	}
}


//////////
//
// VRScript_EnlistQuitCommand
// Install an application-quit command by adding the given information to our linked list.
//
//////////

void VRScript_EnlistQuitCommand (WindowObject theWindowObject, UInt32 theOptions, char *theCmdLine)
{
	VRScriptAtQuitPtr		myPointer;
	ApplicationDataHdl		myAppData;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;

	myPointer = (VRScriptAtQuitPtr)NewPtrClear(sizeof(VRScriptAtQuit));
	if (myPointer != NULL) {
		myPointer->fEntryType = kVREntry_QuitCommand;
		myPointer->fOptions = theOptions;
		myPointer->fCommandLine = malloc(strlen(theCmdLine) + 1);
		strncpy(myPointer->fCommandLine, theCmdLine, strlen(theCmdLine) + 1);
		myPointer->fNextEntry = (VRScriptAtQuitPtr)(**myAppData).fListArray[kVREntry_QuitCommand];
		(**myAppData).fListArray[kVREntry_QuitCommand] = (VRScriptGenericPtr)myPointer;
	}
}


//////////
//
// VRScript_EnlistMouseOverHSCommand
// Install a mouse-over hot spot command by adding the given information to our linked list.
//
//////////

void VRScript_EnlistMouseOverHSCommand (WindowObject theWindowObject, UInt32 theNodeID, UInt32 theHotSpotID, OSType theHotSpotType, SInt32 theMaxTimes, UInt32 theOptions, char *theCmdLine)
{
	VRScriptAtMOHSPtr		myPointer;
	ApplicationDataHdl		myAppData;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;

	myPointer = (VRScriptAtMOHSPtr)NewPtrClear(sizeof(VRScriptAtMOHS));
	if (myPointer != NULL) {
		myPointer->fEntryType = kVREntry_MouseOverHS;
		myPointer->fNodeID = theNodeID;
		myPointer->fOptions = theOptions;							// the hot spot action selector
		myPointer->fSelectByID = theHotSpotID > 0 ? true : false;
		myPointer->fHotSpotID = theHotSpotID;
		myPointer->fHotSpotType = theHotSpotType;
		myPointer->fCommandLine = malloc(strlen(theCmdLine) + 1);
		strncpy(myPointer->fCommandLine, theCmdLine, strlen(theCmdLine) + 1);
		myPointer->fMaxExecutions = theMaxTimes;
		myPointer->fNextEntry = (VRScriptAtMOHSPtr)(**myAppData).fListArray[kVREntry_MouseOverHS];
		(**myAppData).fListArray[kVREntry_MouseOverHS] = (VRScriptGenericPtr)myPointer;
	}
}


//////////
//
// VRScript_EnlistClickHSCommand
// Install a hot spot click command by adding the given information to our linked list.
//
//////////

void VRScript_EnlistClickHSCommand (WindowObject theWindowObject, UInt32 theNodeID, UInt32 theHotSpotID, SInt32 theMaxTimes, UInt32 theOptions, char *theCmdLine)
{
	VRScriptClickHSPtr		myPointer;
	ApplicationDataHdl		myAppData;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;

	myPointer = (VRScriptClickHSPtr)NewPtrClear(sizeof(VRScriptClickHS));
	if (myPointer != NULL) {
		myPointer->fEntryType = kVREntry_ClickHS;
		myPointer->fNodeID = theNodeID;
		myPointer->fOptions = theOptions;
		myPointer->fHotSpotID = theHotSpotID;
		myPointer->fCommandLine = malloc(strlen(theCmdLine) + 1);
		strncpy(myPointer->fCommandLine, theCmdLine, strlen(theCmdLine) + 1);
		myPointer->fMaxExecutions = theMaxTimes;
		myPointer->fNextEntry = (VRScriptClickHSPtr)(**myAppData).fListArray[kVREntry_ClickHS];
		(**myAppData).fListArray[kVREntry_ClickHS] = (VRScriptGenericPtr)myPointer;
	}
}


//////////
//
// VRScript_EnlistNodeEntryCommand
// Install a node-entry command by adding the given information to our linked list.
//
//////////

void VRScript_EnlistNodeEntryCommand (WindowObject theWindowObject, UInt32 theNodeID, SInt32 theMaxTimes, UInt32 theOptions, char *theCmdLine)
{
	VRScriptNodeInPtr		myPointer;
	ApplicationDataHdl		myAppData;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;

	myPointer = (VRScriptNodeInPtr)NewPtrClear(sizeof(VRScriptNodeIn));
	if (myPointer != NULL) {
		myPointer->fEntryType = kVREntry_NodeEntry;
		myPointer->fNodeID = theNodeID;
		myPointer->fOptions = theOptions;
		myPointer->fCommandLine = malloc(strlen(theCmdLine) + 1);
		strncpy(myPointer->fCommandLine, theCmdLine, strlen(theCmdLine) + 1);
		myPointer->fMaxExecutions = theMaxTimes;
		myPointer->fNextEntry = (VRScriptNodeInPtr)(**myAppData).fListArray[kVREntry_NodeEntry];
		(**myAppData).fListArray[kVREntry_NodeEntry] = (VRScriptGenericPtr)myPointer;
	}
}


//////////
//
// VRScript_EnlistNodeExitCommand
// Install a node-exit command by adding the given information to our linked list.
//
//////////

void VRScript_EnlistNodeExitCommand (WindowObject theWindowObject, UInt32 theFromNodeID, UInt32 theToNodeID, SInt32 theMaxTimes, UInt32 theOptions, char *theCmdLine)
{
	VRScriptNodeOutPtr		myPointer;
	ApplicationDataHdl		myAppData;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;

	myPointer = (VRScriptNodeOutPtr)NewPtrClear(sizeof(VRScriptNodeOut));
	if (myPointer != NULL) {
		myPointer->fEntryType = kVREntry_NodeExit;
		myPointer->fOptions = theOptions;
		myPointer->fFromNodeID = theFromNodeID;
		myPointer->fToNodeID = theToNodeID;
		myPointer->fCommandLine = malloc(strlen(theCmdLine) + 1);
		strncpy(myPointer->fCommandLine, theCmdLine, strlen(theCmdLine) + 1);
		myPointer->fMaxExecutions = theMaxTimes;
		myPointer->fNextEntry = (VRScriptNodeOutPtr)(**myAppData).fListArray[kVREntry_NodeExit];
		(**myAppData).fListArray[kVREntry_NodeExit] = (VRScriptGenericPtr)myPointer;
	}
}


//////////
//
// VRScript_EnlistAngleCommand
// Install an angle-triggered command by adding the given information to our linked list.
// Note: this function expects theMinAngle and theMaxAngle to be in *degrees*.
//
//////////

void VRScript_EnlistAngleCommand (WindowObject theWindowObject, UInt32 theKind, UInt32 theNodeID, float theMinAngle, float theMaxAngle, SInt32 theMaxTimes, UInt32 theOptions, char *theCmdLine)
{
	VRScriptAtAnglePtr		myPointer;
	ApplicationDataHdl		myAppData;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;

	theMinAngle = QTVRUtils_DegreesToRadians(theMinAngle);
	theMaxAngle = QTVRUtils_DegreesToRadians(theMaxAngle);
	VRScript_UnpackString(theCmdLine);

	myPointer = (VRScriptAtAnglePtr)NewPtrClear(sizeof(VRScriptAtAngle));
	if (myPointer != NULL) {
		myPointer->fEntryType = theKind;
		myPointer->fNodeID = theNodeID;
		myPointer->fOptions = theOptions;
		myPointer->fMinAngle = theMinAngle;
		myPointer->fMaxAngle = theMaxAngle;
		myPointer->fMaxExecutions = theMaxTimes;
		myPointer->fCommandLine = malloc(strlen(theCmdLine) + 1);
		strncpy(myPointer->fCommandLine, theCmdLine, strlen(theCmdLine) + 1);
		myPointer->fNextEntry = (VRScriptAtAnglePtr)(**myAppData).fListArray[theKind];
		(**myAppData).fListArray[theKind] = (VRScriptGenericPtr)myPointer;
	}
}


//////////
//
// VRScript_EnlistVariable
// Install a variable by adding the given information to our linked list.
//
//////////

void VRScript_EnlistVariable (WindowObject theWindowObject, char *theVarName, UInt32 theVarValue)
{
	VRScriptVariablePtr		myPointer;
	ApplicationDataHdl		myAppData;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;
		
	myPointer = (VRScriptVariablePtr)NewPtrClear(sizeof(VRScriptVariable));
	if (myPointer != NULL) {
		myPointer->fEntryType = kVREntry_Variable;
		myPointer->fValue = theVarValue;
		myPointer->fVarName = malloc(strlen(theVarName) + 1);
		strncpy(myPointer->fVarName, theVarName, strlen(theVarName) + 1);
		myPointer->fNextEntry = (VRScriptVariablePtr)(**myAppData).fListArray[kVREntry_Variable];
		(**myAppData).fListArray[kVREntry_Variable] = (VRScriptGenericPtr)myPointer;
	}
}


//////////
//
// VRScript_EnlistTransition
// Install a transition effect by adding the given information to our linked list.
//
//////////

void VRScript_EnlistTransitionEffect (WindowObject theWindowObject, UInt32 theFromNodeID, UInt32 theToNodeID, SInt32 theMaxTimes, OSType theEffectType, long theEffectNum, UInt32 theOptions)
{
	VRScriptTransitionPtr	myPointer;
	ApplicationDataHdl		myAppData;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;
		
	myPointer = (VRScriptTransitionPtr)NewPtrClear(sizeof(VRScriptTransition));
	if (myPointer != NULL) {
		myPointer->fEntryType = kVREntry_TransitionEffect;
		myPointer->fOptions = theOptions;
		myPointer->fFromNodeID = theFromNodeID;
		myPointer->fToNodeID = theToNodeID;
		myPointer->fMaxExecutions = theMaxTimes;
		myPointer->fEffectType = theEffectType;
		myPointer->fEffectNum = theEffectNum;
		myPointer->fNumberOfSteps = (long)kDefaultNumSteps;
		myPointer->fNextEntry = (VRScriptTransitionPtr)(**myAppData).fListArray[kVREntry_TransitionEffect];
		(**myAppData).fListArray[kVREntry_TransitionEffect] = (VRScriptGenericPtr)myPointer;
	}
}


//////////
//
// VRScript_DelistEntry
// Remove an entry from a linked list.
//
//////////

void VRScript_DelistEntry (WindowObject theWindowObject, VRScriptGenericPtr theEntryPtr)
{
	ApplicationDataHdl		myAppData;
	VRScriptGenericPtr		*myListHead;	// the *address* of the list head
	
	if (theEntryPtr == NULL)
		return;
	
	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData != NULL) {
		
		// get the head of the specified list
		myListHead = &((**myAppData).fListArray[theEntryPtr->fEntryType]);
		
		// get rid of any memory associated with this entry
		switch (theEntryPtr->fEntryType) {
			case kVREntry_Sound:
				VRSound_DumpEntryMem((VRScriptSoundPtr)theEntryPtr);			break;
			case kVREntry_QTMovie: 
				VRMoov_DumpEntryMem((VRScriptMoviePtr)theEntryPtr);				break;
			case kVREntry_QD3DObject: 
#if QD3D_AVAIL
				VR3DObjects_DumpEntryMem((VRScript3DObjPtr)theEntryPtr);
#endif
				break;
			case kVREntry_OverlayPicture: 
				VRPicture_DumpEntryMem((VRScriptPicturePtr)theEntryPtr);		break;
			case kVREntry_TimedCommand: 
				free(((VRScriptAtTimePtr)theEntryPtr)->fCommandLine);			break;
			case kVREntry_QuitCommand: 
				free(((VRScriptAtQuitPtr)theEntryPtr)->fCommandLine);			break;
			case kVREntry_MouseOverHS: 
				free(((VRScriptAtMOHSPtr)theEntryPtr)->fCommandLine);			break;
			case kVREntry_ClickHS: 
				free(((VRScriptClickHSPtr)theEntryPtr)->fCommandLine);			break;
			case kVREntry_NodeEntry: 
				free(((VRScriptNodeInPtr)theEntryPtr)->fCommandLine);			break;
			case kVREntry_NodeExit: 
				free(((VRScriptNodeOutPtr)theEntryPtr)->fCommandLine);			break;
			case kVREntry_PanAngleCmd: 
				free(((VRScriptAtAnglePtr)theEntryPtr)->fCommandLine);			break;
			case kVREntry_TiltAngleCmd: 
				free(((VRScriptAtAnglePtr)theEntryPtr)->fCommandLine);			break;
			case kVREntry_FOVAngleCmd: 
				free(((VRScriptAtAnglePtr)theEntryPtr)->fCommandLine);			break;
			case kVREntry_Variable: 
				free(((VRScriptVariablePtr)theEntryPtr)->fVarName);				break;
			case kVREntry_TransitionEffect: 
				VREffects_DumpEntryMem((VRScriptTransitionPtr)theEntryPtr);		break;
			case kVREntry_Generic: 
			case kVREntry_Unknown: 
			default: 
				return;
				break;
		}
	
		// now drop the specified entry from the correct list

		if (*myListHead == theEntryPtr) {
			// the entry to remove is the head of the list; make the next entry the head	
			*myListHead = theEntryPtr->fNextEntry;
		} else {
			// the entry to remove is not the head of the list;
			// walk the list until we find the entry *before* the one to be removed
			VRScriptGenericPtr		myPointer;

			myPointer = *myListHead;
			while (myPointer != NULL) {
				if (myPointer->fNextEntry == theEntryPtr) {
					myPointer->fNextEntry = theEntryPtr->fNextEntry;
					break;
				}
				myPointer = (VRScriptGenericPtr)(myPointer->fNextEntry);
			}
		}
	}
	
	DisposePtr((Ptr)theEntryPtr);
}


//////////
//
// VRScript_DeleteListOfType
// Delete the list containing entries of the specified type.
//
//////////

void VRScript_DeleteListOfType (WindowObject theWindowObject, UInt32 theEntryType)
{
	ApplicationDataHdl		myAppData;
	VRScriptGenericPtr		*myListHead;	// the *address* of the list head
	VRScriptGenericPtr		myPointer;
	
	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;
		
	// get the head of specified list
	myListHead = &((**myAppData).fListArray[theEntryType]);
		
	myPointer = *myListHead;
	while (myPointer != NULL) {
		VRScriptGenericPtr		myNext;
		
		myNext = myPointer->fNextEntry;
		
		// get rid of any memory associated with this entry
		switch (theEntryType) {
			case kVREntry_Sound: 
				VRSound_DumpEntryMem((VRScriptSoundPtr)myPointer);				break;
			case kVREntry_QTMovie: 
				VRMoov_DumpEntryMem((VRScriptMoviePtr)myPointer);				break;
			case kVREntry_QD3DObject: 
#if QD3D_AVAIL
				VR3DObjects_DumpEntryMem((VRScript3DObjPtr)myPointer);
#endif
																				break;
			case kVREntry_OverlayPicture: 
				VRPicture_DumpEntryMem((VRScriptPicturePtr)myPointer);			break;
			case kVREntry_TimedCommand: 
				free(((VRScriptAtTimePtr)myPointer)->fCommandLine);				break;
			case kVREntry_QuitCommand: 
				free(((VRScriptAtQuitPtr)myPointer)->fCommandLine);				break;
			case kVREntry_MouseOverHS: 
				free(((VRScriptAtMOHSPtr)myPointer)->fCommandLine);				break;
			case kVREntry_ClickHS: 
				free(((VRScriptClickHSPtr)myPointer)->fCommandLine);			break;
			case kVREntry_NodeEntry: 
				free(((VRScriptNodeInPtr)myPointer)->fCommandLine);				break;
			case kVREntry_NodeExit: 
				free(((VRScriptNodeOutPtr)myPointer)->fCommandLine);			break;
			case kVREntry_PanAngleCmd: 
				free(((VRScriptAtAnglePtr)myPointer)->fCommandLine);			break;
			case kVREntry_TiltAngleCmd: 
				free(((VRScriptAtAnglePtr)myPointer)->fCommandLine);			break;
			case kVREntry_FOVAngleCmd: 
				free(((VRScriptAtAnglePtr)myPointer)->fCommandLine);			break;
			case kVREntry_Variable: 
				free(((VRScriptVariablePtr)myPointer)->fVarName);				break;
			case kVREntry_TransitionEffect: 
				VREffects_DumpEntryMem((VRScriptTransitionPtr)myPointer);		break;
			case kVREntry_Generic: 
			case kVREntry_Unknown: 
			default: 
				return;
		}
		
		DisposePtr((Ptr)myPointer);
		myPointer = myNext;
	}
	
	*myListHead = NULL;
}


//////////
//
// VRScript_DeleteAllLists
// Delete all lists used by this window.
//
//////////

void VRScript_DeleteAllLists (WindowObject theWindowObject)
{
	UInt32		myCount;
	
	for (myCount = kVRScript_FirstEntryType; myCount <= kVRScript_FinalEntryType; myCount++)
		VRScript_DeleteListOfType(theWindowObject, myCount);
}


//////////
//
// VRScript_GetObjectByEntryID
// Get the (first) list entry of the specified type having a specified entry ID.
//
//////////

VRScriptGenericPtr VRScript_GetObjectByEntryID (WindowObject theWindowObject, UInt32 theEntryType, UInt32 theEntryID)
{
	ApplicationDataHdl	myAppData;
	VRScriptGenericPtr	myPointer = NULL;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);	
	if (myAppData == NULL)
		return(NULL);
	
	// walk our linked list to find the target object
	myPointer = (**myAppData).fListArray[theEntryType];
	while (myPointer != NULL) {
		if (myPointer->fEntryID == theEntryID)
			return(myPointer);
		myPointer = myPointer->fNextEntry;
	}
	
	return(NULL);
}


//////////
//
// VRScript_GetVariableEntry
// Get the (first) variable having the specified name.
//
//////////

VRScriptVariablePtr VRScript_GetVariableEntry (WindowObject theWindowObject, char *theVarName)
{
	ApplicationDataHdl	myAppData;
	VRScriptVariablePtr	myPointer = NULL;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);	
	if (myAppData == NULL)
		return(NULL);
	
	// walk our linked list to find the target object
	myPointer = (VRScriptVariablePtr)(**myAppData).fListArray[kVREntry_Variable];
	while (myPointer != NULL) {
		if (strcmp(myPointer->fVarName, theVarName) == 0)
			return(myPointer);
		myPointer = myPointer->fNextEntry;
	}
	
	return(NULL);
}


//////////
//
// VRScript_InstallHotSpotInterceptProc
// Install an intercept procedure for handling hot spot triggerings.
//
//////////

void VRScript_InstallHotSpotInterceptProc (QTVRInstance theInstance, WindowObject theWindowObject)
{
	QTVRInterceptUPP		myInterceptProc;
	
	myInterceptProc = NewQTVRInterceptProc(VRScript_HotSpotInterceptProc);
	QTVRInstallInterceptProc(theInstance, kQTVRTriggerHotSpotSelector, myInterceptProc, (long)theWindowObject, 0);
}


//////////
//
// VRScript_InstallMouseOverHotSpotProc
// Install an intercept procedure for handling mouse-over hot spots.
//
//////////

void VRScript_InstallMouseOverHotSpotProc (QTVRInstance theInstance, WindowObject theWindowObject)
{
	QTVRMouseOverHotSpotUPP		myInterceptProc;
	
	myInterceptProc = NewQTVRMouseOverHotSpotProc(VRScript_MouseOverHotSpotProc);
	QTVRSetMouseOverHotSpotProc(theInstance, myInterceptProc, (long)theWindowObject, 0);
}


//////////
//
// VRScript_InstallPrescreenRoutine
// Install a prescreen buffer imaging complete procedure.
//
//////////

void VRScript_InstallPrescreenRoutine (QTVRInstance theInstance, WindowObject theWindowObject)
{
	ApplicationDataHdl	myAppData;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData != NULL)
		QTVRSetPrescreenImagingCompleteProc(theInstance, (**myAppData).fPrescreenProc, (SInt32)theWindowObject, 0);
}


//////////
//
// VRScript_InstallBackBufferImagingProc
// Install a back buffer imaging procedure.
// (This routine might sometimes be called to move or resize the area of interest within the panorama.)
//
//////////

void VRScript_InstallBackBufferImagingProc (QTVRInstance theInstance, WindowObject theWindowObject)
{
	ApplicationDataHdl		myAppData;
	QTVRAreaOfInterest		myArea;
	float					myWidth, myHeight;
	VRScriptMoviePtr		myPointer;
	OSErr					myErr = noErr;
	
	if ((theInstance == NULL) || (theWindowObject == NULL)) 
		return;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL) 
		return;

	HLock((Handle)myAppData);
	
	// remove any existing back buffer imaging procedure and dispose of its routine descriptor
	if ((**myAppData).fBackBufferProc != NULL)
		VRScript_RemoveBackBufferImagingProc((**theWindowObject).fInstance, theWindowObject);

	// create a new routine descriptor
	(**myAppData).fBackBufferProc = NewQTVRBackBufferImagingProc(VRScript_BackBufferImagingProc);
	
	// set the area of interest:
	// the application data structure holds the desired width, center, size, and scale of the movie
	myPointer = (VRScriptMoviePtr)(**myAppData).fListArray[kVREntry_QTMovie];
	myWidth = myPointer->fMovieWidth * myPointer->fMovieScale;
	myHeight = myWidth * (((float)myPointer->fMovieRect.right) / ((float)myPointer->fMovieRect.bottom));
	
	if (myPointer->fUseMovieCenter) {
		// use the stored movie center
		myArea.panAngle = myPointer->fMovieCenter.x + (myWidth/2);
		myArea.tiltAngle = myPointer->fMovieCenter.y + (myHeight/2);
	} else {
		// center the movie on the current pan and tilt angles
		myArea.panAngle = QTVRGetPanAngle(theInstance) + (myWidth/2);
		myArea.tiltAngle = QTVRGetTiltAngle(theInstance) + (myHeight/2);
	}
	
	myArea.width = myWidth;
	myArea.height = myHeight;
	
	// *** insertion (B) ***
	
	// make sure we get called on every idle event, so we can keep playing the embedded movie;
	// also make sure we get called on every back buffer update
	if (myPointer->fCompositeMovie)
		myArea.flags = kQTVRBackBufferEveryIdle | kQTVRBackBufferEveryUpdate | kQTVRBackBufferAlwaysRefresh;
	else
		myArea.flags = kQTVRBackBufferEveryIdle | kQTVRBackBufferEveryUpdate;
			
	// install our procedure
	myErr = QTVRSetBackBufferImagingProc(theInstance, (**myAppData).fBackBufferProc, 1, &myArea, (SInt32)theWindowObject);

	// make sure we successfully installed the procedure;
	// if an error occurred, clear the saved routine descriptor.
	if (myErr != noErr)
		VRScript_RemoveBackBufferImagingProc((**theWindowObject).fInstance, theWindowObject);
	
	HUnlock((Handle)myAppData);
}
	

//////////
//
// VRScript_InstallInterceptRoutine
// Install our QTVR intercept procedure.
//
//////////

void VRScript_InstallInterceptRoutine (QTVRInstance theInstance, WindowObject theWindowObject)
{
	QTVRInterceptUPP	myInterceptProc;
	
	myInterceptProc = NewQTVRInterceptProc(VRScript_InterceptRoutine);	
	
	// we'll just use the same intercept proc for each intercepted procedure
	QTVRInstallInterceptProc(theInstance, kQTVRSetPanAngleSelector, myInterceptProc, (SInt32)theWindowObject, 0);
	QTVRInstallInterceptProc(theInstance, kQTVRSetTiltAngleSelector, myInterceptProc, (SInt32)theWindowObject, 0);
	QTVRInstallInterceptProc(theInstance, kQTVRSetFieldOfViewSelector, myInterceptProc, (SInt32)theWindowObject, 0);
	QTVRInstallInterceptProc(theInstance, kQTVRSetViewCenterSelector, myInterceptProc, (SInt32)theWindowObject, 0);
}


//////////
//
// VRScript_RemoveBackBufferImagingProc
// Remove the back buffer imaging procedure.
//
//////////

void VRScript_RemoveBackBufferImagingProc (QTVRInstance theInstance, WindowObject theWindowObject)
{
	ApplicationDataHdl		myAppData;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL) 
		return;

	// clear the existing back buffer imaging proc
	QTVRSetBackBufferImagingProc(theInstance, NULL, 0, NULL, 0);
#if TARGET_OS_MAC
	DisposeRoutineDescriptor((**myAppData).fBackBufferProc);
#endif
	(**myAppData).fBackBufferProc = NULL;
}


//////////
//
// VRScript_HotSpotInterceptProc
// An intercept procedure for handling hot spot triggerings.
//
//////////

PASCAL_RTN void VRScript_HotSpotInterceptProc (QTVRInstance theInstance, QTVRInterceptPtr theMsg, WindowObject theWindowObject, Boolean *theCancel)
{
	VRScriptClickHSPtr		myPointer;
	VRScriptClickHSPtr		myNext;
	ApplicationDataHdl		myAppData;
	UInt32					myNodeID;
	Boolean					myCancelLink = false;
	
	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;

	myNodeID = QTVRGetCurrentNodeID(theInstance);
	
	// walk the linked list and find any commands for this hot spot
	myPointer = (VRScriptClickHSPtr)(**myAppData).fListArray[kVREntry_ClickHS];
	while (myPointer != NULL) {
		myNext = myPointer->fNextEntry;
		
		if ((myPointer->fNodeID == myNodeID) || (myPointer->fNodeID == kVRAnyNode))
			if (myPointer->fHotSpotID == (UInt32)(theMsg->parameter[0])) {
				if (myPointer->fOptions != 0)
					myCancelLink = true;
				VRScript_ProcessScriptCommandLine(theWindowObject, myPointer->fCommandLine);
				VRScript_CheckForExpiredCommand(theWindowObject, (VRScriptGenericPtr)myPointer);
			}

		myPointer = myNext;
	}

	*theCancel = myCancelLink;
}


///////////
//
// VRScript_PrescreenRoutine
// 	* alter the position of the 3D sound listener based on the user's panning or tilting
// 	* set balance and volume of embedded QuickTime movie
//	* draw the rendered 3D scene into the prescreen buffer
//	* draw any enlisted overlay picture into the prescreen buffer
//
//////////

PASCAL_RTN OSErr VRScript_PrescreenRoutine (QTVRInstance theInstance, WindowObject theWindowObject)
{
	float				myPan;
	float				myTilt;
	float				myFOV;
	UInt32				myNodeID;
	ApplicationDataHdl	myAppData;

	// get the application-specific data associated with the window
	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return(paramErr);
		
	// get the current environment
	myPan = QTVRGetPanAngle(theInstance);
	myTilt = QTVRGetTiltAngle(theInstance);
	myFOV = QTVRGetFieldOfView(theInstance);
	myNodeID = QTVRGetCurrentNodeID(theInstance);

	// process any localized sounds in the current node
	if ((**myAppData).fViewHasChanged || (**myAppData).fSoundHasChanged) 
		if ((**myAppData).fListArray[kVREntry_Sound] != NULL) {
			if (gHasSoundSprockets) {
#if SOUNDSPROCKET_AVAIL
				TQ3Vector3D			myOrientation;
			
				// figure out the orientation	
				myOrientation.x = -sin(myPan) * cos(myTilt);
				myOrientation.y = sin(myTilt);
				myOrientation.z = -cos(myPan) * cos(myTilt);
					
				// set the new orientation of the listener
				SSpListener_SetOrientation((**myAppData).fListener, &myOrientation);
				// update the virtual audio environment
				VRSound_Update3DSoundEnv(theWindowObject);
#endif
			} else {
				// we don't have SoundSprockets, so....
				// handle sound balance and volume attentuation ourselves
				VRSound_SetBalanceAndVolume(theWindowObject, myPan, myTilt);
			}
		}
		
#if QD3D_AVAIL
	// process any 3D objects in the current node
	if (gHasQuickDraw3D)
		if ((**myAppData).fListArray[kVREntry_QD3DObject] != NULL)
			VR3DObjects_PrescreenRoutine(theInstance, theWindowObject);
#endif

	// handle directionality and volume attenuation for QuickTime movie sound and texture-mapped QuickTime movie sound
	if ((**myAppData).fViewHasChanged || (**myAppData).fSoundHasChanged)
		if (((**myAppData).fListArray[kVREntry_QTMovie] != NULL) || ((**myAppData).fListArray[kVREntry_QD3DObject] != NULL))
			VRMoov_SetAllBalanceAndVolume(theWindowObject, myPan, myTilt);

	// draw any overlay pictures
	VRPicture_DrawNodePictures(theWindowObject);

	// reset our flags
	(**myAppData).fViewHasChanged = false;
	(**myAppData).fSoundHasChanged = false;
	
	return(noErr);
}


//////////
//
// VRScript_BackBufferImagingProc
// The back buffer imaging procedure:
//	* handle any QuickTime movies playing in the back buffer
//	* handle any pictures embedded in the back buffer [to be provided]
//
//////////

PASCAL_RTN OSErr VRScript_BackBufferImagingProc (QTVRInstance theInstance, Rect *theRect, UInt16 theAreaIndex, UInt32 theFlagsIn, UInt32 *theFlagsOut, WindowObject theWindowObject)
{
#pragma unused(theAreaIndex)

	ApplicationDataHdl		myAppData;
	OSErr					myErr = noErr;
	
	if ((theInstance == NULL) || (theWindowObject == NULL)) 
		return(paramErr);

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL) 
		return(paramErr);

	// if we've got an active movie to play, do so
	if (VRMoov_GetEmbeddedVideo(theWindowObject) != NULL)
		myErr = VRMoov_BackBufferImagingProc(theInstance, theRect, theAreaIndex, theFlagsIn, theFlagsOut, theWindowObject);
			
	return(myErr);
}


//////////
//
// VRScript_InterceptRoutine
// 	* signal that the view parameters have changed
//
//////////

PASCAL_RTN void VRScript_InterceptRoutine (QTVRInstance theInstance, QTVRInterceptPtr theMsg, WindowObject theWindowObject, Boolean *cancel)
{
#pragma unused(theInstance)

	Boolean				myCancelInterceptedProc = false;	// true == do NOT call thru; false == call thru
	ApplicationDataHdl	myAppData;
	
	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;

	switch (theMsg->selector) {
		case kQTVRSetPanAngleSelector:
		case kQTVRSetTiltAngleSelector:
		case kQTVRSetFieldOfViewSelector:
		case kQTVRSetViewCenterSelector:
			(**myAppData).fViewHasChanged = true;
			break;
			
		default:
			break;
	}
	
	*cancel = myCancelInterceptedProc;
}


//////////
//
// VRScript_EnteringNodeProc
// A node-entering procedure:
//	* initialize node-specific global variables (e.g., the node timer)
//	* execute the standard node-entry procedure
//	* execute any node-entry commands
//	* execute any custom transitions from the previous node to the current node
//
//////////

PASCAL_RTN OSErr VRScript_EnteringNodeProc (QTVRInstance theInstance, UInt32 theNodeID, WindowObject theWindowObject)
{
	VRScriptNodeInPtr		myPointer = NULL;
	VRScriptNodeInPtr		myNext = NULL;
	ApplicationDataHdl		myAppData;

	// initialize our node timer
	gNodeStartingTime = TickCount();

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return(paramErr);

	// call the standard node-entry procedure
	if ((**theWindowObject).fController != NULL)
		QTVRUtils_StandardEnteringNodeProc(theInstance, theNodeID, (**theWindowObject).fController);

	// node-entry commands might invoke QTVRUpdate, so enable update-streaming
	if (QTVRUtils_IsPanoNode(theInstance))
		QTVRBeginUpdateStream(theInstance, kQTVRCurrentMode);

	// walk the node-entry command list and execute any that apply to this node
	myPointer = (VRScriptNodeInPtr)(**myAppData).fListArray[kVREntry_NodeEntry];
	while (myPointer != NULL) {
		myNext = myPointer->fNextEntry;
		
		if ((myPointer->fNodeID == theNodeID) || (myPointer->fNodeID == kVRAnyNode)) {
			VRScript_ProcessScriptCommandLine(theWindowObject, myPointer->fCommandLine);
			VRScript_CheckForExpiredCommand(theWindowObject, (VRScriptGenericPtr)myPointer);
		}
	
		myPointer = myNext;
	}
	
	// disable update-streaming, if previously enabled
	if (QTVRUtils_IsPanoNode(theInstance))
		QTVREndUpdateStream(theInstance);

	// if a node transition effect is pending, force the movie image to be updated;
	// this draws the destination image into an offscreen GWorld
	if ((**myAppData).fActiveTransition != NULL)
		MoviesTask((**theWindowObject).fMovie, 0L);
	
	// execute any custom transitions from the previous node to the current node
	if (gHasQTVideoEffects)
		VREffects_RunTransitionEffect(theWindowObject);

	return(noErr);
}


//////////
//
// VRScript_LeavingNodeProc
// A node-leaving procedure:
//	* execute any node-exit commands
//	*stop any sounds attached to the current node
//	* dump any pictures attached to the current node
//	* dump any movies attached to the current node
//	* dump any unexpired AtTime commands that shouldn't be orphaned
//	* set up for any custom transitions from the current node to the target node

//////////

PASCAL_RTN OSErr VRScript_LeavingNodeProc (QTVRInstance theInstance, UInt32 fromNodeID, UInt32 toNodeID, Boolean *theCancel, WindowObject theWindowObject)
{
	VRScriptNodeOutPtr		myPointer;
	VRScriptNodeOutPtr		myNext;
	ApplicationDataHdl		myAppData;
	Boolean					myCancelExit = false;
	
	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return(paramErr);
		
	// walk the node-leaving command list and execute any that apply to this node
	myPointer = (VRScriptNodeOutPtr)(**myAppData).fListArray[kVREntry_NodeExit];
	while (myPointer != NULL) {
		myNext = myPointer->fNextEntry;
		
		if ((myPointer->fFromNodeID == fromNodeID) || (myPointer->fFromNodeID == kVRAnyNode))
			if ((myPointer->fToNodeID == toNodeID) || (myPointer->fToNodeID == kVRAnyNode)) {
				VRScript_ProcessScriptCommandLine(theWindowObject, myPointer->fCommandLine);
				if (myPointer->fOptions != 0)
					myCancelExit = true;
				VRScript_CheckForExpiredCommand(theWindowObject, (VRScriptGenericPtr)myPointer);
			}
	
		myPointer = myNext;
	}
	
	// set output parameter to indicate whether we're leaving the node or not
	*theCancel = myCancelExit;
	
	// if we're not leaving the node, just return
	if (myCancelExit)
		return(noErr);
	
	// set up for any custom transitions from the current node to the target node
	// (since we're going to get the first frame of our effect by rendering a frame
	// of the movie into an offscreen GWorld, we need to do this before we dump the
	// node pictures, movies, etc.)
	if (gHasQTVideoEffects)
		VREffects_SetupTransitionEffect(theWindowObject, fromNodeID, toNodeID);
	
	// if we really are leaving the node, dump node-specific sounds, pictures, and movies
	VRSound_DumpNodeSounds(theWindowObject);
	VRPicture_DumpNodePictures(theWindowObject);
	VRMoov_DumpNodeMovies(theWindowObject);
	
	// dump any unexpired time-based commands that are marked as to-be-killed
	VRScript_DumpUnexpiredCommands(theWindowObject, fromNodeID);
	
	// work around a bug in QTVR: 
	// moving from a panoramic node to an object node effectively hoses the prescreen routine,
	// so we need to reinstall that routine if we're moving from an object to a panoramic node
	if (QTVRGetNodeType(theInstance, fromNodeID) == kQTVRObjectType)
		if (QTVRGetNodeType(theInstance, toNodeID) == kQTVRPanoramaType)
			VRScript_InstallPrescreenRoutine(theInstance, theWindowObject);

	return(noErr);
}


//////////
//
// VRScript_MouseOverHotSpotProc
// Walk through our linked list of mouse-over hot spot commands and see whether any need to be executed.
//
// We support hot spots specified by ID (myPointer->fSelectByID == true) or by type (myPointer->fSelectByID == false).
//
//////////

PASCAL_RTN OSErr VRScript_MouseOverHotSpotProc (QTVRInstance theInstance, UInt32 theHotSpotID, UInt32 theFlags, WindowObject theWindowObject)
{
	VRScriptAtMOHSPtr		myPointer;
	VRScriptAtMOHSPtr		myNext;
	ApplicationDataHdl		myAppData;
	UInt32					myNodeID;
	OSType					myType;
	
	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return(paramErr);
	
	// get the current node ID
	myNodeID = QTVRGetCurrentNodeID(theInstance);
	
	// get the type of the hot spot, given its ID
	QTVRUtils_GetHotSpotType(theInstance, myNodeID, theHotSpotID, &myType);
	
	// walk the linked list and find any commands for this hot spot
	myPointer = (VRScriptAtMOHSPtr)(**myAppData).fListArray[kVREntry_MouseOverHS];
	while (myPointer != NULL) {
		myNext = myPointer->fNextEntry;
		
		if ((myPointer->fNodeID == myNodeID) || (myPointer->fNodeID == kVRAnyNode))
			if (((myPointer->fSelectByID) && (myPointer->fHotSpotID == theHotSpotID))	// hot spot is specified by ID
			|| ((!myPointer->fSelectByID) && (myPointer->fHotSpotType == myType)))		// hot spot is specified by type
				if (myPointer->fOptions == theFlags) {
					VRScript_ProcessScriptCommandLine(theWindowObject, myPointer->fCommandLine);
					VRScript_CheckForExpiredCommand(theWindowObject, (VRScriptGenericPtr)myPointer);
				}
	
		myPointer = myNext;
	}

	return(noErr);		// returning noErr means QTVR should do its normal mouse-over-hot-spot stuff
}


//////////
//
// VRScript_CheckForTimedCommands
// Walk through our linked list of timed commands and see whether any need to be executed.
//
//////////

void VRScript_CheckForTimedCommands (WindowObject theWindowObject)
{
	ApplicationDataHdl		myAppData;
	UInt32					myNodeID;
	VRScriptAtTimePtr		myPointer;
	VRScriptAtTimePtr		myNext;
	Boolean					myRunCommand;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;

	myNodeID = QTVRGetCurrentNodeID((**theWindowObject).fInstance);
	
	// walk the linked list and find any commands ready to be executed
	myPointer = (VRScriptAtTimePtr)(**myAppData).fListArray[kVREntry_TimedCommand];
	while (myPointer != NULL) {
		myNext = myPointer->fNextEntry;
		myRunCommand = false;

		if (myPointer->fMode == kVRUseNodeTime) {
			if ((myPointer->fNodeID == myNodeID) || (myPointer->fNodeID == kVRAnyNode))
				if (gNodeElapsedTime >= myPointer->fTime)
					myRunCommand = true;
		} else if (myPointer->fMode == kVRUseAbsoluteTime) {
			if (gAbsoluteElapsedTime >= myPointer->fTime)
				myRunCommand = true;
		} else if (myPointer->fMode == kVRUseInstallTime) {
			if ((gAbsoluteElapsedTime - myPointer->fTimeInstalled) >= myPointer->fTime)
				myRunCommand = true;
		}
		
		if (myRunCommand) {
			VRScript_ProcessScriptCommandLine(theWindowObject, myPointer->fCommandLine);
			if (myPointer->fRepeat) {
				myPointer->fTime += myPointer->fPeriod;
				VRScript_CheckForExpiredCommand(theWindowObject, (VRScriptGenericPtr)myPointer);
			} else {
				VRScript_DelistEntry(theWindowObject, (VRScriptGenericPtr)myPointer);
			}
		}
		
		myPointer = myNext;
	}
}


//////////
//
// VRScript_CheckForAngleCommands
// Walk through our linked lists of angle-triggered commands and see whether any need to be executed.
//
//////////

void VRScript_CheckForAngleCommands (WindowObject theWindowObject)
{
	ApplicationDataHdl		myAppData;
	VRScriptAtAnglePtr		myNext;
	VRScriptAtAnglePtr		myPointer;
	UInt32					myNodeID;
	float					myPan;
	float					myTilt;
	float					myFOV;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;
		
	// get the current environment
	myPan = QTVRGetPanAngle((**theWindowObject).fInstance);
	myTilt = QTVRGetTiltAngle((**theWindowObject).fInstance);
	myFOV = QTVRGetFieldOfView((**theWindowObject).fInstance);
	myNodeID = QTVRGetCurrentNodeID((**theWindowObject).fInstance);

	// walk the pan angle linked list
	myPointer = (VRScriptAtAnglePtr)(**myAppData).fListArray[kVREntry_PanAngleCmd];
	while (myPointer != NULL) {
		myNext = myPointer->fNextEntry;

		if ((myPointer->fNodeID == myNodeID) || (myPointer->fNodeID == kVRAnyNode))
			if (myPointer->fMinAngle <= myPan)
				if (myPointer->fMaxAngle >= myPan) {
					VRScript_ProcessScriptCommandLine(theWindowObject, myPointer->fCommandLine);
					VRScript_CheckForExpiredCommand(theWindowObject, (VRScriptGenericPtr)myPointer);
				}
		myPointer = myNext;
	}

	// walk the tilt angle linked list
	myPointer = (VRScriptAtAnglePtr)(**myAppData).fListArray[kVREntry_TiltAngleCmd];
	while (myPointer != NULL) {
		myNext = myPointer->fNextEntry;

		if ((myPointer->fNodeID == myNodeID) || (myPointer->fNodeID == kVRAnyNode))
			if (myPointer->fMinAngle <= myTilt)
				if (myPointer->fMaxAngle >= myTilt) {
					VRScript_ProcessScriptCommandLine(theWindowObject, myPointer->fCommandLine);
					VRScript_CheckForExpiredCommand(theWindowObject, (VRScriptGenericPtr)myPointer);
				}
		myPointer = myNext;
	}

	// walk the zoom angle linked list
	myPointer = (VRScriptAtAnglePtr)(**myAppData).fListArray[kVREntry_FOVAngleCmd];
	while (myPointer != NULL) {
		myNext = myPointer->fNextEntry;

		if ((myPointer->fNodeID == myNodeID) || (myPointer->fNodeID == kVRAnyNode))
			if (myPointer->fMinAngle <= myFOV)
				if (myPointer->fMaxAngle >= myFOV) {
					VRScript_ProcessScriptCommandLine(theWindowObject, myPointer->fCommandLine);
					VRScript_CheckForExpiredCommand(theWindowObject, (VRScriptGenericPtr)myPointer);
				}
		myPointer = myNext;
	}

}


//////////
//
// VRScript_CheckForExpiredCommand
// See whether the specified command entry has expired.
//
//////////

void VRScript_CheckForExpiredCommand (WindowObject theWindowObject, VRScriptGenericPtr thePointer)
{
	if (thePointer->fMaxExecutions != kVRDoIt_Forever)
		thePointer->fMaxExecutions--;
	if (thePointer->fMaxExecutions == 0)
		VRScript_DelistEntry(theWindowObject, (VRScriptGenericPtr)thePointer);
}


//////////
//
// VRScript_DumpUnexpiredCommands
// Dump any unexpired AtTime commands, if the list entry so indicates.
// This is called only when leaving a node, to removed unexecuted ("orphaned") commands
// which would be executed the next time the user entered the node.
//
//////////

void VRScript_DumpUnexpiredCommands (WindowObject theWindowObject, UInt32 theNodeID)
{
	ApplicationDataHdl		myAppData;
	VRScriptAtTimePtr		myPointer;
	VRScriptAtTimePtr		myNext;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;

	// walk the linked list of timed commands and find any commands to be removed from list
	myPointer = (VRScriptAtTimePtr)(**myAppData).fListArray[kVREntry_TimedCommand];
	while (myPointer != NULL) {
		myNext = myPointer->fNextEntry;

		if (myPointer->fNodeID == theNodeID)
			if (myPointer->fOptions == kVROrphan_Kill)
				VRScript_DelistEntry(theWindowObject, (VRScriptGenericPtr)myPointer);
		
		myPointer = myNext;
	}
}


//////////
//
// VRScript_PackString
// Pack a string.
//
//////////

void VRScript_PackString (char *theString)
{
	short		myLength;
	short		myIndex;
	
	myLength = strlen(theString);
	for (myIndex = 0; myIndex < myLength; myIndex++)
		if (isspace(theString[myIndex]))
			theString[myIndex] = '#';
}


//////////
//
// VRScript_UnpackString
// Unpack a string (for instance, to make a packed command line suitable for parsing by VRScript_ProcessScriptCommandLine).
//
// We allow embedding commands in other commands: '*' -> '&' -> '%' -> '#' -> ' '
//
//////////

void VRScript_UnpackString (char *theString)
{
	short		myLength;
	short		myIndex;
	
	myLength = strlen(theString);
	for (myIndex = 0; myIndex < myLength; myIndex++) {
		if (theString[myIndex] == '#')
			theString[myIndex] = ' ';
		if (theString[myIndex] == '%')
			theString[myIndex] = '#';
		if (theString[myIndex] == '&')
			theString[myIndex] = '%';
		if (theString[myIndex] == '*')
			theString[myIndex] = '&';
	}
}


//////////
//
// VRScript_DecodeString
// Decode a string, assumed to have been encoded using a simple rotate-11 scheme.
//
//////////

void VRScript_DecodeString (char *theString)
{
	short		myLength;
	short		myIndex;
	
	myLength = strlen(theString);
	for (myIndex = 0; myIndex < myLength; myIndex++)
		if (theString[myIndex] != '\n')
			theString[myIndex] = theString[myIndex] - 11;
}


//////////
//
// VRScript_StringToOSType
// Convert a string to an OSType.
//
//////////

OSType VRScript_StringToOSType (char *theString)
{
	unsigned long		myType = 0L;
	
	if (strlen(theString) < kMaxOSTypeLength - 1)
		return((OSType)myType);
		
	myType += theString[3] << 0;
	myType += theString[2] << 8;
	myType += theString[1] << 16;
	myType += theString[0] << 24;
	
	return((OSType)myType);
}


//////////
//
// VRScript_OSTypeToString
// Convert an OSType to a string.
//
// The caller is responsible for disposing of the pointer returned by this function (by calling free).
//
//////////

char *VRScript_OSTypeToString (OSType theType)
{
	char			*myString = malloc(kMaxOSTypeLength);
	
	if (myString == NULL)
		return(NULL);
		
	myString[0] = (theType & 0xff000000) >> 24;
	myString[1] = (theType & 0x00ff0000) >> 16;
	myString[2] = (theType & 0x0000ff00) >> 8;
	myString[3] = (theType & 0x000000ff) >> 0;
	myString[4] = '\0';
	
	return(myString);
}