// Persistence of Vision Ray Tracer Scene Description File
// File: rope.inc
// Vers: 3.6
// Desc: Include file for a rope macro.
// Date: Feb 2006
// Auth: Bruno Cabasson

#ifndef (_rope_inc)
#declare _rope_inc = version;

#include "math.inc"

// Quality constants (per length unit)
#declare LOW_DENSITY    = 100;
#declare MEDIUM_DENSITY = 200;
#declare HIGH_DENSITY   = 300;

// The main blob macro that builds a thread
#macro O_Thread(_spline, _segments, _thread_radius, _strand_radius, _thread_frequency, _strand_phase, _rope_radius, _strand_frequency, _rope_phase)
blob
{
    #local n = _segments;
    #local thres = 0.65;
    #local thread_radius = _thread_radius/sqrt(1-sqrt(thres)); // See field formula for blobs.
    
    // Compute first direction
    #local p0 = _spline(0);
    #local p1 = _spline(1/n);
    #local dir = p1-p0;
    #local prev_dir = dir;
    
    // Compute initial vectors for twisting threads and strands
    #local p_perp = vnormalize(VPerp_To_Vector(dir));
    #local p_ref1 = _strand_radius*p_perp;
    #local p_ref2 = _rope_radius*p_perp;

    sphere {p0+p_ref1+p_ref2, thread_radius, 1}
    
    // Loop over segments
    #local i = 1;
    #while (i<=n)
        // Compute direction for current segment
        #local p0 = p1;
        #local p1 = _spline((i-1)/n);
        #local dir = p1-p0;
        #local cross = vcross (prev_dir, dir);
        
        // Compute displacement vectors for current segment
        #if (vlength(cross)>0)
            #local rot = VAngleD (prev_dir, dir);
            #local p_ref1 = vaxis_rotate(p_ref1, cross, rot);
            #local p_ref2 = vaxis_rotate(p_ref2, cross, rot);
        #end
        
        #local k = 2*pi*i/n;
        #local thread_disp = vaxis_rotate(p_ref1, dir, degrees(-k*_thread_frequency) + _strand_phase);
        #local strand_disp = vaxis_rotate(p_ref2, dir, degrees(k*_strand_frequency) + _rope_phase);
       
        // Add blob element
        sphere {p0+thread_disp+strand_disp, thread_radius, 1}
        
        // Prepare next segment
        #local prev_dir = dir;

        #local i=i+1;
    #end

    threshold thres
    sturm
}
#end

// Intermediate macro for strand
#macro O_Strand (_spline, _length, _nb_threads, _thread_radius, _strand_radius, _thread_frequency, _rope_radius, _strand_frequency, _rope_phase, _density)
union
{
    // strand = twisted threads
    #local i=0;
    #while (i<_nb_threads)
        object {O_Thread(_spline, _length*_density, _thread_radius, _strand_radius, _thread_frequency, degrees(2*pi*i/_nb_threads), _rope_radius, _strand_frequency, _rope_phase)}
        #local i=i+1;
    #end
}
#end

#macro SplineLength(_spline, _segments)
    #local ret = 0;
    #local p0 = _spline(0);
    #local i=1;
    #while (i<=_segments)
        #local p1 = _spline(i/_segments);
        #local ret = ret + vlength(p1-p0);
        #local p0 = p1;
        #local i=i+1;
    #end
    ret
#end


// --------------------------------------------------------------------------------------
/*
    Macro that builds a rope that follows a given spline.
    
    The rope is composed of twisted strands of twisted threads. Each thread is a blob.
     
    The macro automatically calculates the radii for strands and rope,
    as well as the linear length of the spline (by integrating segments).
    
    Parameters:
        
        _spline:            a #declared spline
        _nb_strands:        number of strands in the rope
        _nb_threads:        number of threads per strand
        _thread_radius:     radius of a single thread
        _thread_frequency:  number of turns by length unit for a thread in its strand
        _strand_frequency:  number of turns by length unit for a strand in the rope
        _density:           density of blob elements per length unit. See constants above.
    
*/
#macro MakeRope (_spline, _nb_strands, _nb_threads, _thread_radius, _thread_frequency, _strand_frequency, _rope_quality)
union
{
    #local l = SplineLength(_spline, 1000);
    #local strand_radius = _thread_radius/sin(pi/_nb_threads);
    #local rope_radius = strand_radius/sin(pi/_nb_strands)+strand_radius;
    #local i=0;
    #while (i<_nb_strands)
        object {O_Strand(_spline, l, _nb_threads, _thread_radius, strand_radius, l*_thread_frequency, rope_radius, l*_strand_frequency, degrees(2*pi*i/_nb_strands), _rope_quality)}
        #local i=i+1;
    #end
}
#end
// --------------------------------------------------------------------------------------

#end