// -------------------------------------------------------------------------
// Crystals Cluster v1.0
// --
// Jaime Vives Piqueres, Feb. 2016
// --
// License: http://creativecommons.org/licenses/by-sa/4.0/
// --
// Criticism, comments and suggestions welcome at jaime@ignorancia.org or at
// this forum thread:
// http://news.povray.org/povray.binaries.images/thread/%3C56c46524%40news.povray.org%3E/
// --
// If there is ever any update to this code, it will be hosted at 
// http://www.ignorancia.org
// -------------------------------------------------------------------------
#version 3.7;

#include "rad_def.inc"
#include "transforms.inc"
#include "CIE.inc"

// *** control center ***
#declare use_blur=7*0;        // blur samples (0=off)
#declare use_area=0;          // area light on/off
#declare use_phot=0;          // photons 0:off, 1:load, 2:save
#declare use_disp=1.05*0;     // dispersion 0:off, >1:on & amount
#declare use_media=0;         // crystal media 0:off, 1:on-bottom, 2:on-top, 3:on-uniform
#declare media_int=1;         // media intensity
#declare use_crystal_color=0; // texture color on/off (will use same color as media)
#declare saturation=.9;       // color saturation if the above is set
#declare use_irid=0;          // iridescence 0:off, >0:on & thickness amount
#declare test_text=0;         // use simple opaque texture for tests
#declare r_t=seed(1693);      // seed for random crystals placement
#declare r_c=seed(2750);      // seed for random crystal color (texture & media)

global_settings{
  max_trace_level 9 
  //radiosity{Rad_Settings(Radiosity_Normal, off, off)}
  //radiosity{Rad_Settings(Radiosity_Fast, off, off)}
  #if (use_phot)
    photons{
      spacing .025
      #if (use_phot>1)
        save_file "saved.ph"
      #else
        load_file "saved.ph"
      #end
    }
  #end
}
#default{texture{finish{ambient 0 emission 0 diffuse 1}}}


// *** base rock ***
// -------------------------------------------------------------------------
// the crystal prisms are traced against this object, so their orientations
// resemble the growth of natural crystals
// -------------------------------------------------------------------------
#declare base_width=50;
#declare rock_base=
  isosurface{
    function{
      f_sphere(x,y*2,z,1)
      +f_bumps(x*2,y*2,z*2)*.33
      -f_granite(x,y,z)*.25
    }
    max_gradient 3
    contained_by{sphere{0,1}}
    scale base_width*.5
}
object{rock_base
  texture{
    pigment{
      // you will get an error here: please use any rock/soil image you have at hand
      image_map{jpeg "taller-piedra_multicolorrustica"}
      warp{spherical}
    }
  }
}

// *** crystals ***
// -------------------------------------------------------------------------
// created at the origin, traced against the rock and then reoriented and 
// placed acordingly to the data obtained
// -------------------------------------------------------------------------
// crystal settings
#declare prism_facets=6;  // only 4 or 6 supported at the moment
#declare num_crystals=40;
#declare max_width=4;
#declare min_width=2;
#declare max_height=30;
#declare min_height=1;
#declare cap_extra_factor=-.1;
#declare efective_base_width=10; // crystals are trace only on this area
#declare crystal_color=rgb vnormalize(Wavelength2RGB(380+380*rand(r_c))*rand(r_c)+Wavelength2RGB(380+380*rand(r_c))*rand(r_c)+(1-saturation));
// generation loop
#declare start_y=100;
#declare cnt_crystals=0;
#while (cnt_crystals<num_crystals)
  // set this crystal dimensions randomly
  #declare this_width=min_width+(max_width-min_width)*rand(r_t);
  #declare this_height=(min_height+(max_height-min_height)*rand(r_t)*rand(r_t));
  #declare cap_extra=this_width*cap_extra_factor;
  // create the prism at origin   
  #switch (prism_facets)
    #case (4)
      #include "prism_cubic.inc"
      #break
    #case (6)
      #include "prism_hexagonal.inc" 
      #break
    #else
      #include "prism_cubic.inc"
      #break
  #end
    inside_vector <1,0,0>
  }
  // trace base point
  #declare norm=<0,0,0>;
  #declare start=<
    -efective_base_width*.5 + efective_base_width*rand(r_t),
    start_y,
    -efective_base_width*.5 + efective_base_width*rand(r_t)
  >;
  #declare inter=trace(rock_base,start,-y,norm);  
  // place and reorient the crystal
  #if (vlength(norm)!=0) 
  // intersection with some random planes to avoid boring regular caps
  intersection{
    plane{y,this_height rotate (15+15*rand(r_t))*x rotate 360*rand(r_t)*y}
    plane{y,this_height rotate (15+15*rand(r_t))*x rotate 360*rand(r_t)*y}
    object{this_prism}    
    hollow
    texture{
    #if(test_text)
      pigment{White}
    #else
      pigment{
        granite scale 10
        color_map{
        [0 rgb 1 transmit .99]
        [1 rgb 1 transmit .85]
        }
        turbulence .33
      }
      finish{
       specular 0.6
       diffuse 0.4
       reflection{0,1 fresnel}
       #if(use_irid)
         irid{0.25 thickness use_irid}
       #end
       conserve_energy
      }
    #end
      normal{
        marble turbulence .33
        normal_map{
          //[0 crackle solid scale .1*y]
          [0 cells .2 scale 2 turbulence .33]
          [0.5 wrinkles 0]
          [1 granite .1 scale 1 turbulence .1]
        }
      }
      rotate 360*rand(r_t)*y
      translate 100*rand(r_t)
    }
    interior{ior 1.54  // quartz-like ior
      #if(use_crystal_color)
        fade_distance 1 fade_power 1002 fade_color crystal_color
      #end
      #if(use_disp)
        dispersion use_disp
      #end
      #if(use_media)
        media{
          intervals 1 samples 2,16
          scattering{1, 
            crystal_color*media_int
          }
          absorption .5*media_int
          density{
            spherical turbulence .33
            density_map{
              #if(use_media=1)
                [rand(r_c) rgb 0]
                [1 granite scale .1]
              #end
              #if(use_media=2)
                [0 granite scale .1]
                [rand(r_c) rgb 0]
              #end
              #if(use_media=3)
                [0 rgb 1]
                [1 rgb 1]
              #end
            }
            scale this_height
          }
        }
      #end
    }
    photons{
     target
     refraction on
     reflection on
     collect off
    }
    Reorient_Trans(y,norm)
    translate inter-.5*y // translated a bit down to embed them on the rock
  }
  #declare cnt_crystals=cnt_crystals+1;
  #end
#end


// *** lighting ***
sphere{
 0,500
 texture{
  // http://www.openfootage.net/?tag=360degree
  pigment{image_map{hdr "OpenfootageNET_Croatia_low_hdr" map_type 1 interpolate 2}}
  finish{emission .5 diffuse 0}
  rotate 0*y
 }
 hollow no_shadow 
}
#declare r_l=seed(5691);
light_source { 
 <0,490,0>
 rgb 150000
 #if (use_area)
   area_light 40*x,40*y,6,6 jitter adaptive 1 orient circular
 #end
 fade_distance 1
 fade_power 2
 rotate 90*rand(r_l)*x
 rotate 360*rand(r_l)*y
} 


// *** camera ***
#declare clo=<80,46,-9>;
#declare cla=<0,20,0>;
camera{
  location clo
  up 1*y
  right 4/3*x
  angle 28.8 // 28.8 39.6 54.4
  look_at cla
  #if (use_blur)
    focal_point cla
    aperture pi*2
    blur_samples use_blur
  #end
}
