/*
 ____________________________
  OTOc.inc 'Odd Tiles Object'
          version 3
      2011 Samuel Benge
 
 
 This macro will construct a wall of interlocking, irregularly shaped bricks in the x/y plane.
 It will integerize your corner settings.
 
 
 The Macro
 
  OTO(C1, C2, Bev, Texture)
   *C1      = corner 1, a 3D vector
   *C2      = corner 2, a 3D vector
   *Bev     = amount of edge rounding
   *Texture = a texture that will be scaled/translated randomly for each brick  
  
   Min_Extent.x must be less than Max_Extent.x, likewise for the y components. The z
   components control the depth, or thickness.
  
   Example:
  
    OTO( <-5,-5,0>, <5,5,1>, 0.1, texture{pigment{granite scale 10}} )
  
  
  Declarables
  
   #macro OTO_Object(C1, C2, Bev)
    *some object between <0,0,0>-<1,1,1>
     -or-
    *some object with two corners (like a Wire_Box_Union)
    *the optional OTO_Transform_Brick() macro (for objects like superellipsoids)
   #end
   
    You can create a macro named "OTO_Object" to make your own brick types. If you wish to
    give them new materials, make a union where one object is invisible, and the other has
    the material. See "OTOTest.pov" for an example.
   
   OTO_Seed
    
    A random seed to change the overall apperance. #declare it before calling OTO().
    
    Example:
     
      #declare OTO_Seed = seed(1007);
  
*/

#version 3.7;

#include "math.inc"

// default brick object
#macro OTO_Object(C1,C2,Bev)
 #include "shapes.inc"
 Round_Box_Union(C1,C2,Bev)
#end

/* When using a custom brick between <0,0,0>-<1,1,1>,
   call this to scale and translate it correctly.     */
#macro OTO_Transform_Brick()
 scale <C2.x-C1.x,C2.y-C1.y,C2.z-C1.z>
 translate Pt1+z*Min_Extent.z
#end

#ifndef(OTO_Offset_Cells) #declare OTO_Offset_Cells = <0,0,0>; #end
#ifndef(OTO_DE) #declare OTO_DE = false; #end

#macro OTO(Min_Extent2, Max_Extent2, Bev2, Texture)
 
 #local Min_Extent = <int(Min_Extent2.x),int(Min_Extent2.y),Min_Extent2.z>;
 #local Max_Extent = <int(Max_Extent2.x),int(Max_Extent2.y),Max_Extent2.z>;
 
 #local MinETmp = Min_Extent;
 #local MaxETmp = Max_Extent;
 
 #local Min_Extent = <min(MinETmp.x,MaxETmp.x)+.5, min(MinETmp.y,MaxETmp.y)+.5, Min_Extent.z>;
 #local Max_Extent = <max(MinETmp.x,MaxETmp.x)-.5, max(MinETmp.y,MaxETmp.y)-.5, Max_Extent.z>;
 
 #undef MinETmp #undef MaxETmp 
 
 #ifndef(OTO_Seed)
  #local OTO_Seed = seed(1005);
 #end
 
 #local Bev = max(0,min(.5,Bev2/2));
 
 #local OTO_FMask = function{ pattern{ pigment_pattern{ checker 0, 1 warp{planar} } } }
 
 // improved variation
 #ifndef(OTO_FOffs)
  #local OTO_FOffs =
   function{
    pattern{
     pigment_pattern{
      pigment_pattern{checker scale 2 warp{planar}}
      pigment_map{
       [0
        cells translate z*int(rand(OTO_Seed)*1000)
        color_map{[0 rgb -0.25][1 rgb .25]}
       ]
       [1
        cells translate z*int(rand(OTO_Seed)*1000)
        color_map{[0 rgb 0.75][1 rgb 1.25]}
       ]
      }
      warp{planar}
     }
    }
   }
 #end
 
 #macro OTO_Get_Offset(Cell)
  #local CX = Cell.x;
  #local CY = Cell.y;
  (1-Bev*2)*<1,1,0>*(.5-OTO_FOffs(CX,CY,0))
 #end

 #macro OTO_Get_Mask(Cell)
  #local CX = Cell.x;
  #local CY = Cell.y;
  OTO_FMask(CX,CY,0)
 #end
 
 union{
  #local Y=Min_Extent.y;
  #while(Y<=Max_Extent.y)
   #local X=Min_Extent.x;
   #while(X<=Max_Extent.x)
    #local Vec = <X-.5,Y-.5,0>;
   
    // corner finding
    #if(OTO_Get_Mask(Vec)=1 )
     #local Pt1 = Vec + y*(OTO_Get_Offset(Vec).y) + x*(OTO_Get_Offset(Vec+y).x);
     #local Pt2 = Vec + y*(OTO_Get_Offset(Vec+x+y).y) + x*(OTO_Get_Offset(Vec-y+x+y).x) +x+y;
    #else
     #local Pt1 = Vec + x*(OTO_Get_Offset(Vec).x) + y*(OTO_Get_Offset(Vec+x).y); 
     #local Pt2 = Vec + x*(OTO_Get_Offset(Vec+x+y).x) + y*(OTO_Get_Offset(Vec-x+x+y).y) +x+y;
    #end
    
    // edge trimming
    #if(X=Min_Extent.x) #local Pt1 = <Min_Extent.x-.5,Pt1.y>; #end
    #if(X=Max_Extent.x) #local Pt2 = <Max_Extent.x+.5,Pt2.y>; #end
    #if(Y=Min_Extent.y) #local Pt1 = <Pt1.x, Min_Extent.y-.5>; #end
    #if(Y=Max_Extent.y) #local Pt2 = <Pt2.x, Max_Extent.y+.5>; #end
     
    object{
     OTO_Object(Pt1+z*Min_Extent.z,Pt2+z*Max_Extent.z,Bev)
     texture{
      Texture
      scale .5+rand(OTO_Seed)
      translate 100*<rand(OTO_Seed),rand(OTO_Seed),rand(OTO_Seed)>
     }
    }
    
    #local X=X+1;
   #end
   #local Y=Y+1;
  #end
 }

#end