/*

 stbHexPatterns.inc
 #include"stbHexPatterns.inc"
 
 2008 Sam Benge stbenge@hotmail.com 
 
 Description: A few useful hexagonal and triangular patterns, plus a few trigonal
 positioning utilities. All the patterns are 2d; they are scaled infinitely in the
 y-direction. They can be used in any material block (texture, pigment, etc.).
 
 Dependencies:
  - functions.inc
  - POV-Ray 3.6 and above

Contents
 
 1.0 Identifiers
  1.1 float h1
  1.2 float h2
  1.3 pigment myHexPattern
 
 2.0 Vector macros
  2.1 ht(x,y)
  2.2 ht2(x,y)
 
 3.0 Pattern macros
  3.1 hexGrad()
  3.2 hexGradGrouped()
  3.3 randomHexCells()
  3.4 hexPattern(hprot)
  3.5 hexPatternGrouped()
  3.6 triAngle()
  3.7 triAngle2()
  3.8 triTessellation()
  3.9 kaleidoscope()
  3.10 skewedGrid()
  3.11 randomTriCells()

Contents Explained

 1.0 Identifiers
  
  1.1 float h1
   Offset value for placing elements in an triangular fashion. Do not redeclare.
  
  1.2 float h2
   Another offset value for placing elements in an triangular fashion. Useful for
   scaling user-defined patterns and placing a pattern to the corner or edge of an
   hexagon. Do not redeclare.
   
   Usage:
    #declare myHexPattern=
    pigment{
     cylindrical
     scale h2*2 - scales the pattern to fill an hexagon
     translate<h2/2,0,.5> - places the pattern at the further-right corner of an hexagon.
    }
  
  1.3 pigment myHexPattern
   A pattern used with the hexPattern() macro. It is automatically stretched
   infinitely in the y-direction. The default is a modified gradient pattern.
   
   Usage:
   #declare myHexPattern=
   pigment{
    gradient x
    scale .5
    rotate y*30
    triangle_wave
   }
 
 2.0 Vector macros
  
  2.1 ht(x,y)
   This macro returns a 2d vector aligned to a skewed grid (see skewedGrid()
   below). The points returned will be placed in trigonal space . It's perfectly
   suited for adding points to a prism object.
   
   Usage:
   // This example produces a triangular prism.
   prism{
    -1,1,3,
    ht( -1, 0.5 ),
    ht( 0.5, 0.5 ),
    ht( 0.5, -1 )
    pigment{rgb 1}
   }
  
  2.2 ht2(x,y)
   This is just like ht(), except that it returns a 3d vector suitable for placing
   objects, patterns, etc. All points returned will lie at y=0.
   
   Usage:
   // This example places six spheres in an hexagonal fashion.
   union{
    sphere{ht2( -1, 0 ),.25}
    sphere{ht2( -1, 1 ),.25}
    sphere{ht2( 0, 1 ),.25}
    sphere{ht2( 1, 0 ),.25}
    sphere{ht2( 1, -1 ),.25}
    sphere{ht2( 0, -1 ),.25}
    pigment{rgb 1}
   }
  
 3.0 Pattern macros

  3.1 hexGrad()
   This pattern is a shortcut to "f_hex_x" in functions.inc. The edge of each hexagonal
   cell is black, blending to white near the center. The gradient is adjusted to work
   with other patterns, making things like "hexGradGrouped()" (below) work. You can use it
   to create mortar between hexagons.
   
   Usage:
    my_object{
     pigment{
      hexGrad()
     }
    }

  3.2 hexGradGrouped()
   This is an hybrid between "hexGrad()" and "hexPatternGrouped()" (below). You can use it
   to create a mortar between groups of hexagons.
  
   Usage:
    my_object{
     pigment{
      hexGradGrouped()
     }
    }
 
  3.3 randomHexCells()
   This is an hybrid between POV-Ray's "cells" and "hexagon" patterns. Each hexagonal cell
   is filled with a random shade of gray.
   
   Usage:
    my_object{
     pigment{
      randomHexCells()
      pigment_map{
       [.5 crackle metric 1]
       [.5 crackle form x]
      }
     }
    }
 
  3.4 hexPattern(hprot)
   A pigment which places a user defined pattern (myHexPattern) in each hexagonal cell.
   There is an option to have the pattern in each cell randomly rotated by 0 degrees, 0,
   120 and 240 degrees or 0, 60, 120, 180, 240 and 300 degrees.
   
   hprot: rotate randomly? 0 = no, 1 = rotate by 120 degrees, 2 = rotate by 60 degrees
   
   Usage:
    my_object{
     pigment{
      hexPattern(2)
     }
    }
 
  3.5 hexPatternGrouped()
   Same as "hexPattern()", but places the pattern in groups of three to form a more complex
   tessellation. It does not support random rotation.
  
   Usage:
    #declare myHexPattern=
    pigment{
     hexGrad()
     triangle_wave
     scale h2*2
     translate<-h2/2,0,-.5>
    }
    my_object{
     pigment{
      hexPatternGrouped()
      color_map{[.5 rgb 0][1 rgb 1]}
     }
    }
  
  3.6 triAngle()
   A triangular version of POV's "hexagon" pattern. It's composed of a checker pattern of
   triangles alternating between black/white. It is not pigment list friendly.
   
   Usage:
    my_object{
     pigment{
      triAngle()
      pigment_map{
       [0 crackle metric 1]
       [1 crackle form x]
      }
     }
    }
  
  3.7 triAngle2()
   This is the same as triAngle(), only rotated 30 degrees so that its lines intersect the
   edges of each hexagon, instead of each corner. The result is slightly larger, and fits
   perfectly with the other patterns.
   
   Usage:
    my_object{
     pigment{
      triAngle2()
      pigment_map{
       [0 hexGrad() color_map{[0 rgb 0][.2 rgb .25]} ]
       [1 hexGrad() color_map{[0 rgb 0][.2 rgb 1]}   ]
      }
     }
    }
    
  3.8 triTessellation()
   A variant of triAngle(), but uses the spiral1 pattern to produce a curved tessellation.
   It's composed of a checker pattern of curved triangles alternating between black/white.
   It is not pigment list friendly.
   
   Usage:
    my_object{
     pigment{
      triTessellation()
      pigment_map{
       [0 crackle metric 1]
       [1 crackle form x]
      }
     }
    }
   
  3.9 kaleidoscope()
   This will make a kaleidoscope out of "myHexPattern".
   
   Usage:
    #declare myHexPattern=
    pigment{
     crackle
     scale .25
     triangle_wave
    }
    my_object{
     pigment{
      kaleidoscope()
     }
    }
  
  3.10 skewedGrid()
   Here's a helper texture for visualizing trigonal space. It was developed for use with
   the ht() and ht2() macros. Apply it to a plane positioned over your texture.
   
   Usage:
    my_object{
     pigment{
      skewedGrid()
      color_map{
       [0 rgb 0]
       [.5 rgb 0 transmit 1]
      }
     }
    }
  
  3.11 randomTriCells()
   Thrown in for completeness. Each triangular cell is filled with a random shade of gray.
   
   Usage:
    my_object{
     pigment{
      randomTriCells()
     }
    }
 
*/

#include"functions.inc"


#declare h1 = sin(pi*2/6);
#declare h2 = sin(pi*2/3)/1.5;

#macro ht(ht1, ht2)
 <ht1+ht2/2, ht2*h1>
#end

#macro ht2(htb1, htb2)
 <htb1+htb2/2,0,htb2*h1>
#end

#ifndef(myHexPattern)
 #declare myHexPattern=
 pigment{
  gradient z triangle_wave frequency 2
 }
#end
#macro singleCell(crot)
 pigment_pattern{
  myHexPattern
  rotate x*270 warp{planar} rotate x*90
 }
 scale h1/1.5 
 rotate y*crot
 scale 1/<1,1,h1>
 translate<.5,0,.5>
 warp{repeat x}
 warp{repeat z}
#end
#macro planar_cells(pcv)
 cells
 translate z*pcv
 warp{planar}
 rotate x*90
#end
#macro hexCells(hcrot, hcv)
 pigment_pattern{planar_cells(hcv) }
 pigment_map{
  #switch(hcrot)
   #case(1)
    [1/3 singleCell(0)]
   
    [1/3 singleCell(120)]
    [2/3 singleCell(120)]
   
    [2/3 singleCell(240)]
   #break
   
   #case(2)
    [1/6 singleCell(0)]
   
    [1/6 singleCell(60)]
    [2/6 singleCell(60)]
   
    [2/6 singleCell(120)]
    [3/6 singleCell(120)]
   
    [3/6 singleCell(180)]
    [4/6 singleCell(180)]
   
    [4/6 singleCell(240)]
    [5/6 singleCell(240)]
   
    [5/6 singleCell(300)]
   #break
   
   #else
    [1/3 singleCell(0)]
    [1/3 singleCell(0)]
    [2/3 singleCell(0)]
    [2/3 singleCell(0)]
  #end
 }
#end
#macro adjustedCells(acrot, acv)
 pigment_pattern{gradient z scale 2}
 pigment_map{
  [.5
   hexCells(acrot, acv)
  ]
  [.5
   hexCells(acrot, acv)
   translate x/2
  ]
 }
 translate z/2
 scale 3*<1,1,h1>
#end

//----------------------------------

#macro hexPattern(hprot)
 pigment_pattern{
  hexagon
  pigment{
   adjustedCells(hprot,0)
  }
  pigment{
   adjustedCells(hprot,-3)
   translate<1.5,0,-h1>
  }
  pigment{
   adjustedCells(hprot,3)
   translate<1.5,0,h1>
  }
 }
#end

#macro hexPatternGrouped()
 pigment_pattern{
  hexagon
  pigment{
   adjustedCells(0,0)
  }
  pigment{
   adjustedCells(0,0)
   rotate y*120
   translate<1.5,0,-h1>
  }
  pigment{
   adjustedCells(0,0)
   rotate y*240
   translate<1.5,0,h1>
  }
 }
#end

#macro adjustedCells2(acv1, acv2)
 pigment_pattern{gradient z scale 2}
 pigment_map{
  [.5
   planar_cells(acv1)
  ]
  [.5
   planar_cells(acv2)
   translate x/2
  ]
 }
 translate z/2
 scale 3*<1,1,h1>
#end
#macro randomHexCells()
 pigment_pattern{
  hexagon
  pigment{adjustedCells2(-9,-6)}
  pigment{adjustedCells2(-3,3) translate<1.5,0,-h1>}
  pigment{adjustedCells2(6,9) translate<1.5,0,h1>}
 }
#end

#macro hexGrad()
 pigment_pattern{
  function{-f_hex_x(x,y,z,0)/0.8660254}
  rotate x*270+y*30
 }
#end

#macro hexGradGrouped()
 #declare myHexPattern_temp=myHexPattern;
 #declare myHexPattern=
 pigment{
  pigment_pattern{
   pigment_pattern{
    gradient z translate z/2
    rotate y*30
    scale 2
   }
   pigment_map{
    [.5 hexGrad() scale h2]
    [.5 planar rotate x*90+y*120 scale .5]
   }
  }
 }
 hexPatternGrouped()
 #declare myHexPattern=myHexPattern_temp;
 #undef myHexPattern_temp
#end

#macro triAngle()
 #declare myHexPattern_temp=myHexPattern;
 #declare myHexPattern=
 pigment{
  pigment_pattern{
   radial frequency 3 color_map{[.5 rgb 0][.5 rgb 1]}
  }
 }
 hexPatternGrouped()
 #declare myHexPattern=myHexPattern_temp;
 #undef myHexPattern_temp
#end

#macro triAngle2()
 #declare myHexPattern_temp=myHexPattern;
 #declare myHexPattern=
 pigment{
  pigment_pattern{
   radial frequency 3 rotate y*30 color_map{[.5 rgb 0][.5 rgb 1]}
  }
 }
 hexPatternGrouped()
 #declare myHexPattern=myHexPattern_temp;
 #undef myHexPattern_temp
#end

#macro triTessellation()
 #declare myHexPattern_temp=myHexPattern;
 #declare myHexPattern=
 pigment{
  pigment_pattern{
   spiral1 3 rotate x*270 scale 2/3  color_map{[.5 rgb 0][.5 rgb 1]}
  }
 }
 hexPatternGrouped()
 #declare myHexPattern=myHexPattern_temp;
 #undef myHexPattern_temp
#end

#macro kaleidoscope()
 #declare myHexPattern_temp=myHexPattern;
 #declare myHexPattern=
 pigment{
  myHexPattern
  
  #declare V=0;
  #while(V<360)
   rotate y*V/2
   warp{repeat x flip x}
   #declare V=V+360/6;
  #end
 }
 hexPattern(0)
 #declare myHexPattern=myHexPattern_temp;
 #undef myHexPattern_temp

#end

#macro skewedGrid()
 pigment_pattern{
  boxed scale .5 translate(x+z)/2
  warp{repeat x}
  warp{repeat z}
  scale h1*<h2*2,1,1>
  matrix<1,0,0,0,1,0,h2,0,1,0,0,0>
  rotate x*270 warp{planar} rotate x*90
 }
#end

#macro randomTriCells()
 pigment_pattern{
  triAngle()
  pigment_map{
   [.5
    pigment_pattern{gradient z scale 2}
    pigment_map{
     #declare ct=
     pigment{cells translate -y*2 rotate x*270 warp{planar} rotate x*90}
     [.5
      ct
     ]
     [.5
      ct
      translate x/2
     ]
    }
    translate z
    scale<1,1,h1>
   ]
   [.5
    pigment_pattern{gradient z scale 2}
    pigment_map{
     #declare ct=
     pigment{cells translate y*2 rotate x*270 warp{planar} rotate x*90}
     [.5
      ct
     ]
     [.5
      ct
      translate x/2
     ]
    }
    scale<1,1,h1>
   ]
  }
 }
 #undef ct
#end