
/* macro write_ascii_df3_file
*
* Version: 1.0
* Author : David  El Tom
* License : GPL
*
* Description:
* This macro adresses the generation of density files based on pigment.
* As it is not possible to write binary data out of SDL in an easy way, especially if it comes to write a binary 0,
* the final density file have to be created by the means of an external program or script.
* Opposed to the approach used by the makecloud macro (Gilles Tran), there is no need to render
* a bunch of slices and merge these images with the external tool (tga2df3) into the binary df3 file.
* Instead the macro will evaluate the density using the .gray operator at each voxel-midpoint and
* write this value to a text file.
* This text file later can be converted into the binary df3 file with an external program or script.
* A PERL implementation to do so is appended at the end of this file.
* 
* Usage:
* Include this macro into your scene file.
* Declare the pigment representing the density and run the macro.
*  NAME : the name of the ascii-df3 file to write the data to
* PIGMENT : pre-declared pigment representing the density distribution
* RES_X : resolution of voxels in x - direction
* RES_Y : resolution of voxels in y - direction
* RES_Z : resolution of voxels in z - direction
* LOWER_LEFT : vector pointing to the lower left extent of the density (think of a box surrounding the pigment to evaluate)
* UPPER_RIGHT : vector pointing to the upper right extent of the density
*/

#ifndef(DF3_INC)
#declare DF3_INC = 1;

#ifdef(View_POV_Include_Stack)
   #debug "including df3.inc\n"
#end

#include "strings.inc"

#macro write_ascii_df3_file(NAME,PIGMENT,RES_X,RES_Y,RES_Z,LOWER_LEFT,UPPER_RIGHT)
	#debug concat("ascii df3 file generation started; output will be written to <",NAME,">\n")
	// declare a vector function based on pigment as eval_pigment leaded to
	// memory problems (could not find why).
	// evaluating the function instead seems to work fine ...
	#local wdf_pig_fcn = function{pigment{PIGMENT}}
	// open file for writing
	#fopen DF3FILE NAME write

	// write header
	#local tmp_string = concat(str(RES_X,0,0),"\n");
	#write(DF3FILE,tmp_string)
	#local tmp_string = concat(str(RES_Y,0,0),"\n");
	#write(DF3FILE,tmp_string)
	#local tmp_string = concat(str(RES_Z,0,0),"\n");
	#write(DF3FILE,tmp_string)

	// calculate step wide
	#local length_vector = UPPER_RIGHT-LOWER_LEFT;
	#local step_x = length_vector.x/RES_X;
	#local step_y = length_vector.y/RES_Y;
	#local step_z = length_vector.z/RES_Z;

	#debug concat("going from <",vstr(3,LOWER_LEFT,", ",0,-1),"> to <",vstr(3,UPPER_RIGHT,", ",0,-1),"\n")
	#debug concat("step width: <",str(step_x,0,-1),", ",str(step_y,0,-1),", ",str(step_z,0,-1),">\n")

	// write data
	#local counter_x = 0;
	#local counter_y = 0;
	#local counter_z = 0;

	#while (counter_z < RES_Z)
		#debug concat("writing slice ", str(counter_z,0,0),"\n")
		#while (counter_y < RES_Y)
			#while (counter_x < RES_X)
				// calculate the point to test
				#local pos = LOWER_LEFT + (.5*<step_x,step_y,step_z>) +  <counter_x*step_x , counter_y*step_y, counter_z*step_z>;
				#declare color_value = wdf_pig_fcn(pos.x,pos.y,pos.z);
				#declare tmp_value = floor(min(color_value.gray,1)*65535);
				#write(DF3FILE,concat(str(tmp_value,0,0),"\n"))
				#local counter_x = counter_x + 1;
			#end
			#local counter_x = 0;
			#local counter_y = counter_y +1;
		#end
		#local counter_y = 0;
		#local counter_z = counter_z +1;
	#end

	// close file
	#fclose DF3FILE
#end

#end//df3.inc

/* PERL -Script to convert the generated ascii file into binary df3 file
*
* you may extract the following lines and save them as ascii2df3.pl
*
* ascii2df3.pl ------> START
#!/usr/bin/perl

# ascii2df3
# parameter:
# - src_file : string containing the file name to read from
# - dest_file: string containing the file name to write to
sub ascii2df3
{
	my ($src_file, $dest_file) = @_;
	my $buffer;
	my $counter = 1;
	
	open (INFILE, "<$src_file") or die "can't open $src_file \n";
	open (OUTFILE, ">$dest_file") or die "can't open $dest_file \n";
	
	binmode OUTFILE;
	
	# write header
	$buffer = <INFILE>;
	print OUTFILE pack(n,$buffer);
	$buffer = <INFILE>;
	print OUTFILE pack(n,$buffer);
	$buffer = <INFILE>;
	print OUTFILE pack(n,$buffer);
	
	# write data
	while ($buffer = <INFILE>)
	{
		#print "$counter : $buffer\n";
		print OUTFILE pack(n,$buffer);
		$counter++;
	}	
	
	close INFILE;
	close OUTFILE;
}

sub main
{
	my $param_count = @ARGV;

	if ($param_count == 0)
	{
		print "\nusage: perl ascii2df3.pl infile [outfile]\n\n";
	}
	else
	{
		my $infile_name = $ARGV[0];
		my $outfile_name;
	
		if ($param_count > 1)
		{
			$outfile_name = $ARGV[1];
		}
		else
		{
			$outfile_name = $infile_name;
			$outfile_name =~ s/\..+\Z/\.df3/i;
		}
		print "will read from $infile_name\nand write to $outfile_name\n";
	
		ascii2df3($infile_name, $outfile_name );
	}
}

# invoke main routine
main()

* ascii2df3.pl ------> END
*/
