/************************************************************************* 

filters.inc
v.1.0  

Author: Gennady Obukhov 
obukhov@mail.com 
http://propro.ru/go/
date: October 19 2002
Walnut Creek, California

*************************************************************************/

#include "functions.inc"


/**********************  General-purpose filters  ***********************/


/*************************************************************************

                            Mosaic Linear

*************************************************************************/

#macro MosaicLinear (pigm, obj, strokesX, strokesY, strokeFinish, bgr)
	#local widthX = 4/3;
	#local heightY = 1;
	#local cntX = 0;
	#local cntY = 0;     
	#local lseed = seed(1000);       
	#local incrX = widthX/strokesX;
	#local incrY = heightY/strokesY;
	
	#while(cntX < strokesX)
		#local cntY = 0;           
		#while(cntY < strokesY)
			#local currentPoint = <incrX*cntX, incrY*cntY, 0>;   
			#local P = eval_pigment(pigm, <currentPoint.x, currentPoint.y, 0>);
			#local T = texture{pigment{P} finish{strokeFinish}};
	  		object{
				obj	
				texture{T}
				scale<incrX, incrY, min(incrX, incrY)>
				translate<incrX/2, incrY/2, 0>
				translate<currentPoint.x, currentPoint.y, 0>
			}
			#if(bgr = 1)
				box{0, 1 texture{T} scale<incrX, incrY, min(incrX, incrY)> translate<currentPoint.x, currentPoint.y, 0>}
			#end
			#local cntY = cntY + 1;
		#end
		#local cntX = cntX + 1;
	#end   
#end

/*************************************************************************

                            Mosaic Diagonal

*************************************************************************/

#macro MosaicDiagonal (pigm, obj, strokesX, strokesY, strokeFinish, bgr)
	#local widthX = 4/3;
	#local heightY = 1;
	#local cntX = 0;
	#local cntY = 0;     
	#local lseed = seed(1000);       
	#local incrX = widthX/strokesX;
	#local incrY = heightY/strokesY;
	#local even = 0;
	
	#while(cntX < strokesX)
		#local cntY = 0;           
		#while(cntY < strokesY)
			#local currentPoint = <incrX*cntX, incrY*cntY, 0>;   
			#local P = eval_pigment(pigm, <currentPoint.x, currentPoint.y, 0>);
			#local T = texture{pigment{P} finish{strokeFinish}};
	  		object{
				obj	
				texture{T}
				scale<incrX, incrY, min(incrX, incrY)>
				translate<incrX/2, incrY/2, 0>
				translate<currentPoint.x, currentPoint.y, 0>
				#if(even = 1)
    				translate -y*(incrY*0.5) 
    			#end
			}
			#if(bgr = 1)
				box{0, 1 texture{T} scale<incrX, incrY, min(incrX, incrY)> translate<currentPoint.x, currentPoint.y, 0>
					#if(even = 1)
    					translate -y*(incrY*0.5) 
    				#end
				}
			#end
			#local cntY = cntY + 1;
		#end
		#if(even = 1)
    		#local even = 0;
    	#else   
    		#local even = 1;
    	#end
		#local cntX = cntX + 1;
	#end   
#end

/*************************************************************************

                            Multipleton

*************************************************************************/    

#macro Multipleton (imageMap, strokeArray, strokesX, strokesY, strokeFinish, bgr)
	#local widthX = 4/3;
	#local heightY = 1;
	#local cntX = 0;
	#local cntY = 0;     
	#local lseed = seed(1000);       
	#local incrX = widthX/strokesX;
	#local incrY = heightY/strokesY;
	#local arraySize = dimension_size(strokeArray, 1);
	
	#while(cntX < strokesX)
		#local cntY = 0;           
		#while(cntY < strokesY)
			#local currentPoint = <incrX*cntX, incrY*cntY, 0>;   
			
			#local P = eval_pigment(imageMap, <currentPoint.x, currentPoint.y, 0>);
    		#local T = texture{pigment{P} finish{strokeFinish}};
	        
			object{
				strokeArray[arraySize * rand(lseed)]	
				texture{T}
				scale<incrX, incrY, min(incrX, incrY)>
				translate<incrX/2, incrY/2, 0>
				translate<currentPoint.x, currentPoint.y, 0>
			}   
			
			#if(bgr = 1)
				box{0, 1 texture{T} scale<incrX, incrY, min(incrX, incrY)> translate<currentPoint.x, currentPoint.y, 0>}
			#end
						
			#local cntY = cntY + 1;
		#end
		#local cntX = cntX + 1;
	#end   
#end

/*************************************************************************

                            POV-Ray objects

*************************************************************************/

#macro PovRay (imageMap, strokesX, strokesY, strokeFinish, bgr)
	#local StrokeArray = array[10];
	#local StrokeArray[0] = difference {
		box{<-0.5, -0.5, -0.5>, 0.5}
		sphere{<0, 0, -0.5>, 0.5}
		scale<1, 1, 0.6>
		no_shadow
	};
	#local StrokeArray[1] = sphere{0, 0.5 scale<1, 1, 0.4> no_shadow};
	#local StrokeArray[2] = torus{0.3, 0.2 rotate x*90 scale<1, 1, 0.6> no_shadow};
	#local StrokeArray[3] = cone{0, 0.5, <0, 0, -0.5>, 0 scale<1, 1, 0.5>};
	#local StrokeArray[4] = cylinder{<-0.5, 0, 0>, <0.5, 0, 0>, 0.5 scale<1, 1, 0.6> no_shadow};
	#local StrokeArray[5] = cylinder{<0, -0.5, 0>, <0, 0.5, 0>, 0.5 scale<1, 1, 0.6> no_shadow};
	#local StrokeArray[6] = cone{0, 0.5, <0, 0, -0.5>, 0.3 scale<1, 1, 0.4> no_shadow};
	#local StrokeArray[7] = union {
		cylinder{<-0.5, 0, 0>, <0.5, 0, 0>, 0.25 scale<1, 1, 0.6> translate y*0.25}
		cylinder{<-0.5, 0, 0>, <0.5, 0, 0>, 0.25 scale<1, 1, 0.6> translate -y*0.25}
		no_shadow
	};
	#local StrokeArray[8] = union {
		cylinder{<0, -0.5, 0>, <0, 0.5, 0>, 0.25 scale<1, 1, 0.6> translate -x*0.25}
		cylinder{<0, -0.5, 0>, <0, 0.5, 0>, 0.25 scale<1, 1, 0.6> translate x*0.25}
		no_shadow
	};
	#local StrokeArray[9] = union {
		torus{0.375, 0.125 rotate x*90 scale<1, 1, 0.6> no_shadow}
   		torus{0.125, 0.125 rotate x*90 scale<1, 1, 0.6> no_shadow}	
	};

	Multipleton (imageMap, StrokeArray, strokesX, strokesY, strokeFinish, bgr)
#end

/*************************************************************************

                            Labyrinth

*************************************************************************/

#macro Labyrinth (pigm, strokeArray, strokesX, strokesY, strokeFinish, bgr)
	#local widthX = 4/3;
	#local heightY = 1;
	#local cntX = 0;
	#local cntY = 0; 
	#local lseed = seed(1000);
	#local incrX = widthX/strokesX;
	#local incrY = heightY/strokesY;
	
	#local item0 = strokeArray[0];
	#local item1 = strokeArray[1];
	#local item2 = strokeArray[2];
	#local item3 = strokeArray[3];
	#local item4 = strokeArray[4];
	#local item5 = strokeArray[5];
	#local item6 = strokeArray[6];
	#local item7 = strokeArray[7];
	#local item8 = strokeArray[8];
	#local item9 = strokeArray[9];
	#local item10 = strokeArray[10];
             
	#local group1 = array[1] {item2};
	#local group2 = array[1] {item3};
	#local group3 = array[1] {item5};
	#local group4 = array[1] {item4};
	#local group5 = array[2] {item0, item6};
	#local group6 = array[2] {item0, item8};
	#local group7 = array[2] {item1, item9};
	#local group8 = array[2] {item1, item7};
	#local group9 = array[1] {item2};
	#local group10 = array[3] {item1, item5, item9};
	#local group11 = array[3] {item0, item3, item6};
	#local group12 = array[4] {item4, item7, item8, item10};

	#local groupSize1 = dimension_size(group1, 1);
	#local groupSize2 = dimension_size(group2, 1);
	#local groupSize3 = dimension_size(group3, 1);
	#local groupSize4 = dimension_size(group4, 1);
	#local groupSize5 = dimension_size(group5, 1);
	#local groupSize6 = dimension_size(group6, 1);
	#local groupSize7 = dimension_size(group7, 1);
	#local groupSize8 = dimension_size(group8, 1);
	#local groupSize9 = dimension_size(group9, 1);
	#local groupSize10 = dimension_size(group10, 1);
	#local groupSize11 = dimension_size(group11, 1);
	#local groupSize12 = dimension_size(group12, 1);

	#local previousColumn = array[strokesY];
	#local i = 0;

	#while(cntX < strokesX)
		#local cntY = 0;
		#while(cntY < strokesY)
			#local randomNum = rand(lseed);

			// left column
			#if(cntX = 0) 
				// left-bottom corner
				#if(cntY = 0)
					#local i = int(groupSize1 * randomNum);
					#local currentStroke = object{group1[i]}; 
					#local previousColumn[cntY] = 1;
				#end
				// left border
				#if(cntY > 0 & cntY < (strokesY - 1))
					#local i = int(groupSize5 * randomNum);
					#local currentStroke = object{group5[i]};
					#if(i = 0)
						#local previousColumn[cntY] = 0;
					#else 
						#local previousColumn[cntY] = 1;
					#end
				#end
				// left-top corner
				#if(cntY = (strokesY - 1)) 
					#local i = int(groupSize2 * randomNum);
					#local currentStroke = object{group2[i]};
					#local previousColumn[cntY] = 1;
				#end
			#end
			// right column
			#if(cntX = (strokesX - 1))
				// right-bottom corner
				#local randomNum = rand(lseed);
				#if(cntY = 0)
					#local i = int(groupSize3 * randomNum);
					#local currentStroke = object{group3[i]};
				#end
				// right border 
				#if(cntY > 0 & cntY < (strokesY - 1))
					#if(previousColumn[cntY] = 1)
						#local currentStroke = item8;
					#else
						#local currentStroke = item0;
					#end                         
				#end
				// right-top corner
				#if(cntY = (strokesY - 1))
					#local i = int(groupSize4 * randomNum);
					#local currentStroke = object{group4[i]};
				#end
			#end
			// body
			#if(cntX > 0 & cntX < (strokesX - 1))
				#local rightConnection = previousColumn[cntY];
				#if(cntY = 0 | cntY = (strokesY - 1)) 
					// bottom border
					#if(cntY = 0)
						#local i = int(groupSize7 * randomNum);
						#local currentStroke = object{group7[i]};
						#if(int(i)= 1)
							#local previousStroke = 1;
						#else 
							#local previousStroke = 0;
						#end
					// top border	
					#else        
						#if(previousStroke = 0)
							 #local currentStroke = item1;
						#else
							 #local currentStroke = item7;
						#end
					#end
				#else         				
					// variant 1
					#if(rightConnection = 0 & previousStroke = 0)
						#local currentStroke = item2;
						#local previousStroke = 1;
						#local previousColumn[cntY] = 1;
					#else
						// variant 2
						#if(rightConnection = 1 & previousStroke = 0)
							#local i = int(groupSize10 * randomNum);
							#local currentStroke = object{group10[i]};
							#if(i = 0)
								#local previousStroke = 0;      
								#local previousColumn[cntY] = 1;
							#end                             
							#if(i = 1)
								#local previousStroke = 1;      
								#local previousColumn[cntY] = 0;
							#end
							#if(i = 2)
								#local previousStroke = 1;      
								#local previousColumn[cntY] = 1;
							#end
						#else
							// variant 3
							#if(rightConnection = 0 & previousStroke = 1)
								#local i = int(groupSize11 * randomNum);
								#local currentStroke = object{group11[i]};
								#if(i = 0)
									#local previousStroke = 1;      
									#local previousColumn[cntY] = 0;
								#end                             
								#if(i = 1)
									#local previousStroke = 0;      
									#local previousColumn[cntY] = 1;
								#end
								#if(i = 2)
									#local previousStroke = 1;      
									#local previousColumn[cntY] = 1;
								#end 
							#else
								// variant 4
								#if(rightConnection = 1 & previousStroke = 1)
									#local i = int(groupSize12 * randomNum);
									#local currentStroke = object{group12[i]};
									#if(i = 0)
										#local previousStroke = 0;      
										#local previousColumn[cntY] = 0;
									#end                             
									#if(i = 1)
										#local previousStroke = 0;      
										#local previousColumn[cntY] = 1;
									#end
									#if(i = 2)
										#local previousStroke = 1;      
										#local previousColumn[cntY] = 0;
									#end 
									#if(i = 3)
										#local previousStroke = 1;      
										#local previousColumn[cntY] = 1;
									#end
								#end
							#end
						#end
					#end
				#end
			#end

			#local currentPoint = <cntX * incrX, cntY * incrY, 0>;
			#local P = eval_pigment(pigm, <currentPoint.x, currentPoint.y, 0>);
			#local T = texture{pigment{P} finish{strokeFinish}}
			object{currentStroke 
				texture{T} 
				scale<incrX, incrY, min(incrX, incrY)> 
				translate<currentPoint.x, currentPoint.y, 0>
			}
			#if(bgr = 1)
				box{0, 1 texture{T} scale<incrX, incrY, min(incrX, incrY)> translate<currentPoint.x, currentPoint.y, 0>}
			#end
			#local cntY = cntY + 1;
		#end
		#local cntX = cntX + 1;
	#end
#end

/*************************************************************************

                            Oriental

*************************************************************************/

#macro Oriental (pigm, strokesX, strokesY, ornamentRadius, strokeFinish, bgr)
	#local StrokeArray = array[11];
	// vertical
	#local StrokeArray[0] = cylinder{<0.5, 0, 0>, <0.5, 1, 0>, ornamentRadius};
	// horizontal
	#local StrokeArray[1] = cylinder{<0, 0.5, 0>, <1, 0.5, 0>, ornamentRadius}; 
	// right-top corner
	#local StrokeArray[2] = difference{torus{0.5, ornamentRadius rotate x*90} plane{-x, 0} plane{-y, 0} translate<1, 1, 0>};
	// right-bottom corner
	#local StrokeArray[3] = difference{torus{0.5, ornamentRadius rotate x*90} plane{-x, 0} plane{y, 0} translate<1, 0, 0>};
	// left-bottom corner
	#local StrokeArray[4] = difference{torus{0.5, ornamentRadius rotate x*90} plane{x, 0} plane{y, 0} translate<0, 0, 0>};
	// left-top corner
	#local StrokeArray[5] = difference{torus{0.5, ornamentRadius rotate x*90} plane{x, 0} plane{-y, 0} translate<0, 1, 0>};
	// triple right
	#local StrokeArray[6] = union{cylinder{<0.5, 0, 0>, <0.5, 1, 0>, ornamentRadius} cylinder{<0.5, 0.5, 0>, <1, 0.5, 0>, ornamentRadius}};
	// triple down
	#local StrokeArray[7] = union{cylinder{<0, 0.5, 0>, <1, 0.5, 0>, ornamentRadius} cylinder{<0.5, 0, 0>, <0.5, 0.5, 0>, ornamentRadius}};
	// triple left
	#local StrokeArray[8] = union{cylinder{<0.5, 0, 0>, <0.5, 1, 0>, ornamentRadius} cylinder{<0, 0.5, 0>, <0.5, 0.5, 0>, ornamentRadius}};
	// triple up
	#local StrokeArray[9] = union{cylinder{<0, 0.5, 0>, <1, 0.5, 0>, ornamentRadius} cylinder{<0.5, 0.5, 0>, <0.5, 1, 0>, ornamentRadius}};
	// cross
	#local StrokeArray[10] = union{cylinder{<0, 0.5, 0>, <1, 0.5, 0>, ornamentRadius} cylinder{<0.5, 0, 0>, <0.5, 1, 0>, ornamentRadius}}; 		
	Labyrinth (pigm, StrokeArray, strokesX, strokesY, strokeFinish, bgr)
#end 

/*************************************************************************

                            Pompidou

*************************************************************************/

#macro Pompidou (pigm, strokesX, strokesY, pipeRadius, joinMultiplier, joinHeight, joinStart, strokeFinish, bgr)
	#local StrokeArray = array[11];
	#local joinTotal = joinHeight + joinStart;
	#local wheelRadius1 = pipeRadius*1.1;
	#local wheelRadius2 = wheelRadius1*0.15;
	
	// wheel - works only without orthographic camera
	#local Wheel = union {
		torus{wheelRadius1, wheelRadius2 rotate x*90}
		cylinder{<-wheelRadius1, 0, 0>, <wheelRadius1, 0, 0>, wheelRadius2*0.8 rotate z*45}
		cylinder{<-wheelRadius1, 0, 0>, <wheelRadius1, 0, 0>, wheelRadius2*0.8 rotate -z*45}
		sphere{0, wheelRadius1*0.4 scale<1, 1, 0.6>}
		translate<0.5, 0.5, -pipeRadius*1.4> no_shadow
	}
	
	// vertical
	#local StrokeArray[0] = union {
		cylinder{<0.5, 0, 0>, <0.5, 1, 0>, pipeRadius}
		cylinder{<0.5, joinStart, 0>, <0.5, joinTotal, 0>, pipeRadius*joinMultiplier}
		cylinder{<0.5, 1 - joinStart, 0>, <0.5, 1 - joinTotal, 0>, pipeRadius*joinMultiplier}
	};
	// horizontal
	#local StrokeArray[1] = union {
		cylinder{<0, 0.5, 0>, <1, 0.5, 0>, pipeRadius}
		cylinder{<joinStart, 0.5, 0>, <0.1, 0.5, 0>, pipeRadius*joinMultiplier}
		cylinder{<1 - joinStart, 0.5, 0>, <1 - joinTotal, 0.5, 0>, pipeRadius*joinMultiplier}
	};
	// right-top corner
	#local StrokeArray[2] = union {
		difference{torus{0.5, pipeRadius rotate x*90} plane{-x, 0} plane{-y, 0} translate<1, 1, 0>}
		cylinder{<0.5, 1 - joinStart, 0>, <0.5, 1 - joinTotal, 0>, pipeRadius*joinMultiplier}
		cylinder{<1 - joinStart, 0.5, 0>, <1 - joinTotal, 0.5, 0>, pipeRadius*joinMultiplier}
	}
	// right-bottom corner
	#local StrokeArray[3] = union {
		difference{torus{0.5, pipeRadius rotate x*90} plane{-x, 0} plane{y, 0} translate<1, 0, 0>}
		cylinder{<0.5, joinStart, 0>, <0.5, joinTotal, 0>, pipeRadius*joinMultiplier}
		cylinder{<1 - joinStart, 0.5, 0>, <1 - joinTotal, 0.5, 0>, pipeRadius*joinMultiplier}
	};
	// left-bottom corner
	#local StrokeArray[4] = union {
		difference{torus{0.5, pipeRadius rotate x*90} plane{x, 0} plane{y, 0} translate<0, 0, 0>}
		cylinder{<0.5, joinStart, 0>, <0.5, joinTotal, 0>, pipeRadius*joinMultiplier}
		cylinder{<joinStart, 0.5, 0>, <joinTotal, 0.5, 0>, pipeRadius*joinMultiplier}
	}
	// left-top corner
	#local StrokeArray[5] = union {
		difference{torus{0.5, pipeRadius rotate x*90} plane{x, 0} plane{-y, 0} translate<0, 1, 0>}
		cylinder{<0.5, 1 - joinStart, 0>, <0.5, 1 - joinTotal, 0>, pipeRadius*joinMultiplier}
		cylinder{<joinStart, 0.5, 0>, <joinTotal, 0.5, 0>, pipeRadius*joinMultiplier}
	};
	// triple right
	#local StrokeArray[6] = union {
		cylinder{<0.5, 0, 0>, <0.5, 1, 0>, pipeRadius} cylinder{<0.5, 0.5, 0>, <1, 0.5, 0>, pipeRadius}
		cylinder{<0.5, 1 - joinStart, 0>, <0.5, 1 - joinTotal, 0>, pipeRadius*joinMultiplier}
		cylinder{<0.5, joinStart, 0>, <0.5, joinTotal, 0>, pipeRadius*joinMultiplier}
		cylinder{<1 - joinStart, 0.5, 0>, <1 - joinTotal, 0.5, 0>, pipeRadius*joinMultiplier}
		object{Wheel}
	};
	// triple down
	#local StrokeArray[7] = union {
		cylinder{<0, 0.5, 0>, <1, 0.5, 0>, pipeRadius} cylinder{<0.5, 0, 0>, <0.5, 0.5, 0>, pipeRadius}
		cylinder{<1 - joinStart, 0.5, 0>, <1 - joinTotal, 0.5, 0>, pipeRadius*joinMultiplier}
		cylinder{<joinStart, 0.5, 0>, <joinTotal, 0.5, 0>, pipeRadius*joinMultiplier}
		cylinder{<0.5, joinStart, 0>, <0.5, joinTotal, 0>, pipeRadius*joinMultiplier} 
	};
	// triple left
	#local StrokeArray[8] = union {
		cylinder{<0.5, 0, 0>, <0.5, 1, 0>, pipeRadius} cylinder{<0, 0.5, 0>, <0.5, 0.5, 0>, pipeRadius}
		cylinder{<0.5, 1 - joinStart, 0>, <0.5, 1 - joinTotal, 0>, pipeRadius*joinMultiplier}
		cylinder{<0.5, joinStart, 0>, <0.5, joinTotal, 0>, pipeRadius*joinMultiplier}
		cylinder{<joinStart, 0.5, 0>, <joinTotal, 0.5, 0>, pipeRadius*joinMultiplier} 
		object{Wheel}
	};
	// triple up
	#local StrokeArray[9] = union {
		cylinder{<0, 0.5, 0>, <1, 0.5, 0>, pipeRadius} cylinder{<0.5, 0.5, 0>, <0.5, 1, 0>, pipeRadius}
		cylinder{<1 - joinStart, 0.5, 0>, <1 - joinTotal, 0.5, 0>, pipeRadius*joinMultiplier}
		cylinder{<joinStart, 0.5, 0>, <joinTotal, 0.5, 0>, pipeRadius*joinMultiplier}
		cylinder{<0.5, 1 - joinStart, 0>, <0.5, 1 - joinTotal, 0>, pipeRadius*joinMultiplier} 
	};
	// cross
	#local StrokeArray[10] = union {
		cylinder{<0, 0.5, 0>, <1, 0.5, 0>, pipeRadius} cylinder{<0.5, 0, 0>, <0.5, 1, 0>, pipeRadius}
		cylinder{<1 - joinStart, 0.5, 0>, <1 - joinTotal, 0.5, 0>, pipeRadius*joinMultiplier}
		cylinder{<joinStart, 0.5, 0>, <joinTotal, 0.5, 0>, pipeRadius*joinMultiplier}
		cylinder{<0.5, 1 - joinStart, 0>, <0.5, 1 - joinTotal, 0>, pipeRadius*joinMultiplier}
		cylinder{<0.5, joinStart, 0>, <0.5, joinTotal, 0>, pipeRadius*joinMultiplier}
	};
	Labyrinth (pigm, StrokeArray, strokesX, strokesY, strokeFinish, bgr)
#end

       
/************************** Artistic filters ****************************/


/*************************************************************************

                            Claude Monet

*************************************************************************/

#macro Monet (pigm, strokesX, strokesY, strokeFinish)
	#local widthX = 4/3;
	#local heightY = 1;
	#local cntX = 0;
	#local cntY = 0;     
	#local lseed = seed(1000);       
	#local incrX = widthX/strokesX;
	#local incrY = heightY/strokesY;
	
	blob {
		threshold 0.65
		#while(cntX < strokesX)
			#local cntY = 0;           
			#while(cntY < strokesY)
				#local currentPoint = <incrX*cntX, incrY*cntY, 0>;   
				#local P = eval_pigment(pigm, <currentPoint.x, currentPoint.y, 0>);				
    			#local T = texture{pigment{P} finish{strokeFinish}};   
    			sphere {<0, 0, 0>, 1, 2 
    				texture{T}
    				scale<1.6, 1, 1>
					scale <1.6 * incrX, 1.25 * incrX, (1.25 * incrX) + (1.2 * incrX * rand(lseed))>
					scale 1 + (0.2 * rand(lseed))  
					rotate z*(-20 + (20 * rand(lseed)))
					translate<currentPoint.x + (incrX*rand(lseed)), currentPoint.y + (incrY*rand(lseed)), 0>
				}   			
				#local cntY = cntY + 1;
			#end
			#local cntX = cntX + 1;
		#end
		scale z*0.14   
	}
#end

/*************************************************************************

                            Vincent Van Gogh

*************************************************************************/
    
#macro VanGogh (pigm, strokesX, strokesY, strokeFinish)
	#local widthX = 4/3;
	#local heightY = 1;
	#local cntX = 0;
	#local cntY = 0;     
	#local lseed = seed(1000);       
	#local incrX = widthX/strokesX;
	#local incrY = heightY/strokesY;
	
	#while(cntX < strokesX)
		#local cntY = 0;           
		#while(cntY < strokesY)
			#local currentPoint = <incrX*cntX, incrY*cntY, 0>;   
			
			#if(cntY >= (strokesY - 5))
				#local P1 = eval_pigment(pigm, <currentPoint.x, currentPoint.y, 0>);	
				#local P2 = eval_pigment(pigm, <currentPoint.x, currentPoint.y - incrY*1, 0>); 
				#local P3 = eval_pigment(pigm, <currentPoint.x, currentPoint.y - incrY*2, 0>);
				#local P4 = eval_pigment(pigm, <currentPoint.x, currentPoint.y - incrY*3, 0>);
				#local P5 = eval_pigment(pigm, <currentPoint.x, currentPoint.y - incrY*4, 0>);
			#else
				#local P1 = eval_pigment(pigm, <currentPoint.x, currentPoint.y, 0>);	
				#local P2 = eval_pigment(pigm, <currentPoint.x, currentPoint.y + incrY*1, 0>); 
				#local P3 = eval_pigment(pigm, <currentPoint.x, currentPoint.y + incrY*2, 0>);
				#local P4 = eval_pigment(pigm, <currentPoint.x, currentPoint.y + incrY*3, 0>);
				#local P5 = eval_pigment(pigm, <currentPoint.x, currentPoint.y + incrY*4, 0>);
			#end
			
    		#local T1 = texture{pigment{P1} finish{strokeFinish}};   
    		#local T2 = texture{pigment{P2} finish{strokeFinish}};      
    		#local T3 = texture{pigment{P3} finish{strokeFinish}}; 
    		#local T4 = texture{pigment{P4} finish{strokeFinish}}; 
    		#local T5 = texture{pigment{P5} finish{strokeFinish}}; 
    		
   			blob {
				threshold 0.65
				sphere { <0, 0, 0>, 1, 2 scale<5, 1, 1> texture{T5} translate<1, 0, 1.4>} 
				sphere { <0, 0, 0>, 1, 2 scale<5, 1, 1> texture{T4} translate<2, 0, 0.7>} 
				sphere { <0, 0, 0>, 1, 2 scale<5, 1, 1> texture{T3}}
				sphere { <0, 0, 0>, 1, 2 scale<5, 1, 1> texture{T2} translate<3, 0, -0.6>} 
				sphere { <0, 0, 0>, 1, 2 scale<4, 1, 1> texture{T1} translate<0.8, 0, -1.2>}
				rotate -x*90
				scale<0.4 * incrX, 0.4 * incrX, 0.001 + (0.001 * rand(lseed))>
				rotate z*(-5 + (30 * rand(lseed)))
				translate<currentPoint.x + (incrX*rand(lseed)), currentPoint.y + (incrY*rand(lseed)), 0>
			}   			
			#local cntY = cntY + 1;
		#end
		#local cntX = cntX + 1;
	#end   
#end

/*************************************************************************

                            Paul Signac

*************************************************************************/

#macro Signac (pigm, strokesX, strokesY, strokeFinish)
	#local widthX = 4/3;
	#local heightY = 1;
	#local cntX = 0;
	#local cntY = 0;     
	#local lseed = seed(1000);       
	#local incrX = widthX/strokesX;
	#local incrY = heightY/strokesY;
	
	#while(cntX < strokesX)
		#local cntY = 0;           
		#while(cntY < strokesY)
			#local currentPoint = <incrX*cntX, incrY*cntY, 0>;   
			
			#local P = eval_pigment(pigm, <currentPoint.x, currentPoint.y, 0>);				
    		#local T = texture{pigment{P} finish{strokeFinish}};   
    		
    		sphere{0, 2 
    			texture{T}
				scale<0.45 * incrX, 0.45 * incrX, 0.0002 + (0.0002 * rand(lseed))>
				translate<currentPoint.x + (incrX*rand(lseed)), currentPoint.y + (incrY*rand(lseed)), 0>
			}   			
			#local cntY = cntY + 1;
		#end
		#local cntX = cntX + 1;
	#end   
#end

/*************************************************************************

                            Pablo Picasso

*************************************************************************/

#macro Picasso (pigm, strokesX, strokesY, strokeFinish)
	#local widthX = 4/3;
	#local heightY = 1;
	#local cntX = 0;
	#local cntY = 0;     
	#local lseed = seed(1000);       
	#local incrX = widthX/strokesX;
	#local incrY = heightY/strokesY;
	
	#while(cntX < strokesX)
		#local cntY = 0;           
		#while(cntY < strokesY)
			#local currentPoint = <incrX*cntX, incrY*cntY, 0>;   
			
			#local P = eval_pigment(pigm, <currentPoint.x, currentPoint.y, 0>);				
    		#local T = texture{pigment{P} finish{strokeFinish}};   
    		
    		box{-1, 1
    			texture{T}
				scale<0.9 * incrX, 0.9 * incrX, 0.002 + (0.002 * rand(lseed))>
				translate<currentPoint.x, currentPoint.y, 0>
			}   			
			#local cntY = cntY + 1;
		#end
		#local cntX = cntX + 1;
	#end   
#end

/*************************************************************************

                            Piet Mondrian

*************************************************************************/

#macro Mondrian (pigm, strokesX, strokesY, strokeFinish)
	#local widthX = 4/3;
	#local heightY = 1; 
	#local cntX = 0;
	#local cntY = 0;     
	#local lseed = seed(1000);       
	#local incrX = widthX/strokesX;
	#local incrY = heightY/strokesY;
	
	#while(cntX < strokesX)
		#local cntY = 0;           
		#while(cntY < strokesY)
			#local currentPoint = <incrX*cntX, incrY*cntY, 0>;   
			
			#local P = eval_pigment(pigm, <currentPoint.x, currentPoint.y, 0>);				
    		#local T = texture{pigment{P} finish{strokeFinish}};   
    		
    		union {
    			difference{
    				box{-1.1, 1.1}
    				box{<-1, -1, -2>, <1, 1, 2>}
    				pigment{color rgb 0} 
    				finish{ambient 0.4 phong 0.5}
    			}
    			box{-1.01, 1.01
    				texture{T}
				}   			       
				scale<1.2 * incrX, 1.2 * incrY, 0.01 * rand(lseed)>
				translate<currentPoint.x, currentPoint.y, 0>
			}
			#local cntY = cntY + 1;
		#end
		#local cntX = cntX + 1;
	#end   
#end

/*************************************************************************

                            Jackson Pollock

*************************************************************************/

#macro Pollock (pigm, strokesX, strokesY, strokeFinish)
	#local widthX = 4/3;
	#local heightY = 1;
	#local cntX = 0;
	#local cntY = 0;     
	#local lseed = seed(1000);       
	#local incrX = widthX/strokesX;
	#local incrY = heightY/strokesY;
	
	blob {
		threshold 0.65
		#while(cntX < strokesX)
			#local cntY = 0;           
			#while(cntY < strokesY)
				#local currentPoint = <incrX*cntX, incrY*cntY, 0>;   
				#local P = eval_pigment(pigm, <currentPoint.x, currentPoint.y, 0>);				
    			#local T = texture{pigment{P} finish{strokeFinish}};   
    			cylinder {<-1, 0, 0>, <1, 0, 0>, 1, 2 
    				texture{T}
					scale <2.5 * incrX, 0.6 * incrX, (0.6 * incrX) + (0.04 * rand(lseed))>
					rotate z*360*rand(lseed)
					translate<currentPoint.x + (incrX*rand(lseed)), currentPoint.y + (incrY*rand(lseed)), 0>
				}   			
				#local cntY = cntY + 1;
			#end
			#local cntX = cntX + 1;
		#end
		scale z*0.2   
	}
#end

/*************************************************************************

                            Roy Lichtenstein

*************************************************************************/

#macro Lichtenstein (pigm, strokesX, strokesY, strokeFinish, bw)
	#local widthX = 4/3;
	#local heightY = 1;
	#local cntX = 0;
	#local cntY = 0;     
	#local lseed = seed(1000);       
	#local incrX = widthX/strokesX;
	#local incrY = heightY/strokesY;
	
	#while(cntX < strokesX)
		#local cntY = 0;           
		#while(cntY < strokesY)
			#local currentPoint = <incrX*cntX, incrY*cntY, 0>;   
			
			#local P = eval_pigment(pigm, <currentPoint.x, currentPoint.y, 0>);
			#local C = (P.red + P.green + P.blue)/3;
			#if(bw)
    			#local T = texture{pigment{color rgb 1} finish{strokeFinish}};
    		#else
    			#local T = texture{pigment{P} finish{ambient 1 diffuse 0}};
    		#end
    		
    		cylinder{0, <0, 0, 0.2>, 0.5
    			
    			texture{T}
    			scale C
    			scale<incrX, incrY, min(incrX, incrY)> 
    			translate<incrX/2, incrY/2, 0>
				translate<currentPoint.x, currentPoint.y, 0>
			}   			
			#local cntY = cntY + 1;
		#end
		#local cntX = cntX + 1;
	#end   
#end

/*************************************************************************

                            M.C. Escher

*************************************************************************/

#macro Escher (pigm, strokesX, strokesY, strokeFinish)
	#local widthX = 4/3;
	#local heightY = 1;
	#local cntX = 0;
	#local cntY = 0;     
	#local lseed = seed(1000);       
	#local incrX = widthX/strokesX;
	#local incrY = heightY/strokesY;
	#local even = 0;
	
	#while(cntX < strokesX)
		#local cntY = 0;           
		#while(cntY < strokesY)
			#local currentPoint = <incrX*cntX, incrY*cntY, 0>;
			#local P = eval_pigment(pigm, <currentPoint.x, currentPoint.y, 0>);
			#local T = texture{pigment{P} finish{strokeFinish}};
			#if(P.red > 0.5 & P.green > 0.5 & P.blue > 0.5) 
    			#local T1 = texture{pigment{P*0.9} finish{strokeFinish}};
    		#else
    			#local T1 = texture{pigment{P*1.1} finish{strokeFinish}};
    		#end
    		
    		box{0, <1, 1, 0.000001>
    			texture {
      				material_map {
   						png "birds.png"
         	        	map_type 0
						interpolate 0
						once
						texture {pigment{transmit 1}}
						texture {T}
						texture {T1}
         			}
				}
				scale 1.5
    			scale<incrX, incrY, 1>
    			translate x * -incrX * 0.3                   
    			translate z*(0.04 * rand(lseed))
    			#if(even = 1)
    				translate -y*(incrY*0.5) 
    			#end
				translate<currentPoint.x, currentPoint.y, 0>
				no_shadow
			}			   			
			#local cntY = cntY + 1;
		#end                    
		#if(even = 1)
    		#local even = 0;
    	#else   
    		#local even = 1;
    	#end
		#local cntX = cntX + 1;
	#end   
#end

/*************************************************************************

                         Guiseppe Arcimboldo

*************************************************************************/

#macro Arcimboldo (imageMap, strokesX, strokesY, strokeFinish, bgr)
	#local StrokeArray = array[7];

	// apricot
	#local StrokeArray[0] = union {
		sphere {0, 0.45}
		sphere {0, 0.45 translate<-0.1, 0, 0.1>}
		cylinder{<0, 0, 0>, <0, 0.6, 0>, 0.03}
		scale<0.8, 1, 0.5>
		scale 1.2
		rotate -z*35
		translate<0.05, -0.03, 0>
	}
	// petal
	#local Petal = union {
		difference {
			sphere{0, 1}
			plane{-y, 0}
			scale<0.2, 1, 0.2>
		}                 
		sphere{0, 0.2}
		scale<0.5, 1, 0.14>
		translate y*1
		scale<1, 0.4, 1>
	}
	// flower
	#local Flower = union {
		#local cnt = 0;               
		#local num = 8;
		#local incr = 360/num;
	
		#while(cnt < num)
			object{Petal rotate z*incr*cnt}
			#local cnt = cnt + 1;
		#end
		sphere{0, 0.2 scale<1, 1, 0.5>}
	}
	#local StrokeArray[1] = Flower;

	// apple
	#local StrokeArray[2] = union {
		torus{0.2, 0.3}                                      
		cylinder{0, <0, 0.46, 0>, 0.03}
		rotate x*-25 
		scale<1, 1.2, 0.6> 
		scale 1.2
		rotate z*20 
		no_shadow
	}

	// flower
	#local StrokeArray[3] = union {
		#local cnt = 0;               
		#local num = 12;
		#local incr = 360/num;
	
		#while(cnt < num)
			object{Petal scale<0.7, 1, 1> rotate z*incr*cnt}
			#local cnt = cnt + 1;
		#end 
		#local cnt = 0;
		#while(cnt < num)
			object{Petal scale<0.7, 1, 1> rotate z*incr*cnt}
			#local cnt = cnt + 1;
		#end
	}

	// pear
	#local StrokeArray[4] = union {
		blob {
			threshold 0.65
			sphere{<0, 0.4, 0>, 0.5, 2 scale<1.4, 1, 1.4>}
			sphere{<0, 0.5, 0>, 0.46, 2}
			sphere{<0, 0.8, 0>, 0.3, 2}
		}
		cylinder{<0, 0.5, 0>, <0, 1.1, 0>, 0.024}
	
		translate -y*0.5
		scale<1, 1, 0.6>
		scale 1.2
		rotate z*45
	}

	// berry
	#local StrokeArray[5] = union {
		sphere{0, 0.1 translate<0.23, 0.1, 0>}
		sphere{0, 0.11 translate<0.3, 0.1, 0>}
		sphere{0, 0.1 translate<0.37, 0.1, 0>}
		
		sphere{0, 0.1 translate<0.15, 0.2, 0>}
		sphere{0, 0.13 translate<0.25, 0.2, 0>}
		sphere{0, 0.13 translate<0.35, 0.2, 0>}
		sphere{0, 0.1 translate<0.45, 0.2, 0>}
		
		sphere{0, 0.1 translate<0.1, 0.3, 0>}
		sphere{0, 0.13 translate<0.2, 0.3, 0>}
		sphere{0, 0.15 translate<0.3, 0.3, 0>}
		sphere{0, 0.13 translate<0.4, 0.3, 0>}
		sphere{0, 0.1 translate<0.5, 0.3, 0>}
		
		sphere{0, 0.1 translate<0.07, 0.4, 0>}
		sphere{0, 0.13 translate<0.15, 0.4, 0>}
		sphere{0, 0.15 translate<0.25, 0.4, 0>}
		sphere{0, 0.15 translate<0.35, 0.4, 0>}
		sphere{0, 0.13 translate<0.45, 0.4, 0>}
		sphere{0, 0.1 translate<0.52, 0.4, 0>}
		
		sphere{0, 0.1 translate<0.1, 0.5, 0>}
		sphere{0, 0.13 translate<0.2, 0.5, 0>}
		sphere{0, 0.15 translate<0.3, 0.5, 0>}
		sphere{0, 0.13 translate<0.4, 0.5, 0>}
		sphere{0, 0.1 translate<0.5, 0.5, 0>}
		
		sphere{0, 0.1 translate<0.17, 0.6, 0>}
		sphere{0, 0.12 translate<0.25, 0.6, 0>}
		sphere{0, 0.12 translate<0.35, 0.6, 0>}
		sphere{0, 0.1 translate<0.43, 0.6, 0>}
	
		sphere{0, 0.2 scale<1, 0.1, 0.2> translate<0.3, 0.72, 0>}
		cylinder{<0.3, 0.5, 0>, <0.3, 0.9, 0>, 0.014}

		translate<-0.3, -0.35, 0>
		scale 1.4
		rotate -z*40
	}

	// flowers
	#local StrokeArray[6] = union {
		object{Flower scale 0.5 translate<0, 0, 0>}
		object{Flower scale 0.5 translate<0, 0.5, 0>}
		object{Flower scale 0.5 translate<0.5, 0.5, 0>}
		object{Flower scale 0.5 translate<0.5, 0, 0>}
		object{Flower scale 0.3 rotate z*15 translate<0.25, 0.25, 0>}
		translate<-0.25, -0.25, 0>
	}

	Multipleton (imageMap, StrokeArray, strokesX, strokesY, strokeFinish, bgr)
#end
 



