// -----------------------------------------------------------------------------
// Scatter Macros for POV-Ray 3.5
//
// Version	: 0.1            
// Author	: Jonathan Rafael Ghiglia 
// Date		: 25th November 2001
//
// --- Conditions d'utilisations -----------------------------------------------
//
// * Vous pouvez utiliser librement ces macros dans vos travaux.
// * Vous pouvez librement utiliser et modifier le code.
// * Credit is always welcome :).
//
// Les commentaires sont fait pour vous faciliter l'adaptation a vos 
// propres besoins. Bien sur, si vous apportez des améliorations ou
// trouvez des erreurs, j'aimerai en être informé.
//
// -----------------------------------------------------------------------------
//
// Comment la macro fonctionne (explication sommaire) :
//
// Cette macro emploie un algorithme géométrique assez simple.
//
// D'abord le macro crée un ensemble (tableau) de points appelés "object_wire[]".
// Ce tableau contient les coordonnées de chaque point inférieur de l'objet dans
// une grille "accuracy_x*accuracy_z".
//
// La macro utilise la fonction "trace()" pour y stocker les coordonnées des
// points. Ces coordonnées seront  plus tard employées pour placer l'objet. Si
// aucune intersection ne se produit, la macro attribue à la composante "y" une
// valeur arbitrairement trés grande, ceci afin d'éviter que ce point soit pris
// en compte dans la recherche des points les plus bas (voir ci-dessous). 
//
// Puis, une position est choisie par la macro "position_macro()", ainsi qu'une
// rotation aléatoire de rotation autour de l'axe de y. Les coordonnées contenues
// dans le tableau "object_wire[]" sont transformées et stockées dans un autre
// tableau nommé "present_wire[]".
// Notez que le tableau "object_wire[]" n'est jamais modifié. 
//
// Les coordonnées contenues dans le tableau "present_wire[]" sont employées pour
// trouver les coordonnées correspondantes dans le tableau "scatter_land[]".
//
// A nouveau, la macro "trace()" est employée. Les coordonnées trouvées sont
// stockées dans un tableau appelée "land_wire[]". Si aucune intersection ne se
// produit, la composante "y" de ce point prend une valeur négative très grande
// (voir ci-dessus). 
//
// La macro recherche ensuite la distance minimum entre les deux points
// correspondants dans "present_wire[]" et "land_wire[]". C'est le premier point
// de contact entre le sol et l'objet.
//
// C'est le seul point de contact qui a 100% de probabilité 100% d'être vraiment
// un point de contact :) 
//
// La macro utilise ce point pour trouver les deux autres points de contact.
// La recherche est limitée à la partie de la grille dans le tableau "present_wire[]"
// bornée par le premier point de contact et par le barycentre (centre idéal de
// la masse) des points de la grille.
//
// Concrètement, la macro recherche deux angles. Le premier angle représente la
// valeur de la rotation exécutée autour d'un axe fixe passant par le premier
// point de contact (cet axe est parallèle à l'axe de z avant d'être transformé
// par une valeur aléatoire).
// Une fois que cette première rotation effectuée, le deuxième angle représente
// la valeur de rotation autour de l'axe passant par le premier et le deuxième
// point de contact. 
// Le macro recherche alors le plus petit angle de rotation dans une direction
// spécifique en calculant l'angle entre la projection des vecteurs de
// "present_wire[]" sur le plan passant par le premier point de contact et
// normale à cette direction, et les vecteurs correspondants dans "land_wire[]".
//
// Le premier point de contact est traité comme origine (ainsi, les vecteur
// auquels il est fait référence sont ceux liant chaque premier point de contact
// du tableau "present_wire[]" aux points du tableau "land_wire[]"). 
//
// La macro écrit alors dans une fichier l'objet "scatter_object" avec toutes les
// tranformations exécutées.
//
// Le même objet est ajouté au tableau "scatter_land[]".
//
// C'était juste une brève explication. Le code parle de lui même... 
//
// -----------------------------------------------------------------------------




// -----------------------------------------------------------------------------
//	The macro ang_vec is used to calculate the angle between two vectors.
//	The formula is elementary :
//		vdot (v1,v2) := vlength(v1)*vlength(v2)*costheta
//		==> theta = acos (vdot(v1,v2)/(vlength(v1)*vlength(v2)))
// -----------------------------------------------------------------------------
#macro ang_vec (v1,v2)
	#if (vdot(v1,v2)/(vlength (v1)*vlength(v2))>1)
		// Inside approximations can lead to this error,
		// which is theoretically impossible. 
		#local ang=0;
	#else
		#local ang= acos(vdot(v1,v2)/(vlength (v1)*vlength(v2)));
	#end

	ang/pi*180 
#end 




// -----------------------------------------------------------------------------
// --- SCATTER MACRO -----------------------------------------------------------
// -----------------------------------------------------------------------------
#macro scatter (name_file)  
	// The objects cannot be placed higher than max_height from the ground
	#ifdef (max_height)
		// The scatter_land grows, but I need to know the y distance from the real ground
		#local old_land= object {scatter_land} 
	#end

	// Using trace for the edges of the bounding box might lead to unexpected results. So I use trace just inside it.
	#declare epsilon = 0.0001; 
	#declare huge = 10000;  // a huge value
	#declare _min = min_extent (scatter_object);
	#declare _max = max_extent (scatter_object);
	// I think of accuracy as the parts which the object is divided in. For this reason, the number of points is accuracy +1
	#declare object_wire = array[x_accuracy+1][z_accuracy+1]  
	#declare present_wire = array[x_accuracy+1][z_accuracy+1]
	#declare land_wire = array[x_accuracy+1][z_accuracy+1]

	#local norm = <0,0,0>;
	#local i=0;
	#while (i<=x_accuracy)
		#local j=0; 
		#while (j<=z_accuracy)
			#declare object_wire[i][j] = trace (scatter_object, _min +epsilon*(x+z) +i*(_max.x-_min.x-2*epsilon)/x_accuracy*x +j*(_max.z-_min.z-2*epsilon)/z_accuracy*z -huge*y,y, norm);
			#if (vlength (norm)=0)
				// If intersection does not occur I use a huge y coordinate.
				#declare object_wire[i][j]=_min +epsilon*(x+z) +i*(_max.x-_min.x-2*epsilon)/x_accuracy*x +j*(_max.z-_min.z-2*epsilon)/z_accuracy*z +huge*y;
			#end
			#local j=j+1;
		#end  
		#local i=i+1;
	#end

	#fopen scatter_file name_file write
	#write (scatter_file, "/*------Scatter array file------*/ \n") 
	#write (scatter_file,"union { \n" )

	#declare present_object = 1;
	#while (present_object<= total_objects)
		//  random rotation amount around y axis
		#declare rand_rot = 360*rand(RS);
		// the chosen position 
		#declare pos = position_macro();
		#declare ok_pos=0;
		// If the position isn't okay, this variable will be set to 0. 
		#while (ok_pos=0) 
			#declare ok_pos=1;  
			// --- the object_wire's coordinates are transformed and stored into present_wire
			#local i=0;
			#while (i<=x_accuracy)
				#local j=0; 
				#while (j<=z_accuracy)
					#declare present_wire[i][j] = vtransform (object_wire[i][j], transform {rotate rand_rot*y translate pos}); 
					#local j=j+1;
				#end  
				#local i=i+1;
			#end
			// --- the present_wire's coordinates are used to calculate the corresponding coordinates of land_wire
			#local norm = <0,0,0>;
			#local i=0;
			#while (i<=x_accuracy)
				#local j=0; 
				#while (j<=z_accuracy)
					#declare land_wire[i][j] = trace (scatter_land,present_wire[i][j]+huge*y,-y,norm) ;
					#if (vlength (norm)=0)
						// If intersection does not occur I use a huge (negative) y coordinate.
						#declare land_wire[i][j]=land_wire[i][j]-huge*y;
					#end 
					#local j=j+1;
				#end  
				#local i=i+1;
			#end

		//------- find the minimum distance between the present_wire's points and the corresponding land_wire's ones
		#declare i_min=0;
		#declare j_min=0;
		#declare min_distance = huge;
		#local i=0;
		#while (i<=x_accuracy)
			#local j=0; 
			#while (j<=z_accuracy)
				#if (present_wire[i][j].y-land_wire[i][j].y<min_distance) #declare min_distance = present_wire[i][j].y-land_wire[i][j].y; #declare i_min=i; #declare j_min=j; #end
					#local j=j+1;
				#end  
				#local i=i+1;
			#end 

		//------- check whether the first point of contact (land_wire[i_min][j_min]) is higher than max_height or not. 
		// If it's higher the macro looks for the lowest point in the four directions and changes the position accordingly.
		#ifdef (max_height)
			#if (land_wire[i_min][j_min].y>trace(old_land,land_wire[i_min][j_min]+huge*y,-y).y+max_height)
				#local vec_n=trace(scatter_land,land_wire[i_min][j_min]+huge*y+vlength(_max-_min)*z,-y);
				#local vec_s=trace(scatter_land,land_wire[i_min][j_min]+huge*y-vlength(_max-_min)*z,-y);
				#local vec_e=trace(scatter_land,land_wire[i_min][j_min]+huge*y+vlength(_max-_min)*x,-y);
				#local vec_w=trace(scatter_land,land_wire[i_min][j_min]+huge*y-vlength(_max-_min)*x,-y);
				#if (min(vec_n.y,vec_s.y)<min(vec_e.y,vec_w.y))
					#declare pos=pos+(vec_n.y<vec_s.y?vec_n:vec_s)*<1,0,1>;
				#else
					#declare pos=pos+(vec_e.y<vec_w.y?vec_e:vec_w)*<1,0,1>;
				#end
				#declare ok_pos=0; 
				#debug "\nObject's position too high! Position changed.\n"
			#end
		#end

		#if (ok_pos!=0) 

		//----- Notice that the present_wire's points are not transformed here for speed reasons. When the other transformations are performed, 
		// I will have to remember that the real coordinates have to be translated by -min_distance*y.

		/*-----find the first angle of rotation (here called second_angle because it refers to the second point of contact)
		The object will be rotated by this angle around the axis passing through land_wire[i_min][j_min] (the first point of contact) and parallel
		to vtransform (z, transform {rotate rand_rot*y})
		The projection of a generic vector present_wire[i][j]-present_wire[i_min][j_min] onto that plane (passing through the first point of contact
		and normal to that direction) is here <present_wire[i][j_min].x,present_wire[i][j].y,present_wire[i][j_min].z>-present_wire[i_min][j_min].
		*/
#if (i_min< ceil (x_accuracy/2)) #declare i_step=1; #else #declare i_step=-1; #end // if the first point of contact is *on the left* of the medium point you go to the right and vice versa 
#if (j_min< ceil (z_accuracy/2)) #declare j_step=1; #else #declare j_step=-1; #end // same here
#declare second_angle=huge;
#declare i=i_min+i_step;
#while (i<= x_accuracy & i>=0)
#declare j=j_min+j_step;
#while (j<=z_accuracy & j>=0)
#if (ang_vec(present_wire[i][j_min]*<1,0,1>+present_wire[i][j].y*y-present_wire[i_min][j_min],land_wire[i][j_min]*<1,0,1>+land_wire[i][j].y*y-land_wire[i_min][j_min])<second_angle)//ang_vec(present_wire[second_i][second_j]-present_wire[i_min][j_min],land_wire[second_i][second_j]-land_wire[i_min][j_min]))
#declare second_angle = ang_vec(present_wire[i][j_min]*<1,0,1>+present_wire[i][j].y*y-present_wire[i_min][j_min],land_wire[i][j_min]*<1,0,1>+land_wire[i][j].y*y-land_wire[i_min][j_min]); 
#declare second_i=i;
#declare second_j=j;
#end
#declare j=j+j_step;
#end
#declare i=i+i_step;
#end  

//-----transform present_wire
#declare dir1 = -i_step*vtransform (z, transform {rotate rand_rot*y}); // The sign of the angle depends on i_step

#local i=0;
#while (i<=x_accuracy)
#local j=0; 
#while (j<=z_accuracy)
#declare present_wire[i][j] = vtransform (present_wire[i][j], transform {translate -(land_wire[i_min][j_min]+min_distance*y) Axis_Rotate_Trans (dir1,second_angle) translate land_wire[i_min][j_min]+min_distance*y}); 
#local j=j+1;
#end  
#local i=i+1;
#end   

//----recalculate land_wire: this can lead to more accurate results, but slows things down... :(
#local norm = <0,0,0>;
#local i=0;
#while (i<=x_accuracy)
#local j=0; 
#while (j<=z_accuracy)
#declare land_wire[i][j] = trace (scatter_land,present_wire[i][j]+huge*y,-y,norm) ;
#if (vlength (norm)=0) #declare land_wire[i][j]=land_wire[i][j]-huge*y; #end // see above
#local j=j+1;
#end  
#local i=i+1;
#end




/*-----find the second angle of rotation (here called third_angle because it refers to the third point of contact).
The object will be rotated by this angle around the axis passing through land_wire[i_min][j_min] (the first point of contact) and
land_wire[second_i][second_j] (the second point of contact).
The projection of a generic vector present_wire[i][j]-present_wire[i_min][j_min] onto that plane (passing through the first point of contact
and normal to that direction) is here present_wire[i][j]-present_wire[i_min][j_min]-vdot(present_wire[i][j]-present_wire[i_min][j_min],dir1)*dir1
where dir1 is the normal of that plane.
*/

#declare i=i_min+i_step; // start from i_min
#if (j_step=1)  // this and the following two lines do have to be improved. The macro tries to find the more *logical* rotation direction.
#if (second_j< ceil (j_min+(z_accuracy-j_min)/2)) #declare j_step=1; #else #declare j_step=-1; #end
#else #if (second_j< ceil(j_min/2)) #declare j_step=1; #else #declare j_step=-1; #end #end

#declare third_angle=huge;
#while (i<= x_accuracy & i>=0 & i!=i_min)
#declare j=second_j+j_step;
#while (j<=z_accuracy & j>=0 & j!=j_min)
#if (ang_vec(present_wire[i][j]-present_wire[i_min][j_min]-vdot(present_wire[i][j]-present_wire[i_min][j_min],dir1)*dir1,land_wire[i][j]-land_wire[i_min][j_min]-vdot(land_wire[i][j]-land_wire[i_min][j_min],dir1)*dir1)<third_angle)//ang_vec(present_wire[second_i][second_j]-present_wire[i_min][j_min],land_wire[second_i][second_j]-land_wire[i_min][j_min]))
#declare third_angle = ang_vec(present_wire[i][j]-present_wire[i_min][j_min]-vdot(present_wire[i][j]-present_wire[i_min][j_min],dir1)*dir1,land_wire[i][j]-land_wire[i_min][j_min]-vdot(land_wire[i][j]-land_wire[i_min][j_min],dir1)*dir1);

#declare third_i=i;
#declare third_j=j;

#end
#declare j=j+j_step;
#end
#declare i=i+i_step;
#end  

#declare dir2=-j_step*vnormalize(present_wire[second_i][second_j]-present_wire[i_min][j_min]);

//---- Here I don't have to tranform all the present_wire, just the third point of contact (the other two points lay on the rotation axis)
#declare present_wire[third_i][third_j] = vtransform (present_wire[third_i][third_j], transform {translate -(land_wire[i_min][j_min]+min_distance*y) Axis_Rotate_Trans (dir2,third_angle) translate land_wire[i_min][j_min]+min_distance*y}); 

//---- Check if the three points of contact are really touching the corresponding points of land_wire. Since these points are rotated, and 
// the corresponding points of land_wire are just their projection, they truly never touch. That's why a height_accuracy float is needed. 
// the first point of contact doesn't need checking.
#if (present_wire[second_i][second_j].y>land_wire[second_i][second_j].y+min_distance+height_accuracy | present_wire[third_i][third_j].y>land_wire[third_i][third_j].y+min_distance+height_accuracy)
#declare pos=pos+vnormalize(present_wire[second_i][second_j]/2-present_wire[i_min][j_min]+present_wire[third_i][third_j]/2)*<1,0,1>*vlength(_max-_min);
#declare ok_pos=0;
#debug concat("\nObject ", str(present_object,0,0), " badly placed on point ",str(i,5,3),", "str(j,5,3),". Changing...\n")
#else
#declare ok_pos=1;
#end 
#end
#end  

//----- write file 

#debug concat ("\nplace object ",str(present_object,0,0),".\n")

#ifndef (scatter_texture_macro)
#write (scatter_file,"object {scatter_object_macro() rotate ",rand_rot,"*y translate ",pos-min_distance*y," translate ",-land_wire[i_min][j_min]," Axis_Rotate_Trans(",dir1,",",second_angle,") Axis_Rotate_Trans(",dir2,",",third_angle,") translate ",land_wire[i_min][j_min]," }\n")

#else
#write (scatter_file,"object {scatter_object rotate ",rand_rot,"*y translate ",pos-min_distance*y," translate ",-land_wire[i_min][j_min]," Axis_Rotate_Trans(",dir1,",",second_angle,") Axis_Rotate_Trans(",dir2,",",third_angle,") translate ",land_wire[i_min][j_min]," scatter_texture_macro() }\n")
#end

//----- add object to scatter_land
#declare scatter_land=
union {
object {scatter_land}
object {scatter_object rotate rand_rot*y translate pos-min_distance*y translate -land_wire[i_min][j_min] Axis_Rotate_Trans(dir1,second_angle) Axis_Rotate_Trans(dir2,third_angle) translate land_wire[i_min][j_min]}
}

#declare present_object= present_object+1;
#end  

#write (scatter_file, "}")
#fclose scatter_file  

#end 






// -----------------------------------------------------------------------------------
// --- multiple scatter macro --------------------------------------------------------
// -----------------------------------------------------------------------------------
#macro scatter_multiple (N,name_file)
	#debug concat("\nscatter_multiple(",str(N,0,0),",",name_file,"\n\n")

	#ifdef (max_height)
		#local old_land= object {scatter_land}
	#end

#declare epsilon = 0.0001;
#declare huge = 10000; 
#declare object_wire = array[x_accuracy+1][z_accuracy+1][N]
#declare _min=array[N]
#declare _max=array[N]   
#declare scatter_object=array[N]
#declare present_wire = array[x_accuracy+1][z_accuracy+1]
#declare land_wire = array[x_accuracy+1][z_accuracy+1]
#local _n=0;
#while (_n<N) 
#declare scatter_object[_n]=object {scatter_object_macro(_n+1)}
#declare _min[_n] = min_extent (scatter_object[_n]);
#declare _max[_n] = max_extent (scatter_object[_n]);
#local norm = <0,0,0>;
#local i=0;
#while (i<=x_accuracy)
#local j=0; 
#while (j<=z_accuracy)
#declare object_wire[i][j][_n] = trace (scatter_object[_n], _min[_n] +epsilon*(x+z) +i*(_max[_n].x-_min[_n].x-2*epsilon)/x_accuracy*x +j*(_max[_n].z-_min[_n].z-2*epsilon)/z_accuracy*z -huge*y,y, norm);
#if (vlength (norm)=0)
#declare object_wire[i][j][_n]=_min[_n] +epsilon*(x+z) +i*(_max[_n].x-_min[_n].x-2*epsilon)/x_accuracy*x +j*(_max[_n].z-_min[_n].z-2*epsilon)/z_accuracy*z +huge*y;
#end // so that it's too high
#local j=j+1;
#end  
#local i=i+1;
#end 
#local _n=_n+1;
#end

#fopen scatter_file name_file write
#write (scatter_file, "/*------Scatter array file------*/ \n") 
#write (scatter_file,"union { \n" )

#declare present_object = 1;
#while (present_object<= total_objects)

#declare _N=floor(rand(RS)*N);   //-----choose an object
#declare rand_rot = 360*rand(RS);
#declare pos= position_macro();  

#declare ok_pos=0;
#while (ok_pos=0)   
#declare ok_pos=1;  

#local i=0;
#while (i<=x_accuracy)
#local j=0; 
#while (j<=z_accuracy)
#declare present_wire[i][j] = vtransform (object_wire[i][j][_N], transform {rotate rand_rot*y translate pos}); 
#local j=j+1;
#end  
#local i=i+1;
#end

#local norm = <0,0,0>;
#local i=0;
#while (i<=x_accuracy)
#local j=0; 
#while (j<=z_accuracy)
#declare land_wire[i][j] = trace (scatter_land,present_wire[i][j]+huge*y,-y,norm) ;
#if (vlength (norm)=0)
#declare land_wire[i][j]=land_wire[i][j]-huge*y;
#end // so that it's too low
#local j=j+1;
#end  
#local i=i+1;
#end

#declare i_min=0;
#declare j_min=0;
#declare min_distance = huge;
#local i=0;
#while (i<=x_accuracy)
#local j=0; 
#while (j<=z_accuracy)
#if (present_wire[i][j].y-land_wire[i][j].y<min_distance)
#declare min_distance = present_wire[i][j].y-land_wire[i][j].y;
#declare i_min=i;
#declare j_min=j;
#end
#local j=j+1;
#end  
#local i=i+1;
#end 

#ifdef (max_height)
#if (land_wire[i_min][j_min].y>trace(old_land,land_wire[i_min][j_min]+huge*y,-y).y+max_height)
#local vec_n=trace(scatter_land,land_wire[i_min][j_min]+huge*y+vlength(_max[_N]-_min[_N])*z,-y);
#local vec_s=trace(scatter_land,land_wire[i_min][j_min]+huge*y-vlength(_max[_N]-_min[_N])*z,-y);
#local vec_e=trace(scatter_land,land_wire[i_min][j_min]+huge*y+vlength(_max[_N]-_min[_N])*x,-y);
#local vec_w=trace(scatter_land,land_wire[i_min][j_min]+huge*y-vlength(_max[_N]-_min[_N])*x,-y);
#if (min(vec_n.y,vec_s.y)<min(vec_e.y,vec_w.y))
#declare pos=pos+(vec_n.y<vec_s.y?vec_n:vec_s)*<1,0,1>;
#else
#declare pos=pos+(vec_e.y<vec_w.y?vec_e:vec_w)*<1,0,1>;
#end
#declare ok_pos=0; 
#debug "\nObject's position too high! Position changed.\n"
#end
#end

#if (ok_pos!=0) 
#if (i_min< ceil (x_accuracy/2)) #declare i_step=1; #else #declare i_step=-1; #end
#if (j_min< ceil (z_accuracy/2)) #declare j_step=1; #else #declare j_step=-1; #end
#declare second_angle=huge;
#declare i=i_min+i_step;
#while (i<= x_accuracy & i>=0)
#declare j=j_min+j_step;
#while (j<=z_accuracy & j>=0)
#if (ang_vec(present_wire[i][j_min]*<1,0,1>+present_wire[i][j].y*y-present_wire[i_min][j_min],land_wire[i][j_min]*<1,0,1>+land_wire[i][j].y*y-land_wire[i_min][j_min])<second_angle)//ang_vec(present_wire[second_i][second_j]-present_wire[i_min][j_min],land_wire[second_i][second_j]-land_wire[i_min][j_min]))
#declare second_angle = ang_vec(present_wire[i][j_min]*<1,0,1>+present_wire[i][j].y*y-present_wire[i_min][j_min],land_wire[i][j_min]*<1,0,1>+land_wire[i][j].y*y-land_wire[i_min][j_min]); 
#declare second_i=i;
#declare second_j=j;
#end
#declare j=j+j_step;
#end
#declare i=i+i_step;
#end  

#declare dir1 = -i_step*vtransform (z, transform {rotate rand_rot*y}); 

#local i=0;
#while (i<=x_accuracy)
#local j=0; 
#while (j<=z_accuracy)
#declare present_wire[i][j] = vtransform (present_wire[i][j], transform {translate -(land_wire[i_min][j_min]+min_distance*y) Axis_Rotate_Trans (dir1,second_angle) translate land_wire[i_min][j_min]+min_distance*y}); 
#local j=j+1;
#end  
#local i=i+1;
#end   

#local norm = <0,0,0>;
#local i=0;
#while (i<=x_accuracy)
#local j=0; 
#while (j<=z_accuracy)
#declare land_wire[i][j] = trace (scatter_land,present_wire[i][j]+huge*y,-y,norm) ;
#if (vlength (norm)=0) #declare land_wire[i][j]=land_wire[i][j]-huge*y; #end // so that it's too low
#local j=j+1;
#end  
#local i=i+1;
#end


#declare i=i_min+i_step;
#if (j_step=1)
#if (second_j< ceil (j_min+(z_accuracy-j_min)/2)) #declare j_step=1; #else #declare j_step=-1; #end
#else #if (second_j< ceil(j_min/2)) #declare j_step=1; #else #declare j_step=-1; #end #end

#declare third_angle=huge;
#while (i<= x_accuracy & i>=0 & i!=i_min)
#declare j=second_j+j_step;
#while (j<=z_accuracy & j>=0 & j!=j_min)
#if (ang_vec(present_wire[i][j]-present_wire[i_min][j_min]-vdot(present_wire[i][j]-present_wire[i_min][j_min],dir1)*dir1,land_wire[i][j]-land_wire[i_min][j_min]-vdot(land_wire[i][j]-land_wire[i_min][j_min],dir1)*dir1)<third_angle)//ang_vec(present_wire[second_i][second_j]-present_wire[i_min][j_min],land_wire[second_i][second_j]-land_wire[i_min][j_min]))
#declare third_angle = ang_vec(present_wire[i][j]-present_wire[i_min][j_min]-vdot(present_wire[i][j]-present_wire[i_min][j_min],dir1)*dir1,land_wire[i][j]-land_wire[i_min][j_min]-vdot(land_wire[i][j]-land_wire[i_min][j_min],dir1)*dir1);

#declare third_i=i;
#declare third_j=j;

#end
#declare j=j+j_step;
#end
#declare i=i+i_step;
#end  

#declare dir2=-j_step*vnormalize(present_wire[second_i][second_j]-present_wire[i_min][j_min]);


#declare present_wire[third_i][third_j] = vtransform (present_wire[third_i][third_j], transform {translate -(land_wire[i_min][j_min]+min_distance*y) Axis_Rotate_Trans (dir2,third_angle) translate land_wire[i_min][j_min]+min_distance*y}); 

#if (present_wire[second_i][second_j].y>land_wire[second_i][second_j].y+min_distance+height_accuracy | present_wire[third_i][third_j].y>land_wire[third_i][third_j].y+min_distance+height_accuracy)
#declare pos=pos+vnormalize(present_wire[second_i][second_j]/2-present_wire[i_min][j_min]+present_wire[third_i][third_j]/2)*<1,0,1>*vlength(_max[_N]-_min[_N]);
#declare ok_pos=0;
#debug concat("\nObject ", str(present_object,0,0), " badly placed on point ",str(i,5,3),", "str(j,5,3),". Changing...\n")
#else
#declare ok_pos=1;
#end 
#end
#end  

#debug concat ("\nobject ",str (present_object,0,0)," placed.\n")

#ifndef (scatter_texture_macro)
#write (scatter_file,"object {scatter_object_macro(",_N+1,") rotate ",rand_rot,"*y translate ",pos-min_distance*y," translate ",-land_wire[i_min][j_min]," Axis_Rotate_Trans(",dir1,",",second_angle,") Axis_Rotate_Trans(",dir2,",",third_angle,") translate ",land_wire[i_min][j_min]," }\n")

#else
#write (scatter_file,"object {scatter_object_macro(",_N+1,") rotate ",rand_rot,"*y translate ",pos-min_distance*y," translate ",-land_wire[i_min][j_min]," Axis_Rotate_Trans(",dir1,",",second_angle,") Axis_Rotate_Trans(",dir2,",",third_angle,") translate ",land_wire[i_min][j_min]," scatter_texture_macro (",_N+1,") }\n")
#end

#declare scatter_land=
union {
object {scatter_land}
object {scatter_object[_N] rotate rand_rot*y translate pos-min_distance*y translate -land_wire[i_min][j_min] Axis_Rotate_Trans(dir1,second_angle) Axis_Rotate_Trans(dir2,third_angle) translate land_wire[i_min][j_min]}
}

#declare present_object= present_object+1;
#end  

#write (scatter_file, "}")
#fclose scatter_file  

#end 






















