/* sor_extents.pov
 * Persistence of Vision Raytracer scene description file
 *
 * Reveals extents and tests tracing of various solids of revolution.
 *
 * Copyright © 2022 Richard Callwood III.  Some rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * 2022-Apr-11  Created.
 * 2022-Apr-18  2 variations to the flat-topped Bézier are added.
 * 2022-Apr-18  All cases are inverted in y.
 * 2022-Apr-23  Inset parameter doubles as x offset.
 * 2022-Apr-25  Inset parameter is renamed to Offset.
 * 2022-Apr-25  Prepared for public comment: private include files are removed
 *              and a license is added.
 */
// +KFF14 +W200 +H266 +GDsor_extents.log
#version max (3.5, min (3.8, version));

#ifndef (Offset) #declare Offset = 0; #end

#include "math.inc"
#include "screen.inc"
#include "shapes.inc"

//--------------------- ENVIRONMENT ------------------------
global_settings
{ assumed_gamma 1
  charset utf8
}
Set_Camera (<0, 6, -34.028>, <0, 6, 0>, 15.8499)
Set_Camera_Aspect_Ratio (0.7519)
#default
{ finish
  { diffuse 0.75 ambient rgb 0.33545
    #if (version >= 3.7) emission 0 #end
  }
}
light_source
{ <-35.395, 76.790, -61.306>,
  rgb 3208
  fade_power 2 fade_distance 1.25
  spotlight point_at <0, 6, 0> radius 45 falloff 90
}
box
{ -<72, 0, 72>, <72, 96, 72>
  pigment { rgb 1 }
  hollow
}

//------------------ DEFINE THE SHAPES ---------------------
#declare NCASES = 7;
#ifndef (Case) #declare Case = max (0, floor ((frame_number - 1) / 2)); #end
#ifndef (Invert) #declare Invert = 1 - mod (frame_number, 2); #end

#declare s_Cases = array [NCASES]
{ "sor", "cubic lathe 1", "Bézier lathe 1a", "Bézier lathe 1b",
  "Bézier lathe 1c", "cubic lathe 2", "Bézier lathe 2",
}
#declare SOR = 0;
#declare CUBIC = 1;
#declare BEZIER = 2;
#declare Types = array [NCASES]
{ SOR, CUBIC, BEZIER, BEZIER, BEZIER, CUBIC, BEZIER,
}
#declare uv_All_ctrls = array [NCASES]
{ array[8] // from sor tutorial
  { <0.0, -0.5>, <3.0,  0.0>, <1.0,  0.2>, <0.5,  0.4>,
    <0.5,  4.0>, <1.0,  5.0>, <3.0, 10.0>, <4.0, 11.0>
  },
  array[6]
  { <-3, 1>, <0, 2>, <3, 4.5>, <3, 9.5>, <0, 12>, <-3, 9.5>,
  },
  array[8]
  { <0, 1>, <2, 1>, <4, 5>, <4, 11>,
    <4, 11>, <3, 11>, <1, 11>, <0, 11>,
  },
  array[8]
  { <0, 1>, <2, 1>, <4, 5>, <4, 11>,
    <4, 11>, <8/3, 11>, <4/3, 11>, <0, 11>,
  },
  array[8]
  { <0, 1>, <2, 1>, <4, 5>, <4, 11>,
    <4, 11>, <2.5, 11>, <1.5, 11>, <0, 11>,
  },
  array[6]
  { <-2, 2.5>, <0, 1>, <3, 1>, <3, 10>, <0, 9>, <-3, 10>,
  },
  array[12]
  { <0, 2>, <1, 1>, <2, 1>, <3, 2>,
    <3, 2>, <4, 3>, <4, 11>, <3, 11>,
    <3, 11>, <2, 11>, <1, 10>, <0, 10>,
  },
}

//------------------- CHOOSE THE SHAPE ---------------------
#declare NCTRLS = dimension_size (uv_All_ctrls[Case], 1);
#if (Invert)
  #declare uv_Ctrls = array [NCTRLS]
  #declare Max = uv_All_ctrls[Case][0].y;
  #declare Min = Max;
  #declare I = 1;
  #while (I < NCTRLS)
    #declare Min = min (Min, uv_All_ctrls[Case][I].y);
    #declare Max = max (Max, uv_All_ctrls[Case][I].y);
    #declare I = I + 1;
  #end
  #declare Subtrd = Min + Max;
  #declare I = 0;
  #while (I < NCTRLS)
    #declare uv_Ctrls [NCTRLS - 1 - I] =
      <uv_All_ctrls[Case][I].x, Subtrd - uv_All_ctrls[Case][I].y>;
    #declare I = I + 1;
  #end
#else
  #declare uv_Ctrls = uv_All_ctrls[Case]
#end

//------------------- CREATE THE SHAPE ---------------------
#macro List (A)
  #local I = 0;
  dimension_size (A, 1),
  #while (I < dimension_size (A, 1))
    A[I]
    #local I = I + 1;
  #end
#end

#declare Form = #switch (Types[Case])
  #case (SOR)
    sor { List (uv_Ctrls) sturm } #break
  #case (CUBIC)
    lathe { cubic_spline List (uv_Ctrls) sturm } #break
  #case (BEZIER)
    lathe { bezier_spline List (uv_Ctrls) sturm } #break
#end

//--------------- FIND THE SHAPE'S MIDDLE ------------------
#declare v_Min = min_extent (Form);
#declare v_Max = max_extent (Form);
#declare yMin = v_Min.y;
#declare yMax = v_Max.y;
#local I = 0;
#while (I < NCTRLS)
  #declare yMin = min (yMin, uv_Ctrls[I].y);
  #declare yMax = max (yMax, uv_Ctrls[I].y);
  #local I = I + 1;
#end
#declare yCtr = (yMin + yMax) / 2;

//---------- ANALYZE AND INSTANTIATE THE SHAPE -------------
#macro Plot (A, Bez_fmt)
  #local RDOT = 0.15;
  #local RCTRL = 0.05;
  #local C_CTRL = rgb <0, 0.4, 0.1>;
  union
  { #if (Bez_fmt)
      #local I = 0;
      #while (I < dimension_size (A, 1))
        sphere { A[I], RDOT }
        cylinder { A[I], A[I+1], RCTRL pigment { C_CTRL } }
        cylinder { A[I+2], A[I+3], RCTRL pigment { C_CTRL } }
        #local I = I + 4;
      #end
      sphere { A[I-1], RDOT }
    #else
      #local I = 0;
      #while (I < dimension_size (A, 1))
        sphere { A[I], RDOT }
        #local I = I + 1;
      #end
    #end
    pigment { blue 1 }
  }
#end

#macro VStr0z (V, Places)
  concat ("<", vstr (2, V, ", ", 0, Places), ", ", str (V.z, 0, 1), ">")
#end

#macro Intersect (v_Start, v_Dir, v_Err)
object
{ #local Norm = <0,0,0>;
  #local adjStart = v_Start + Offset * x;
  #local Ypt = trace (Form, adjStart, v_Dir, Norm);
  #if (VZero (Norm))
    Center_Object (text { ttf "ariblk" "??" 0.003, 0 }, x+y+z)
    translate v_Err
    #debug concat
    ( "Missed vertical trace from ", VStr0z (adjStart, 6),
      " toward <", vstr (3, v_Dir, ", ", 0, 1), ">.\n"
    )
  #else
    #local Side = <v_Min.x - 1, Ypt.y + v_Dir.y * Offset, 0>;
    #local SectPt = trace (Form, Side, x, Norm);
    #if (VZero (Norm))
      Center_Object (text { ttf "ariblk" "?" 0.003, 0 }, x+y+z)
      translate Ypt - x
      #debug concat ("  Missed side trace from ", VStr0z (Side, 16), ".\n")
    #else
      union
      { cylinder { -x, x, 1/3 }
        cylinder { -y, y, 1/3 }
        rotate 45 * z
        scale <0.4, 0.4, 0.001>
        translate SectPt
      }
      #debug concat
      ( "  Side trace from ", VStr0z (Side, 16),
        "\n    ", VStr0z (SectPt, 16), "\n"
      )
    #end
  #end
  pigment { red 0.5 }
}
#end

#declare M = 0.4;
union
{ intersection
  { object
    { Form
      pigment { rgb <1, 0.4, 0.2> }
    }
    plane
    { -z, 0
      pigment { rgb <1, 0.4, 0.2> }
      normal { marble 1 rotate -45 * z scale 0.3 }
    }
  }
  Plot (uv_Ctrls, (Types[Case] = BEZIER))
  difference
  { box { <v_Min.x - M, v_Min.y - M, 0>, <v_Max.x + M, v_Max.y + M, 0.001> }
    box { <v_Min.x, v_Min.y, -0.001>, <v_Max.x, v_Max.y, 0.002> }
    pigment { rgbt <0, 0.5, 0.5, 0.7> }
  }
  #debug concat ("Frame ", str (frame_number, 0, 0), ": ", s_Cases[Case])
  #if (Invert) #debug " inverted" #end
  #debug "\n"
  Intersect (v_Max * y + y, -y, (v_Max - 2) * y)
  Intersect (v_Min * y - y, y, (v_Min + 2) * y)
  translate -yCtr * y
  translate <0, 6, 0>
}

//================================= ANNOTATE ===================================
#macro Annotate (S, V)
  Screen_Object
  ( text
    { ttf "arial" S 0.001, 0
      pigment { rgb 0 }
      scale 15 / image_height
    },
    V, <0.01, 0.01>, yes, 1
  )
#end

Annotate (s_Cases[Case], <0, 1>)
#if (Invert) Annotate ("inverted", <1, 1>) #end
Annotate (concat ("Offset ", str (Offset, 0, 6)), <0.5, 0.56>)
