// Persistence of Vision Ray Tracer Scene Description File
// File: Geomorph_test.pov
// Vers: 3.6
// Desc: Test scene file for the Geomorph macro
// Date: January 2005
// Auth: Thomas de Groot
//

#version 3.7;

#include "colors.inc"

global_settings {
  assumed_gamma 1.0
  radiosity {
    pretrace_start 0.08           // start pretrace at this size
    pretrace_end   0.004          // end pretrace at this size
    count 50, 1000                // higher -> higher quality (1..1600) [35]
    nearest_count 10, 5           // higher -> higher quality (1..10) [5]
    error_bound 1                 // higher -> smoother, less accurate [1.8]
    recursion_limit 2             // how much interreflections are calculated (1..5+) [2]
    low_error_factor .3           // reduce error_bound during last pretrace step
    gray_threshold 0.0            // increase for weakening colors (0..1) [0]
    minimum_reuse 0.015           // reuse of old radiosity samples [0.015]
    maximum_reuse 0.1							// new parameter 3.7 [0.2]
    brightness 1                  // brightness of radiosity effects (0..1) [1]

    adc_bailout 0.01/2
    normal off                   // take surface normals into account [off]
    media off                    // take media into account [off]
    always_sample off           // turn sampling in final trace off [on]
    //max_sample 1.0              // maximum brightness of samples
  }
}

//----------------------------------------------------------
#declare Trees      = false;
#declare Simpletree = false;
#declare Sea        = false;
#declare Fog        = false;
#declare A					= 0;
#declare Quick      = on;
//----------------------------------------------------------
// ----------------------------------------

camera {
  location  <0.0, 5.0, -30>
  direction 1.5*z
  right     x*image_width/image_height
  look_at   <0.0, 5.0,  100.0>
  angle     54
}

sky_sphere {
  pigment {
    gradient y
    color_map {
      [0.0 rgb <0.6,0.7,1.0>]
      [0.7 rgb <0.0,0.1,0.8>]
    }
  }
}

light_source {<-1000,2000,-1000> colour rgb 1*2}
light_source {<1000,1000,-1000> colour rgb 0.2 shadowless}

#if (Fog)
  //fog {fog_type 1 distance 200 color rgb <0.6, 0.6, 0.7>}
  fog {
    fog_type   2
    distance   30
    color      rgb <0.6, 0.6, 0.7> 
    fog_offset -0.5
    fog_alt    2.0
    turbulence 0.8
  }
#end

//=====================================================================================================================================
// Quick use!
#if (Quick)
  #declare Landform = 1;
  #declare Standalone = false;
  #include "Geomorph_mcr2B.pov"
  #switch (Landform)
  #include "Geomorph_Database.inc"
  #end
  Geomorph()
  object {Landscape}
  
//=====================================================================================================================================
#else
#declare Standalone = false;
#if (A)
  #include "Geomorph_mcr2A.pov"
#else
  #include "Geomorph_mcr2B.pov"
#end

// substrate:
#declare HeteroInf = true;  // in function, when true: y scaled infinitely 
#declare HetH = 0.8;
#declare HetLac = 2;
#declare HetOct = 1;
#declare HetOff = 0;
#declare HetT = 0.9;
#declare HetNG = 3;

#declare Noised = false;

#declare Ridged = true;
#declare RidgedInf = true;  // in function, when true: y scaled infinitely
#declare RidH = 1;
#declare RidLac = 1;
#declare RidOct = 1;
#declare RidOff = 0;
#declare RidG = 1;
#declare RidNG = 2;

#declare Fractal = false;
#declare Mandelbrot = off;
#declare Complex = <0.1, 0.2>;
#declare Iterations = 10;
#declare Exponent = 3;
#declare Exterior = 1;
#declare ExFactor = 1.0;
#declare Interior = 0;
#declare InFactor = 1.0;

// noise:
#declare Noise = 0.1;										// controller for the f_noise3d function

// patterns:
#declare Wrinkled = true;
#declare Rippled = false;
#declare Waved = true;
#declare Dented = true;
#declare Isolated = off;									// when true, generates an island or an isolated rock
#declare Isolator = 1;
#declare Crackled = true;
#declare Hexed = false;
#declare Honeycomb = true;

// warp:
#declare Wrinkleswarp = true;
#declare Rippleswarp = false;
#declare Waveswarp = true;
#declare Dentswarp = true;
#declare Cracklewarp = false;

// gain:
#declare Wrinklesgain = 0.4;  
#declare Ripplesgain = 0.2;  
#declare Wavesgain = 0.5;  
#declare Dentsgain = 0.5;  
#declare Cracklegain = 0.5;
#declare Hexgain = 1.0; 

// warp turbulence:
#declare Wrinklesturb = <0.5, 0.9, 0.9>;
#declare Ripplesturb = <0.5, 2, 0.9>;
#declare Wavesturb = <0.5, 0.9, 0.9>;
#declare Dentsturb = <0.5, 0.9, 0.9>;
#declare Crackleturb = <0.5, 0.2, 0.9>;

// scale:
#declare Substratescale = <1.0, 0.8, 1.0>;   //<5.0, 5, 5.0>;
#declare Ridgescale = <1.5, 2.0, 5.0>;
#declare Fractalscale = <3, -1.5, 3>;

#declare Wrinklescale = <1.0, 0.5, 1.0>;
#declare Ripplescale = <0.1, 0.2, 0.1>;
#declare Wavescale = <5, 0.2, 15>;
#declare Dentscale = <0.5, 0.2, 0.5>;			
#declare Cracklescale = <0.5, 0.2, 0.5>;
#declare Cracksize = 0.8;										// multiplier for Cracklescale
#declare Hexscale = <0.5, 1, 0.5>;

// angle of rotation:
#declare Fractalangle = <0, 0, 0>;

#declare Wrinklesangle = <10, 0, -20>;
#declare Ripplesangle = <0, 0, -30>;
#declare Wavesangle = <0, 0, 20>;
#declare Dentsangle = <-10, 0, 0>;
#declare Crackleangle = <0, 0, 0>;
#declare Hexangle = <0, 0, 0>;

// fractal translation:
#declare Fractaltrans = <0, 0, 0>;

// island:
#declare IslandShearX = 30;							// shear of the island function in the x direction
#declare IslandShearZ = 0;							// shear of the island function in the z direction
#declare Islandgain = 0.5;  
#declare Diameter = 15;									// diameter of the island function

// isosurface:
#declare Evaluate = true;
#declare UpperBound = 6;								// Bounding box upper limit
#declare LowerBound = -2;								// Bounding box lower limit
#declare MaxGrad = 10;									// max_gradient of the isosurface (change according to need)
#declare Xsize = 10;										// increase size of the bounding box (not scale!) for X
#declare Zsize = 10;										// increase size of the bounding box (not scale!) for Z
#declare Isoscale = <1, 2, 1>;					// scale of the isosurface
#declare Isorot = <0, 0, 0>;						// rotation of the isosurface
#declare Isotrans = <0, 0, 20>;					// translation of the isosurface

// texture:
//#declare Landtex =										// define your own texture here
// for the internal textures, use:
#declare Simpletex = false;							// just a choice between two nice textures
#declare Slope = true;									// true = slope & altitude; false = gradient texture
#declare SlopeValue = 0.5;							// (0-1) higher values: more slope, less altitude, and vice versa


// now call the macro:
Geomorph()
// ...and the landscape:
object {Landscape}
// remember that you can use this object for the trace function (see below)
#end  //of Quick
//===================================================================================================================

#if (Sea)
//Deep Water:
plane { y, -10.0 pigment {color rgb <0.119608, 0.137255, 0.556863>*0.01}}
//plane { y, -10.0 pigment {color rgb <1,0,0>}}

//----------------------------------------------------------
// Iso_Sea by Marc Jacquier (2003)
// Originally: Isosurface plugin 1beta by S. Shonfield, morayplugin@yahoo.ca
// adapted to left-handed orientation:

#declare Scale_x=7;
#declare Scale_z=5;
#declare Damp=1.2;
#declare Lacunarity=3;
#declare Octave=6;
#declare Offset=1.18;
#declare Gain=5;
#declare Wave_Height=0.8;
isosurface {
   function {
      y +1-f_ridged_mf(x/Scale_x, 0, z/Scale_z,Damp,Lacunarity,Octave,Offset,Gain,2)*Wave_Height
   }
   contained_by {box{<-500,-11,-500>,<500,1,500>}}
   threshold 0
   accuracy 0.001
   max_gradient 20
   //max_trace 1
   //all_intersections
   TdG_NorthSeaWater (
  		      					0.9,  // filter
                      0.25, // transparency
                      3,    // ScaleX
                      2,    // ScaleY
                      1,    // ScaleZ
                      1,    // fade distance
                      3     // fade power
                    )
   scale <1,1,1>
   rotate <0,0,0>
   translate <0, 0, 0>
} // end isosurface

#end

#if (Trees)
// -------- random seeds for the Trees -------------
#declare Seed=seed(2);
#declare Seed2=seed(1);
#declare Test_Render = false;

// -------- Tree textures -------------
#if (Test_Render)
  // simple textures for test renders
  #declare T_Wood=
  texture {
    pigment { color rgb <1, 0, 0> }
  }

  #declare T_Tree=
  texture {
    pigment { color rgb <0, 1, 0> }
  }

#else
  #declare T_Wood=
  texture {
    pigment { color rgb <0.4, 0.2, 0.05> }
    finish {
      specular 0.3
      diffuse 0.5
    }
    normal {
      bozo 0.6
      scale <0.1, 0.1, 0.5>
    }
  }

  #declare T_Tree=
  texture {
    pigment {
      agate
      color_map {
        [0.77 color rgbt 1]
        [0.77 color rgb <0.2, 0.5, 0.10> ]
        [0.85 color rgb <0.4, 0.6, 0.15> ]
        [0.97 color rgb <0.4, 0.6, 0.15> ]
        [0.99 color rgb <0.4, 0.2, 0.05> ]
      }
      scale 0.5
      warp { turbulence 0.4 }
    }
    finish {
      diffuse 0.5
      brilliance 1.5
      ambient 0.07
    }
    normal {
      wrinkles 0.6
      scale 0.5
    }
  }
#end

// -------- Tree macro -------------
#if (Simpletree)
 #macro TreeA (Size)
  union {
    cylinder { 0, Size*y, Size*0.04 }       // Trunk
    union {                                 // Roots
      cylinder { 0, -Size*0.30*y, Size*0.025 rotate (40+rand(Seed)*20)*x rotate rand(Seed2)*360*y }
      cylinder { 0, -Size*0.25*y, Size*0.020 rotate (40+rand(Seed)*20)*x rotate rand(Seed2)*360*y }
      cylinder { 0, -Size*0.27*y, Size*0.022 rotate (40+rand(Seed)*20)*x rotate rand(Seed2)*360*y }
    }

    union {                                 // Branches
      cylinder {
        0, Size*0.35*y, Size*0.025
        rotate (40+rand(Seed)*35)*x
        rotate rand(Seed2)*360*y
        translate Size*(0.7+0.3*rand(Seed))*y
      }
      cylinder {
        0, Size*0.40*y, Size*0.026
        rotate (40+rand(Seed)*35)*x
        rotate rand(Seed2)*360*y
        translate Size*(0.7+0.3*rand(Seed))*y
      }
      cylinder {
        0, Size*0.27*y, Size*0.022
        rotate (40+rand(Seed)*35)*x
        rotate rand(Seed2)*360*y
        translate Size*(0.7+0.3*rand(Seed))*y
      }
    }

    #if (Test_Render)                       // Foliage
      sphere {
        Size*y, Size*(0.4+rand(Seed)*0.15)
        scale <rand(Seed)*0.5+0.5, 1, rand(Seed)*0.5+0.5>
        texture { T_Tree scale Size translate rand(Seed)*6 }
      }
    #else
      union {
        sphere {
          Size*y, Size*(0.4+rand(Seed)*0.15)
          scale <rand(Seed)*0.5+0.5, 1, rand(Seed)*0.5+0.5>
          texture { T_Tree scale Size translate rand(Seed)*6 }
        }
        sphere {
          Size*y, Size*(0.3+rand(Seed)*0.15)
          scale <rand(Seed)*0.5+0.5, 1, rand(Seed)*0.5+0.5>
          texture { T_Tree scale Size translate rand(Seed)*6 }
        }
        sphere {
          Size*y, Size*(0.2+rand(Seed)*0.15)
          scale <rand(Seed)*0.5+0.5, 1, rand(Seed)*0.5+0.5>
          texture { T_Tree scale Size translate rand(Seed)*6 }
        }
        sphere {
          Size*y, Size*(0.3+rand(Seed)*0.15)
          scale <rand(Seed)*0.5+0.5, 1, rand(Seed)*0.5+0.5>
          texture { T_Tree scale Size translate rand(Seed)*6 }
        }
        sphere {
          Size*y, Size*(0.2+rand(Seed)*0.15)
          scale <rand(Seed)*0.5+0.5, 1, rand(Seed)*0.5+0.5>
          texture { T_Tree scale Size translate rand(Seed)*6 }
        }
        sphere {
          Size*y, Size*(0.3+rand(Seed)*0.15)
          scale <rand(Seed)*0.5+0.5, 1, rand(Seed)*0.5+0.5>
          texture { T_Tree scale Size translate rand(Seed)*6 }
        }
      }
    #end

    texture { T_Wood scale Size }
  }
 #end // end of macro
#end // end of Simpletree


#include "C:\VirtualReality\Data files\POV-Tree files\TdG_PinParasol_mesh.inc"
#declare TreeB = object {TREE}

#if (Simpletree)
  #declare Treeobj = object {TreeA (0.5+rand(Seed)*0.02)}
#else
  #declare Treeobj = object {TREE}
#end

// -------- Placing the trees -------------

#declare Spacing=0.7;
#declare Cnt=0;


#declare PosX=-20;

#while (PosX < 20)

  #declare PosZ=-25;

  #while (PosZ < 25)

    // trace function
    #declare Norm = <0, 0, 0>;
    #declare Start = <PosX+(rand(Seed)-0.5)*Spacing, UpperBound+Isoscale.y+1, PosZ+(rand(Seed)-0.5)*Spacing>;
    #declare Pos = trace (
                  Landscape,     	 // object to test
                  Start,           // starting point
                  -y,              // direction
                  Norm );          // normal


    #if (Norm.x != 0 | Norm.y != 0 | Norm.z != 0)   // if intersection is found, normal differs from 0
      #if ((vdot(Norm, y)>0.5) & (Pos.y > 0.2))  			// criteria for placing trees: not too steep and not too high
        object {
            Treeobj
          scale RRand(1.0, 2.5, Seed)
          rotate RRand(0, 360, Seed2)*y
          translate Pos
        }
        #declare Cnt=Cnt+1;
      #end
    #end

    #declare PosZ=PosZ+Spacing;

  #end

  #declare PosX=PosX+Spacing;
#end

#end // end of if Trees

//=====================================================================================================================================

