/* icls.inc
 *
 * "i Chain Link solver" code.
 *
 * 202504.1
 */

#ifndef (icls__include_temp)

#declare icls__include_temp = version;
#version 3.8;

#ifdef (View_POV_Include_Stack)
  #debug "including 'icls.inc'\n"
#end

/* --[constants & helpers]-------------------------------------------------- */

#include "bsvars.inc"
#include "pvars.inc"

#macro iclsIsBool(v_) (1 = v_ | 0 = v_) #end

#macro iclsIsInt(v_) (v_ = int(v_)) #end

/* "epsilon" */
#declare icls__eps_ = 1e-3;

/* grid centre points */
#declare icls__data_centres_ = array [6][6] {
  {<.5,-.5>,<2,-.5>,<3.5,-.5>,<5,-.5>,<6.5,-.5>,<8,-.5>},
  {<.5,-2>,<2,-2>,<3.5,-2>,<5,-2>,<6.5,-2>,<8,-2>},
  {<.5,-3.5>,<2,-3.5>,<3.5,-3.5>,<5,-3.5>,<6.5,-3.5>,<8,-3.5>},
  {<.5,-5>,<2,-5>,<3.5,-5>,<5,-5>,<6.5,-5>,<8,-5>},
  {<.5,-6.5>,<2,-6.5>,<3.5,-6.5>,<5,-6.5>,<6.5,-6.5>,<8,-6.5>},
  {<.5,-8>,<2,-8>,<3.5,-8>,<5,-8>,<6.5,-8>,<8,-8>}
};

/* colours for found digits (gray-ish), grid and clues (blue), and scene
 * background and "boxes" (white-ish).
 */
#declare icls__data_colours_ = array [3] {
  color rgb <.175,.175,.175>,
  color rgb <0,0,1>,
  color rgb <.956562,.947969,1>
};

#macro icls__objTexture(i_)
  texture {pigment {icls__data_colours_[i_]} finish {ambient 0 emission 1}}
#end

/* display digits */
#declare icls__data_ddisp_ = array [7] {"","1","2","3","4","5","6"};

#macro icls__incr(v_,optional n_)
  #if (!defined(local.n_)) #local n_ = 1; #end
  #local v_ = v_ + n_;
#end

#macro icls__withZ(v_,z_) <v_.x, v_.y, z_> #end

#macro icls__n2rc(i_)
  #local r_ = int(i_/6);
  #local c_ = mod(i_,6);
  (r_,c_)
#end

#macro icls__rc2rc(rc_)
  #local r_ = int(rc_/10) - 1;
  #local c_ = mod(rc_,10) - 1;
  (r_,c_)
#end

#macro icls__cl2point(v_) icls__data_centres_[v_.x][v_.y] #end

#macro icls__hyphen(n_)
  #local s_ = "";
  #for (i_,1,n_) #local s_ = concat(s_,"-"); #end
  s_
#end

#macro icls__debugEmit(s_)
  #if (iclsOptDebug) #debug concat("[II] ", s_, ".\n") #end
#end

#macro icls__newDigEmit(r_,c_,v_)
  #if (iclsOptDebug & iclsOptVerbose)
    #debug concat("[II] insert digit '", str(v_,0,0), "' @ [", str(r_,0,0),
            "][", str(c_,0,0), "].\n")
  #end
#end

#macro icls__passSepEmit(n_)
  #local s_ = concat("[pass ",str(n_,0,0),"]");
  #debug concat("\n",icls__hyphen(5),s_,icls__hyphen(67-strlen(s_)),"\n")
#end

/* --[drawing the grid]----------------------------------------------------- */

/* seen from the camera, the objects lie in three planes: the digits, the
 * cells, and the cell "chain" links.  all coords are '<x,y>'s only, use
 * a wrapper macro to provide the 'z'.
 */

/* cells @ z = .25 +- width (same as chain link radius) */
#macro icls__drawCell(i_,elem_)
  #local w_ = .0265;
  #local ctr_ = icls__data_centres_[pvGet(elem_.idx_,2)][pvGet(elem_.idx_,3)];
  #local pi1_ = ctr_ - (.5 - w_);
  #local pi2_ = ctr_ + (.5 - w_);
  #local po1_ = ctr_ - (.5 + w_);
  #local po2_ = ctr_ + (.5 + w_);
  union {
    box {
      icls__withZ(pi1_, .25 - w_), icls__withZ(pi2_, .25 + w_)
      icls__objTexture(2)
    }
    box {
      icls__withZ(po1_, .25 - w_), icls__withZ(po2_, .25 + w_)
      icls__objTexture(1)
      scale 1 + icls__eps_
    }
  }
#end

/* linking chains @ z = .75, "magic number" radii */
#macro icls__drawChain(i_,elem_,arg_)
  #if (i_)
    #local p1_ = icls__withZ(icls__cl2point(arg_), .75);
    #local p2_ = icls__withZ(icls__cl2point(elem_), .75);
    union {
      cylinder {p1_,p2_,.0265 icls__objTexture(1)}
      sphere {icls__withZ(p2_, .75), .0265 icls__objTexture(1)}
    }
  #else
    sphere {icls__withZ(elem_, .75), .0265 icls__objTexture(1)}
  #end
  #local arg_ = elem_;
#end

#macro icls__buildChain(i_,elem_,arg_)
  #local old_ = <-1,-1>;
  #local ta_ = array [6];
  #for (k_, 0, 5)
    #local ta_[k_] = <pvGet(puzzle_.cells_[elem_[0][k_]].idx_,2),
            pvGet(puzzle_.cells_[elem_[0][k_]].idx_,3)>;
  #end
  Foreach(ta_, dictionary {.Macro: "icls__drawChain", .Arg: "old_"})
#end

/* numbers @ z = 0, guard against negative '.cix' */
#macro icls__drawDigit(i_,elem_)
  #local ctr_ = icls__data_centres_[pvGet(elem_.idx_,2)][pvGet(elem_.idx_,3)];
  #local s_ = text {
    ttf "cyrvetic.ttf", icls__data_ddisp_[elem_.digit_], .1, 0
    scale .75
  };
  #local sx_ = (max_extent(s_) - min_extent(s_)) / 2;
  #local at_ = <ctr_.x - sx_.x, ctr_.y - sx_.y, 0>;
  #if (elem_.digit_)
    object {s_ icls__objTexture(elem_.cix_) translate at_}
  #end
#end

#macro iclsDrawGrid(pd_)
  icls__debugEmit("drawing puzzle grid from current")
union {
  Foreach(pd_.chains_, dictionary {.Macro: "icls__buildChain", .Arg: "pd_"})
  Foreach(pd_.cells_, dictionary {.Macro: "icls__drawCell"})
  Foreach(pd_.cells_, dictionary {.Macro: "icls__drawDigit"})
}
#end

/* --[debug output]--------------------------------------------------------- */

/* debug output the whole puzzle dictionary */

#macro icls__dumpSarray(i_,elem_)
  #local s_ = "";
  #for (j_,0,5)
    #local s_ = concat(s_," ",str(elem_[0][j_],2,0));
  #end
  #local s_ = concat(s_,"  ",elem_[1],".");
  #debug concat(s_,"\n")
#end

#macro iclsDumpPuzzle(d_)
  #if (iclsOptDebug & iclsOptVerbose)
    #local i_ = 0;
    #debug concat("label: ",d_.legend_,"\n")
    #debug concat("grid/cell state:\n")
    #while (i_ < 36)
      #local s_ = concat("cell[",str(i_,-2,0),"]  ",
              str(d_.cells_[i_].digit_,0,0));
      #local s_ = concat(s_,"  {",vstr(4,d_.cells_[i_].idx_," ",0,0));
      #local s_ = concat(s_,"}  ",str(d_.cells_[i_].cix_,2,0));
      #local s_ = concat(s_,"  ",d_.cells_[i_].avail_);
      #debug concat(s_,".\n")
      icls__incr(i_,)
    #end
    #debug concat("by row:\n")
    Foreach(d_.rows_, dictionary {.Macro: "icls__dumpSarray"})

    #debug concat("by column:\n")
    Foreach(d_.cols_, dictionary {.Macro: "icls__dumpSarray"})

    #debug concat("by chain:\n")
    Foreach(d_.chains_, dictionary {.Macro: "icls__dumpSarray"})
    #debug concat("cell avails gridded:\n")
    #for (j_, 0, 5)
      #local s_ = "";
      #for (i_, 0, 5)
        #local s_ = concat(s_, " ", d_.cells_[d_.grid_[j_][i_]].avail_);
      #end
      #debug concat(s_, "\n")
    #end
  #end
#end

/* --[make new puzzle]------------------------------------------------------ */

/* build & return a "ready-to-go" puzzle dictionary */

/* empty as in no clues yet, use chains data only */
#macro icls__makeEmptyPuzzle(fdat_)
  #local rd_ = dictionary;
  #local rd_.grid_ = array [6][6];
  #for (j_, 0, 5) #for (i_, 0, 5) #local rd_.grid_[j_][i_] = -1; #end #end
  #local rd_.chains_ = array [6];
  #local rd_.cols_ = array [6];
  #local rd_.rows_ = array [6];
  #for (i_, 0, 5)
    #local rd_.chains_[i_] =
            array mixed [2] {array [6] {-1,-1,-1,-1,-1,-1},"111111"};
    #local rd_.cols_[i_] =
            array mixed [2] {array [6] {-1,-1,-1,-1,-1,-1},"111111"};
    #local rd_.rows_[i_] =
            array mixed [2] {array [6] {-1,-1,-1,-1,-1,-1},"111111"};
  #end
  #local rd_.ccnt_ = 0;
  #local k_ = 0;
  #local rd_.cells_ = array [36];
  #while (k_ < 36)
    #local (ch_, ln_) = icls__n2rc(k_);
    #local (r_, c_) = icls__rc2rc(fdat_[k_]);
    #local rd_.grid_[r_][c_] = k_;
    #local rd_.cols_[c_][0][r_] = k_;
    #local rd_.rows_[r_][0][c_] = k_;
    #local rd_.chains_[ch_][0][ln_] = k_;
    #local rd_.cells_[k_] = dictionary {
        .digit_ : 0,
        .idx_   : pvPack(ch_, ln_, r_, c_),
        .cix_   : -1,
        .avail_ : "111111"
    };
    icls__incr(k_,)
  #end
  #for (j_, 0, 5)
    #for (i_, 0, 5)
      #if (-1 = rd_.grid_[j_][i_])
        #error "oops, bad file data, bailing out."
      #end
    #end
  #end
  rd_
#end

/* _setDigit(puzzle-dict, row, col, digit, colour-index)
 * must use to insert every new digit into the grid.
 */
#macro icls__setDigit(d_, j_, i_, v_, ci_)
  #local n_ = d_.grid_[j_][i_];
  #if (d_.cells_[n_].digit_)
    #error concat("oops, cell[",str(n_,0,0),"] already set.")
  #end
  #local (ch_, ln_, r_, c_) = pvUnpack(d_.cells_[n_].idx_);
  bsvClear(d_.chains_[ch_][1], v_)
  bsvClear(d_.cols_[c_][1], v_)
  bsvClear(d_.rows_[r_][1], v_)
  #for (k_, 0, 5)
    bsvClear(d_.cells_[d_.chains_[ch_][0][k_]].avail_, v_)
    bsvClear(d_.cells_[d_.cols_[c_][0][k_]].avail_, v_)
    bsvClear(d_.cells_[d_.rows_[r_][0][k_]].avail_, v_)
  #end
  #local d_.cells_[n_].digit_ = v_;
  #local d_.cells_[n_].cix_ = ci_;
  #local d_.cells_[n_].avail_ = "000000";
  icls__incr(d_.ccnt_,)
  icls__newDigEmit(r_, c_, v_)
#end

#macro iclsMakePuzzle(fd_)
  icls__debugEmit("building puzzle dict from file data")
  #local rtn_ = icls__makeEmptyPuzzle(fd_.Data);
  #local rtn_.legend_ = fd_.Names[0];
  #for (i_, 36, dimension_size(fd_.Data,1)-1, 2)
    #local (r_, c_) = icls__rc2rc(fd_.Data[i_]);
    icls__setDigit(rtn_, r_, c_, fd_.Data[1+i_], 1)
  #end
  rtn_
#end

/* ------------------------------------------------------------------------- */

/* strategy #1 - singletons.  first find a sole avail digit in a row,
 * column, and chain, and in cells.
 */

#macro icls__countAvail(i_,elem_,arg_)
  #if (bsvTrue(puzzle_.cells_[elem_].avail_, arg_.digit_))
    icls__incr(arg_.n_,)
    #local arg_.at_ = i_;
  #end
#end

#macro icls__singleInChain(d_, digit_)
  #local rtn_ = 0;
  #for (i_, 0, 5)
    #local td_ = dictionary {.digit_: digit_, .n_: 0, .at_: 0};
    #if (bsvTrue(d_.chains_[i_][1], digit_))
      Foreach(d_.chains_[i_][0],
              dictionary {.Macro: "icls__countAvail", .Arg: "td_"})
      #if (1 = td_.n_)
        #local (r_, c_) = (pvGet(d_.cells_[d_.chains_[i_][0][td_.at_]].idx_,2),
                pvGet(d_.cells_[d_.chains_[i_][0][td_.at_]].idx_,3));
        icls__setDigit(d_, r_, c_, digit_, 0)
        icls__incr(rtn_,)
        tmcAdd(tmcn_,"singleInChain")
      #end
    #end
    #undef td_
  #end
  rtn_
#end

#macro icls__singleInCol(d_, digit_)
  #local rtn_ = 0;
  #for (i_, 0, 5)
    #local td_ = dictionary {.digit_: digit_, .n_: 0, .at_: 0};
    #if (bsvTrue(d_.cols_[i_][1], digit_))
      Foreach(d_.cols_[i_][0],
              dictionary {.Macro: "icls__countAvail", .Arg: "td_"})
      #if (1 = td_.n_)
        #local (r_, c_) = (pvGet(d_.cells_[d_.cols_[i_][0][td_.at_]].idx_,2),
                pvGet(d_.cells_[d_.cols_[i_][0][td_.at_]].idx_,3));
        icls__setDigit(d_, r_, c_, digit_, 0)
        icls__incr(rtn_,)
        tmcAdd(tmcn_,"singleInCol")
      #end
    #end
    #undef td_
  #end
  rtn_
#end

#macro icls__singleInRow(d_, digit_)
  #local rtn_ = 0;
  #for (i_, 0, 5)
    #local td_ = dictionary {.digit_: digit_, .n_: 0, .at_: 0};
    #if (bsvTrue(d_.rows_[i_][1], digit_))
      Foreach(d_.rows_[i_][0],
              dictionary {.Macro: "icls__countAvail", .Arg: "td_"})
      #if (1 = td_.n_)
        #local (r_, c_) = (pvGet(d_.cells_[d_.rows_[i_][0][td_.at_]].idx_,2),
                pvGet(d_.cells_[d_.rows_[i_][0][td_.at_]].idx_,3));
        icls__setDigit(d_, r_, c_, digit_, 0)
        icls__incr(rtn_,)
        tmcAdd(tmcn_,"singleInRow")
      #end
    #end
    #undef td_
  #end
  rtn_
#end

#macro icls__singleInCell(d_, digit_)
  #local rtn_ = 0;
  #for (i_, 0, 35)
    #if (bsvTrue(d_.cells_[i_].avail_, digit_)
            & 1 = bsvCount(d_.cells_[i_].avail_))
      #local (r_, c_) = (pvGet(d_.cells_[i_].idx_,2),
              pvGet(d_.cells_[i_].idx_,3));
      icls__setDigit(d_, r_, c_, digit_, 0)
      icls__incr(rtn_,)
      tmcAdd(tmcn_,"singleInCell")
    #end
  #end
  rtn_
#end

#macro iclsSingletons(d_)
  tmcAdd(tmcn_,"singletons")
  #local rtn_ = 0;
  #for (i_, 1, 6)
    icls__incr(rtn_, icls__singleInChain(d_, i_))
    icls__incr(rtn_, icls__singleInCol(d_, i_))
    icls__incr(rtn_, icls__singleInRow(d_, i_))
    icls__incr(rtn_, icls__singleInCell(d_, i_))
  #end
  rtn_
#end


/* ------------------------------------------------------------------------- */

/* strategy #2 - to be supplied */

#macro iclsStub(d_)
  #local rtn_ = 0;
  /* the rules if you fancy a "go", renaming aside:
   * 1) your code must treat the puzzle dictionary 'd_' as read-only.
   * 2) if you find a new digit and want to add it, you must use the
   *    'icls__setDigit()' macro; see the "Singleton" macros above for
   *    reference.
   * 3) your code must return the number of new digits it's added.
   */
  rtn_
#end


/* ------------------------------------------------------------------------- */

#version icls__include_temp;

#end

/* -------------------------------------------------------------------- *
  * the content above is covered by the GNU General Public License v3+ *
  * copyright (c) 2025 jr <creature.eternal@gmail.com>.                *
  * all rights reserved.                                               *
 * -------------------------------------------------------------------- */
