/* filed.inc
 *
 * a flexible, easy-to-use macro for reading and writing text data
 * files formatted as CSV of the POV-Ray variant.
 *
 * Persistence of Vision Raytracer version 3.8 or later.
 *
 *
 * macro Filed(dict)
 *
 *   dict   - a 'dictionary {}'.
 *
 *     .File     - the (string) name of the text file to use.
 *
 *     .Access   - the file access (string): read, write, or append.
 *                 only the (lowercase) initial is needed.
 *
 *     .Fields   - an array of "little language" strings describing
 *                 the record's fields, in order.
 *
 *     .Data     - required for write/append, must not exist when read.
 *                 array 1D/2D, or 1D of array mixed.
 *
 *     .hasNames - opt bool, default 'false'.  whether file read has
 *                 column name labels in first row, or 'Names' should
 *                 be written as first row.
 *
 *     .Names    - opt array of strings.
 *                 must not exist when read, when write requires same
 *                 number of elements as 'Fields'.
 *
 *     .Range    - opt array [2] of ints.  the 'from' and 'to' record
 *                 numbers (inclusive) to read.  first record always '1'
 *                 irrespective of existing 'Names' row.
 *
 *     .Strict   - some warnings become errors.  default 'true'.
 *
 *     .Verbose  - enable (basic) feedback, and parse warnings.
 *                 default 'false'.
 *
 *
 * variable fild_workingDir
 *
 *   this optional string provides a working directory where file data
 *   will be read and written.  it should end with the path separator
 *   appropriate for the operating system.  used in macro calls from
 *   point of (re-)declaration.
 *
 *
 * version: 202108.2
 */

#ifndef (fild__include_temp)

#declare fild__include_temp = version;

#version 3.8;

#ifdef (View_POV_Include_Stack)
  #debug "including 'filed.inc'\n"
#end

/* no default, ie CWD */
#ifndef (global.fild_workingDir)
  #declare fild_workingDir = "";
#end

/* -----[constants + helpers]------------------------------------------------ *
 * maxf    - maximum fields per record.
 * modes   - file access mode strings.
 * incr    - incrementer.
 * isBool  - bool (0|1) test.
 * isInt   - int test.
 * warning - "Parse Warning"s when verbose.
 */

#declare fild__data_maxf_ = 50;

#declare fild__data_modes_ = array [3] {"read", "write", "append"};

#macro fild__incr(v_,optional n_)
  #if (!defined(local.n_))
    #local n_ = 1;
  #end
  #local v_ = v_ + n_;
#end

#macro fild__isBool(v_) (1 = v_ | 0 = v_) #end

#macro fild__isInt(v_) (v_ = int(v_)) #end

#macro fild__warning(s_)
  #if (idict_.verbose_)
    #warning s_
  #end
#end

/* -----[formats + pretty printing]----------------------------------------- */

#macro fild__boolFmt(f_)
  #local s_ = "no";
  #if (f_) #local s_ = "yes"; #end
  s_
#end

#macro fild__strFmt(s_)
  #local w_ = 20;
  #local n_ = strlen(s_);
  #if (w_ > n_)
    #local s_ = concat(substr("                    ",1,(w_-n_)),s_);
  #end
  s_
#end

#macro fild__vectFmt(n_,v_,p_)
  concat("<",vstr(n_,v_,",",0,p_),">")
#end

#macro fild__hyphen(n_)
  #local s_ = "";
  #for (i_,1,n_)
    #local s_ = concat(s_,"-");
  #end
  s_
#end

#macro fild__emitBool(s_,v_)
  #debug concat(fild__strFmt(s_),": ",fild__boolFmt(v_),"\n")
#end

#macro fild__emitInt(s_,v_)
  #debug concat(fild__strFmt(s_),": ",str(v_,0,0),"\n")
#end

#macro fild__emitStr(s_,v_)
  #debug concat(fild__strFmt(s_),": ",v_,"\n")
#end

#macro fild__emitHeader()
  #debug concat(fild__hyphen(5),"[Filed]",fild__hyphen(60),"\n")
  fild__emitStr("file name",idict_.fname_)
  fild__emitStr("access mode",fild__data_modes_[idict_.mode_])
  fild__emitBool("strict",idict_.strict_)
  fild__emitInt("# columns",idict_.rcdn_)
  fild__emitBool("column labels",idict_.hasnames_)
  fild__emitBool("array mixed",!idict_.homogene_)
  #local s_ = concat("# records ",fild__data_modes_[idict_.mode_]);
  fild__emitInt(s_,dimension_size(idict_.data_,1))
  #if (idict_.range_[1])
    fild__emitBool("range filter",true)
    fild__emitInt("# records",dimension_size(dict_.Data,1))
  #end
  #debug concat(fild__hyphen(72),"\n")
#end

/* -----[record field coding]------------------------------------------------ *
 * array 'rcdf_' element {type, precision, Nvector}.
 * type 0 "S"   string, regular quoted.
 *      1 "B"   boolean type.
 *      2 "I"   integer, read/write with 0 precision.
 *      3 "Fn"  float, write with given precision, eg "F2" --> "n.nn",
 *              or use "G" for fixed six decimals.
 *      4 "Vcn" vector, where 2<=c<=5, and n as for float.
 */

#macro fild__mkSBIG(s_)
  #if ("S" = s_)
    #local el_ = array [3] {0,0,0};
  #elseif ("B" = s_)
    #local el_ = array [3] {1,0,0};
  #elseif ("I" = s_)
    #local el_ = array [3] {2,0,0};
  #elseif ("G" = s_)
    #local el_ = array [3] {3,6,0};
  #else
    #error concat("oops, unknown/bad field type '",s_,"'.")
  #end
  el_
#end

#macro fild__mkFV(s_)
  #if ("F" = substr(s_,1,1))
    #local tt_ = 3;
    #local tn_ = 0;
    #local tp_ = val(substr(s_,2,strlen(s_)-1));
  #elseif ("V" = substr(s_,1,1))
    #local tt_ = 4;
    #local tn_ = val(substr(s_,2,1));
    #if (2 > tn_ | 5 < tn_)
      #error "oops, bad vector spec, expected 2,..,5 components."
    #end
    #local tp_ = 0;
    #if (2 < strlen(s_))
      #local tp_ = val(substr(s_,3,strlen(s_)-2));
    #end
  #else
    #error concat("oops, unknown/bad field type '",s_,"'.")
  #end
  #if (0 > tp_ | 16 < tp_)
    #error "oops, precision out-of-range (0,..16)."
  #end
  array [3] {tt_,tp_,tn_}
#end

/* -----[filtering records]-------------------------------------------------- *
 * if no range (ie {0,0}) return all records, else return given set;
 * if fewer records than 'to' emit warning.  the selected range of data is
 * stored in either an 'array mixed', or as simple 1D or 2D array if
 * "homogeneous".
 */

#macro fild__rdmkrng()
  #local n_ = dimension_size(idict_.data_,1);
  #if (!(idict_.range_[0] + idict_.range_[1]))
    #local vs_ = 0;
    #local ve_ = n_;
  #elseif (dimension_size(idict_.data_,1) < idict_.range_[1])
    fild__warning("note - read fewer records than given in 'Range'.")
    #local vs_ = idict_.range_[0] - 1;
    #local ve_ = n_;
  #else
    #local vs_ = idict_.range_[0] - 1;
    #local ve_ = idict_.range_[1];
  #end
  (vs_, ve_)
#end

#macro fild__rdFilter()
  #local (from_,to_) = fild__rdmkrng();
  #local n_ = to_ - from_;
  #local k_ = 0;
  #if (!idict_.homogene_)
    #local a_ = array [n_];
    #for (i_, from_, to_-1)
      #local a_[k_] = idict_.data_[i_];
      fild__incr(k_,)
    #end
  #elseif (1 = idict_.rcdn_)
    #local a_ = array [n_];
    #for (i_, from_, to_-1)
      #local a_[k_] = idict_.data_[i_][0];
      fild__incr(k_,)
    #end
  #else
    #local a_ = array [n_][idict_.rcdn_];
    #for (j_, from_, to_-1)
      #for (i_, 0, idict_.rcdn_-1)
        #local a_[k_][i_] = idict_.data_[j_][i_];
      #end
      fild__incr(k_,)
    #end
  #end
  a_
#end

/* -----[reading a file]----------------------------------------------------- *
 * slurp file into temp array, #elems must be multiple of record size,
 * break into set of records, and, if applies, fill labels.
 */

#macro fild__rdmknew()
  #local a_ = array mixed [idict_.rcdn_];
  #for (i_,0,idict_.rcdn_-1)
    #switch (idict_.rcdf_[i_][0])
      #case (0)    #local a_[i_] = "";  #break
      #range (1,2) #local a_[i_] = 0;   #break
      #case (3)    #local a_[i_] = 0.0; #break
      #case (4)
        #switch (idict_.rcdf_[i_][2])
          #case (2) #local a_[i_] = <0,0>;       #break
          #case (3) #local a_[i_] = <0,0,0>;     #break
          #case (4) #local a_[i_] = <0,0,0,0>;   #break
          #case (5) #local a_[i_] = <0,0,0,0,0>; #break
        #end
        #break
      #else
        #error "oops, \"cannot happen\" error in 'rdmknew'."
    #end
  #end
  a_
#end

#macro fild__rdNames(d_,a_)
  #if (d_.hasnames_)
    #for (i_, 0, d_.rcdn_-1)
      #local d_.names_[i_] = a_[i_];
    #end
  #end
#end

#macro fild__rdRecords(d_,a_,n_)
  #if (d_.hasnames_)
    #local j_ = d_.rcdn_;
  #else
    #local j_ = 0;
  #end
  #while (j_ < n_)
    #local tmp_ = fild__rdmknew();
    #for (i_,0,d_.rcdn_-1)
      #switch (d_.rcdf_[i_][0])
        #case (1)
          #local tv_ = a_[(j_ + i_)];
          #if (!fild__isBool(tv_))
            #error "oops, bad \"B\" field in file data."
          #end
          #local tmp_[i_] = tv_;
          #break
        #case (2)
          #local tv_ = a_[(j_ + i_)];
          #if (!fild__isInt(tv_))
            #error "oops, bad \"I\" field in file data."
          #end
          #local tmp_[i_] = tv_;
          #break
        #case (0)
        #case (3)
        #case (4)
          #local tmp_[i_] = a_[(j_ + i_)];
          #break
        #else
          #error "oops, \"cannot happen\" error in 'rdRecords'."
      #end
    #end
    #local d_.data_[dimension_size(d_.data_,1)] = tmp_;
    #undef tmp_
    fild__incr(j_,d_.rcdn_)
  #end
#end

#macro fild__rdFile(a_)
  #fopen fild__fp_ idict_.fname_ read
  #while (defined(fild__fp_))
    #read (fild__fp_,fild__tmp_)
    #local a_[dimension_size(a_,1)] = fild__tmp_;
    #undef fild__tmp_
  #end
  #fclose fild__fp_
#end

#macro fild__reading(d_)
  #local ta_ = array mixed;
  fild__rdFile(ta_)
  #local n_ = dimension_size(ta_,1);
  #if (mod(n_,d_.rcdn_))
    #error "oops, #items read is not exact multiple of #fields."
  #end
  fild__rdNames(d_,ta_)
  fild__rdRecords(d_,ta_,n_)
#end

/* -----[write/append a file]------------------------------------------------ *
 * POV-Ray doc says nothing on what to expect on IO errors like 'disk full',
 * assume fp will be closed/undefined.
 */

#macro fild__wrout(v_,nl_)
  #local s_ = v_;
  #if (nl_) #local s_ = concat(s_,"\n"); #end
  #if (!defined(fild__fp_))
    #error concat("oops, IO error, file '",idict_.fname_,"' found closed.")
  #end
  #write (fild__fp_,s_)
#end

#macro fild__wrNames()
  #for (i_,0, idict_.rcdn_-1)
    #local s_ = concat("\"",idict_.names_[i_],"\",");
    fild__wrout(s_,false)
  #end
  fild__wrout("",true)
#end

#macro fild__wrRecord(j_,last_)
  #for (i_,0,idict_.rcdn_-1)
    #switch (idict_.rcdf_[i_][0])
      #case (0)
        #local s_ = concat("\"",idict_.data_[j_][i_],"\"");
        #break
      #case (1)
      #case (2)
        #local s_ = str(idict_.data_[j_][i_],0,0);
        #break
      #case (3)
        #local s_ = str(idict_.data_[j_][i_],0,idict_.rcdf_[i_][1]);
        #break
      #case (4)
        #local s_ = fild__vectFmt(idict_.rcdf_[i_][2],
                idict_.data_[j_][i_], idict_.rcdf_[i_][1]);
        #break
      #else
        #error "oops, \"cannot happen\" error in 'wrRecord'."
    #end
    #local eol_ = ((1 + i_) = idict_.rcdn_);
    #if (!(eol_ & last_))
      #local s_ = concat(s_,",");
    #end
    fild__wrout(s_,eol_)
  #end
#end

#macro fild__writing()
  #if (2 = idict_.mode_)
    #fopen fild__fp_ idict_.fname_ append
    fild__wrout(",",false)
  #else
    #fopen fild__fp_ idict_.fname_ write
    #if (idict_.hasnames_)
      fild__wrNames()
    #end
  #end
  #local n_ = dimension_size(idict_.data_,1);
  #for (i_,0,n_-1)
    fild__wrRecord(i_,((1 + i_) = n_))
  #end
  #fclose fild__fp_
#end

/* -----[build dictionary from input]---------------------------------------- *
 * verify existing keys, supply defaults where can.
 * for fields check if identical (bar precision).
 * for range check "shape" only, values once file read.
 */

#macro fild__chkAccess(d_)
  #if (!defined(dict_.Access) | !strlen(dict_.Access))
    #error "oops, bad/missing 'Access' key."
  #end
  #local a_ = substr(dict_.Access,1,1);
  #for (i_,0,2)
    #if (substr(fild__data_modes_[i_],1,1) = a_)
      #local d_.mode_ = i_;
    #end
  #end
  #if (!defined(d_.mode_))
    #error "oops, bad 'Access' value."
  #end
#end

#macro fild__chkData(d_)
  #local f_ = defined(dict_.Data);
  #if (!d_.mode_)
    #if (f_)
      #error "oops, array 'Data' already exists."
    #end
    #local d_.data_ = array;
  #elseif (!f_)
    #error "oops, missing 'Data' array/key."
  #elseif (2 = dimensions(dict_.Data))
    #if (d_.rcdn_ != dimension_size(dict_.Data,2))
      #error "oops, bad 'Data' array."
    #end
    #local d_.data_ = dict_.Data;
  #elseif (1 != dimensions(dict_.Data))
    #error "oops, bad #dimensions in 'Data' array."
  #elseif (1 = d_.rcdn_)
    #local d_.data_ = array [dimension_size(dict_.Data,1)][1];
    #for (i_,0,dimension_size(dict_.Data,1)-1)
      #local d_.data_[i_][0] = dict_.Data[i_];
    #end
  #else
    #if (d_.rcdn_ != dimension_size(dict_.Data[0],1))
      #error "oops, bad 'Data' array."
    #end
    #local d_.data_ = dict_.Data;
  #end
#end

#macro fild__chkFields(d_)
  #if (!defined(dict_.Fields))
    #error "oops, missing 'Fields' key."
  #end
  #local n_ = dimension_size(dict_.Fields,1);
  #if (1 != dimensions(dict_.Fields))
    #error "oops, expected 1D 'Fields' array."
  #elseif (1 > n_ | fild__data_maxf_ < n_)
    #error concat("oops, bad #elems in 'Fields' array (1,..,",
            str(fild__data_maxf_,0,0),").")
  #end
  #local d_.rcdn_ = n_;
  #local d_.rcdf_ = array [n_];
  #for (i_,0,n_-1)
    #local j_ = strlen(dict_.Fields[i_]);
    #if (1 = j_)
      #local d_.rcdf_[i_] = fild__mkSBIG(dict_.Fields[i_]);
    #elseif (1 < j_ | 5 > j_)
      #local d_.rcdf_[i_] = fild__mkFV(dict_.Fields[i_]);
    #else
      #error concat("oops, bad field spec '",dict_.Fields[i_],"'.")
    #end
  #end
  #local d_.homogene_ = true;
  #local cmp_ = d_.rcdf_[0];
  #for (i_,1,n_-1)
    #if (!(d_.rcdf_[i_][0] = cmp_[0] & d_.rcdf_[i_][2] = cmp_[2]))
      #local d_.homogene_ = false;
    #end
  #end
#end

#macro fild__chkFile(d_)
  #if (!defined(dict_.File) | !strlen(dict_.File))
    #error "oops, bad/missing 'File' key."
  #end
  #local a_ = concat(fild_workingDir,dict_.File);
  #if (!file_exists(a_))
    #if ((2 = d_.mode_ & d_.strict_) | !d_.mode_)
      #error concat("oops, file '",a_,"' not found.")
    #end
    #if (2 = d_.mode_)
      fild__warning(concat("note - file '",a_,"' will be created."))
      #local d_.mode_ = 1;
    #end
  #elseif (1 = d_.mode_)
    #if (d_.strict_)
      #error concat("oops, file '",a_,"' already exists.")
    #else
      fild__warning(concat("note - overwriting file '",a_,"'."))
    #end
  #end
  #local d_.fname_ = a_;
#end

#macro fild__chkNames(d_)
  #local f_ = defined(dict_.Names);
  #if (!d_.hasnames_)
    #if (f_)
      fild__warning("note - key 'Names' not used.")
    #end
    #break
  #end
  #if (!f_)
    #if (1 = d_.mode_)
      #error "oops, missing key 'Names'."
    #end
    #local d_.names_ = array [d_.rcdn_];
  #elseif (1 != dimensions(dict_.Names)
          & dimension_size(dict_.Names,1) != d_.rcdn_)
    #error "oops, bad key 'Names' array."
  #else
    #if (2 = d_.mode_)
      fild__warning("note - key 'Names' ignored in append.")
    #elseif (!d_.mode_)
      #error "oops, key 'Names' already exists."
    #end
    #local d_.names_ = dict_.Names;
  #end
#end

#macro fild__chkRange(d_)
  #if (!defined(dict_.Range))
    #local d_.range_ = array [2] {0,0};
  #elseif (!(1 = dimensions(dict_.Range) & 2 = dimension_size(dict_.Range,1)))
    #error "oops, bad 'Range' array."
  #elseif (!(fild__isInt(dict_.Range[0]) & fild__isInt(dict_.Range[1])))
    #error "oops, non-integers in 'Range' array."
  #elseif (1 > dict_.Range[1] | dict_.Range[1] < dict_.Range[0])
    #error "oops, bad range value(s)."
  #else
    #local d_.range_ = dict_.Range;
  #end
#end

/* checks depend on order */
#macro fild__dictBuild(d_)
  #if (!defined(dict_.Strict))
    #local d_.strict_ = true;
  #else
    #local d_.strict_ = dict_.Strict;
  #end
  #if (!defined(dict_.hasNames))
    #local d_.hasnames_ = false;
  #else
    #local d_.hasnames_ = dict_.hasNames;
  #end
  #if (!defined(dict_.Verbose))
    #local d_.verbose_ = false;
  #else
    #local d_.verbose_ = dict_.Verbose;
  #end
  fild__chkAccess(d_)
  fild__chkFile(d_)
  fild__chkFields(d_)
  fild__chkData(d_)
  fild__chkNames(d_)
  fild__chkRange(d_)
#end

/* -------------------------------------------------------------------------- *
 * "the public interface" macro.  returns nothing.
 * build up internal dictionary from inputs, then write or append (same code),
 * or read and copy filtered set of records, and labels.
 * give feedback if requested, and done.
 */
#macro Filed(dict_)
  #local idict_ = dictionary;
  fild__dictBuild(idict_)
  #if (idict_.mode_)
    fild__writing()
  #else
    fild__reading(idict_)
    #local dict_.Data = fild__rdFilter();
    #if (idict_.hasnames_)
      #local dict_.Names = idict_.names_;
    #end
  #end
  #if (idict_.verbose_)
    fild__emitHeader()
  #end
#end

#version fild__include_temp;

#end

/* -------------------------------------------------------------------- *
  * the content above is covered by the GNU General Public License v3+ * 
  * copyright (c) 2021 jr <creature.eternal@gmail.com>.                * 
  * all rights reserved.                                               * 
 * -------------------------------------------------------------------- */
