/* fore_demo.pov
 * the code is organised in blocks, one of which can be chosen either via
 * a 'declare=N=n' on the command-line (1 <= n <= 5), or the corresponding
 * section in 'fore_demo.ini'.  each block demonstrates some feature(s) of
 * 'Foreach()' usage, and has (hopefully) relevant comments.
 * four of the five code blocks (only) output information during parsing
 * (and use a 'box {0,1}' to suppress the "no objects in scene" warning),
 * the final renders a simple scene (the default).
 */

#version 3.8;

global_settings {assumed_gamma 1}

#include "foreach.inc"

/* default is render image */
#ifndef (N)
#declare N = 5;
#end

/* ------------------------------------------------------------------------- */

#switch (N)
/* 1) povray 'fore_demo[bool_vs_void]'
 * by default 'Foreach()' expects your macro to return nothing ('void').
 * the 'Boolean' flag indicates that the macro does return a true/false
 * (ie non-zero/zero status) value.  unless the 'Strict' flag is given,
 * a false return is simply reported[*] and execution continues, with
 * 'Strict' the 'Foreach()' macro terminates immediately.
 * a 'void' type macro can always "bail out" with '#break', but there is
 * no message.
 * [*] requires 'Verbose' on.
 */

  #case (1)
    box {0,1}

    #declare A = array [2][2][3] {
      {{1, 2, 3}, {4, 5, 6}}, {{7, 8, 9}, {10, 11, 12}}
    };

    #macro m_void(i_, elem_)
      #local n_ = elem_ * elem_;
//    #if (n_ > 30)
//      #break
//    #end
      #debug concat("m_void i = ",str(i_,0,0),", elem = ",str(elem_,0,0),
              "  sqr = ",str(n_,0,0),".\n")
    #end

    #macro m_bool(i_, elem_)
      #local n_ = elem_ * elem_;
      #debug concat("m_bool i = ",str(i_,0,0),", elem = ",str(elem_,0,0),
              "  sqr = ",str(n_,0,0),".\n")
      #if (n_ > 30)
        #local r_ = 0;
      #else
        #local r_ = 1;
      #end
      r_
    #end

//  #declare fore_debug = on;

    #declare D = dictionary {
      .Macro:   "m_void",
      .Walk:    2,
      .To:      3,
      .Verbose: on
    };

    Foreach(A, D)

    #declare D.Macro = "m_bool";
    #declare D.Boolean = on;

    Foreach(A, D)

    /* silent, break on false */
    #declare D.Verbose = off;
    #declare D.Strict = on;

    Foreach(A, D)

    /* calling a void-type macro with '.Boolean', or vice versa,
     * causes POV-Ray error(s).
     */
//  Foreach(A, dictionary {.Macro: "m_void", .Boolean: true})
//  Foreach(A, dictionary {.Macro: "m_bool"})

    #break

/* 2) povray 'fore_demo[walk_modes]'
 * the first/leftmost dimension ('.From') of any given array is always
 * "walked" for each element.  different '.Walk' settings can access
 * additional elements.  the default '0' setting results in just the '.From'
 * dimension being used.  when walk is '1', elements of the 'To' dimension,
 * too, are visited.  when walk is '2', all elements through all dims, from
 * '.From' up to and including the '.To' dimension are visited.
 * eg for array 'X[2][2][2][2]', from 2nd to 4th:
 *
 *   Walk = 0       Walk = 1       Walk = 2
 *  ------------   ------------   ------------
 *  [0][0][0][0]   [0][0][0][0]   [0][0][0][0]
 *  [0][1][0][0]   [0][0][0][1]   [0][0][0][1]
 *                 [0][1][0][0]   [0][0][1][0]
 *                 [0][1][0][1]   [0][0][1][1]
 *                                [0][1][0][0]
 *                                [0][1][0][1]
 *                                [0][1][1][0]
 *                                [0][1][1][1]
 */

  #case (2)
    box {0,1}

    #declare A = array [2][2][3] {
      {{1, 2, 3}, {4, 5, 6}}, {{7, 8, 9}, {10, 11, 12}}
    };

    #macro m_walk(i_, elem_)
      #local n_ = elem_ * elem_;
      #debug concat("m_walk i = ",str(i_,0,0),", elem = ",str(elem_,0,0),
              "  sqr = ",str(n_,0,0),".\n")
    #end

    /* default walk, or an explicit '.Walk: 0', (re)sets 'To' to 'From' */
    #declare D = dictionary {
      .Macro:   "m_walk",
      .To:      3,
      .Verbose: on
    };

    Foreach(A, D)

    #declare D.Walk = 1;
    #declare D.To = 3;

    Foreach(A, D)

    #declare D.Walk = 2;

    Foreach(A, D)

    #break

/* 3) povray 'fore_demo[extra_indices]'
 * the 'Extra' flag signals a changed "signature" of the payload macro.
 * when used in conjunction with 'Indices' _and_ a non-zero walk mode,
 * parameters for all current indices into array are included.
 * eg for array 'X[2][2][2][2]', from 1st to 3rd:
 *
 *   walk=[012], extra=no, indices=no   -->  (first, elem)
 *   walk=[12], extra=yes, indices=no   -->  (first, third, elem)
 *   walk=[12], extra=yes, indices=yes  -->  (first, second, third, elem)
 */

  #case (3)
    box {0,1}

    #declare A = array [2][2][3] {
      {{1, 2, 3}, {4, 5, 6}}, {{7, 8, 9}, {10, 11, 12}}
    };

    #macro m_2arg(i_, elem_)
      #local n_ = elem_ * elem_;
      #debug concat("m_3arg i = ",str(i_,0,0),", elem = ",str(elem_,0,0),
              "  sqr = ",str(n_,0,0),".\n")
    #end

    #macro m_3arg(i_, j_, elem_)
      #local n_ = elem_ * elem_;
      #debug concat("m_3arg i = ",str(i_,0,0),", j = ",str(j_,0,0),
              ", elem = ",str(elem_,0,0),"  sqr = ",str(n_,0,0),".\n")
    #end

    #macro m_4arg(i_, j_, k_, elem_)
      #local n_ = elem_ * elem_;
      #debug concat("m_4arg i = ",str(i_,0,0),", j = ",str(j_,0,0),
              ", k = ",str(k_,0,0),", elem = ",str(elem_,0,0),
              "  sqr = ",str(n_,0,0),".\n")
    #end

//  #declare fore_debug = on;

    #declare D = dictionary {
      .Macro:   "m_2arg",
      .Walk:    2,
      .To:      3,
      .Verbose: on
    };

    Foreach(A, D)

    #declare D.Macro = "m_3arg";
    #declare D.Extra = on;

    Foreach(A, D)

    #declare D.Macro = "m_4arg";
    #declare D.Indices = on;

    Foreach(A, D)

    #break

/* 4) povray 'fore_demo[extra_arg]'
 * adding the optional 'Arg' dictionary allows the payload macro to
 * function somewhat akin to a traditional "callback".
 */

  #case (4)
    box {0,1}

    #declare A = array [2][2][3] {
      {{1, 2, 3}, {4, 5, 6}}, {{7, 8, 9}, {10, 11, 12}}
    };

    #macro m_dict(i_, j_, elem_, arg_)
      #local n_ = elem_ * elem_;
      #if (elem_ = arg_.at_)
        #local arg_.sqr_ = n_;
      #end
      #debug concat("m_3arg i = ",str(i_,0,0),", j = ",str(j_,0,0),
              ", elem = ",str(elem_,0,0),"  sqr = ",str(n_,0,0),".\n")
    #end

//  #declare fore_debug = on;

    #declare D = dictionary {
      .Macro:   "m_dict",
      .Walk:    2,
      .To:      3,
      .Extra:   on,
      .Arg:     "R",
      .Verbose: on
    };

    #declare R = dictionary {
      .at_: 10
    }; 

    Foreach(A, D)

    #debug concat(str(R.at_,0,0)," squared is ",str(R.sqr_,0,0),".\n")

    #break

/* 5) povray fore_demo
 * a small scene using 'Foreach' to place some spheres.
 */

  #case (5)

    camera {
      location <3.5,0,-5>
      direction z
      right x * (4/3)
      up y
      angle 30
      look_at <1,.5,0>
    }

    background {color rgb .65}

    #declare vectors_ = array [8] {
      <0, 0, 0>, <1, 0, 0>, <0, 1, 0>, <1, 1, 0>,
      <0, 0, 1>, <1, 0, 1>, <0, 1, 1>, <1, 1, 1>
    };

    #declare stuff_ = array {
      array mixed {0, 0, "one",   7},
      array mixed {2, 1, "two",   6},
      array mixed {4, 2, "three", 5},
      array mixed {6, 3, "four",  3},
      array mixed {1, 4, "five",  2},
      array mixed {3, 5, "six",   4},
      array mixed {5, 6, "seven", 1},
      array mixed {7, 7, "eight", 0}
    };

    #macro mkSphere(p_,d_,c_)
      sphere {
        p_, d_
        texture {
          pigment {color rgb vectors_[c_]}
          finish {ambient 0 emission 1}
        }
      }
    #end

    #macro littleSphere(i_,elem_)
      mkSphere(elem_[0], .05, elem_[1])
    #end

    #macro littleSpheres(i_, elem_)
      #local tp_ = vectors_[elem_[1]];
      #debug concat("littleSpheres '",elem_[2],"' around <",
              vstr(3,tp_,",",0,0),">.\n")
      #local ta_ = array;
      #local ta_[0] = array mixed {<tp_.x - .35, tp_.y, tp_.z>, elem_[3]};
      #local ta_[1] = array mixed {<tp_.x + .35, tp_.y, tp_.z>, elem_[3]};
      #local ta_[2] = array mixed {<tp_.x, tp_.y - .35, tp_.z>, elem_[3]};
      #local ta_[3] = array mixed {<tp_.x, tp_.y + .35, tp_.z>, elem_[3]};
      #local ta_[4] = array mixed {<tp_.x, tp_.y, tp_.z - .35>, elem_[3]};
      #local ta_[5] = array mixed {<tp_.x, tp_.y, tp_.z + .35>, elem_[3]};
      #for (i_,0,11)
      union {
        Foreach(ta_, dictionary {.Macro: "littleSphere"})
        rotate (15 * i_)
      }
      #end
    #end

    #macro bigSphere(i_, elem_)
      #debug concat("bigSphere number '",elem_[2],"'.\n")
      mkSphere(vectors_[elem_[1]], .25, elem_[0])
    #end

//  #declare fore_debug = on;

    Foreach(stuff_, dictionary {.Macro: "bigSphere"})

    Foreach(stuff_, dictionary {.Macro: "littleSpheres"})

    #break

  #else

    #error "oops, arg 'N' must be in range 1,..,5."

#end

