/* foreach.inc
 *
 * iterate over an array, call the given macro for every element of
 * a given dimension, with options to select dimension(s), early
 * termination, etc.  for documentation see 'foreach.html'.
 *
 * Persistence of Vision Raytracer version 3.8 or later.
 *
 * macro Foreach(array, dict)
 *
 *   array  - name of array.  either old-style declared, ie sized, or the
 *            new since version 3.8 auto-sizing type.
 *
 *   dict   - a 'dictionary {}'.  all keys optional, except '.Macro'.
 *
 *     .Walk    - default '0'.  governs how the array will be "walked".
 *                0 - the payload macro is called for every element in
 *                    the '.From' dimension, '.To' is ignored.
 *                1 - this walk visits all elements in '.To' for every
 *                    element in '.From'.
 *                2 - this walk visits every element, from the '.From'
 *                    dimension through to the '.To' dimension.
 *
 *     .From    - default '1'.  the first/leftmost dimension of array to
 *                be "walked".
 *
 *     .To      - default as '.From'.  the last/rightmost dimension of
 *                array to be "walked".
 *
 *     .Macro   - the name (string) of your "payload" macro, called either
 *                as '(from_idx,element)' or '(from_idx,to_idx,element)',
 *                depending on the '.Extra' flag.
 *
 *     .Boolean - default 'off'.  when enabled indicates that the payload
 *                macro returns a boolean value.
 *
 *     .Strict  - default 'off'.  used when the payload macro provides
 *                a boolean return value.  when the macro returns 'false'
 *                and this flag is enabled, execution of 'Foreach' halts
 *                immediately.
 *
 *     .Extra   - default 'off'.  changes the payload macro's arguments
 *                from two to three when enabled, see '.Macro'.
 *
 *     .Verbose - default 'off'.  enable for (basic) feedback.
 *
 * version: 202104.2
 */

#ifndef (fore_include_temp)

#declare fore_include_temp = version;
#version 3.8;

#ifdef (View_POV_Include_Stack)
#debug "including 'foreach.inc'\n"
#end

#ifndef (global.fore_debug)
#declare fore_debug = off;
#end

/* pretty printing */
#macro fore_hyphens(n_)
  #local s_ = "";
  #for (i_,1,n_)
    #local s_ = concat(s_,"-");
  #end
  s_
#end
#macro fore_boolFmt(f_)
  #if (f_)
    #local s_ = "yes";
  #else
    #local s_ = "no";
  #end
  s_
#end
#macro fore_strFmt(s_)
  #local w_ = 20;
  #local n_ = strlen(s_);
  #if (w_ > n_)
    #local s_ = concat(substr("                    ",1,(w_-n_)),s_);
  #end
  s_
#end
#macro fore_emitBool(s_,v_)
  #debug concat(fore_strFmt(s_),": ",fore_boolFmt(v_),"\n")
#end
#macro fore_emitStr(s_,v_)
  #debug concat(fore_strFmt(s_),": ",v_,"\n")
#end
#macro fore_emitVal(s_,v_,p_)
  #debug concat(fore_strFmt(s_),": ",str(v_,0,p_),"\n")
#end

/* local version of 'strings.inc:Parse_String' */
#macro fore_exec(s_)
  #local fn_ = "parse_fore.tmp";
  #fopen fp_ fn_ write
  #write (fp_, s_)
  #fclose fp_
  #include fn_
#end
#macro fore_mkExec(d_,aa_)
  #local s_ = concat(d_.Macro,"(",str(d_.ix_[2][d_.ll_],0,0));
  #if (d_.Extra)
    #local s_ = concat(s_,",",str(d_.ix_[2][d_.lr_],0,0));
  #end
  #local s_ = concat(s_,",a_",aa_,")");
  s_
#end

/* make array access strings, eg "[3][3][1][0]" */
#macro fore_aaStr(d_)
  #local s_ = "";
  #for (i_, 1, d_.ndims_)
    #local s_ = concat(s_,"[",str(d_.ix_[2][i_],0,0),"]");
  #end
  s_
#end
#macro fore_aaNext(d_)
  #local carry_ = false;
  #for (i_, d_.lr_, d_.ll_, -1)
    #if (d_.ix_[0][i_])
      #if (i_ = d_.lr_)
        #local d_.ix_[2][i_] = d_.ix_[2][i_] + 1;
        #if (!mod(d_.ix_[2][i_], d_.ix_[1][i_]))
          #local d_.ix_[2][i_] = 0;
          #local carry_ = true;
        #end
      #elseif (carry_)
        #local carry_ = false;
        #local d_.ix_[2][i_] = d_.ix_[2][i_] + 1;
        #if (!mod(d_.ix_[2][i_], d_.ix_[1][i_]))
          #local d_.ix_[2][i_] = 0;
          #local carry_ = true;
        #end
      #end
    #end
  #end
#end

#macro fore_dictCheck(d_)
  #ifdef (d_.ttl_)
    #error "oops, found stale data in dictionary."
  #end
  #ifndef (d_.Macro)
    #error "oops, missing \"payload\" macro."
  #end
  #ifndef (d_.Walk)
    #local d_.Walk = 0;
  #elseif (0 > d_.Walk | 2 < d_.Walk | d_.Walk != int(d_.Walk))
    #error "oops, bad 'Walk' value."
  #end
  #ifndef (d_.From)
    #local d_.From = 1;
  #elseif (1 > d_.From | d_.ndims_ < d_.From | d_.From != int(d_.From))
    #error "oops, bad 'From' value."
  #end
  #ifndef (d_.To)
    #local d_.To = d_.From;
  #elseif (1 = d_.ndims_ & 1 < d_.To)
    #error "oops, multi-dimensional unsized arrays not yet implemented."
  #elseif (1 > d_.To | d_.From > d_.To | d_.ndims_ < d_.To)
    #error "oops, bad 'To' value."
  #elseif (!d_.Walk)
    #local d_.To = d_.From;
  #end
  #ifndef (d_.Boolean)
    #local d_.Boolean = off;
  #end
  #ifndef (d_.Strict)
    #local d_.Strict = off;
  #end
  #ifndef (d_.Extra)
    #local d_.Extra = off;
  #elseif (d_.Extra & !d_.Walk)
    #error "oops, 'Extra' requires a 'Walk' mode."
  #end
  #ifndef (d_.Verbose)
    #local d_.Verbose = off;
  #end
#end
#macro fore_dictClear(d_)
  #undef d_.ix_ 
  #undef d_.ll_ 
  #undef d_.lr_ 
  #undef d_.ndims_ 
  #undef d_.ttl_ 
#end
/* index array: [0] active, [1] size, [2] current.
 * dummy 1st element to avoid subtraction(s).
 */
#macro fore_dictSet(d_)
  #local d_.ll_ = d_.From;
  #local d_.lr_ = d_.To;
  #local d_.ix_ = array {
          array {-1, 0, 0, 0, 0, 0},
          array {-1, 0, 0, 0, 0, 0},
          array {-1, 0, 0, 0, 0, 0}};
  #for (i_, 1, d_.ndims_)
    #local d_.ix_[1][i_] = dimension_size(arr_, i_);
  #end
  #if (2 = d_.Walk)
    #for (i_, d_.ll_, d_.lr_)
      #local d_.ix_[0][i_] = 1;
    #end
  #elseif (1 = d_.Walk)
    #local d_.ix_[0][d_.ll_] = 1;
    #local d_.ix_[0][d_.lr_] = 1;
  #else
    #local d_.ix_[0][d_.ll_] = 1;
  #end
  /* product selected dims */
  #local d_.ttl_ = d_.ix_[1][d_.ll_];
  #if (d_.Walk)
    #for (i_, d_.ll_ + 1, d_.lr_)
      #if (d_.ix_[0][i_])
        #local d_.ttl_ = d_.ttl_ * d_.ix_[1][i_];
      #end
    #end
  #end
#end

#macro fore_boolRun(a_,d_)
  #local cont_ = true;
  #for (i_, 0, dict_.ttl_ - 1)
    #if (cont_)
      #if (i_)
        fore_aaNext(dict_)
      #end
      #local cmd_ = fore_mkExec(dict_,fore_aaStr(dict_));
      #if (!fore_exec(cmd_))
        #if (d_.Strict)
          #debug concat("break after call '",cmd_,"'.\n")
          #local cont_ = false;
        #end
        #if (d_.Verbose)
          #debug concat("false return from call '",cmd_,"'.\n")
        #end
      #end
      #if (fore_debug)
        #debug concat("called '",cmd_,"'.\n")
      #end
    #end
  #end
#end
#macro fore_voidRun(a_,d_)
  #for (i_, 0, dict_.ttl_ - 1)
    #if (i_)
      fore_aaNext(dict_)
    #end
    #local cmd_ = fore_mkExec(dict_,fore_aaStr(dict_));
    fore_exec(cmd_)
    #if (fore_debug)
      #debug concat("called '",cmd_,"'.\n")
    #end
  #end
#end

/* -------------------------------------------------------------------------- *
 * "the public interface" macro.  returns nothing.
 */

#macro Foreach(arr_,dict_)
  #ifdef (fore_running)
    #error "oops, cannot nest 'Foreach()' calls."
  #end
  #local fore_running = 1;
  /* error if array is not, always '1' when new-style */
  #local dict_.ndims_ = dimensions(arr_);
  /* check inputs, set defaults */
  fore_dictCheck(dict_)
  /* compute internally used stuff */
  fore_dictSet(dict_)
  /* summary */
  #if (dict_.Verbose | fore_debug)
    #debug concat(fore_hyphens(5),"[Foreach]",fore_hyphens(58),"\n")
    fore_emitBool("walk n'th dim",(1 = dict_.Walk))
    fore_emitBool("walk thru to n'th",(2 = dict_.Walk))
    fore_emitVal("first dim",dict_.From,0)
    fore_emitVal("n'th dim",dict_.To,0)
    fore_emitStr("payload macro",dict_.Macro)
    fore_emitBool("extra arg",dict_.Extra)
    fore_emitBool("returns bool",dict_.Boolean)
    fore_emitBool("break on 'false'",dict_.Strict)
  #end
  #if (dict_.Boolean)
    fore_boolRun(arr_,dict_)
  #else
    fore_voidRun(arr_,dict_)
  #end
  fore_dictClear(dict_)
  #if (dict_.Verbose)
    #debug concat(fore_hyphens(72),"\n")
  #end
#end

#version fore_include_temp;

#end

/* --------------------------------------------------------------- *
  * the content above is covered by the GPLv3+, see file COPYING. * 
  * copyright (c) 2021 jr <creature.eternal@gmail.com>.           * 
  * all rights reserved.                                          * 
 * --------------------------------------------------------------- */

