/* foreach.inc
 *
 * iterate over an array, call the given macro for every primary element,
 * ie 'A[n]', options for additional dimension(s), early termination, etc.
 *
 * Persistence of Vision Raytracer version 3.8 or later.
 *
 * for documentation see 'foreach.html'.
 *
 *
 *     macro Foreach(array, dict)
 *
 * arguments:
 *
 *    array  - name of array.  either "old" style declared, ie sized,
 *             or the new since version 3.8 auto-sizing type.
 *
 *    dict   - a 'dictionary {}'.  all keys optional except '.Macro'.
 *
 *      .Walk    - default '0'.  governs how the array will be "walked".
 *                 the payload macro is always called for every element in
 *                 the primary dimension (ie 'A[n]').  a type '1' walk
 *                 visits all elements in '.Nth' for every primary element.
 *                 a type '2' walk visits every element, from the primary
 *                 thru to the '.Nth' dimension.
 *                 if given, a value outside range '0..2' is an error.
 *
 *      .Nth     - default '1'.  the rightmost dimension of array to be
 *                 "walked".
 *
 *      .Macro   - name (string) of a "payload" macro, called either as
 *                 'foo(index1, element)' or 'foo(index1, index2, element)',
 *                 depending on '.Extra' flag.
 *
 *      .Boolean - default 'off'.  when enabled indicates that the payload
 *                 macro returns a boolean value.
 *
 *      .Strict  - default 'off'.  used when the payload macro provides
 *                 a boolean return value.  when the macro returns 'false'
 *                 and the is flag is enabled, execution of 'Foreach' halts
 *                 immediately.
 *
 *      .Extra   - default 'off'.  when enabled changes the payload
 *                 macro's arguments from two to three.
 *
 *      .Verbose - default 'off'.  enable for (basic) feedback.
 *
 *
 * version: 202104.1
 *
 */

#ifndef (fore_include_temp)

#declare fore_include_temp = version;
#version 3.8;

#ifdef (View_POV_Include_Stack)
#debug "including 'foreach.inc'\n"
#end

#ifndef (global.fore_debug)
#declare fore_debug = off;
#end

/* pretty printing */
#macro fore_hyphens(n_)
  #local s_ = "";
  #for (i_,1,n_)
    #local s_ = concat(s_,"-");
  #end
  s_
#end
#macro fore_boolFmt(f_)
  #if (f_)
    #local s_ = "yes";
  #else
    #local s_ = "no";
  #end
  s_
#end
#macro fore_strFmt(s_)
  #local w_ = 20;
  #local n_ = strlen(s_);
  #if (w_ > n_)
    #local s_ = concat(substr("                    ",1,(w_-n_)),s_);
  #end
  s_
#end
#macro fore_emitBool(s_,v_)
  #debug concat(fore_strFmt(s_),": ",fore_boolFmt(v_),"\n")
#end
#macro fore_emitStr(s_,v_)
  #debug concat(fore_strFmt(s_),": ",v_,"\n")
#end
#macro fore_emitVal(s_,v_,p_)
  #debug concat(fore_strFmt(s_),": ",str(v_,0,p_),"\n")
#end

/* 'exec' is local copy of 'strings.inc:Parse_String' */
#macro fore_exec(s_)
  #local fn_ = "/tmp/parse_fore.tmp";
  #fopen fp_ fn_ write
  #write (fp_, s_)
  #fclose fp_
  #include fn_
#end

#macro fore_mkCmd(d_,aa_)
  #local s_ = concat(d_.Macro,"(",str(d_.ix_[2][1],0,0));
  #if (d_.Extra)
    #local s_ = concat(s_,",",str(d_.ix_[2][d_.Nth],0,0));
  #end
  #local s_ = concat(s_,",a_",aa_,")");
  s_
#end

/* make array access strings, eg "[3][3][1][0]" */
#macro fore_aaStr(d_)
  #local s_ = "";
  #for (i_, 1, d_.ndims_)
    #local s_ = concat(s_, "[", str(d_.ix_[2][i_],0,0), "]");
  #end
  s_
#end
/* make index for next array access */
// TODO elseif?
#macro fore_aaNext(d_)
  #local carry_ = false;
  #for (i_, d_.Nth, 1, -1)
    #if (d_.ix_[0][i_])
      #if (i_ = d_.Nth)
        #local d_.ix_[2][i_] =  d_.ix_[2][i_] +  1;
        #if (!mod(d_.ix_[2][i_], d_.ix_[1][i_]))
          #local d_.ix_[2][i_] =  0;
          #local carry_ = true;
        #end
      #else
        #if (carry_)
          #local carry_ = false;
          #local d_.ix_[2][i_] =  d_.ix_[2][i_] +  1;
          #if (!mod(d_.ix_[2][i_], d_.ix_[1][i_]))
            #local d_.ix_[2][i_] =  0;
            #local carry_ = true;
          #end
        #end
      #end
    #end
  #end
#end

/* cf docs: allow other user keys but must/should begin with uppercase */
#macro fore_checkDict(d_)
  #if (defined(d_.ix_) | defined(d_.ttl_))
    #error "oops, found stale info in dictionary."
  #end
  #ifndef (d_.Macro)
    #error "oops, missing \"payload\" macro."
  #end
  #ifndef (d_.Walk)
    #local d_.Walk = 0;
  #elseif (0 > d_.Walk | 2 < d_.Walk | d_.Walk != int(d_.Walk))
    #error "oops, bad 'Walk' value."
  #end
  #ifndef (d_.Nth)
    #local d_.Nth = 1;
  #elseif (1 = d_.ndims_ & 1 < d_.Nth)
    #error "oops, multi-dimensional unsized arrays not yet implemented."
  #elseif (1 > d_.Nth | 5 < d_.Nth | d_.ndims_ < d_.Nth)
    #error "oops, bad 'Nth' value."
  #elseif (!d_.Walk)
    #local d_.Nth = 1;
  #end
  #ifndef (d_.Boolean)
    #local d_.Boolean = off;
  #end
  #ifndef (d_.Strict)
    #local d_.Strict = off;
  #end
  #ifndef (d_.Extra)
    #local d_.Extra = off;
  #elseif (d_.Extra & !d_.Walk)
    #error "oops, 'Extra' requires a 'Walk' mode."
  #end
  #ifndef (d_.Verbose)
    #local d_.Verbose = off;
  #end
#end

#macro fore_cleanDict(d_)
  #undef d_.ttl_ 
  #undef d_.ix_ 
#end

/* index array: [0] active, [1] size, [2] current.
 * dummy 1st element to get rid of subtraction(s).
 */
#macro fore_prepDict(d_)
  #local d_.ix_ = array {
          array {-1, 1, 0, 0, 0, 0},
          array {-1, 0, 0, 0, 0, 0},
          array {-1, 0, 0, 0, 0, 0}
  };
  #for (i_, 1, d_.Nth)
    #local d_.ix_[1][i_] = dimension_size(arr_, i_);
  #end
  #if (2 = d_.Walk)
    #for (i_, 1, d_.Nth)
      #local d_.ix_[0][i_] = 1;
    #end
  #elseif (1 = d_.Walk)
    #local d_.ix_[0][d_.Nth] = 1;
  #end
  /* limit for walker */
  #local d_.ttl_ = d_.ix_[1][1];
  #if (d_.Walk)
    #for (i_, 2, d_.Nth)
      #if (d_.ix_[0][i_])
        #local d_.ttl_ = d_.ttl_ * d_.ix_[1][i_];
      #end
    #end
  #end
#end

/* separate "walkers" for 'bool' and 'void' */
#macro fore_runBool(a_,d_)
  #local cont_ = 1;
  #for (i_, 0, dict_.ttl_ - 1)
    #if (cont_)
      #if (i_)
        fore_aaNext(dict_)
      #end
      #local cmd_ = fore_mkCmd(dict_,fore_aaStr(dict_));
      #if (!fore_exec(cmd_))
        #if (d_.Strict)
          #debug concat("break after call '",cmd_,"'.\n")
          #local cont_ = 0;
        #end
        #if (d_.Verbose)
          #debug concat("false return from call '",cmd_,"'.\n")
        #end
      #end
      #if (fore_debug)
        #debug concat("called '",cmd_,"'.\n")
      #end
    #end
  #end
#end

#macro fore_runVoid(a_,d_)
  #local cont_ = 1;
  #for (i_, 0, dict_.ttl_ - 1)
    #if (cont_)
      #if (i_)
        fore_aaNext(dict_)
      #end
      #local cmd_ = fore_mkCmd(dict_,fore_aaStr(dict_));
      fore_exec(cmd_)
      #if (fore_debug)
        #debug concat("called '",cmd_,"'.\n")
      #end
    #end
  #end
#end

/* -------------------------------------------------------------------------- *
 * "the public interface" macro.  returns nothing.
 */

#macro Foreach(arr_,dict_)
  #ifdef (fore_running_)
    #error "oops, cannot nest 'Foreach()' calls."
  #end
  #local fore_running_ = 1;
  /* error if array is not, always '1' when new-style */
  #local dict_.ndims_ = dimensions(arr_);
  /* check inputs, set defaults */
  fore_checkDict(dict_)
  /* compute internally used stuff */
  fore_prepDict(dict_)
  /* summary */
  #if (dict_.Verbose | fore_debug)
    #debug concat(fore_hyphens(5),"[Foreach()]",fore_hyphens(56),"\n")
    fore_emitBool("walk n'th dim",(1 = dict_.Walk))
    fore_emitBool("walk thru to n'th",(2 = dict_.Walk))
    fore_emitVal("dimension n",dict_.Nth,0)
    fore_emitStr("payload macro",dict_.Macro)
    fore_emitBool("extra arg",dict_.Extra)
    fore_emitBool("returns bool",dict_.Boolean)
    fore_emitBool("break on 'false'",dict_.Strict)
  #end
  #if (dict_.Boolean)
    fore_runBool(arr_,dict_)
  #else
    fore_runVoid(arr_,dict_)
  #end
  fore_cleanDict(dict_)
#end

#version fore_include_temp;

#end

/* --------------------------------------------------------------- *
  * the content above is covered by the GPLv3+, see file COPYING. * 
  * copyright (c) 2021 jr <creature.eternal@gmail.com>.           * 
  * all rights reserved.                                          * 
 * --------------------------------------------------------------- */
