/* queues.inc
 *
 * FIFO, LIFO, and simple priority type queues.
 *
 * documentation see 'queues.html'.
 *
 * version 201811.3
 */

#ifndef (Queues_include_temp)

#declare Queues_include_temp = version;

#version 3.8;

/* ------------------------------------------------------------------------- */

#ifdef (QQ_MAXQUEUE)
#declare qqMaxLen = QQ_MAXQUEUE;
#else
#declare qqMaxLen = 10000;
#end

#macro qqCreate(qlen_,optional strict_)
  #if ((2 > qlen_) | (qqMaxLen < qlen_))
    #error concat("oops, require 2 <= len <= ",str(qqMaxLen,0,0),".\n")
  #end
  #ifndef (local.strict_)
    #local strict_ = 1;
  #elseif (!((1 = strict_) | (0 = strict_)))
    #error concat("oops, require boolean, ie '0' or '1' flag.\n")
  #end
  dictionary {
    .maxLength: qlen_,
    .length: 0,
    .eexit: strict_,
    .arr: array mixed
  }
#end

#macro qqDiscard(q_)
  #for (i_,0,q_.length-1)
    #undef q_.arr[i_]
  #end
  #declare q_.length = 0;
#end

#macro qqGet(q_)
  #if (!q_.length)
    #if (q_.eexit)
      #error concat("oops, queue empty.")
    #end
    #local r_ = "";
  #else
    #local r_ = q_.arr[0];
    #for (i_,1,q_.length-1)
      #declare q_.arr[i_-1] = q_.arr[i_];
    #end
    #undef q_.arr[q_.length-1]
    #declare q_.length = q_.length - 1;
  #end
  r_
#end

#macro qqLength(q_) q_.length #end

#macro qqPut(q_,el_)
  #if (q_.maxLength <= q_.length)
    #if (q_.eexit)
      #error concat("oops, queue full.")
    #end
  #else
    #declare q_.arr[q_.length] = el_;
    #declare q_.length = q_.length + 1;
  #end
#end

/* ------------------------------------------------------------------------- */

#ifdef (QS_MAXSTACK)
#declare qsMaxElem = QS_MAXSTACK;
#else
#declare qsMaxElem = 10000;
#end

#macro qsCreate(ssz_,optional strict_)
  #if ((2 > ssz_) | (qsMaxElem < ssz_))
    #error concat("oops, require 2 <= len <= ",str(qsMaxElem,0,0),".\n")
  #end
  #ifndef (local.strict_)
    #local strict_ = 1;
  #elseif (!((1 = strict_) | (0 = strict_)))
    #error concat("oops, require boolean, ie '0' or '1' flag.\n")
  #end
  dictionary {
    .maxElems: ssz_,
    .elems: 0,
    .eexit: strict_,
    .arr: array
  }
#end

#macro qsDiscard(s_)
  #for (i_,0,s_.elems-1)
    #undef s_.arr[i_]
  #end
  #declare s_.elems = 0;
#end

#macro qsPop(s_)
  #if (!s_.elems)
    #if (s_.eexit)
      #error concat("oops, stack empty.")
    #end
    #local r_ = "";
  #else
    #local r_ = s_.arr[s_.elems-1];
    #undef s_.arr[s_.elems-1]
    #declare s_.elems = s_.elems - 1;
  #end
  r_
#end

#macro qsPush(s_,el_)
  #if (s_.maxElems <= s_.elems)
    #if (s_.eexit)
      #error concat("oops, stack full.")
    #end
  #else
    #declare s_.arr[s_.elems] = el_;
    #declare s_.elems = s_.elems + 1;
  #end
#end

#macro qsSize(s_) s_.elems #end

/* ------------------------------------------------------------------------- */

#ifdef (QP_MAXPRIO)
#declare qpMaxElem = QP_MAXPRIO;
#else
#declare qpMaxElem = 10000;
#end

#macro qp_goodPrio(p_)
  ((1 <= p_) & (5 >= p_))
#end

/* each priority in its own queue.  cheating, but simplifies things */
#macro qpCreate(qlen_,optional strict_)
  #if ((2 > qlen_) | (qpMaxElem < qlen_))
    #error concat("oops, require 2 <= len <= ",str(qpMaxElem,0,0),".\n")
  #end
  #ifndef (local.strict_)
    #local strict_ = 1;
  #elseif (!((1 = strict_) | (0 = strict_)))
    #error concat("oops, require boolean, ie '0' or '1' flag.\n")
  #end
  dictionary {
    .maxLength: qlen_,
    .eexit: strict_,
    .lengths: array {0,0,0,0,0},
    .arrs: array {
      array mixed, array mixed, array mixed, array mixed, array mixed
    }
  }
#end

#macro qpDiscard(p_)
  #for (j_,0,4)
    #for (i_,0,p_.lengths[j_]-1)
      #undef p_.arrs[j_][i_]
    #end
    #declare p_.lengths[j_] = 0;
  #end
#end

#macro qpDiscardP(p_, prio_)
  #if (!qp_goodPrio(prio_))
    #error concat("oops, expected priority 1,..5, got '",str(prio_,0,0),"'.")
  #end
  #local j_ = prio_ - 1;
  #for (i_,0,p_.lengths[j_]-1)
    #undef p_.arrs[j_][i_]
  #end
  #declare p_.lengths[j_] = 0;
#end

#macro qpGet(p_)
  #if (!qpLength(p_))
    #if (p_.eexit)
      #error concat("oops, queue empty.")
    #end
    #local r_ = "";
  #else
    #for (j_,4,0,-1)
      #if (p_.lengths[j_])
        #break
      #end
    #end
    #local r_ = p_.arrs[j_][0];
    #for (i_,1,p_.lengths[j_]-1)
      #declare p_.arrs[j_][i_-1] = p_.arrs[j_][i_];
    #end
    #undef p_.arrs[j_][p_.lengths[j_]-1]
    #declare p_.lengths[j_] = p_.lengths[j_] - 1;
  #end
  r_
#end

#macro qpGetP(p_, prio_)
  #if (!qp_goodPrio(prio_))
    #error concat("oops, expected priority 1,..5, got '",str(prio_,0,0),"'.")
  #elseif (!qpLengthP(p_,prio_))
    #if (p_.eexit)
      #error concat("oops, queue empty.")
    #end
    #local r_ = "";
  #else
    #local j_ = prio_ - 1;
    #local r_ = p_.arrs[j_][0];
    #for (i_,1,p_.lengths[j_]-1)
      #declare p_.arrs[j_][i_-1] = p_.arrs[j_][i_];
    #end
    #undef p_.arrs[j_][p_.lengths[j_]-1]
    #declare p_.lengths[j_] = p_.lengths[j_] - 1;
  #end
  r_
#end

#macro qpLength(p_)
  #local n_ = 0;
  #for (i_,0,4)
    #local n_ = n_ + p_.lengths[i_];
  #end
  n_
#end

#macro qpLengthP(p_,prio_)
  #if (!qp_goodPrio(prio_))
    #error concat("oops, expected priority 1,..5, got '",str(prio_,0,0),"'.")
  #end
  p_.lengths[prio_-1]
#end

#macro qpPut(p_,el_,optional prio_)
  #if (p_.maxLength <= qpLength(p_))
    #if (p_.eexit)
      #error concat("oops, queue full.")
    #end
    #break
  #elseif (!defined(local.prio_))
    #local prio_ = 1;
  #elseif (!qp_goodPrio(prio_))
    #error concat("oops, expected priority 1,..5, got '",str(prio_,0,0),"'.")
  #end
  #local i_ = prio_ - 1;
  #declare p_.arrs[i_][p_.lengths[i_]] = el_;
  #declare p_.lengths[i_] = p_.lengths[i_] + 1;
#end

#version Queues_include_temp;

#end

/* ------------------------------------------------------- *
 *  the content above is covered by the GPLv3+.            *
 *  copyright (c) 2018-19 jr <creature.eternal@gmail.com>  *
 *  all rights reserved.                                   *
 * ------------------------------------------------------- */
