/* rxor.inc
 * a simple rotor cipher machine, wrapped in an "API" of macros:
 *
 * to convert from a string to an array, and vice versa:
 *
 *   rxor_array_to_string(A)
 *     returns the contents of array 'A' converted to a string;
 *     codes outside 32,..,122 get converted to dots ('.').
 *
 *   rxor_string_to_array(S,A)
 *     takes string 'S' and converts it, one character at a time,
 *     to an integer representation.  array 'A' must have enough
 *     elements to store the whole string.
 *
 * to copy data from a file to an array, and vice versa:
 *
 *   rxor_array_to_file(A,F)
 *     writes the elements of array 'A', as a comma-delimited list
 *     of integer values, to a file named 'F'.
 *     the list is prefixed with the element count.
 *
 *   rxor_file_to_array(F,A)
 *     reads a list of comma-delimited values from file ''F' into array 'A'.
 *     the first value is the count of the subsequent values.
 *
 * to "do work":
 *
 *   rxor_cipher(A,N)
 *     applies an exclusive-or operation (Vernam cipher) to the elements
 *     of array 'A', starting generation from the 'N'th machine step.
 *
 *   rxor_rand(N)
 *     returns a value in range: 0,..,255.  the "seed value" N must have
 *     been declared as a positive integer >= 0.
 *
 *   rxor_mask(N)
 *     returns a value in range: 0,..,255, corresponds to the 'N'th step
 *     in the machine's sequence.
 *
 */

#ifndef (rxor_include_temp)

#declare rxor_include_temp = version;
#version 3.7;

/* -------------------------------------------------------------------------- *
 * the "machine" assembles, on each step, a pseudo random (masking) value,
 * from eight simple "rotors", one for every bit position in a byte.
 * there are 32 bytes (256 bits) of random data per rotor(*), and one
 * corresponding length (ie circumference), in bits; range: 2 <= N <= 255,
 * ideally all lengths differ, and will have been chosen from this list:
 *   67 71 73 79 83 89 97 101 103 107 109 113 127 131 137 139 149 151 157
 *   163 167 173 179 181 191 193 197 199 211 223 227 229 233 239 241 251
 *
 * (*) from the 'rxorincdata.tcl' script, which uses 'urandom(4)',
 */

#declare rxor_rotor_data = array[8][32] {
  {85,147,9,210,124,121,247,40,28,225,115,135,47,87,91,168,
   28,225,115,135,47,87,91,168,53,224,91,48,154,145,109,66},
  {170,155,101,178,73,65,211,205,49,10,85,126,236,249,211,109,
   49,10,85,126,236,249,211,109,107,181,113,46,192,49,106,188},
  {119,171,244,49,131,133,89,16,90,86,40,23,67,70,165,23,
   90,86,40,23,67,70,165,23,118,95,57,90,110,244,66,199},
  {85,149,36,32,21,203,62,31,23,70,8,222,206,150,129,101,
   23,70,8,222,206,150,129,101,16,186,137,182,51,11,111,90},
  {204,91,206,199,42,243,189,195,164,58,216,57,19,204,140,122,
   164,58,216,57,19,204,140,122,107,45,39,226,54,32,71,176},
  {204,49,203,170,247,165,104,77,102,203,210,160,19,71,7,19,
   102,203,210,160,19,71,7,19,176,10,75,159,107,88,23,82},
  {187,113,104,160,242,118,124,180,158,133,81,181,91,105,19,106,
   158,133,81,181,91,105,19,106,36,180,58,105,77,60,186,63},
  {221,75,86,235,80,54,199,195,76,174,82,8,241,140,198,219,
   76,174,82,8,241,140,198,219,28,241,123,40,210,105,239,47}
};

#declare rxor_rotor_length = array[8] {
  211, 103, 173, 197, 83, 191, 239, 101
};

#macro rxor_rotor_machine(na_)
  #local V = 0;
  #for (I, 0, 7)
    #local J = mod(na_, rxor_rotor_length[I]);
    #if (bitwise_and(rxor_rotor_data[I][int(J/8)], pow(2,(8-mod(J,8)-1))))
      #local V = bitwise_or(V, pow(2,I));
    #end
  #end
  (V)
#end

#macro rxor_cipher(aa_, na_)
  #if (0 > na_)
    #error concat("step count must be a positive integer.\n")
  #elseif (1 != dimensions(aa_))
    #error concat("wrong # dimensions in array.\n")
  #end
  #local N = dimension_size(aa_, 1);
  #if (255 < N)
    #error concat("array too large.\n")
  #end
  #for (I, 0, N - 1)
    #declare aa_[I] = bitwise_xor(aa_[I], rxor_rotor_machine(na_ + I));
  #end
#end

#macro rxor_rand(na_)
  #if (0 > na_)
    #error concat("random seed must be a positive integer.\n")
  #end
  #local R = rxor_rotor_machine(na_);
  #declare na_ = na_ + 1;
  (R)
#end

#macro rxor_mask(na_)
  #if (0 > na_)
    #error concat("step count must be a positive integer.\n")
  #end
  (rxor_rotor_machine(na_))
#end

#macro rxor_array_to_string(aa_)
  #if (1 != dimensions(aa_))
    #error concat("wrong # dimensions in array.\n")
  #end
  #local N = dimension_size(aa_, 1);
  #if (255 < N)
    #error concat("array too large.\n")
  #end
  #local S = "";
  #for (I, 0, N - 1)
    #if ((32 > aa_[I]) | (122 < aa_[I]))
      #local X = ".";
    #else
      #local X = chr(aa_[I]);
    #end
    #local S = concat(S, X);
  #end
  S
#end

#macro rxor_string_to_array(sa_, aa_)
  #local N = strlen(sa_);
  #if (1 != dimensions(aa_))
    #error concat("wrong # dimensions in array.\n")
  #elseif (N > dimension_size(aa_, 1))
    #error concat("array too small.\n")
  #end
  #for (I, 1, N)
    #declare aa_[I - 1] = asc(substr(sa_, I, 1));
  #end
#end

#macro rxor_array_to_file(aa_, fa_)
  #if (1 != dimensions(aa_))
    #error concat("wrong # dimensions in array.\n")
  #end
  #local N = dimension_size(aa_, 1);
  #if (255 < N)
    #error concat("array too large.\n")
  #end
  #fopen FP fa_ write
  #write (FP str(N, 0, 0))
  #for (I, 0, N - 1)
    #write (FP concat(", ", str(aa_[I], 0, 0)))
  #end
  #write (FP "\n")
  #fclose FP
#end

#macro rxor_file_to_array(fa_, aa_)
  #fopen FP fa_ read
  #read (FP, N)
  #if ((0 > N) | (255 < N))
    #error concat(" bad data item #1 (count).\n")
  #elseif (1 != dimensions(aa_))
    #error concat("wrong # dimensions in array.\n")
  #elseif (dimension_size(aa_, 1) < N)
    #error concat("array too small.\n")
  #end
  #for (I, 1, N)
    #read (FP, V)
    #if ((0 > V) | (255 < V))
      #error concat(" bad data item #", str(I + 1, 0, 0), ".\n")
    #end
    #declare aa_[I - 1] = V;
  #end
  #fclose FP
#end

#version rxor_include_temp;

#end

/* ---------------------------------------------------- *
 *  the content above is covered by the GPLv3+.         *
 *  copyright (c) 2018 jr <creature.eternal@gmail.com>  *
 *  all rights reserved.                                *
 * ---------------------------------------------------- */
