#declare EdgeCutout = object {
   difference {
      box {<-10, -.1, -.1> <10, EdgeRadius, EdgeRadius>}
      cylinder {<-10.1, EdgeRadius, EdgeRadius> <10.1, EdgeRadius, EdgeRadius>, EdgeRadius}
   } //difference
} //object

#macro ComputeIntersectionPoint (Point1, Angle1, Point2, Angle2)

   #local Point1_Macro = Point1; //Local copies must be used because parameters
   #local Angle1_Macro = Angle1; //are passed by reference.
   #local Point2_Macro = Point2;
   #local Angle2_Macro = Angle2;

   #if (Point2_Macro.x < Point1_Macro.x)
      #local TPoint = Point2_Macro;
      #local TAngle = Angle2_Macro;
      #local Point2_Macro = Point1_Macro;
      #local Angle2_Macro = Angle1_Macro;
      #local Point1_Macro = TPoint;
      #local Angle1_Macro = TAngle;
   #end //#if

   #local XOffset = Point1_Macro.x;
   #local YOffset = Point1_Macro.y;

   #local Point1_Macro = <0, 0, 0>;
   #local Point2_Macro = Point2_Macro - <XOffset, YOffset, 0>;

   #local Theta = degrees (atan (Point2_Macro.y / Point2_Macro.x));
   #local R2 = sqrt (Point2_Macro.x * Point2_Macro.x + Point2_Macro.y * Point2_Macro.y);

   #local Angle1_Macro = Angle1_Macro - Theta;
   #local Angle2_Macro = Angle2_Macro - Theta;

   #local Point2_Macro = <R2, 0, 0>;

   #local Angle2_Macro = 180 - Angle2_Macro;

   #local Angle3_Macro = 180 - Angle1_Macro - Angle2_Macro;

   #local R1 = R2 * sin (radians (Angle2_Macro)) / sin (radians (Angle3_Macro));

   #local Angle1_Macro = Angle1_Macro + Theta;

   #declare IntersectionPoint = <R1 * cos (radians (Angle1_Macro)), R1 * sin (radians (Angle1_Macro)), 0>;

   #declare IntersectionPoint = <IntersectionPoint.x + XOffset, IntersectionPoint.y + YOffset, 0>;

#end //#macro ComputeIntersectionPoint

#macro BeveledCylinder (Top, Bottom, Radius, EdgeRadius)

   union {
      cylinder {<Top.x, Top.y, Top.z + EdgeRadius>, Bottom, Radius}
      cylinder {Top, Bottom, Radius - EdgeRadius}
      torus {
         Radius - EdgeRadius, EdgeRadius
         rotate 90 * x
         translate <Top.x, Top.y, Top.z + EdgeRadius>
         sturm
      } //torus
   } //union

#end //#macro BeveledCylinder

#macro BeveledHollowCylinder (Top, Bottom, InnerRadius, OuterRadius, EdgeRadius)

   #if (OuterRadius - InnerRadius < EdgeRadius * 2)
      #debug "\n\n\n* * * Bad Cylinder Options * * *\n\n\n"
   #end //#if
   union {
      difference {
         BeveledCylinder (Top, Bottom, OuterRadius, EdgeRadius)
         cylinder {<Top.x, Top.y, Top.z - .1>, <Bottom.x, Bottom.y, Bottom.z + .1>, InnerRadius}
         cylinder {<Top.x, Top.y, Top.z - .1>, <Top.x, Top.y, Top.z + EdgeRadius>, InnerRadius + EdgeRadius}
      } //difference
      torus {
         InnerRadius + EdgeRadius, EdgeRadius
         rotate 90 * x
         translate <Top.x, Top.y, Top.z + EdgeRadius>
         sturm
      } //torus
   } //union

#end //#macro BeveledHollowCylinder

#macro NormalizeAngles (Angle1_Macro, Angle2_Macro, CurRotation)

   #while (Angle1_Macro < 0)
      #local Angle1_Macro = Angle1_Macro + 360;
   #end //#while

   #while (Angle2_Macro < 0)
      #local Angle2_Macro = Angle2_Macro + 360;
   #end //#while

   #if (Angle1_Macro > Angle2_Macro)
      #local Temp = Angle1_Macro;
      #local Angle1_Macro = Angle2_Macro;
      #local Angle2_Macro = Temp;
   #end //#if

   #while (Angle1_Macro >= 90)
      #local CurRotation = CurRotation + 90;
      #local Angle1_Macro = Angle1_Macro - 90;
      #local Angle2_Macro = Angle2_Macro - 90;
   #end //#while

   #if (Angle2_Macro - Angle1_Macro > 180)
      #local CurRotation = CurRotation + Angle2_Macro;
      #local AngleOffset = 360 - Angle2_Macro;
      #local Angle2_Macro = Angle1_Macro + AngleOffset;
      #local Angle1_Macro = 0;
   #else
      #local CurRotation = CurRotation + Angle1_Macro;
      #local AngleOffset = Angle1_Macro;
      #local Angle2_Macro = Angle2_Macro - Angle1_Macro;
      #local Angle1_Macro = 0;
   #end //#if

#end //#macro NormalizeAngles

#macro CornerPlane (Origin, Angle1, Angle2, CornerRadius, EdgeRadius, Depth)

   // This macro will create a plane to be used to chop off a sharp point. The
   //    object will be rotated and translated into proper position by the macro.
   //
   // Input variables:
   //    Origin: The point the sides would meet if the corner remained sharp.
   //    Angle1 and Angle2: The angles of the sides. Angles are measured from
   //       0 degrees = positive x axis. The macro will figure out which is the
   //       "Inside."
   //    CornerRadius: The radius of the corner.
   //    EdgeRadius: The radius of the rounded edges. (Not used.)
   //    Depth: Depth of the logo (from z = 0 to z = Depth). (Not used.)

   #local Angle1_Macro = Angle1; //Local copies of Angle1 and Angle2 must be used
   #local Angle2_Macro = Angle2; //because parameters are passed by reference.
   #local CurRotation = 0;

   NormalizeAngles (Angle1_Macro, Angle2_Macro, CurRotation)

   #local XCenter = CornerRadius / tan (radians (Angle2_Macro / 2));

   object {
      plane {y, 0 rotate (Angle2_Macro / 2 + 90) * z translate XCenter * x}
      rotate CurRotation * z
      translate Origin
   } //object

#end //#macro CornerPlane

#macro RoundedPoint (Origin, Angle1, Angle2, PointRadius, EdgeRadius, Depth)

   // This macro will create a semi-circle to be used as a rounded point. The
   //    object will be rotated and translated into proper position by the macro.
   //
   // Input variables:
   //    Origin: The point the sides would meet if the corner remained sharp.
   //    Angle1 and Angle2: The angles of the sides. Angles are measured from
   //       0 degrees = positive x axis. The macro will figure out which is the
   //       "Inside."
   //    PointRadius: The radius of the point.
   //    EdgeRadius: The radius of the rounded edges.
   //    Depth: Depth of the logo (from z = 0 to z = Depth).

   #local Angle1_Macro = Angle1; //Local copies of Angle1 and Angle2 must be used
   #local Angle2_Macro = Angle2; //because parameters are passed by reference.
   #local CurRotation = 0;

   NormalizeAngles (Angle1_Macro, Angle2_Macro, CurRotation)

   #local XCenter = PointRadius / tan (radians (Angle2_Macro / 2));

   object {
      intersection {
         object {BeveledCylinder (<0, 0, 0> <0, 0, Depth>, PointRadius, EdgeRadius) translate <XCenter, PointRadius, 0>}
         plane {y, 0 inverse rotate (Angle2_Macro / 2 + 90) * z translate XCenter * x}
      } //intersection
      rotate CurRotation * z
      translate Origin
   } //object

#end //#macro RoundedPoint

#macro RoundedCorner (Origin, Angle1, Angle2, CornerRadius, EdgeRadius, Depth)

   // This macro will create a rounded corner. The object will be rotated and
   //    translated into proper position by the macro.
   //
   // Input variables:
   //    Origin: The point the sides would meet if the corner remained sharp.
   //    Angle1 and Angle2: The angles of the sides. Angles are measured from
   //       0 degrees = positive x axis. The macro will figure out which is the
   //       "Inside."
   //    CornerRadius: The radius of the corner.
   //    EdgeRadius: The radius of the rounded edges.
   //    Depth: Depth of the logo (from z = 0 to z = Depth).

   #local Angle1_Macro = Angle1; //Local copies of Angle1 and Angle2 must be used
   #local Angle2_Macro = Angle2; //because parameters are passed by reference.
   #local CurRotation = 0;

   NormalizeAngles (Angle1_Macro, Angle2_Macro, CurRotation)

   #local XCenter = CornerRadius / tan (radians (Angle2_Macro / 2));

   #local Corner = object {
      difference {
         union {
            cylinder {<0, 0, -.1> <0, 0, Depth + .1>, CornerRadius}
            cylinder {<0, 0, -.1> <0, 0, EdgeRadius>, CornerRadius + EdgeRadius}
         } //union
         torus {
            CornerRadius + EdgeRadius, EdgeRadius
            rotate 90 * x
            translate EdgeRadius * z
            sturm
         } //torus
      } //difference
   } //object

   object {
      difference {
         intersection {
            box {<-1, -EdgeRadius, 0> <1, 1, Depth>}
            plane {y, 0 translate EdgeRadius * y rotate Angle2_Macro * z}
            plane {y, 0 inverse rotate (Angle2_Macro / 2 + 90) * z translate XCenter * x}
         } //intersection
         object {Corner translate <XCenter, CornerRadius, 0>}
      } //difference
      rotate CurRotation * z
      translate Origin
   } //object

#end //#macro RoundedCorner

#macro ComputeBorderPoint (Origin, Angle1, Angle2, BorderWidth, ReflexAngle)

   // This macro will compute the point that beveled edges with a distance of
   //    "BorderWidth" from the outside edge will meet. The point will be
   //    rotated and translated into proper position by the macro. If the
   //    desired angle is > 180, set ReflexAngle = true.
   //
   // Input variables:
   //    Origin: The point the outside edges meet
   //    Angle1 and Angle2: The angles of the sides. Angles are measured from
   //       0 degrees = positive x axis. The macro will figure out which is the
   //       "Inside."
   //    BorderWidth: The width of the border.

   #local Angle1_Macro = Angle1; //Local copies of Angle1 and Angle2 must be used
   #local Angle2_Macro = Angle2; //because parameters are passed by reference.
   #local CurRotation = 0;

   NormalizeAngles (Angle1_Macro, Angle2_Macro, CurRotation)

   #local Theta = Angle2_Macro / 2;

   #local R_Macro = BorderWidth / sin (radians (Theta));

   #if (ReflexAngle)
      #declare CurRotation = CurRotation + 180;
   #end //#if

   #local NewX = R_Macro * cos (radians (Theta + CurRotation));
   #local NewY = R_Macro * sin (radians (Theta + CurRotation));

   #declare BorderPoint = <Origin.x + NewX, Origin.y + NewY, 0>;

#end //#macro ComputeBorderPoint

#macro Create_Logo (BorderWidth, BorderHeight, Depth, CornerRadius, EdgeRadius, VerticalGap, DeltaTheta, MetalColor1, MetalColor2)

   //DeltaTheta determins the quality of the rounded portion of the D which
   //is created as a union of segments. The value of DeltaTheta MUST be a
   //factor of 90. Lower values create a smoother curve. Possible values are
   //1, 2, 3, 5, 6, 9, 10, 15, as well as fractional values. A value of 1 is
   //sufficient for very high resolution.

   #declare DOKKEN = object {
      union {
         Create_Union (BorderWidth, BorderHeight, Depth, CornerRadius, EdgeRadius, VerticalGap, DeltaTheta, MetalColor2)
         Create_Union (0, 0, Depth, CornerRadius, EdgeRadius, VerticalGap, DeltaTheta, MetalColor1)
      } //union
   } //object

#end //#macro Create_Logo

#macro Create_Union (BorderWidth, BorderHeight, Depth, CornerRadius, EdgeRadius, VerticalGap, DeltaTheta, MetalColor)

   #object {
      union {
         #local X1 = 7.38;
         object {Create_D (BorderWidth, Depth, DeltaTheta)}
         object {Create_O (BorderWidth, Depth) translate <4.05, 1.25, 0>}
         object {Create_Bar (BorderWidth, Depth) translate X1 * x}
         object {Create_Hooks (BorderWidth, Depth) translate X1 * x}
         object {Create_K (BorderWidth, Depth) translate SavePoint5 translate X1 * x}
         object {Create_K (BorderWidth, Depth) translate SavePoint7 translate X1 * x}
         object {Create_E (BorderWidth, Depth) translate <X1 + 3.01, 1 + VerticalGap, 0>}
         object {Create_N (BorderWidth, Depth) translate <X1 + 4.8, 1 + VerticalGap, 0>}
      } //union
      texture {MetalColor}
      translate -BorderHeight * z
   } //object

#end //#macro Create_Union

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
// *                                                                                                           *
// *                                                                                                           *
// *                                                                                                           *
// * D                                                                                                         *
// *                                                                                                           *
// *                                                                                                           *
// *                                                                                                           *
// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

#macro Create_D (BorderWidth, Depth, DeltaTheta)

   #local D_Bottom = object {
      #local Point1 = <0, 0, 0>;                               //Lower left
      #local Point2 = <2.9 - BorderWidth, BorderWidth, 0>;     //Lower right
      #local Point3 = <2.9 - BorderWidth, 1 - BorderWidth, 0>; //Upper right
      #local Point4 = <1, 1, 0>;                               //Upper left
      ComputeBorderPoint (Point1, 0, 45, BorderWidth, false)
      #local Point1 = BorderPoint;
      ComputeBorderPoint (Point4, 0, 225, BorderWidth, false)
      #local Point4 = BorderPoint;
      union {
         intersection {
            box {<0, BorderWidth, 0> <2.9 - BorderWidth, 1 - BorderWidth, Depth>}
            plane {y, 0 rotate 45 * z translate Point1}
            CornerPlane (Point1, 0, 45, CornerRadius, EdgeRadius, Depth)
            CornerPlane (Point3, 180, -90, CornerRadius, EdgeRadius, Depth)
            CornerPlane (Point4, 0, 225, CornerRadius * 3, EdgeRadius, Depth)
         } //intersection
         RoundedPoint (Point1, 0, 45, CornerRadius, EdgeRadius, Depth)
         RoundedPoint (Point3, 180, -90, CornerRadius, EdgeRadius, Depth)
         RoundedPoint (Point4, 0, 225, CornerRadius * 3, EdgeRadius, Depth)
      } //union
   } //object

   #local D_Bottom = object {
      difference {
         object {D_Bottom}
         object {EdgeCutout rotate 0 * z translate Point1}
         object {EdgeCutout rotate 90 * z translate Point2}
         object {EdgeCutout rotate 180 * z translate Point3}
         object {EdgeCutout rotate 225 * z translate Point4}
      } //difference
   } //object

   #local GlitchFixer = object {
      box {<0, BorderWidth, -.1> <2.9 - BorderWidth - EdgeRadius, 1 - BorderWidth, Depth + .1>}
   } //object

   #local D_Back = object {
      #local Point1 = <0, 0, 0>; //Lower left
      #local Point2 = <1, 1, 0>; //Lower right
      #local Point4 = <0, 2.08, 0>; //Upper left
      ComputeBorderPoint (Point1, 90, 45, BorderWidth, false)
      #local Point1 = BorderPoint;
      ComputeBorderPoint (Point2, 90, 225, BorderWidth, false)
      #local Point2 = BorderPoint;
      ComputeBorderPoint (Point4, -90, 45, BorderWidth, false)
      #local Point4 = BorderPoint;
      union {
         intersection {
            box {<BorderWidth, 0, 0> <1 - BorderWidth, 4, Depth>}
            plane {y, 0 inverse rotate 45 * z translate Point1} //Bottom
            plane {y, 0 rotate 45 * z translate Point4} //Top
            CornerPlane (Point2, 90, -135, CornerRadius * 3, EdgeRadius, Depth)
            CornerPlane (Point4, 45, -90, CornerRadius * 3, EdgeRadius, Depth)
         } //intersection
         RoundedPoint (Point2, 90, -135, CornerRadius * 3, EdgeRadius, Depth)
         RoundedPoint (Point4, 45, -90, CornerRadius * 3, EdgeRadius, Depth)
      } //union
   } //object

   #local D_Back = object {
      difference {
         object {D_Back}
         object {EdgeCutout rotate 45 * z translate Point1}
         object {EdgeCutout rotate 90 * z translate Point2}
         object {EdgeCutout rotate 225 * z translate Point4}
         object {EdgeCutout rotate 270 * z translate Point4}
      } //difference
      rotate -20 * z
      translate <1.15, .6, 0>
   } //object

   #local Range = 90; //Loop will go from 0 .. Range degrees.
   #local NSegments = ceil (Range / DeltaTheta) + 1;
   #local NSegments = NSegments + 1; //Last one is a dummy.
   #local SegmentX_Inner = array [NSegments]
   #local SegmentY_Inner = array [NSegments]
   #local SegmentX_Outer = array [NSegments]
   #local SegmentY_Outer = array [NSegments]
   #local R1 = 1.3 + BorderWidth; //Inner radius
   #local R2 = 2.4 - BorderWidth; //Outer radius
   #local Theta = 0;
   #local I = 0;

   #while (Theta <= Range)
      #if (Theta < 90)
         #local R1Extension = .5 * (1 - sin (radians (Theta)));
      #else
         #local R1Extension = 0;
      #end //#if
      #local SegmentX_Inner [I] = (R1 + R1Extension) * cos (radians (-90 + Theta));
      #local SegmentY_Inner [I] = (R1 + R1Extension) * sin (radians (-90 + Theta));;
      #local SegmentX_Outer [I] = (R2 - .2) * cos (radians (-90 + Theta));
      #local SegmentY_Outer [I] = (R2 - .2) * sin (radians (-90 + Theta));;
      #local Theta = Theta + DeltaTheta;
      #local I = I + 1;
   #end //#while

   #local SegmentX_Inner [NSegments - 1] = SegmentX_Inner [NSegments - 2]; //Last element is a dummy.
   #local SegmentY_Inner [NSegments - 1] = SegmentY_Inner [NSegments - 2] - 1; //Change last value to prevent division by 0.
   #local SegmentX_Outer [NSegments - 1] = SegmentX_Outer [NSegments - 2];
   #local SegmentY_Outer [NSegments - 1] = SegmentY_Outer [NSegments - 2] - 1;

   #local D_Front = object {
      union {
         #local Extra = 2;
         #local I = 0;
         #while (I <= NSegments - 3) //"- 2" gives full curve
            #local CurX_Inner = SegmentX_Inner [I];
            #local CurY_Inner = SegmentY_Inner [I];
            #local NextX_Inner = SegmentX_Inner [I + 1];
            #local NextY_Inner = SegmentY_Inner [I + 1];
            #local DeltaX_Inner = NextX_Inner - CurX_Inner;
            #local DeltaY_Inner = NextY_Inner - CurY_Inner;
            #if (DeltaX_Inner = 0)
               #local Angle_Inner = 90;
            #else
               #local Angle_Inner = degrees (atan (DeltaY_Inner / DeltaX_Inner));
            #end //#if
            #if (DeltaX_Inner < 0)
               #local Angle_Inner = Angle_Inner - 180;
            #end //#if
            #local CurX_Outer = SegmentX_Outer [I];
            #local CurY_Outer = SegmentY_Outer [I];
            #local NextX_Outer = SegmentX_Outer [I + 1];
            #local NextY_Outer = SegmentY_Outer [I + 1];
            #local DeltaX_Outer = NextX_Outer - CurX_Outer;
            #local DeltaY_Outer = NextY_Outer - CurY_Outer;
            #if (DeltaX_Outer = 0)
               #local Angle_Outer = 90;
            #else
               #local Angle_Outer = degrees (atan (DeltaY_Outer / DeltaX_Outer));
            #end //#if
            #if (DeltaX_Outer < 0)
               #local Angle_Outer = Angle_Outer - 180;
            #end //#if
            #local DeltaX_Bottom = CurX_Outer - CurX_Inner;
            #local DeltaY_Bottom = CurY_Outer - CurY_Inner;
            #local Angle_Bottom = degrees (atan (DeltaY_Bottom / DeltaX_Bottom));
            #local DeltaX_Top = NextX_Outer - NextX_Inner;
            #local DeltaY_Top = NextY_Outer - NextY_Inner;
            #local Angle_Top = degrees (atan (DeltaY_Top / DeltaX_Top));
            #local Segment = intersection {
               box {<CurX_Inner - Extra, CurY_Inner - Extra, 0> <CurX_Outer + Extra, CurY_Outer + Extra, Depth>}
               plane {y, 0 rotate Angle_Inner * z translate <CurX_Inner, CurY_Inner, 0>} //Left side
               plane {y, 0 inverse rotate Angle_Outer * z translate <CurX_Outer, CurY_Outer, 0>} //Right side
               plane {y, 0 inverse rotate Angle_Bottom * z translate <CurX_Inner, CurY_Inner, 0>} //Bottom
               plane {y, 0 rotate Angle_Top * z translate <NextX_Inner, NextY_Inner, 0>} //Top
            } //intersection
            #local PercentDone = I / (NSegments - 3);
            #if (PercentDone > .3) //Don't need to bevel part that will be chopped off
               #local Segment = object {
                  difference {
                     object {Segment}
                     object {EdgeCutout rotate 90 * x rotate Angle_Inner * z translate <CurX_Inner, CurY_Inner, 0>} //Left side
                  } //difference
               } //object
            #end //#if
            object {Segment}
            #local I = I + 1;
         #end //#while
      } //union
   } //object

   #local D_Front = object {
      union {
         object {D_Front}
         difference {
            intersection {
               object {BeveledHollowCylinder (<0, 0, 0>, <0, 0, Depth>, R1, R2, EdgeRadius)}
               plane {x, 0 inverse}
            } //intersection
            intersection {
               cylinder {<0, 0, -.1> <0, 0, Depth + .1>, R2 - .3}
               plane {y, 0}
            } //intersection
         } //difference
      } //union
   } //object

   #local D_Front = object {
      difference {
         intersection {
            object {D_Front}
            #local XOffset = 1.6 - BorderWidth / sin (radians (37.5));
            plane {y, 0 rotate 37.5 * z translate -XOffset * x} //Very top
         } //intersection
         object {EdgeCutout rotate 90 * x rotate 37.5 * z translate -XOffset * x}
      } //difference
   } //object

   #local D_Front = object {
      union {
         intersection {
            object {D_Front}
            #local TSin = sin (radians (37.5));
            #local TCos = cos (radians (37.5));
            #local O = XOffset * TSin;
            #local A1 = XOffset * TCos;
            #local A2 = sqrt (R1 * R1 - O * O);
            #local R = A1 + A2;
            #local X1 = R * TCos - XOffset;
            #local Y1 = R * TSin;
            #local Point1 = <X1, Y1, 0>; //Upper left
            #local Point1Slope = -5 - BorderWidth * 100; //Fudge factor.
            #local A2 = sqrt (R2 * R2 - O * O);
            #local R = A1 + A2;
            #local X1 = R * TCos - XOffset;
            #local Y1 = R * TSin;
            #local Point2 = <X1, Y1, 0>; //Upper right
            union {
               CornerPlane (Point1, Point1Slope, 37.5, CornerRadius, EdgeRadius, Depth) //This plane chops off the bottom part of the curve.
               plane {y, 0 rotate 40 * z}                                               //This union fixes that problem.
            } //union
            CornerPlane (Point2, -35, 180 + 37.5, CornerRadius * 2, EdgeRadius, Depth)
         } //intersection
         RoundedPoint (Point1, Point1Slope, 37.5, CornerRadius, EdgeRadius, Depth)
         RoundedPoint (Point2, -35, 180 + 37.5, CornerRadius * 2, EdgeRadius, Depth)
      } //union
   } //object

   #local D_Front = object {
      intersection {
         object {D_Front translate <1.8, 1.8, 0>} //Move into final position
         plane {y, 0 inverse translate BorderWidth * y}
      } //intersection
   } //object

   #local D_Front = object { //Clean up bottom
      difference {
         object {D_Front}
         object {EdgeCutout translate BorderWidth * y}
         object {GlitchFixer}
      } //difference
   } //object

   #local D_Whole = object {
      union {
         object {D_Bottom}
         object {D_Back}
         object {D_Front}
      } //union
   } //object

   #local D_Whole = object {
      #local TSin = sin (radians (36.7));
      #local TCos = cos (radians (36.7));
      #local R = 1.62; //Estimated
      #local X1 = R * TCos + 2.75;
      #local Y1 = R * TSin;
      #local Point3 = <X1, Y1, 0>; //Middle right
      ComputeBorderPoint (Point3, 180 + 36.7, 72, BorderWidth, false)
      #local Point3 = BorderPoint;
      #local Point4 = <2.75, 0, 0>; //Lower right
      ComputeBorderPoint (Point4, 36.7, 180, BorderWidth, false)
      #local Point4 = BorderPoint;
      union {
         intersection {
            object {D_Whole}
            plane {y, 0 inverse rotate 36.7 * z translate Point4} //Bottom right angle
            CornerPlane (Point3, 180 + 36.7, 72, CornerRadius * 5, EdgeRadius, Depth)
            CornerPlane (Point4, 180, 36.7, CornerRadius * 5, EdgeRadius, Depth)
         } //intersection
         RoundedPoint (Point3, 75, 180 + 36.7, CornerRadius * 5, EdgeRadius, Depth)
         RoundedPoint (Point4, 180, 36.7, CornerRadius * 5, EdgeRadius, Depth)
      } //union
   } //object

   #object {
      difference {
         object {D_Whole}
         object {EdgeCutout rotate 36.7 * z translate Point4}
      } //difference
   } //object

#end //#macro Create_D

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
// *                                                                                                           *
// *                                                                                                           *
// *                                                                                                           *
// * O                                                                                                         *
// *                                                                                                           *
// *                                                                                                           *
// *                                                                                                           *
// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

#macro Create_O (BorderWidth, Depth)

   #local O_ = object {
      union {
         intersection {
            ComputeBorderPoint (<1, 0, 0>, 40, 180 - 3.5, BorderWidth, false)
            #local Point1 = BorderPoint;
            ComputeBorderPoint (<0, tan (radians (3.5)), 0>, -3.5, 90, BorderWidth, false)
            #local Point2 = BorderPoint;
            ComputeBorderPoint (<0, 1.3 + tan (radians (3.5)), 0>, -90, 40, BorderWidth, false)
            #local Point3 = BorderPoint;
            ComputeBorderPoint (<.75, 1.3 + tan (radians (3.5)) + .75 * tan (radians (40)), 0>, -3.5, 180 + 40, BorderWidth, false)
            #local Point4 = BorderPoint;
            ComputeBorderPoint (<1.75, 1.3 + .75 * tan (radians (40)), 0>, 180 - 3.5, -90, BorderWidth, false)
            #local Point5 = BorderPoint;
            ComputeBorderPoint (<1.75, .75 * tan (radians (40)), 0>, 90, -90 - 50, BorderWidth, false)
            #local Point6 = BorderPoint;
            box {<BorderWidth, 0, 0> <1.75 - BorderWidth, 2.5, Depth>}
            plane {y, 0 inverse rotate -3.5 * z translate Point1}
            plane {y, 0 rotate 40 * z translate Point3}
            plane {y, 0 rotate -3.5 * z translate Point4}
            plane {y, 0 inverse rotate 40 * z translate Point6}
            CornerPlane (Point1, 40, 180 - 3.5, CornerRadius * 4, EdgeRadius, Depth)
            CornerPlane (Point2, 90, -3.5, CornerRadius * 3, EdgeRadius, Depth)
            CornerPlane (Point3, -90, 40, CornerRadius * 4, EdgeRadius, Depth)
            CornerPlane (Point4, 180 + 40, -3.5, CornerRadius * 4, EdgeRadius, Depth)
            CornerPlane (Point5, -90, 180 - 3.5, CornerRadius * 3, EdgeRadius, Depth)
            CornerPlane (Point6, 90, 180 + 40, CornerRadius * 4, EdgeRadius, Depth)
         } //intersection
         RoundedPoint (Point1, 40, 180 - 3.5, CornerRadius * 4, EdgeRadius, Depth)
         RoundedPoint (Point2, 90, -3.5, CornerRadius * 3, EdgeRadius, Depth)
         RoundedPoint (Point3, -90, 40, CornerRadius * 4, EdgeRadius, Depth)
         RoundedPoint (Point4, 180 + 40, -3.5, CornerRadius * 4, EdgeRadius, Depth)
         RoundedPoint (Point5, -90, 180 - 3.5, CornerRadius * 3, EdgeRadius, Depth)
         RoundedPoint (Point6, 90, 180 + 40, CornerRadius * 4, EdgeRadius, Depth)
      } //union
   } //object

   #local O_ = object {
      difference {
         object {O_}
         object {EdgeCutout rotate 90 * z translate Point6}
         object {EdgeCutout rotate (180 - 3.5) * z translate Point5}
         object {EdgeCutout rotate (180 + 40) * z translate Point4}
         object {EdgeCutout rotate 270 * z translate Point3}
         object {EdgeCutout rotate (270 + 90 - 3.5) * z translate Point2}
         object {EdgeCutout rotate 40 * z translate Point1}
      } //difference
   } //object

   #local O_Center = object {
      #local XC = 1.75 / 2;
      #local YC = (1.3 + tan (radians (3.5)) + .75 * tan (radians (40))) / 2;
      #local O_CenterHeight = 1.2;
      intersection {
         box {<XC - .05/2 - BorderWidth, 0, -.1> <XC + .05/2 + BorderWidth, 2.5, Depth + .1>}
         plane {y, 0 inverse rotate -3.5 * z translate <XC, YC - O_CenterHeight / 2 - BorderWidth, 0>} //Bottom
         plane {y, 0 rotate -3.5 * z translate <XC, YC + O_CenterHeight / 2 + BorderWidth, 0>} //Top
      } //intersection
   } //object

   #local O_Center = object {
      union {
         object {O_Center}
         intersection {
            box {<XC - .05/2 - BorderWidth - EdgeRadius, 0, -.1> <XC + .05/2 + BorderWidth + EdgeRadius, 2.5, EdgeRadius>}
            plane {y, 0 inverse rotate -3.5 * z translate <XC, YC - O_CenterHeight / 2 - BorderWidth - EdgeRadius, 0>} //Bottom
            plane {y, 0 rotate -3.5 * z translate <XC, YC + O_CenterHeight / 2 + BorderWidth + EdgeRadius, 0>} //Top
         } //intersection
      } //union
   } //object

   #local O_Center = object {
      difference {
         object {O_Center}
         cylinder {<0, 0, 0> <0, 2.5, 0>, EdgeRadius translate <XC - .05/2 - BorderWidth - EdgeRadius, 0, EdgeRadius>}
         cylinder {<0, 0, 0> <0, 2.5, 0>, EdgeRadius translate <XC + .05/2 + BorderWidth + EdgeRadius, 0, EdgeRadius>}
         cylinder {<-10.1, 0, 0> <10.1, 0, 0>, EdgeRadius rotate -3.5 * z translate <XC, YC - O_CenterHeight / 2 - BorderWidth - EdgeRadius, EdgeRadius>}
         cylinder {<-10.1, 0, 0> <10.1, 0, 0>, EdgeRadius rotate -3.5 * z translate <XC, YC + O_CenterHeight / 2 + BorderWidth + EdgeRadius, EdgeRadius>}
      } //difference
   } //object

   #object {
      difference {
         object {O_}
         object {O_Center}
      } //difference
      rotate -20 * z
   } //object

#end //#macro Create_O

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
// *                                                                                                           *
// *                                                                                                           *
// *                                                                                                           *
// * BAR                                                                                                       *
// *                                                                                                           *
// *                                                                                                           *
// *                                                                                                           *
// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

#macro Create_Bar (BorderWidth, Depth)

   #local BarLength_Left = 3.25;
   #local BarLength_Middle = 1.77;
   #local BarLength_Right = 6.121;
   #local BarHoleWidth = .5;

   #local Bar_Left = object {
      ComputeBorderPoint (<0, 1, 0>, 180, -90 - 20, BorderWidth, false) //Upper right
      #local Point1 = BorderPoint;
      ComputeBorderPoint (<-BarLength_Left, 1, 0>, 0, 180 + 36.7, BorderWidth, false) //Upper left
      #local Point2 = BorderPoint;
      ComputeBorderPoint (<-BarLength_Left - 1 / tan (radians (36.7)), 0, 0>, 0, 36.7, BorderWidth, false) //Lower left
      #local Point3 = BorderPoint;
      ComputeBorderPoint (<-1 / tan (radians (70)), 0, 0>, 180, 70, BorderWidth, false) //Lower right
      #local Point4 = BorderPoint;
      #if (BorderWidth > 0)
         ComputeBorderPoint (Point1, 180, -90 - 20, BorderWidth, false)
         #declare SavePoint5 = BorderPoint;
         ComputeBorderPoint (Point4, 180, 70, BorderWidth, false)
         #declare SavePoint1 = BorderPoint;
      #end //#if
      union {
         intersection {
            box {<-6, BorderWidth, 0> <0, 1 - BorderWidth, Depth>}
            plane {y, 0 inverse rotate 70 * z translate Point1}
            plane {y, 0 rotate 36.7 * z translate Point2}
            CornerPlane (Point2, 0, 180 + 36.7, CornerRadius * 3, EdgeRadius, Depth)
            CornerPlane (Point3, 0, 36.7, CornerRadius, EdgeRadius, Depth)
         } //intersection
         RoundedPoint (Point2, 0, 180 + 36.7, CornerRadius * 3, EdgeRadius, Depth)
         RoundedPoint (Point3, 0, 36.7, CornerRadius, EdgeRadius, Depth)
      } //union
   } //object

   #local Bar_Left = object {
      difference {
         object {Bar_Left}
         object {EdgeCutout translate Point3}
         object {EdgeCutout rotate 70 * z translate Point4}
         object {EdgeCutout rotate 180 * z translate Point1}
         object {EdgeCutout rotate (180 + 36.7) * z translate Point2}
      } //difference
   } //object

   #local Bar_Middle = object {
      ComputeBorderPoint (<0, 1, 0>, 180, -90 - 20, BorderWidth, false) //Upper right
      #local Point1 = BorderPoint;
      ComputeBorderPoint (<-BarLength_Middle, 1, 0>, 0, -90 - 20, BorderWidth, false) //Upper left
      #local Point2 = BorderPoint;
      ComputeBorderPoint (<-BarLength_Middle - 1 / tan (radians (70)), 0, 0>, 0, 70, BorderWidth, false) //Lower left
      #local Point3 = BorderPoint;
      ComputeBorderPoint (<-1 / tan (radians (70)), 0, 0>, 180, 70, BorderWidth, false) //Lower right
      #local Point4 = BorderPoint;
      #if (BorderWidth > 0)
         ComputeBorderPoint (Point1, 180, -90 -20, BorderWidth, false)
         #declare SavePoint7 = <BorderPoint.x + BarHoleWidth + BarLength_Middle, BorderPoint.y, BorderPoint.z>;
         ComputeBorderPoint (Point2, 0, -90 -20, BorderWidth, false)
         #declare SavePoint6 = <BorderPoint.x + BarHoleWidth + BarLength_Middle, BorderPoint.y, BorderPoint.z>;
         ComputeBorderPoint (Point3, 0, 70, BorderWidth, false)
         #declare SavePoint2 = <BorderPoint.x + BarHoleWidth + BarLength_Middle, BorderPoint.y, BorderPoint.z>;
         ComputeBorderPoint (Point4, 180, 70, BorderWidth, false)
         #declare SavePoint3 = <BorderPoint.x + BarHoleWidth + BarLength_Middle, BorderPoint.y, BorderPoint.z>;
      #end //#if
      intersection {
         box {<-6, BorderWidth, 0> <0, 1 - BorderWidth, Depth>}
         plane {y, 0 inverse rotate 70 * z translate Point1}
         plane {y, 0 rotate 70 * z translate Point2}
      } //intersection
   } //object

   #local Bar_Middle = object {
      difference {
         object {Bar_Middle}
         object {EdgeCutout translate Point3}
         object {EdgeCutout rotate 70 * z translate Point4}
         object {EdgeCutout rotate 180 * z translate Point1}
         object {EdgeCutout rotate (180 + 70) * z translate Point2}
      } //difference
   } //object

   #local Bar_Right = object {
      ComputeBorderPoint (<0, 1, 0>, 0, -90 - 20, BorderWidth, false) //Upper left
      #local Point1 = BorderPoint;
      ComputeBorderPoint (<-1 / tan (radians (70)), 0, 0>, 0, 70, BorderWidth, false) //Lower left
      #local Point2 = BorderPoint;
      ComputeBorderPoint (<BarLength_Right, 0, 0>, 180, 90 + 45, BorderWidth, false) //Lower right
      #local Point3 = BorderPoint;
      ComputeBorderPoint (<BarLength_Right - 1 / tan (radians (45)), 1, 0>, 180, -45, BorderWidth, false) //Upper right
      #local Point4 = BorderPoint;
      #if (BorderWidth > 0)
         ComputeBorderPoint (Point1, 0, -90 - 20, BorderWidth, false)
         #declare SavePoint8 = <BorderPoint.x + BarHoleWidth * 2 + BarLength_Middle, BorderPoint.y, BorderPoint.z>;
         ComputeBorderPoint (Point2, 0, 70, BorderWidth, false)
         #declare SavePoint4 = <BorderPoint.x + BarHoleWidth * 2 + BarLength_Middle, BorderPoint.y, BorderPoint.z>;
      #end //#if
      union {
         intersection {
            box {<-1, BorderWidth, 0> <8, 1 - BorderWidth, Depth>}
            plane {y, 0 rotate 70 * z translate Point1}
            plane {y, 0 inverse rotate (90 + 45) * z translate Point3}
            CornerPlane (Point3, 180, 90 + 45, CornerRadius, EdgeRadius, Depth)
            CornerPlane (Point4, 180, -45, CornerRadius * 3, EdgeRadius, Depth)
         } //intersection
         RoundedPoint (Point3, 180, 90 + 45, CornerRadius, EdgeRadius, Depth)
         RoundedPoint (Point4, 180, -45, CornerRadius * 3, EdgeRadius, Depth)
      } //union
   } //object

   #local Bar_Right = object {
      difference {
         object {Bar_Right}
         object {EdgeCutout translate Point2}
         object {EdgeCutout rotate (90 + 45) * z translate Point3}
         object {EdgeCutout rotate 180 * z translate Point4}
         object {EdgeCutout rotate (180 + 70) * z translate Point1}
      } //difference
   } //object

   #object {
      union {
         object {Bar_Left}
         object {Bar_Middle translate (BarHoleWidth + BarLength_Middle) * x}
         object {Bar_Right translate (BarHoleWidth * 2 + BarLength_Middle) * x}
      } //union
   } //object

#end //#macro Create_Bar

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
// *                                                                                                           *
// *                                                                                                           *
// *                                                                                                           *
// * HOOKS                                                                                                     *
// *                                                                                                           *
// *                                                                                                           *
// *                                                                                                           *
// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

#macro Create_Hooks (BorderWidth, Depth)

   #local HookHeight = 1.5;

   #local Hook_Left = object {
      #local Point1 = SavePoint1;
      #local Point2 = SavePoint2;
      #local Point3 = <SavePoint2.x - HookHeight * tan (radians (20)), SavePoint2.y - HookHeight, 0>;
      ComputeIntersectionPoint (Point1, -90 - 20, Point3, 180 - 62)
      #local Point4 = IntersectionPoint;
      ComputeBorderPoint (Point1, 0, -90 - 20, BorderWidth, false)
      #local Point1 = BorderPoint;
      ComputeBorderPoint (Point2, 180, -90 - 20, BorderWidth, false)
      #local Point2 = BorderPoint;
      ComputeBorderPoint (Point3, 70, 180 - 62, BorderWidth, false)
      #local Point3 = BorderPoint;
      ComputeBorderPoint (Point4, 70, -62, BorderWidth, false)
      #local Point4 = BorderPoint;
      union {
         intersection {
            box {<Point4.x, Point1.y, 0> <Point2.x, Point3.y, Depth>}
            plane {y, 0 rotate 70 * z translate Point1}
            plane {y, 0 inverse rotate 70 * z translate Point2}
            plane {y, 0 rotate (180 - 62) * z translate Point3}
            CornerPlane (Point3, 70, 180 - 62, CornerRadius, EdgeRadius, Depth)
            CornerPlane (Point4, 70, -62, CornerRadius * 3, EdgeRadius, Depth)
         } //intersection
         RoundedPoint (Point3, 70, 180 - 62, CornerRadius, EdgeRadius, Depth)
         RoundedPoint (Point4, 70, -62, CornerRadius * 3, EdgeRadius, Depth)
      } //union
   } //object

   #local Hook_Left = object {
      difference {
         object {Hook_Left}
         object {EdgeCutout rotate 70 * z translate Point3}
         object {EdgeCutout rotate 180 * z translate Point2}
         object {EdgeCutout rotate (180 + 70) * z translate Point1}
         object {EdgeCutout rotate -62 * z translate Point4}
      } //difference
   } //object

   #local Hook_Right = object {
      #local Point1 = SavePoint3;
      #local Point2 = SavePoint4;
      #local Point4 = <SavePoint3.x - HookHeight * tan (radians (20)), SavePoint3.y - HookHeight, 0>;
      ComputeIntersectionPoint (Point2, -90 - 20, Point4, 36.7)
      #local Point3 = IntersectionPoint;
      ComputeBorderPoint (Point1, 0, -90 - 20, BorderWidth, false)
      #local Point1 = BorderPoint;
      ComputeBorderPoint (Point2, 180, -90 - 20, BorderWidth, false)
      #local Point2 = BorderPoint;
      ComputeBorderPoint (Point3, 70, 180 + 36.7, BorderWidth, false)
      #local Point3 = BorderPoint;
      ComputeBorderPoint (Point4, 70, 36.7, BorderWidth, false)
      #local Point4 = BorderPoint;
      union {
         intersection {
            box {<Point4.x, Point4.y, 0> <Point2.x, Point2.y, Depth>}
            plane {y, 0 rotate 70 * z translate Point4}
            plane {y, 0 rotate 36.7 * z inverse translate Point4}
            plane {y, 0 rotate (-90 - 20) * z translate Point2}
            CornerPlane (Point3, 70, 180 + 36.7, CornerRadius * 4, EdgeRadius, Depth)
            CornerPlane (Point4, 70, 36.7, CornerRadius, EdgeRadius, Depth)
         } //intersection
         RoundedPoint (Point3, 70, 180 + 36.7, CornerRadius * 4, EdgeRadius, Depth)
         RoundedPoint (Point4, 70, 36.7, CornerRadius, EdgeRadius, Depth)
      } //union
   } //object

   #local Hook_Right = object {
      difference {
         object {Hook_Right}
         object {EdgeCutout rotate 70 * z translate Point3}
         object {EdgeCutout rotate 180 * z translate Point2}
         object {EdgeCutout rotate (180 + 70) * z translate Point1}
         object {EdgeCutout rotate 36.7 * z translate Point4}
      } //difference
   } //object

   #object {
      union {
         object {Hook_Left}
         object {Hook_Right}
      } //union
   } //object

#end //#macro Create_Hooks

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
// *                                                                                                           *
// *                                                                                                           *
// *                                                                                                           *
// * K                                                                                                         *
// *                                                                                                           *
// *                                                                                                           *
// *                                                                                                           *
// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

#macro Create_K (BorderWidth, Depth)

   #local KWidth = SavePoint6.x - SavePoint5.x;

   #local K_Left = object {
      #local R14 = 2.45;
      #local Point1 = <0, 0, 0>;
      #local Point2 = <KWidth, 0, 0>;
      #local Point4 = <Point1.x + R14 * cos (radians (70)), Point1.y + R14 * sin (radians (70)), 0>;
      ComputeIntersectionPoint (Point4, 21.4, Point2, 70)
      #local Point3 = IntersectionPoint;
      ComputeBorderPoint (Point1, 0, 70, BorderWidth, false)
      #local Point1 = BorderPoint;
      ComputeBorderPoint (Point2, 180, 70, BorderWidth, false)
      #local Point2 = BorderPoint;
      ComputeBorderPoint (Point3, -90 - 20, 180 + 21.4, BorderWidth, false)
      #local Point3 = BorderPoint;
      ComputeBorderPoint (Point4, 21.4, -90 - 20, BorderWidth, false)
      #local Point4 = BorderPoint;
      union {
         intersection {
            box {<0, BorderWidth, 0> <Point3.x, Point3.y, Depth>}
            plane {y, 0 rotate 70 * z translate Point1}
            plane {y, 0 inverse rotate 70 * z translate Point2}
            plane {y, 0 rotate 21.4 * z translate Point4}
            CornerPlane (Point1, 0, 70, CornerRadius, EdgeRadius, Depth)
            CornerPlane (Point3, 180 + 21.4, -90 - 20, CornerRadius, EdgeRadius, Depth)
            CornerPlane (Point4, 21.4, -90 - 20, CornerRadius * 3, EdgeRadius, Depth)
         } //intersection
         RoundedPoint (Point1, 0, 70, CornerRadius, EdgeRadius, Depth)
         RoundedPoint (Point3, 180 + 21.4, -90 - 20, CornerRadius, EdgeRadius, Depth)
         RoundedPoint (Point4, 21.4, -90 - 20, CornerRadius * 3, EdgeRadius, Depth)
      } //union
   } //object

   #local K_Left = object {
      difference {
         object {K_Left}
         object {EdgeCutout rotate 0 * z translate Point1}
         object {EdgeCutout rotate 70 * z translate Point2}
         object {EdgeCutout rotate (180 + 21.4) * z translate Point3}
         object {EdgeCutout rotate (180 + 70) * z translate Point4}
      } //difference
   } //object

   #local K_Left_Bar = object {
      #local BarWidth_Bottom = .19;
      #local BarHeight = .63;
      #local Point1 = <0, BorderWidth, 0>;
      #local Point2 = <BarWidth_Bottom, 0, 0>;
      #local Point4 = <-BorderWidth, BarHeight, 0>;
      ComputeIntersectionPoint (Point2, 52, Point4, 0)
      #local Point3 = IntersectionPoint;
      ComputeBorderPoint (Point2, 180, 52, BorderWidth, false)
      #local Point2 = BorderPoint;
      ComputeBorderPoint (Point3, 180, 180 + 52, BorderWidth, false)
      #local Point3 = BorderPoint;
      ComputeBorderPoint (Point4, 0, -90, BorderWidth, false)
      #local Point4 = BorderPoint;
      union {
         intersection {
            box {<-KWidth / 2, BorderWidth, 0> <Point3.x, Point3.y, Depth>}
            plane {y, 0 inverse rotate 52 * z translate Point2}
            CornerPlane (Point2, 180, 52, CornerRadius * 3, EdgeRadius, Depth)
            CornerPlane (Point3, 180, 180 + 52, CornerRadius, EdgeRadius, Depth)
         } //intersection
         RoundedPoint (Point2, 180, 52, CornerRadius * 3, EdgeRadius, Depth)
         RoundedPoint (Point3, 180, 180 + 52, CornerRadius, EdgeRadius, Depth)
      } //union
   } //object

   #local K_Left_Bar = object {
      difference {
         object {K_Left_Bar}
         object {EdgeCutout rotate 0 * z translate Point1}
         object {EdgeCutout rotate 52 * z translate Point2}
         object {EdgeCutout rotate 180 * z translate Point3}
      } //difference
   } //object

   #local K_Left = object {
      union {
         object {K_Left}
         object {K_Left_Bar translate (KWidth - EdgeRadius) * x}
      } //union
   } //object

   #local K_Right = object {
      #local R1 = .1 + BorderWidth; //Inner radius
      #local R2 = .35 - BorderWidth; //Outer radius
      #local Point1 = <0, 0, 0>;
      #local Point2 = <KWidth, 0, 0>;
      #local R = 1.175;
      #local X1 = R * cos (radians (70));
      #local Y1 = R * sin (radians (70));
      #local Point3 = <Point2.x + X1, Point2.y + Y1, 0>;
      #local R = .82;
      #local X1 = R * cos (radians (70));
      #local Y1 = R * sin (radians (70));
      #local Point5 = <Point1.x + X1, Point1.y + Y1, 0>;
      #local R = .62;
      #local X1 = R * cos (radians (180 - 30));
      #local Y1 = R * sin (radians (180 - 30));
      #local Point4 = <Point5.x + X1, Point5.y + Y1, 0>;
      ComputeBorderPoint (Point1, 0, 70, BorderWidth, false)
      #local Point1 = BorderPoint;
      ComputeBorderPoint (Point2, 180, 70, BorderWidth, false)
      #local Point2 = BorderPoint;
      ComputeBorderPoint (Point3, -90 - 20, 180 - 30, BorderWidth, false)
      #local Point3 = BorderPoint;
      ComputeBorderPoint (Point4, 21.4, -30, BorderWidth, false)
      #local Point4 = BorderPoint;
      ComputeBorderPoint (Point5, -90 - 20, 180 - 30, BorderWidth, true)
      #local Point5 = BorderPoint;
      union {
         difference {
            intersection {
               box {<0, BorderWidth, 0> <2, 2, Depth>}
               plane {y, 0 rotate 70 * z translate Point1}
               plane {y, 0 inverse rotate 70 * z translate Point2}
               plane {y, 0 inverse rotate (180 - 30) * z translate Point3}
               //plane {y, 0 rotate 21.4 * z translate Point4} //Remove small hook at top of curve.
               CornerPlane (Point3, -90 - 20, 180 - 30, R2, EdgeRadius, Depth)
            } //intersection
            object {EdgeCutout rotate (-90 - 20) * z translate Point1}
         } //difference
         RoundedPoint (Point3, -90 - 20, 180 - 30, R2, EdgeRadius, Depth)
         difference {
            intersection {
               box {<Point4.x, Point5.y, 0> <Point2.x * .7, 2, Depth>}
               plane {y, 0 rotate (180 - 30) * z translate Point5}
               plane {y, 0 rotate 21.4 * z translate Point4}
               CornerPlane (Point4, 21.4, -30, CornerRadius, EdgeRadius, Depth)
            } //intersection
            object {EdgeCutout rotate -30 * z translate Point4}
            object {EdgeCutout rotate (180 + 21.4) * z translate Point4}
         } //difference
         RoundedPoint (Point4, 21.4, -30, CornerRadius, EdgeRadius, Depth)
      } //union
   } //object

   #local K_Right = object {
      difference {
         object {K_Right}
         object {EdgeCutout translate Point1}
         object {EdgeCutout rotate 70 * z translate Point2}
         object {EdgeCutout rotate (180 - 30) * z translate Point3}
      } //difference
   } //object

   #local K_Right = object {
      union {
         object {K_Right}
         RoundedCorner (Point5, 180 - 30, -90 - 20, R1, EdgeRadius, Depth)
      } //union
   } //object

   #local K_RightTop = object {
      #local Point1 = <0, 0, 0>;
      #local R12 = .95;
      #local Point2 = <Point1.x + R12 * cos (radians (180 - 28.2)), Point1.y + R12 * sin (radians (180 - 28.2)), 0>;
      #local R14 = 1.15;
      #local Point4 = <Point1.x + R14 * cos (radians (180 + 21.4)), Point1.y + R14 * sin (radians (180 + 21.4)), 0>;
      ComputeBorderPoint (Point1, 180 - 28.2, 180 + 21.4, BorderWidth, false)
      #local Point1 = BorderPoint;
      ComputeBorderPoint (Point2, -28.2, 180 + 21.4, BorderWidth, false)
      #local Point2 = BorderPoint;
      ComputeBorderPoint (Point4, 21.4, 180 - 28.2, BorderWidth, false)
      #local Point4 = BorderPoint;
      union {
         intersection {
            box {<Point1.x, Point4.y, 0> <-6, Point2.y, Depth>}
            plane {y, 0 inverse rotate (180 - 28.2) * z translate Point1}
            plane {y, 0 rotate 21.4 * z translate Point2}
            plane {y, 0 inverse rotate 21.4 * z translate Point1}
            plane {y, 0 rotate (180 - 28.2) * z translate Point4}
            CornerPlane (Point1, 180 - 28.2, 180 + 21.4, CornerRadius, EdgeRadius, Depth)
            CornerPlane (Point2, -28.2, 180 + 21.4, CornerRadius * 3, EdgeRadius, Depth)
         } //intersection
         RoundedPoint (Point1, 180 - 28.2, 180 + 21.4, CornerRadius, EdgeRadius, Depth)
         RoundedPoint (Point2, -28.2, 180 + 21.4, CornerRadius * 3, EdgeRadius, Depth)
      } //union
   } //object

   #local K_RightTop = object {
      difference {
         object {K_RightTop}
         object {EdgeCutout rotate (180 - 28.2) * z translate Point1}
         object {EdgeCutout rotate (180 + 21.4) * z translate Point2}
         object {EdgeCutout rotate 21.4 * z translate Point1}
         object {EdgeCutout rotate -28.2 * z translate Point4}
      } //difference
   } //object

   #object {
      union {
         object {K_Right}
         object {K_RightTop translate <1.65, 1.62, 0>}
         object {K_Left translate <-KWidth - .32, 1 - SavePoint5.y + VerticalGap, 0>}
      } //union
   } //object

#end //#macro Create_K

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
// *                                                                                                           *
// *                                                                                                           *
// *                                                                                                           *
// * E                                                                                                         *
// *                                                                                                           *
// *                                                                                                           *
// *                                                                                                           *
// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

#macro Create_E (BorderWidth, Depth)

   #local R12 = 1.42;
   #local R23 = 1.42;
   #local R65 = .59;
   #local R54 = .9;
   #local R89 = .12;

   #local Point1 = <0, 0, 0>;
   #local Point2 = <Point1.x + R12 * cos (radians (70)), Point1.y + R12 * sin (radians (70)), 0>;
   #local Point3 = <Point2.x + R23 * cos (radians (21.4)), Point2.y + R23 * sin (radians (21.4)), 0>;
   #local Point7 = <1.52, 0, 0>;
   #local Point6 = <Point7.x + .63 / tan (radians (52)), .63, 0>;
   #local Point5 = <Point6.x - R65, .63, 0>;
   ComputeIntersectionPoint (Point3, -63.2, Point5, 21.4)
   #local Point4 = IntersectionPoint;
   #local Point8 = <1.25, 1.07, 0>;
   #local Point9 = <Point8.x + R89 * cos (radians (21.4)), Point8.y + R89 * sin (radians (21.4)), 0>;

   ComputeBorderPoint (Point1, 0, 70, BorderWidth, false)
   #local Point1 = BorderPoint;
   ComputeBorderPoint (Point2, 21.4, -90 - 20, BorderWidth, false)
   #local Point2 = BorderPoint;
   ComputeBorderPoint (Point3, 180 + 21.4, -63.2, BorderWidth, false)
   #local Point3 = BorderPoint;
   ComputeBorderPoint (Point4, 180 + 21.4, 180 - 63.2, BorderWidth, false)
   #local Point4 = BorderPoint;
   ComputeBorderPoint (Point5, 0, 21.4, BorderWidth, true)
   #local Point5 = BorderPoint;
   ComputeBorderPoint (Point6, 180, 180 + 52, BorderWidth, false)
   #local Point6 = BorderPoint;
   ComputeBorderPoint (Point7, 180, 52, BorderWidth, false)
   #local Point7 = BorderPoint;
   ComputeBorderPoint (Point8, 70, 21.4, BorderWidth, true)
   #local Point8 = BorderPoint;
   ComputeBorderPoint (Point9, 180 - 63.2, 180 - 21.4, BorderWidth, true)
   #local Point9 = BorderPoint;

   #local E_Center = object {
      intersection {
         box {<Point8.x, Point8.y, -.1> <Point9.x, Point9.y + 1, Depth + .1>}
         plane {y, 0 inverse rotate 21.4 * z translate Point8}
         plane {y, 0 rotate 70 * z translate Point8}
         plane {y, 0 inverse rotate (180 - 63.2) * z translate Point9}
      } //intersection
   } //object

   ComputeBorderPoint (Point8, 70, 21.4, EdgeRadius, true)
   #local Point8 = BorderPoint;
   ComputeBorderPoint (Point9, 180 - 63.2, 180 - 21.4, EdgeRadius, true)
   #local Point9 = BorderPoint;

   #local E_Center = object {
      union {
         object {E_Center}
         intersection {
            box {<Point8.x, Point8.y, -.1> <Point9.x, Point9.y + 1, EdgeRadius>}
            plane {y, 0 inverse rotate 21.4 * z translate Point8}
            plane {y, 0 rotate 70 * z translate Point8}
            plane {y, 0 inverse rotate (180 - 63.2) * z translate Point9}
         } //intersection
      } //union
   } //object

   #local E_Center = object {
      difference {
         object {E_Center}
         cylinder {<-2, 0, 0> <2, 0, 0>, EdgeRadius rotate 21.4 * z translate <Point8.x, Point8.y, EdgeRadius>}
         cylinder {<-2, 0, 0> <2, 0, 0>, EdgeRadius rotate 70 * z translate <Point8.x, Point8.y, EdgeRadius>}
         cylinder {<-2, 0, 0> <2, 0, 0>, EdgeRadius rotate (180 - 63.2) * z translate <Point9.x, Point9.y, EdgeRadius>}
      } //intersection
   } //object

   #local E_Top = object {
      union {
         intersection {
            box {<Point1.x, BorderWidth, 0> <Point4.x, Point3.y, Depth>}
            plane {y, 0 rotate 70 * z translate Point1}
            plane {y, 0 rotate 21.4 * z translate Point2}
            plane {y, 0 inverse rotate (180 - 63.2) * z translate Point3}
            plane {y, 0 inverse rotate 21.4 * z translate Point4}
            CornerPlane (Point2, 21.4, -90 - 20, CornerRadius * 3, EdgeRadius, Depth)
            CornerPlane (Point3, 180 + 21.4, -63.2, CornerRadius * 2, EdgeRadius, Depth)
            CornerPlane (Point4, 180 + 21.4, 180 - 63.2, CornerRadius * 2, EdgeRadius, Depth)
         } //intersection
         RoundedPoint (Point2, 21.4, -90 - 20, CornerRadius * 3, EdgeRadius, Depth)
         RoundedPoint (Point3, 180 + 21.4, -63.2, CornerRadius * 2, EdgeRadius, Depth)
         RoundedPoint (Point4, 180 + 21.4, 180 - 63.2, CornerRadius * 2, EdgeRadius, Depth)
      } //union
   } //object

   #local E_Top = object {
      difference {
         object {E_Top}
         object {EdgeCutout rotate 21.4 * z translate Point4}
         object {EdgeCutout rotate (180 - 63.2) * z translate Point4}
         object {EdgeCutout rotate (180 + 21.4) * z translate Point3}
         object {EdgeCutout rotate (180 + 70) * z translate Point2}
         object {E_Center}
      } //difference
   } //object

   #local E_Bottom = object {
      union {
         intersection {
            box {<Point1.x, BorderWidth, 0> <Point6.x, Point6.y, Depth>}
            plane {y, 0 rotate 70 * z translate Point1}
            plane {y, 0 inverse rotate 52 * z translate Point7}
            CornerPlane (Point1, 0, 70, CornerRadius, EdgeRadius, Depth)
            CornerPlane (Point7, 180, 52, CornerRadius * 3, EdgeRadius, Depth)
            CornerPlane (Point6, 180, 180 + 52, CornerRadius, EdgeRadius, Depth)
         } //intersection
         RoundedPoint (Point1, 0, 70, CornerRadius, EdgeRadius, Depth)
         RoundedPoint (Point7, 180, 52, CornerRadius * 3, EdgeRadius, Depth)
         RoundedPoint (Point6, 180, 180 + 52, CornerRadius, EdgeRadius, Depth)
      } //union
   } //object

   #local E_Bottom = object {
      difference {
         object {E_Bottom}
         object {EdgeCutout rotate 0 * z translate Point1}
         object {EdgeCutout rotate 52 * z translate Point7}
         object {EdgeCutout rotate 180 * z translate Point6}
         object {EdgeCutout rotate (180 + 70) * z translate Point2}
      } //difference
   } //object

   #object {
      union {
         object {E_Top}
         object {E_Bottom}
         RoundedCorner (Point5, 0, 21.4, .02, EdgeRadius, Depth)
      } //union
   } //object

#end //#macro Create_E

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
// *                                                                                                           *
// *                                                                                                           *
// *                                                                                                           *
// * N                                                                                                         *
// *                                                                                                           *
// *                                                                                                           *
// *                                                                                                           *
// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

#macro Create_N (BorderWidth, Depth)

   #local NWidth = SavePoint6.x - SavePoint5.x;

   #local N_Left = object {
      #local R12 = 1.64;
      #local Point1 = <0, 0, 0>;
      #local Point2 = <Point1.x + R12 * cos (radians (70)), Point1.y + R12 * sin (radians (70)), 0>;
      #local Point4 = <Point1.x + NWidth, Point1.y, 0>;
      ComputeIntersectionPoint (Point2, 21.4, Point4, 70)
      #local Point3 = IntersectionPoint;
      ComputeBorderPoint (Point1, 0, 70, BorderWidth, false)
      #local Point1 = BorderPoint;
      ComputeBorderPoint (Point2, -90 -20, 21.4, BorderWidth, false)
      #local Point2 = BorderPoint;
      ComputeBorderPoint (Point3, 180 + 21.4, -90 - 20, BorderWidth, false)
      #local Point3 = BorderPoint;
      ComputeBorderPoint (Point4, 180, 70, BorderWidth, false)
      #local Point4 = BorderPoint;
      union {
         intersection {
            box {<Point1.x, BorderWidth, 0> <Point3.x, Point3.y, Depth>}
            plane {y, 0 rotate 70 * z translate Point1}
            plane {y, 0 inverse rotate 70 * z translate Point4}
            plane {y, 0 rotate 21.4 * z translate Point2}
            CornerPlane (Point1, 0, 70, CornerRadius, EdgeRadius, Depth)
            CornerPlane (Point2, -90 - 20, 21.4, CornerRadius * 4, EdgeRadius, Depth)
            CornerPlane (Point3, 180 + 21.4, -90 - 20, CornerRadius, EdgeRadius, Depth)
            CornerPlane (Point4, 180, 70, CornerRadius * 2, EdgeRadius, Depth)
         } //intersection
         RoundedPoint (Point1, 0, 70, CornerRadius, EdgeRadius, Depth)
         RoundedPoint (Point2, -90 - 20, 21.4, CornerRadius * 4, EdgeRadius, Depth)
         RoundedPoint (Point3, 180 + 21.4, -90 - 20, CornerRadius, EdgeRadius, Depth)
         RoundedPoint (Point4, 180, 70, CornerRadius * 2, EdgeRadius, Depth)
      } //union
   } //object

   #local N_Left = object {
      difference {
         object {N_Left}
         object {EdgeCutout rotate 0 * z translate Point1}
         object {EdgeCutout rotate 70 * z translate Point4}
         object {EdgeCutout rotate (180 + 21.4) * z translate Point3}
         object {EdgeCutout rotate (180 + 70) * z translate Point2}
      } //difference
   } //object

   #local N_Right = object {
      #local R1 = .05 + BorderWidth; //Inner radius
      #local R2 = .3 - BorderWidth; //Outer radius
      #local TailLength = 1.13;
      #local R56 = 1.9;
      #local R98 = 1.48;
      #local R513 = 1.35;
      #local Point5 = <0, 0, 0>;
      #local Point6 = <Point5.x + R56 * cos (radians (70)), Point5.y + R56 * sin (radians (70)), 0>;
      #local Point9 = <Point5.x + NWidth + .04, 0, 0>;
      #local Point8 = <Point9.x + R98 * cos (radians (70)), Point9.y + R98 * sin (radians (70)), 0>;
      ComputeIntersectionPoint (Point6, 21.4, Point8, 180 - 45)
      #local Point7 = IntersectionPoint;
      #local Point12 = <Point9.x + TailLength, 0, 0>;
      #local Point11 = <Point12.x - .63 / tan (radians (45)), .63, 0>;
      ComputeIntersectionPoint (Point9, 70, Point11, 180)
      #local Point10 = IntersectionPoint;
      #local Point13 = <Point5.x + R513 * cos (radians (70)), Point5.y + R513 * sin (radians (70)), 0>;
      ComputeBorderPoint (Point5, 0, 70, BorderWidth, false)
      #local Point5 = BorderPoint;
      ComputeBorderPoint (Point6, -90 -20, 21.4, BorderWidth, false)
      #local Point6 = BorderPoint;
      ComputeBorderPoint (Point7, 180 + 21.4, -45, BorderWidth, false)
      #local Point7 = BorderPoint;
      ComputeBorderPoint (Point8, 180 - 45, -90 - 20, BorderWidth, false)
      #local Point8 = BorderPoint;
      ComputeBorderPoint (Point9, 180, 70, BorderWidth, false)
      #local Point9 = BorderPoint;
      ComputeBorderPoint (Point10, 0, 70, BorderWidth, true)
      #local Point10 = BorderPoint;
      ComputeBorderPoint (Point11, 180, -45, BorderWidth, false)
      #local Point11 = BorderPoint;
      ComputeBorderPoint (Point12, 180, 180 - 45, BorderWidth, false)
      #local Point12 = BorderPoint;
      ComputeBorderPoint (Point13, -90 - 20, 21.4, BorderWidth, true)
      #local Point13 = BorderPoint;
      difference {
         intersection {
            box {<Point5.x, BorderWidth, 0> <Point8.x, Point7.y, Depth>}
            plane {y, 0 rotate 70 * z translate Point5}
            plane {y, 0 inverse rotate 70 * z translate Point9}
            plane {y, 0 rotate 21.4 * z translate Point6}
            plane {y, 0 inverse rotate (180 - 45) * z translate Point8}
         } //intersection
         object {EdgeCutout rotate (180 + 70) * z translate Point1}
         object {EdgeCutout rotate (180 - 45) * z translate Point8}
         object {EdgeCutout rotate 70 * z translate Point9}
      } //difference
   } //object

   #local N_Right = object {
      union {
         object {N_Right}
         difference {
            intersection {
               box {<Point5.x, Point6.y / 2, 0> <Point7.x, Point7.y, Depth>}
               plane {y, 0 rotate 21.4 * z translate Point6}
               plane {y, 0 inverse rotate 21.4 * z translate Point13}
            } //intersection
            object {EdgeCutout rotate 21.4 * z translate Point13}
         } //difference
      } //union
   } //object

   #local N_Right = object {
      difference {
         object {N_Right}
         object {EdgeCutout rotate (180 + 21.4) * z translate Point6}
      } //difference
   } //object

   #local N_Right = object {
      union {
         intersection {
            object {N_Right}
            CornerPlane (Point5, 0, 70, R2, EdgeRadius, Depth)
            CornerPlane (Point7, 180 + 21.4, -45, CornerRadius * 3, EdgeRadius, Depth)
            CornerPlane (Point8, 180 -45, -90 - 20, CornerRadius * 3, EdgeRadius, Depth)
         } //intersection
         RoundedPoint (Point5, 0, 70, R2, EdgeRadius, Depth)
         RoundedPoint (Point7, 180 + 21.4, -45, CornerRadius * 3, EdgeRadius, Depth)
         RoundedPoint (Point8, 180 -45, -90 - 20, CornerRadius * 3, EdgeRadius, Depth)
      } //union
   } //object

   #local N_RightTail = object {
      union {
         intersection {
            box {<Point9.x - EdgeRadius, Point9.y, 0> <Point12.x, Point11.y, Depth>}
            plane {y, 0 inverse rotate (180 - 45) * z translate Point12}
            CornerPlane (Point12, 180, 180 - 45, CornerRadius, EdgeRadius, Depth)
            CornerPlane (Point11, 180, -45, CornerRadius * 3, EdgeRadius, Depth)
         } //intersection
         RoundedPoint (Point12, 180, 180 - 45, CornerRadius, EdgeRadius, Depth)
         RoundedPoint (Point11, 180, -45, CornerRadius * 3, EdgeRadius, Depth)
      } //union
   } //object

   #local N_RightTail = object {
      difference {
         object {N_RightTail}
         object {EdgeCutout rotate (180 - 45) * z translate Point12}
         object {EdgeCutout rotate 180 * z translate Point11}
      } //difference
   } //object

   #local N_Right = object {
      difference {
         union {
            object {N_Right}
            object {N_RightTail}
         } //union
         object {EdgeCutout translate Point5}
      } //difference
   } //object

   #local N_Right = object {
      union {
         object {N_Right}
         RoundedCorner (Point10, 0, 70, R1, EdgeRadius, Depth)
      } //union
   } //object

   #object {
      union {
         object {N_Left}
         object {N_Right translate (NWidth + .05) * x}
      } //union
   } //object

#end //#macro Create_N
