// Persistence of Vision Ray Tracer Scene Description File
// File: subsurface.pov
// Vers: 3.7
// Desc: Subsurface Scattering Demo - Candles on a Checkered Plane
// Date: 2009-04-09
// Auth: John Valentine, modified from "Subsurface Scattering Demo - Candle on a Checkered Plane" by Christoph Lipka
//
// Recommended settings:
//  +W640 +H512 +A0.3
// Rendering time:
//  ~30 min on a 2.4GHz Intel Core 2 Q6600 (Quad Core)

#version 3.7;

#include "colors.inc"

global_settings {
  mm_per_unit 40
  subsurface { samples 400, 40 }
}

// ----------------------------------------

#declare From  = <-1.9,  3.1, -4.0>;    // Basic position for camera
#declare At    = < 0.3,  0.4,  0.0>;    // Camera target
#declare Dolly = 1.0;                   // >1.0 to move away; <1.0 to move in
#declare Zoom  = 1.0;                   // Change field of view while maintaining optical size of target

camera {
  location  At + (From-At) * Dolly * Zoom
  direction 1.5*z
  right     x*image_width/image_height
  look_at   At
  angle     55/Zoom
}

sky_sphere {
  pigment {
    gradient y
    color_map {
      [0.0 rgb <0.6,0.7,1.0>]
      [0.7 rgb <0.0,0.1,0.8>]
    }
  }
}

#declare Exposure = 0.6;

light_source {
  <-30, 30, -30>
  color rgb <1,1,1> * 0.6 * Exposure
}

// ----------------------------------------

// a checkered white marble / grey "plastic" plane
plane {
  y, -0.01
  texture {
    
    checker
    
    texture {
      pigment { color rgb 1 }
      finish{
        specular 1.0
        diffuse 0.4
        reflection { 0.2 }
      }
    }
    
    texture {
      pigment { color rgb 1 }
      finish{
        specular 0.2
        diffuse 0.3
        reflection { 0.2 }
        // marble physical parameters from Jensen et al. "A Practical Model for Subsurface Light Transport", Siggraph 2001
        // scaled by JV - this might be less authentic, but we needed to control the scattering to make the image work.
        subsurface { 0.07*<2.19,2.62,3.00>, 10*<0.0021, 0.0041, 0.0071> }
      }
    }
    
    scale 4
    translate <0.7,0,1>
  }
  interior { ior 1.5 }
}

// the classic chrome sphere
sphere { <0.7, 1, 1.6>, 1
  pigment { color rgb 1 }
  finish {
    ambient 0 diffuse 0
    specular 0.7  roughness 0.01
    reflection { 0.7 metallic }
  }
}

#declare Candle = 
  blob {
    threshold 0.5
    cylinder { <0.0, 0.0,  0.0>,
               <0.0, 2.0,  0.0>,  1.3,   1.0 } // candle "body"
    cylinder { <0.0, 1.0,  0.0>,
               <0.0, 2.0,  0.0>,  0.6,  -4.0 } // hole burned down the middle
    cylinder { <0.2, 0.8,  0.0>,
               <0.5, 2.0,  0.0>,  0.5,  -1.0 } // hole burned down one side
    sphere   { <0.0, 2.5,  0.0>,  0.8,  -1.0 } // (used to shape the candle top)
    sphere   { <0.0,-0.52, 0.0>,  0.8,  -2.0 } // (used to shape the candle bottom)
    sphere   { <0.0, 2.0, -0.5>,  0.1,  -0.2 } // dimple on inside of candle
    texture {
      pigment { color rgb 1 }
      finish{
        // ambient 0.0 diffuse 0.05
        specular 0.6 roughness 0.00081
        // "artificial" wax parameters
        subsurface { < 1.6732, 1.2806, 0.6947>,< 0.0005, 0.0018, 0.0131> }
      }
      // normal { bumps 0.35 scale 0.01 }
    }
    interior { ior 1.45 }
    rotate y*45
  }              

#declare Wick = 
  intersection {
    box { <-1,-1,-1>, <0,1,1> }
    torus { 0.15, 0.03 }
    rotate x*90
    translate <0.15, 0.8, 0.0>
    pigment { color rgb 0 }
    finish { ambient 0 diffuse 1 specular 0 }
  }


// First candle, lit from within
union {
  blob { Candle } 
  intersection { Wick }
  light_source {
    y*1.5         
    color rgb <1,1,0.6> * 4.4 * Exposure
  }
  rotate y*20
  translate <-1,0,1>
}

// Second candle, unlit
union {
  blob { Candle } 
  intersection { Wick }
  rotate y*110
  translate <1.0, 0.0, -0.5>
}