//
// ***** CLIFFORD ATTRACTOR *****
//
// DESCRIPTION:
// ------------ 
// Creates a union of spheres to form the 2D Clifford attractor along the z-axis in 3D space.
// Original source by Paul Nylander, modified by me to be more portable and (sort of) 3D, the
// original source can be found on his homepage "http://www.bugman123.com". The z-axis is
// controlled by a liniar eqation, so it's only 2 thirds chaotic.
//
// Definition - xn+1 = sin(a yn) + c cos(a xn)
//              yn+1 = sin(b xn) + d cos(b yn)
//
// USAGE:
// -----------
// clifford (a1, a2, b1, b2, c1, c2, d1, d2, n, inc, rad, col_map, fin, trans)
//   (Faked Z-axis)
//
// or clifford_2d (a1, a2, b1, b2, c1, c2, d1, d2, n, inc, rad, col_map, fin, trans)
//   (2D version)
//
// or clifford_sweep (a1, a2, b1, b2, c1, c2, d1, d2, n, inc, rad, col_map, fin, trans)
//   (3D version made from shere sweeps rather than individual spheres. This can take a very
//   long time to render, though parsing time isn't effected too badly)
//
//
// ?1 / ?2 - Seeds for the attractor, -2 to 2.
// n       - Number iterations, higher values result in more sphere's and longer parse-time.
//           Every 1 n creates a line of spheres, the density of which is controlled by "inc"
// inc     - Density of the spheres, 0-1, smaller values result in higher density and parse-time
// rad     - Radius of each individual sphere.
// mat     - The material used
// trans   - Transformation
//
// WARNING: Using very high values for 'n' or very low values fo 'inc' will result in an EXTREME parsing
//          time and the possibility of a crash if you run out of memory.
//
// EXAMPLE:
// -----------
// #declare cliff_mat = material { texture { pigment { rgbt <1, 1, 1, 0.8> } } }
// #declare cliff_trans = transform { scale 2 }
//
// clifford (1.6, 1.3, -0.6, 1.7, -1.2, 0.5, 1.6, 1.4, 100, 0.01, 0.02, cliff_mat, cliff_trans)
//

#macro clifford (a1, a2, b1, b2, c1, c2, d1, d2, n, inc, rad, mat, trans)
  
  #macro Interpolate(x1,x2) 2*cos(acos(x1/2)+p*(acos(x2/2)-acos(x1/2))) #end
  
  #local p = 0;
  
  #local clifford_shape = union
  {
  
    #while (p <= 1)
      
      #local Z = <0,0,0>;
      #local i = 0;
  
      #local a = Interpolate(a1, a2);
      #local b = Interpolate(b1, b2);
      #local c = Interpolate(c1, c2);
      #local d = Interpolate(d1, d2);
    
      #while (i < n)
    
        #local Z = <sin(a*Z.y)+c*cos(a*Z.x),sin(b*Z.x)+d*cos(b*Z.y),(p*2)-1>;
      
        sphere
        {
          Z, rad
        }

        #local i = i+1;

      #end

      #local p = p + inc;

    #end

  }

  #declare clifford_center = max_extent(clifford_shape) - min_extent(clifford_shape);

  object
  {
    clifford_shape
    material { mat }
    transform { trans }
    hollow
  }

#end

#macro clifford_2d (a1, a2, b1, b2, c1, c2, d1, d2, n, inc, rad, mat, trans)
  
  #macro Interpolate(x1,x2) 2*cos(acos(x1/2)+p*(acos(x2/2)-acos(x1/2))) #end
  
  #local p = 0;
  
  #local clifford_shape = union
  {
  
    #while (p <= 1)
      
      #local Z = <0,0,0>;
      #local i = 0;
  
      #local a = Interpolate(a1, a2);
      #local b = Interpolate(b1, b2);
      #local c = Interpolate(c1, c2);
      #local d = Interpolate(d1, d2);
    
      #while (i < n)
    
        #local Z = <sin(a*Z.y)+c*cos(a*Z.x),sin(b*Z.x)+d*cos(b*Z.y),0>;
      
        sphere
        {
          Z, rad
        }

        #local i = i+1;

      #end

      #local p = p + inc;

    #end

  }

  #declare clifford_center = max_extent(clifford_shape) - min_extent(clifford_shape);

  object
  {
    clifford_shape
    material { mat }
    transform { trans }
    hollow
  }

#end

#macro clifford_sweep (a1, a2, b1, b2, c1, c2, d1, d2, n, inc, rad, mat, trans)
  
  #macro Interpolate(x1,x2) 2*cos(acos(x1/2)+p*(acos(x2/2)-acos(x1/2))) #end
  
  #local p = 0;
  #local cur = 0;
  #local points = array[n+1][(1/inc)+2];

  #while (p <= 1)

    #local Z = <0,0,0>;
    #local i = 0;
    #local cur2 = 0;

    #local a = Interpolate(a1, a2);
    #local b = Interpolate(b1, b2);
    #local c = Interpolate(c1, c2);
    #local d = Interpolate(d1, d2);

    #while (i < n)

      #local Z = <sin(a*Z.y)+c*cos(a*Z.x),sin(b*Z.x)+d*cos(b*Z.y),(p*2)-1>;

      #local points[i][cur] = Z;

      #local i = i+1;

    #end

    #local p = p + inc;
    #local cur = cur + 1;

  #end

  #local cur = cur - 1;  
  #local i = 0;

  #while(i < n)
    sphere_sweep
    {
      cubic_spline
      (1/inc)+2,
      #local j = 0;
      #while(j < cur)
        #if(j = 0) points[i][j], rad, #end
        points[i][j], rad,
        #local j = j+1;
      #end
      points[i][j-1], rad
      material { mat }
      transform { trans }
      hollow
    }
    #local i = i+1;
  #end

#end

//
// ***** PETER DE JONG ATTRACTOR *****
//
// DESCRIPTION:
// ------------
// It works in the same was as the clifford attractor, but using Peter De Jong's equation. Except
// for the Interpolate macro(from Paul Nylander's Clifford attractor source), this my own work.
// Like the Clifford attractor, this is only 2 thirds chaotic because the Z axis is liniar.
//
// Definition - xn+1 = sin(a yn) - cos(b xn)
//              yn+1 = sin(c xn) - cos(d yn)
//
// USAGE:
// ------------
// dejong (a1, a2, b1, b2, c1, c2, d1, d2, n, inc, rad, col_map, fin, trans)
//   (Faked Z-Axis)
//
// or dejong_2d (a1, a2, b1, b2, c1, c2, d1, d2, n, inc, rad, col_map, fin, trans)
//   (2D version)
//
// or dejong_sweep (a1, a2, b1, b2, c1, c2, d1, d2, n, inc, rad, col_map, fin, trans)
//   (Sphere-sweep version)
//

#macro dejong (a1, a2, b1, b2, c1, c2, d1, d2, n, inc, rad, mat, trans)

  #macro Interpolate(x1,x2) 2*cos(acos(x1/2)+p*(acos(x2/2)-acos(x1/2))) #end

  #local p = 0;

  #local dejong_shape = union
  {

    #while (p <= 1)
      
      #local Z = <0,0,0>;
      #local i = 0;

      #local a = Interpolate(a1, a2);
      #local b = Interpolate(b1, b2);
      #local c = Interpolate(c1, c2);
      #local d = Interpolate(d1, d2);

      #while (i < n)

        #local Z = <sin(a*Z.y)-cos(b*Z.x), sin(c*Z.x)-cos(d*Z.y), (p*2)-1>;

        sphere
        {
          Z, rad
        }

        #local i = i+1;

      #end

      #local p = p + inc;

    #end

  }

  #declare dejong_center = max_extent(dejong_shape) - min_extent(dejong_shape);

  object
  {
    dejong_shape
    material { mat }
    transform { trans }
    hollow
  }

#end

#macro dejong_2d (a1, a2, b1, b2, c1, c2, d1, d2, n, inc, rad, mat, trans)

  #macro Interpolate(x1,x2) 2*cos(acos(x1/2)+p*(acos(x2/2)-acos(x1/2))) #end

  #local p = 0;

  #local dejong_shape = union
  {

    #while (p <= 1)
      
      #local Z = <0,0,0>;
      #local i = 0;

      #local a = Interpolate(a1, a2);
      #local b = Interpolate(b1, b2);
      #local c = Interpolate(c1, c2);
      #local d = Interpolate(d1, d2);

      #while (i < n)

        #local Z = <sin(a*Z.y)-cos(b*Z.x), sin(c*Z.x)-cos(d*Z.y), (p*2)-1>;

        sphere
        {
          Z, rad
        }

        #local i = i+1;

      #end

      #local p = p + inc;

    #end

  }

  #declare dejong_center = max_extent(dejong_shape) - min_extent(dejong_shape);

  object
  {
    dejong_shape
    material { mat }
    transform { trans }
    hollow
  }

#end

#macro dejong_sweep (a1, a2, b1, b2, c1, c2, d1, d2, n, inc, rad, mat, trans)
  
  #macro Interpolate(x1,x2) 2*cos(acos(x1/2)+p*(acos(x2/2)-acos(x1/2))) #end
  
  #local p = 0;
  #local cur = 0;
  #local points = array[n+1][(1/inc)+2];

  #while (p <= 1)

    #local Z = <0,0,0>;
    #local i = 0;
    #local cur2 = 0;

    #local a = Interpolate(a1, a2);
    #local b = Interpolate(b1, b2);
    #local c = Interpolate(c1, c2);
    #local d = Interpolate(d1, d2);

    #while (i < n)

      #local Z = <sin(a*Z.y)-cos(b*Z.x), sin(c*Z.x)-cos(d*Z.y), (p*2)-1>;

      #local points[i][cur] = Z;

      #local i = i+1;

    #end

    #local p = p + inc;
    #local cur = cur + 1;

  #end

  #local cur = cur - 1;  
  #local i = 0;

  #while(i < n)
    sphere_sweep
    {
      cubic_spline
      (1/inc)+2,
      #local j = 0;
      #while(j < cur)
        #if(j = 0) points[i][j], rad, #end
        points[i][j], rad,
        #local j = j+1;
      #end
      points[i][j-1], rad
      material { mat }
      transform { trans }
      hollow
    }
    #local i = i+1;
  #end

#end


//
// ***** PICKOVER ATTRACTOR *****
//
// DESCRIPTION:
// ------------
// Works the same as the previous attractors, using the pickover equation which was 3 dimentional
// in the first place, so I didn't have to fake the z-axis liniarly.
//
// Definition - xn+1 = sin(ay) - z cos(bx)
//              yn+1 = z sin(cx) - cos(dy)
//              zn+1 = sin(x)
//
// USAGE:
// ------------
// pickover (a1, a2, b1, b2, c1, c2, d1, d2, n, inc, rad, col_map, fin, trans)
//
// or pickover_sweep (a1, a2, b1, b2, c1, c2, d1, d2, n, inc, rad, col_map, fin, trans)
//   (Sphere-sweep version)
//

#macro pickover (a1, a2, b1, b2, c1, c2, d1, d2, n, inc, rad, mat, trans)
  
  #macro Interpolate(x1,x2) 2*cos(acos(x1/2)+p*(acos(x2/2)-acos(x1/2))) #end
  
  #local p = 0;
  
  #local pickover_shape = union
  {
  
    #while (p <= 1)
      
      #local Z = <0,0,0>;
      #local i = 0;
  
      #local a = Interpolate(a1, a2);
      #local b = Interpolate(b1, b2);
      #local c = Interpolate(c1, c2);
      #local d = Interpolate(d1, d2);
    
      #while (i < n)
    
        #local Z = <sin(a*Z.y)-Z.z*cos(b*Z.x), Z.z*sin(c*Z.x)-cos(d*Z.y), sin(Z.x)>;
      
        sphere
        {
          Z, rad
        }

        #local i = i+1;

      #end

      #local p = p + inc;

    #end

  }

  #declare pickover_center = max_extent(pickover_shape) - min_extent(pickover_shape);

  object
  {
    pickover_shape
    material { mat scale pickover_center }
    transform { trans }
    hollow
  }

#end

#macro pickover_sweep (a1, a2, b1, b2, c1, c2, d1, d2, n, inc, rad, mat, trans)
  
  #macro Interpolate(x1,x2) 2*cos(acos(x1/2)+p*(acos(x2/2)-acos(x1/2))) #end
  
  #local p = 0;
  #local cur = 0;
  #local points = array[n+1][(1/inc)+2];

  #while (p <= 1)

    #local Z = <0,0,0>;
    #local i = 0;
    #local cur2 = 0;

    #local a = Interpolate(a1, a2);
    #local b = Interpolate(b1, b2);
    #local c = Interpolate(c1, c2);
    #local d = Interpolate(d1, d2);

    #while (i < n)

      #local Z = <sin(a*Z.y)-cos(b*Z.x), sin(c*Z.x)-cos(d*Z.y), (p*2)-1>;

      #local points[i][cur] = Z;

      #local i = i+1;

    #end

    #local p = p + inc;
    #local cur = cur + 1;

  #end

  #local cur = cur - 1;  
  #local i = 0;

  #while(i < n)
    sphere_sweep
    {
      cubic_spline
      (1/inc)+2,
      #local j = 0;
      #while(j < cur)
        #if(j = 0) points[i][j], rad, #end
        points[i][j], rad,
        #local j = j+1;
      #end
      points[i][j-1], rad
      material { mat }
      transform { trans }
      hollow
    }
    #local i = i+1;
  #end

#end


//
// ***** CHAOTIC1 *****
//
// DESCRIPTION:
// ------------
// An attractor of my own creation, though it's simple eqaution, so I expect someone else
// has already used it at some point.
//
// Definition - xn+1 = cos(ay) - sin(bz)
//              yn+1 = sin(ax) - cos(by)
//              zn+1 = cos(x) * cos(y)
//
// USAGE:
// -----------
// chaotic1 (a1,a2, b1,b2, n, inc, rad, mat, trans)
//

#macro chaotic1 (a1, a2, b1, b2, n, inc, rad, mat, trans)
  
  #macro Interpolate(x1,x2) 2*cos(acos(x1/2)+p*(acos(x2/2)-acos(x1/2))) #end
  
  #local p = 0;
  
  #local chaotic_shape = union
  {
  
    #while (p <= 1)
      
      #local Z = <0,0,0>;
      #local i = 0;
  
      #local a = Interpolate(a1, a2);
      #local b = Interpolate(b1, b2);
    
      #while (i < n)
    
        #local Z = <cos(a*Z.y) - sin(b*Z.x), sin(a*Z.x) - cos(b*Z.y), cos(Z.x)*cos(Z.y)>;
      
        sphere
        {
          Z, rad
        }

        #local i = i+1;

      #end

      #local p = p + inc;

    #end

  }

  #declare chaotic_center = max_extent(chaotic_shape) - min_extent(chaotic_shape);

  object
  {
    chaotic_shape
    material { mat }
    transform { trans }
    hollow
  }

#end


//
// ***** CHAOTIC2 *****
//
// DESCRIPTION:
// ------------
// Another attractor of my creation, a litlle more complex than the previous one.
//
// Definition: xn+1 = (cos(y) + a) - b cos(z)
//             yn+1 = (cos(z) + b) - c cos(x)
//             zn+1 = (cos(x) + c) - a cos(y)
//
// USAGE:
// -----------
// chaotic2 (a1, a2, b1, b2, c1, c2, n, inc, rad, mat, trans)
//

#macro chaotic2 (a1, a2, b1, b2, c1, c2, n, inc, rad, mat, trans)
  
  #macro Interpolate(x1,x2) 2*cos(acos(x1/2)+p*(acos(x2/2)-acos(x1/2))) #end
  
  #local p = 0;
  
  #local chaotic_shape = union
  {
  
    #while (p <= 1)
      
      #local Z = <0,0,0>;
      #local i = 0;
  
      #local a = Interpolate(a1, a2);
      #local b = Interpolate(b1, b2);
      #local c = Interpolate(c1, c2);
    
      #while (i < n)

        #local Z = <(cos(Z.y)+a)-(b*cos(Z.z)), (cos(Z.z)+b)-(c*cos(Z.x)), (cos(Z.x)+c)-(a*cos(Z.y))>;
      
        sphere
        {
          Z, rad*2
        }

        #local i = i+1;

      #end

      #local p = p + inc;

    #end

  }

  #declare chaotic_center = max_extent(chaotic_shape) - min_extent(chaotic_shape);

  object
  {
    chaotic_shape
    material { mat }
    scale 0.5
    translate <0,0,-0.5>
    transform { trans }
    hollow
  }

#end
