// Persistence of Vision Ray Tracer POV-Ray scene file "wfc0.inc"
// POV-Ray 3.8
// Macros for processing a basic Wave Function Collapse algorithm
// Many of these macros use the global namespace
// (this may be bad practice)

/* Development History
00: First copy from wfc_dev18.pov into a separate file. Hopefully refactoring
    will be easier

*/

#include "math.inc"
#include "transforms.inc"

#declare Ures = res.u;
#declare Vres = res.v;

// Pigments for the Maketile Macro
#declare P_tileborder = pigment { color rgb 0.8 }
#declare P_clear = pigment { color rgbt 1 }

// Border is yesno to draw a border or not
#macro MakeTile(tile, border)
  #local tile_pigment = A_Pigments[tile["pidx"]]
  box { <0, 0, 0> <1, 1, 0>
  texture { 
    pigment { 
      tile_pigment 
      #if (tile["rot"] > 0)
        translate <-0.5, -0.5, 0>
        rotate 90*tile["rot"]*z
        translate <0.5, 0.5, 0>
      #end
      } 
    }
  #if(border)
    texture {
      pigment { 
        boxed 
        pigment_map { 
          [0.05 P_tileborder] 
          [0.05 P_clear] 
        }
        translate <1, 1, 0>
        scale 0.5
      } // pigment
    } // texture
  #end // if border
  } // box
#end

// Reverse a string 
#macro R(s)
  #local slen = strlen(s);
  #local reversed = ""
  #for(i,slen,1,-1)
    #local reversed = concat(reversed, substr(s,i,1))
  #end
  reversed
#end

// ----------- Create Tile
// tname - name of tile
// north, east, south, west - edge definitions
// pidx - pigment index from A_Pigments (usually imported)
// rot - rotation index (0-3)
// The created tile will have 4 extra items to store
// socket codes based on the edge definitions
#macro CreateTile(tname, north, east, south, west, pidx, rot)
  #local _tile = dictionary;
  #local _tile["name"] = tname;
  #local _tile["north"] = north;
  #local _tile["east"] = east;
  #local _tile["south"] = south;
  #local _tile["west"] = west;
  #local _tile["pidx"] = pidx;
  #local _tile["rot"] = rot;
  #local _tile["s_south"] = R(south);
  #local _tile["s_west"] = R(west);
  #local _tile["s_north"] = R(north);
  #local _tile["s_east"] = R(east);
  _tile
#end


// For each tile, determine if we need to rotate
// There is probably a faster algorithm for this
#macro CheckTileRotation(_tile)
  #local _rotate = 0;
  #if(_tile["north"] != _tile["east"]) 
    #local _rotate = 1; 
  #end
  #if(_tile["east"] != _tile["south"]) 
    #local _rotate = 1; 
  #end
  #if(_tile["south"] != _tile["west"]) 
    #local _rotate = 1; 
  #end
  _rotate
#end


// RotateTile returns a copy of the Tile with a new rotation
#macro RotateTile(_tile, rot)
  #local _edges = array[4] { _tile.north, _tile.east, _tile.south, _tile.west }
  #local _sockets = array[4] { _tile.s_north, _tile.s_east, _tile.s_south, _tile.s_west }
  #local _new_tile = dictionary {
    .name: concat(_tile.name, "_rot_", str(rot,0,0)),
    .north: _edges[mod(rot, 4)],
    .east: _edges[mod(rot+1, 4)],
    .south: _edges[mod(rot+2, 4)],
    .west: _edges[mod(rot+3, 4)],
    .pidx: _tile.pidx,
    .rot: rot,
    .s_north: _sockets[mod(rot, 4)],
    .s_east: _sockets[mod(rot+1, 4)],
    .s_south: _sockets[mod(rot+2, 4)],
    .s_west: _sockets[mod(rot+3, 4)]
  }
  _new_tile
#end


// FUTURE ENHANCEMENT - Update each tile in TileSet with acceptable neighbors
#macro FillTileSet()
  // MaxEntropy is the total number of tiles used in the tile set
  #local MaxEntropy = 0; 
  
  // We can't create a flexible array but we can determine the max size
  #declare TileSet = array[dimension_size(InputTiles,1)*4] 

  // Fill out the TileSet and track Max Entropy
  #for(I,0,dimension_size(InputTiles, 1)-1, 1)
    #if (CheckTileRotation(InputTiles[I]))
      #declare TileSet[MaxEntropy] = RotateTile(InputTiles[I], 0)
      #declare TileSet[MaxEntropy+1] = RotateTile(InputTiles[I], 1)
      #declare TileSet[MaxEntropy+2] = RotateTile(InputTiles[I], 2)
      #declare TileSet[MaxEntropy+3] = RotateTile(InputTiles[I], 3)
      #declare MaxEntropy = MaxEntropy + 4;
    #else
      #declare TileSet[MaxEntropy] = InputTiles[I]
      #local MaxEntropy = MaxEntropy + 1;
    #end
  #end // for I
  #debug concat("Max Entropy for TileSet is ", str(MaxEntropy, 0, 0))
  MaxEntropy
#end

// CreateGrid
// Establishes the global Grid and Possibilities objects
#macro CreateGrid()

  // Grid of Solved Cells
  #declare Grid = array[Ures][Vres];

  // Possibilities is a 3-dimensional array, the last being the length of MaxEntropy
  // Create the grid and mark every possibilty
  #declare Possibilities = array[Ures][Vres][MaxEntropy]
  #for(V, 0, Vres-1, 1)
    #for(U, 0, Ures-1, 1)
      #for(I, 0, MaxEntropy-1, 1)
        #declare Possibilities[U][V][I] = 1;
      #end // for I
    #end // for U
  #end // for V

#end



// returns a 2d array of entropy scores. 
// Entropy of 1 means the cell should be collapsed
// Entropy of more than one means not collapsed
// Entoropy of 0 means something has gone horribly wrong
#macro CalculateEntropies()
  #local _entropy = array[Ures][Vres]
  #for(V, 0, Vres-1, 1)
    #for(U, 0, Ures-1, 1)
      #local _score = 0;
      #for(I, 0, MaxEntropy-1,1)
        #if (Possibilities[U][V][I] = 1) #local _score = _score + 1; #end
      #end // for I
      #local _entropy[U][V] = _score;
    #end // for U
  #end // for V
  _entropy
#end


// figure out which tiles have a lower entropy than the known
// Must have Entropy and Possibilities defined before calling this
#macro GetLowestEntropyTiles()
  #local _highest_entropy = MaxEntropy+1;
  #local _current_cells = array; // undefined so we can add freely
  #for(U, 0, Ures-1, 1)
    #for(V, 0, Vres-1, 1)
      #if (Entropy[U][V] > 1) // skip anything that's resolved
        // calculate a new score
        #local _score = 0;
        #for(I, 0, MaxEntropy-1, 1)
          #if (Possibilities[U][V][I] = 1) #local _score = _score + 1; #end
        #end // for I
        // start over if we found a lower score
        #if (_score < _highest_entropy)
          #local _highest_entropy = _score;
          #undef _current_cells
          #local _current_cells = array; // clear the array
          #local _current_cells[0] = <U, V>; // seems easier to store a 2vec
        #elseif(_score = _highest_entropy)
          #ifndef(_current_cells[0])
            #local _current_cells[0] = <U, V>;
          #else
            #local _current_cells[dimension_size(_current_cells,1)] = <U,V>;
          #end
        #end
      
      #end
    #end // for V
  #end // for U
  _current_cells
#end


// can we highlight the lowest scores
// draws a dot in the cell under the number
// Can we dump this into the DrawEntropies macro?
#macro HighlightLowestScores()

  #local Ulen = MapMax.u - MapMin.u;
  #local Uscale = Ulen/Ures;
  
  #local Vlen = MapMax.v - MapMin.v;
  #local Vscale = Vlen/Vres;
  
  #for(I, 0, dimension_size(LowestScores,1)-1,1)
    #local Vhere = Interpolate(LowestScores[I].v, 0, Vres-1, MapMin.v, (MapMax.v - Vscale), 1);
    #local Uhere = Interpolate(LowestScores[I].u, 0, Ures-1, MapMin.u, (MapMax.u - Uscale), 1);
    sphere { <0, 0, 0> 0.1 
      scale <Ulen/Ures, Vlen/Vres, 0.1> 
      pigment { rgb <1,0,0> } 
      translate <Uhere+Uscale/2, Vhere+Vscale/4, 0>
      }
  
  #end // for I
#end // HighlightLowestScores

// Now we draw the grid either with a tile or a text object showing the entropy
#macro DrawEntropies(Tiles)
  // Calculate actual drawing space with the MapMin and MapMax global variables
  #local Ulen = MapMax.u - MapMin.u;
  #local Uscale = Ulen/Ures;

  #local Vlen = MapMax.v - MapMin.v;
  #local Vscale = Vlen/Vres;
  #for(V, 0, Vres-1, 1)
    #local Vhere = Interpolate(V, 0, Vres-1, MapMin.v, (MapMax.v - Vscale), 1);
    #for(U, 0, Ures-1, 1)
      #local Uhere = Interpolate(U, 0, Ures-1, MapMin.u, (MapMax.u - Uscale), 1);
      #ifdef(Tiles[U][V])
        object { MakeTile(TileSet[Tiles[U][V]], BorderTiles  ) 
          scale <Ulen/Ures, Vlen/Vres, 1> 
          translate <Uhere, Vhere, 0>}
      #else
        #local entropy = text { 
          ttf "arial.ttf", str(Entropy[U][V],0,0), 0.02, 0.0 // thickness, offset
          texture{ pigment{ color rgb 0 } } // end of texture
          scale (Vlen/2)/Vres
        } // end of text object ---------------------------------------------
        object { 
          entropy
          Center_Trans(entropy, x)
          Center_Trans(entropy, y)
          translate <Uhere+Uscale/2, Vhere+Vscale/2, -0.01>
          }

      #end
    #end // for U
  #end // for V
#end // macro DrawTiles



#macro ChooseATile(UV)
  // figure out which options are available to us
  #declare _local_options = array;
  #declare _index = 0;
  #for(I, 0, MaxEntropy-1, 1)
    #if(Possibilities[UV.u][UV.v][I] = 1)
      #declare _local_options[_index] = I;
      #declare _index = _index + 1;
    #end
  #end //for I
  
  #if(_index = 0)
    #local _chosen_one = -1;
  #else
    #local _chosen_one = _local_options[floor(rand(WFCSeed)*0.9999*dimension_size(_local_options,1))];
  #end
  _chosen_one
#end // ChooseATile

// Remove all other Possibilites for the neighbors of the collapsed cell

#macro CollapseCell(UV, Tidx)
  #for(I, 0, MaxEntropy-1, 1)
    #declare Possibilities[UV.u][UV.v][I] = ((I = Tidx)?1:0);
  #end // for I

  // Simplify Options northward

  #if(Tidx >= 0)
  #for(I, 0, MaxEntropy-1, 1)
    #if(UV.v < (Vres-1))
      #if(Entropy[UV.u][UV.v+1] > 1)
        #declare Possibilities[UV.u][UV.v+1][I] = ((TileSet[Tidx].north = TileSet[I].s_south)?(1 & Possibilities[UV.u][UV.v+1][I]):0);
      #end // if
    #end // if

  // Simplify Options eastward
    #if(UV.u < (Ures-1))
      #if(Entropy[UV.u+1][UV.v] > 1)
        #declare Possibilities[UV.u+1][UV.v][I] = ((TileSet[Tidx].east = TileSet[I].s_west)?(1 & Possibilities[UV.u+1][UV.v][I]):0);
      #end
    #end // eastward check

  // Simplify Options southward
    #if(UV.v > 0)
      #if(Entropy[UV.u][UV.v-1] > 1)
        #declare Possibilities[UV.u][UV.v-1][I] = ((TileSet[Tidx].south = TileSet[I].s_north)?(1 & Possibilities[UV.u][UV.v-1][I]):0);
      #end 
    #end // southward check

  // Simplify Options westward
    #if(UV.u > 0)
      #if(Entropy[UV.u-1][UV.v] > 1)
        #declare Possibilities[UV.u-1][UV.v][I] = ((TileSet[Tidx].west = TileSet[I].s_east)?(1 & Possibilities[UV.u-1][UV.v][I]):0);
      #end
    #end // westward check
  #end // for I
  #end // sanity check
#end // CollapseCell


#macro CollapseTheObvious()
  #for(U, 0, Ures-1, 1)
    #for(V, 0, Vres-1, 1)
      #if (Entropy[U][V] = 1)
        #ifndef(Grid[U][V])
          #local _option = ChooseATile(<U,V>);
          CollapseCell(<U,V>, _option)  
          #declare Grid[U][V] = _option;
        #end    
      #end
    #end // for V
  #end // for U
#end
