/*
 _________________________
      PrunedShrub.inc
 ~for POV-Ray version 3.7~
     2011 Samuel Benge
 
 
 Makes a spherical pruned shrub, in a pot.
 
 3.7 features:
  +radiosity sampling importance 0.2
  +backside illumination
 
 Outside Dependencies:
  1) UnderDish.inc
  2) FlowerPot.inc
 
 __________________________
 Potted_PrunedShrub(TreeSeed, Seg)
  *TreeSeed = a random seed (eg. seed(1019))

*/

#include "UnderDish.inc"
#include "FlowerPot.inc"

#macro PrunedShrub_Lerp(A,B,C) A*(1-C)+B*C #end

/* The pruning function.
   Function patterns seem to parse faster than pure functions, object functions or inside() tests. */
#declare PrunedShrub_Pruning = function{ pattern{ spherical scale 2 translate y*3.5  } }

#declare PrunedShrub_Leaf =
 sphere{0,1
  clipped_by{
   sphere{-z,1 scale<.2,.5,1>}
  }
  normal{planar 1 scallop_wave rotate z*90 scale .2}
  finish{specular .2 metallic .5 diffuse .65,.35 roughness 0.07}
  translate <0,.5,.865>
 }

#macro Set_Leaf() // leaf instance
 #if(V>Segments*.25 & PrunedShrub_Pruning(X,Y,Z)<.3 ) // leaves appear on upper levels
  #if(rand(PrunedShrubSeed)>.5 | V>Segments-5) /* average 0.5 leaves per-segment, force upper levels */
   object{
    PrunedShrub_Leaf
    pigment{rgbf<.02,.22+.5*V/Segments,.1,0.5>} // pigment instance
    translate y*max(Rad*Rad_Actual,.01)
    rotate x*( 90 + 30*(1-2*rand(PrunedShrubSeed)) )
    rotate y*720*rand(PrunedShrubSeed)
    scale Rad_Actual
    translate Pos
   }
  #end
 #end
#end

#macro PrunedShrub_Branching(Pos,Norm,Rad,Rad_Actual,Iter,Segments)
 
 #if(Iter>0)
  
  #local V = 0;
  #while(V<Segments)
   
   #local X = Pos.x; #local Y = Pos.y; #local Z = Pos.z;
   
   // enforce pruning, but only after the first 15 segments
   #if( (PrunedShrub_Pruning(X,Y,Z)<.1 & V>15) ) #local V = Segments; #end
   
   #local Norm = vnormalize( PrunedShrub_Lerp(Norm, <0,-pow(V/Segments,2),0>+1.25*(1-2*<rand(PrunedShrubSeed),rand(PrunedShrubSeed),rand(PrunedShrubSeed)>), 0.075));
   
   #local RadOld = Rad;
   #local Rad = Rad*(1-2/Segments);
   
   #local PosOld = Pos;
   #local Pos = Pos + Norm*(.03+.25*Rad);
   
   // branches
   sphere{Pos,max(Rad*Rad_Actual,.01)}
   #if(vlength(PosOld-Pos) != 0)
    cone{PosOld, max(RadOld*Rad_Actual,.01), Pos, max(Rad*Rad_Actual,.01)}
   #end
   
   Set_Leaf()
   
   #if(mod(V,3)=2)
    #local PosNew = Pos;
    
    #local NormNew = vnormalize(Norm+V*.1*(<1,0,1>-2*<rand(PrunedShrubSeed),0,rand(PrunedShrubSeed)>)+.5*(rand(PrunedShrubSeed)-.25));
    
    #local RadNew = Rad*.75;
    #local IterNew = Iter;
    #local SegmentsNew = Segments-V;
    
    PrunedShrub_Branching(PosNew, NormNew, RadNew, Rad_Actual, IterNew-1, SegmentsNew+20)
    
   #end
   
   #local V = V + 1;
  
  #end   // while
  
 #end  // Iter>0
 
#end // PrunedShrub_Branching

#macro Potted_PrunedShrub(PrunedShrubSeed)
 
 #local PrunedShrub_Noise = function{ pattern{ bumps translate z*rand(PrunedShrubSeed)*1000 scale 2} }
 
 union{
  
  // the shrub
  union{
   PrunedShrub_Branching(
    <0,.5,0>, // initial position
    <0,1,0>,  // initial direction (hanging +x = <.5,1,0>)
    .2,       // length
    .5,       // radius (length*radius = branch thickness)
    4,        // number of iterations
    120       // number of segments (affects length and branch count)
   )
   pigment{rgb<.9,.6,.5>} // branch pigment
   radiosity{importance 0.4}
  }
  
  // pot
  union{
   object{UnderDish scale <0.8,1,0.8>}
   object{FlowerPot}
   scale .85
   pigment{rgb<.6,.25,.15>}
   normal{granite .04 scale<30,.1,30> no_bump_scale translate y*rand(PrunedShrubSeed)*100}
  }
  
 }

#end
