/*
 __________________________
         Tree.inc
  -for POV-Ray version 3.7-
     2011 Samuel Benge
 
 Makes a basic tree...
 
 The tree object uses a radiosity sampling importance value of 0.2. The leaves use backside illumination.
 
 dependencies:
 
  1) UnderDish.inc
  2) FlowerPot.inc
 __________________________
 Potted_Tree(TreeSeed, Seg)
  
  *TreeSeed = a random seed (eg. seed(1004))
  *Seg      = number of segments, controls length, # branches (good values: between 40-70)

*/

#include "UnderDish.inc"
#include "FlowerPot.inc"

#macro Tree_Lerp(A,B,C) A*(1-C)+B*C #end

#declare Tree_Leaf =
 sphere{0,1
  clipped_by{
   sphere{-z,1 scale<.2,.5,1>}
  }
  pigment{rgb<.075,.3,.03>}
  normal{planar -.5 scallop_wave rotate z*90 scale .2}
  
  finish{specular .5 diffuse .6,.4}
  translate <0,.5,.8>
 }

#macro Tree_Branching(Pos,Norm,Rad,Rad_Actual,Iter,Segments)
 
 #if(Iter>0)
  
  #local V=0;
  #while(V<Segments)
   
   #local X=Pos.x; #local Y=Pos.y; #local Z=Pos.z;
   
   #local Norm = vnormalize( Tree_Lerp(Norm, <0,1,0>-.33*(Tree_Noise(X,Y,Z)-.5)*V+1.25*(1-2*<rand(TreeSeed),rand(TreeSeed),rand(TreeSeed)>), 0.025));
   
   #local RadOld = Rad;
   #local Rad=Rad*(1-2/Segments);
   
   #local PosOld = Pos;
   
   #local Pos = Pos + Norm*(.03+.25*Rad);
  
   sphere{Pos,Rad*Rad_Actual+.005}
   #if( vlength(PosOld-Pos)!=0 )
    cone{PosOld, RadOld*Rad_Actual+.005, Pos, Rad*Rad_Actual+.005}
   #end
   
   #if(V>Segments-65)
    #if(rand(TreeSeed)>.5)
     object{
      Tree_Leaf
      rotate x*(90+40*(1-2*rand(TreeSeed)))+y*720*rand(TreeSeed)
      scale Rad_Actual/2
      translate Pos
     }
    #end
   #end
   
   #if(mod(V,20)=2)
    #local PosNew = Pos;
    
    #local YRot = y*V*20;
    #local ZRot = z*90;
    
    #local NormNew = vnormalize(Norm+(<1,0,1>-2*<rand(TreeSeed),0,rand(TreeSeed)>)+(rand(TreeSeed)-.25));
    
    #local RadNew = Rad*.75;
    #local IterNew = Iter;
    #local SegmentsNew = Segments-V;
    
    Tree_Branching(PosNew, NormNew, RadNew, Rad_Actual, IterNew-1, SegmentsNew+20)
    
   #end
   
   #local V=V+1;
  
  #end // while
  
 #end // Iter>0
 
#end // Tree_Branching

#macro Potted_Tree(TreeSeed, Seg)
 
 #local Tree_Noise = function{ pattern{ bumps translate z*rand(TreeSeed)*1000 scale 2} }
 
 union{
  
  // the tree
  union{
   Tree_Branching(
    <0,0,0>, // initial position
    <0,1,0>, // initial direction
    .2,      // length
    .5,      // radius (length*radius = branch thickness)
    5,       // number of iterations
    Seg      // number of segments (affects length and branch count)
   )
   pigment{rgb<.9,.6,.5>} // branch pigment
   radiosity{importance 0.2}
  }
  
  // pot
  union{
   object{UnderDish}
   object{FlowerPot}
   scale .75
   pigment{rgb<.6,.25,.15>}
   normal{granite .04 scale<30,.1,30> no_bump_scale translate y*rand(TreeSeed)*100}
  }
  
 }

#end