/*
 
 OddTiles.inc
 
 2010 Samuel Benge

 
 This macro produces a pigment consisting of tightly-fitting rectangular
 tiles of irregular sizes. It can be used to make brick walls and such.


 The Macro
  
  OddTiles(push, mode)
   
   * push = amount of tile size variation, float (0.0 to 1.0)
   * mode = tile mode, int
      0 = basic cells 
      1 = uniform mortar
      2 = uniform mortar and cells combo 
      3 = scaled mortar
      4 = scaled mortar and cells combo
      5 = scaled superellipsoidal mortar
      6 = scaled superellipsoidal mortar and cells combo
   
   There are two methods from which stem all seven modes:
   
    modes 0, 1 & 2
   
     The first (and fastest) is the translation method. It uses basic translations to solve the
     pattern. The mortar it produces is of a uniform thickness across the entire pigment. Due to this
     method's nature, the mortar thickness is inversely proportionate to the amount of 'push' you pass
     to the macro. So if you set 'push' to its maximum value (1.0), there will be no mortar at all.
     Set 'push' to zero, and you will get a maximum mortar thickness, but no tile size variation.
   
    modes 3 through 6:
   
     The second is the scale method. It stretches a pigment at the cell level, allowing 1-unit areas
     to occupy each transformed tile. Because of this, special mortar styles and tile patterns may be
     devised, as long as the given pigment is tiled regularly with unit squares. It is automatically
     used when you declare a special pigment (see OddTiles_Tiles).


 Declarables
  
  Some of the following can break the pattern, and should be trusted only as far as your understanding
  of them...
  
  OddTiles_Repeat
   
   You can cause the pattern to repeat by declaring a 2D vector named 'OddTiles_Repeat'. When both
   integers are even numbers, you will get a square repetition. When both integers are odd numbers,
   you will get a diamond repetition which can also be tiled in a square fashion.
  
  OddTiles_Tiles
   
   Should a pigment entitled 'OddTiles_Tiles' be declared, the macro will kick into scaled mode. Then,
   if the given pigment just happens to be made up of 1-unit square portions, then each portion will
   be distorted to fit into its respective tile without errors. In other words, it's one way to
   provide a custom pattern (or mortar) for the tiles.
  
  OddTiles_Mask (not fully supported yet)
   
   If you declare a pigment named 'OddTiles_Mask', the macro will use it to control the tile size
   variation. This should be a pigment comprised of unit squares of solid color, such as a checker or
   cells pattern. You can use a bitmap for this, provided it has no interpolation and is scaled
   properly. For instance, if your image's resolution is 640x480, scale it to <640,480,1>. Each
   pixel's color will then affect how much the tiles get pushed around.
   
   This pigment will override the amount of push you pass to the macro, but only as far as the cells
   go. As for the mortar, you will have to adjust the 'push' term to accommodate the values you give
   this pigment. This only applies to modes 0, 1 & 2.
  
*/

#macro OddTiles(push, mode)
 
 #local push = max(0,min(1,abs(push)));
 #local mode = int(mode);
 
 #local Cells =
 pigment{
  #ifdef(OddTiles_Mask)
   OddTiles_Mask
  #else
   cells
  #end
  #ifdef(OddTiles_Repeat)
   warp{repeat x*int(OddTiles_Repeat.x)}
   warp{repeat y*int(OddTiles_Repeat.y)}
  #end
 }
 
 #local f_cells =
  function{
   pigment{
    #ifdef(OddTiles_Mask)
     OddTiles_Mask
    #else
     Cells
     color_map{[0 rgb .5-push/2][1 rgb .5+push/2]}
    #end
   }
  }
 
 
 #local f_visible =
  function{
   pigment{
    #ifdef(OddTiles_Tiles)
     OddTiles_Tiles
     #local mode = 3;
    #else
     pigment_pattern{
      #if(mode=0)
       Cells
      #else
       #if(mode=1|mode=2|mode=3|mode=4)
        boxed
       #end
       #if(mode=5|mode=6)
        function{ max(0,min(1,1-sqrt(pow(x,8)+pow(y,8)))) }
       #end
       translate (x+y)
       scale 1/2
       warp{repeat x}
       warp{repeat y}
      #end
     }
     #if(mode!=0)
      #if(mode=1)
       color_map{[0 rgb 0][1-push rgb 1]}
      #else
       #if(mode!=3&mode!=5)
        pigment_map{
         [0 rgb 0]
         [1-push*(mode=2) Cells]
        }
       #end
      #end
     #end
    #end
   }
  }

 #local f_bands = function(dir){ select( -dir, mod(dir,1), 1-mod(-dir,1) ) }
 
 pigment_pattern{
  checker

  #if(mode<=2) // translation mode
   
   pigment{
    function{
     select(
      f_bands(y)+f_cells(x,y,0).x-1,
      f_visible(x+f_cells(x,y-1,0).x,y-1+f_cells(x,y,0).x,0).x,
      f_visible(x+f_cells(x,y+1,0).x,y+f_cells(x,y,0).x-1,0).x
     )
    }
   }
   pigment{
    function{
     select(
      f_bands(x)+f_cells(x,y,0).x-1,
      f_visible(x+f_cells(x,y,0).x,y+f_cells(x-1,y,0).x-1,0).x,
      f_visible(x+f_cells(x,y,0).x,y+f_cells(x+1,y,0).x-1,0).x
     )
    }
   }
  
  #else       // scale mode (there must be a simpler way...)
   
   pigment{
    function{
     select(
      f_bands(y)+f_cells(x,y,0).x-1,
      
      select(
       f_bands(x)+f_cells(x,y-1,0).x-1,
       
       #local xlr = function{ (1-f_cells(x+1,y,0).x+f_cells(x,y-1,0).x) }
       #local xll = function{ (1-f_cells(x,y-1,0).x+f_cells(x-1,y,0).x) }
       #local yll = function{ (1-f_cells(x,y,0).x+f_cells(x-1,y-1,0).x) }
       #local ylr = function{ (1-f_cells(x,y,0).x+f_cells(x+1,y-1,0).x) }
       
       f_visible( (f_bands(x)-1+f_cells(x,y-1,0).x)/xll(x,y,0)+floor(x)+1, (f_bands(y)-1+f_cells(x,y,0).x)/yll(x,y,0)+floor(y), 0 ).x,
       f_visible( (f_bands(x)-1+f_cells(x,y-1,0).x)/xlr(x,y,0)+floor(x)+1, (f_bands(y)-1+f_cells(x,y,0).x)/ylr(x,y,0)+floor(y) ,0).x
      ),
      
      select(
       f_bands(x)+f_cells(x,y+1,0).x-1,
       
       #local xul = function{ (1-f_cells(x,y+1,0).x+f_cells(x-1,y,0).x) }
       #local yul = function{ (f_cells(x,y,0).x-f_cells(x-1,y+1,0).x+1) }
       #local xur = function{ (f_cells(x,y+1,0).x-f_cells(x+1,y,0).x+1) }
       #local yur = function{ (f_cells(x,y,0).x-f_cells(x+1,y+1,0).x+1) }
       
       f_visible( (f_bands(x)-1+f_cells(x,y+1,0).x)/xul(x,y,0)+floor(x)+1, (f_bands(y)-1+f_cells(x,y,0).x)/yul(x,y,0)+floor(y) ,0).x,
       f_visible( (f_bands(x)-1+f_cells(x,y+1,0).x)/xur(x,y,0)+floor(x)+1, (f_bands(y)-1+f_cells(x,y,0).x)/yur(x,y,0)+floor(y) ,0).x
      )
      
     )
    }
   }
   
   #undef xlr #undef xll #undef ylr #undef yll #undef xur #undef xul #undef yur #undef yul
   
   pigment{
    function{
     select(
      f_bands(x)+f_cells(x,y,0).x-1,
      
      select(
       f_bands(y)+f_cells(x-1,y,0).x-1,
       
       #local xll = function{ (1-f_cells(x,y,0).x+f_cells(x-1,y-1,0).x) }
       #local yll = function{ (1-f_cells(x-1,y,0).x+f_cells(x,y-1,0).x) }
       #local xul = function{ (1-f_cells(x,y,0).x+f_cells(x-1,y+1,0).x) }
       #local yul = function{ (-f_cells(x-1,y,0).x+f_cells(x,y+1,0).x-1) }
       
       f_visible( (f_bands(x)-1+f_cells(x,y,0).x)/xll(x,y,0)+floor(x)+1, (f_bands(y)-1+f_cells(x-1,y,0).x)/yll(x,y,0)+floor(y), 0 ).x,
       f_visible( (f_bands(x)-1+f_cells(x,y,0).x)/xul(x,y,0)+floor(x)+1, 1-(f_bands(y)-1+f_cells(x-1,y,0).x)/yul(x,y,0)+floor(y)-1, 0 ).x
       
      ),
      
      select(
       f_bands(y)+f_cells(x+1,y,0).x-1,
       
       #local xlr = function{ (f_cells(x,y,0).x-f_cells(x+1,y-1,0).x+1) }
       #local ylr = function{ (1-f_cells(x+1,y,0).x+f_cells(x,y-1,0).x) }
       #local yur = function{ (f_cells(x+1,y,0).x-f_cells(x,y+1,0).x+1) }
       #local xur = function{ (f_cells(x,y,0).x-f_cells(x+1,y+1,0).x+1) }
       
       f_visible( (f_bands(x)-1+f_cells(x,y,0).x)/xlr(x,y,0)+floor(x)+1, (f_bands(y)-1+f_cells(x+1,y,0).x)/ylr(x,y,0)+floor(y), 0 ).x,
       f_visible( (f_bands(x)-1+f_cells(x,y,0).x)/xur(x,y,0)+floor(x)+1, (f_bands(y)-1+f_cells(x+1,y,0).x)/yur(x,y,0)+floor(y), 0 ).x
      )
       
     )
    }
   }
  
  #end
 
 }
 
#end
