
#include "MainWindow.h"
#include "SplineEditorWidget.h"
#include "PovExporter.h"
#include "ProcessRunner.h"

#include <QMenuBar>
#include <QFileDialog>
#include <QSettings>
#include <QMessageBox>

MainWindow::MainWindow(QWidget* parent): QMainWindow(parent)
{
    m_editor = new SplineEditorWidget(this);
    setCentralWidget(m_editor);
    setWindowTitle("Qt + TinySpline + POV-Ray Starter");

    auto fileMenu = menuBar()->addMenu("&File");
    QAction* actExport = fileMenu->addAction("Export && Render…");
    QAction* actSetRenderer = fileMenu->addAction("Set POV-Ray Binary…");
    fileMenu->addSeparator();
    QAction* actQuit = fileMenu->addAction("Quit");

    connect(actExport, &QAction::triggered, this, &MainWindow::onExportAndRender);
    connect(actSetRenderer, &QAction::triggered, this, &MainWindow::onSetRenderer);
    connect(actQuit, &QAction::triggered, this, &QWidget::close);

    loadSettings();
}

void MainWindow::loadSettings(){
    QSettings s("QtTinySplinePov","Starter");
    m_rendererPath = s.value("renderer","{}").toString();
}

void MainWindow::saveSettings(){
    QSettings s("QtTinySplinePov","Starter");
    s.setValue("renderer", m_rendererPath);
}

void MainWindow::onSetRenderer(){
    QString filter;
#ifdef _WIN32
    filter = "Executables (*.exe);;All Files (*.*)";
#else
    filter = "povray (*)";
#endif
    QString f = QFileDialog::getOpenFileName(this, "Select POV-Ray executable", QString(), filter);
    if(!f.isEmpty()){
        m_rendererPath = f;
        saveSettings();
        QMessageBox::information(this, "Renderer", QString("Using: %1").arg(m_rendererPath));
    }
}

void MainWindow::onExportAndRender(){
    if(m_rendererPath.isEmpty()){
        QMessageBox::warning(this, "Renderer", "Set POV-Ray binary first (File → Set POV-Ray Binary…).");
        return;
    }
    QString dir = QFileDialog::getExistingDirectory(this, "Choose export folder");
    if(dir.isEmpty()) return;

    PovExporter exp;
    if(!exp.exportAll(dir, m_editor->points(), m_editor->isClosed())){
        QMessageBox::warning(this, "Export", "Failed to export POV files.");
        return;
    }

#ifdef _WIN32
    // pvengine.exe /RENDER file.pov /EXIT
    QStringList args;
    args << "/RENDER" << (dir+"/driver.pov") << "/EXIT";
    ProcessRunner::runDetached(m_rendererPath, args);
#else
    // povray +Idriver.pov +H360 +W640 +Oout.png +FN
    QStringList args;
    args << "+Idriver.pov" << "+H360" << "+W640" << "+FN";
    ProcessRunner::runDetached(m_rendererPath, args, dir);
#endif
}
