
#include "PovExporter.h"
#include <QFile>
#include <QTextStream>

static bool writeAll(const QString& path, const QString& txt){
    QFile f(path); if(!f.open(QIODevice::WriteOnly|QIODevice::Truncate|QIODevice::Text)) return false;
    QTextStream s(&f); s.setCodec("UTF-8"); s << txt; return true;
}

bool PovExporter::exportAll(const QString& dir, const QVector<QPointF>& pts, bool closed){
    if(!writeAll(dir+"/model_splines.inc", incText(pts,closed))) return false;
    if(!writeAll(dir+"/driver.pov", driverText())) return false;
    if(!writeAll(dir+"/quick.ini", iniText())) return false;
    return true;
}

QString PovExporter::incText(const QVector<QPointF>& pts, bool closed) const
{
    QString out;
    out += "// Generated by QtTinySplinePov Starter
";
    out += "#declare S_PROFILE_N = "+QString::number(pts.size()+ (closed?1:0))+";
";
    out += "#declare A1D_PROFILE = array[S_PROFILE_N]{
";
    for(int i=0;i<pts.size();++i){
        out += QString("  < %1, %2 >,")
               .arg(pts[i].x()/40.0,0,'f',6) // scale down to reasonable units
               .arg((height()? pts[i].y():pts[i].y())/40.0,0,'f',6);
        out += "
";
    }
    if(closed && pts.size()>0){
        out += QString("  < %1, %2 >
")
               .arg(pts[0].x()/40.0,0,'f',6)
               .arg(pts[0].y()/40.0,0,'f',6);
    } else {
        if(out.endsWith(",
")) out.chop(2), out+="
";
    }
    out += "};

";

    // Lathe expects 2D points in x-y, rotated about y
    out += "#declare OBJ_Lathe = lathe{
";
    out += "  cubic_spline
";
    out += "  S_PROFILE_N,
";
    out += "  #for(i,0,S_PROFILE_N-1) A1D_PROFILE[i] #end
";
    out += "  texture{ pigment{ color rgb <0.8,0.85,1> } finish{ diffuse 0.9 } }
";
    out += "}

";

    // Optional: a simple sphere_sweep along control points (unit radius)
    if(pts.size()>=2){
        out += "#declare OBJ_Path = sphere_sweep{ linear_spline "+QString::number(pts.size())+",
";
        for(int i=0;i<pts.size();++i){
            out += QString("  < %1, 0, %2 >, 0.05
").arg(pts[i].x()/40.0,0,'f',6).arg(pts[i].y()/40.0,0,'f',6);
        }
        out += "  texture{ pigment{ color rgb <0.2,0.8,0.6> } finish{ diffuse 0.9 } }
}";
    }
    out += "
";

    return out;
}

QString PovExporter::driverText() const
{
    return QString(
R"POV(
#include "colors.inc"
#include "model_splines.inc"

camera { location <0,-6,3> look_at <0,0,0> angle 35 }
light_source { <6,-10,8> rgb 1 }
background { color rgb <0.02,0.02,0.03> }

// ground
plane { y, -1 pigment{ color rgb <0.1,0.1,0.12> } finish{ diffuse 0.5 } }

object { OBJ_Lathe }
#ifdef(OBJ_Path)
object { OBJ_Path }
#end
)POV");
}

QString PovExporter::iniText() const
{
    return QString(
R"INI(
Input_File_Name=driver.pov
Width=640
Height=360
Antialias=On
Antialias_Depth=3
Output_File_Type=N
)INI");
}
