
// File: source/base/complex.hpp
#ifndef POV_BASE_COMPLEX_HPP
#define POV_BASE_COMPLEX_HPP

#include <cmath>
#include "base/types.h" // DBL

namespace pov {

struct Complex
{
    DBL r; DBL i;
    constexpr Complex() : r(0), i(0) {}
    constexpr explicit Complex(DBL R) : r(R), i(0) {}
    constexpr Complex(DBL R, DBL I) : r(R), i(I) {}

    Complex operator+() const { return *this; }
    Complex operator-() const { return Complex(-r,-i); }
};

inline Complex operator+(const Complex&a,const Complex&b){ return Complex(a.r+b.r, a.i+b.i);} 
inline Complex operator-(const Complex&a,const Complex&b){ return Complex(a.r-b.r, a.i-b.i);} 
inline Complex operator*(const Complex&a,const Complex&b){ return Complex(a.r*b.r - a.i*b.i, a.r*b.i + a.i*b.r);} 

inline Complex operator/(const Complex&a,const Complex&b){
    DBL ar=fabs(b.r), ai=fabs(b.i);
    if(ar>=ai){ DBL t=b.i/b.r; DBL d=b.r+b.i*t; return Complex((a.r+a.i*t)/d, (a.i-a.r*t)/d); }
    DBL t=b.r/b.i; DBL d=b.r*t+b.i; return Complex((a.r*t+a.i)/d, (a.i*t-a.r)/d);
}

inline Complex operator+(const Complex&z, DBL x){ return Complex(z.r+x, z.i);} 
inline Complex operator-(const Complex&z, DBL x){ return Complex(z.r-x, z.i);} 
inline Complex operator*(const Complex&z, DBL x){ return Complex(z.r*x, z.i*x);} 
inline Complex operator/(const Complex&z, DBL x){ return Complex(z.r/x, z.i/x);} 
inline Complex operator+(DBL x,const Complex&z){ return Complex(x+z.r, z.i);} 
inline Complex operator-(DBL x,const Complex&z){ return Complex(x-z.r, -z.i);} 
inline Complex operator*(DBL x,const Complex&z){ return Complex(x*z.r, x*z.i);} 
inline Complex operator/(DBL x,const Complex&z){ return Complex(x,0)/z;} 

inline DBL abs(const Complex&z){ return std::hypot(z.r, z.i);} 
inline DBL arg(const Complex&z){ return std::atan2(z.i, z.r);} 
inline Complex conj(const Complex&z){ return Complex(z.r, -z.i);} 

inline Complex cexp(const Complex&z){ DBL ex=std::exp(z.r); return Complex(ex*std::cos(z.i), ex*std::sin(z.i)); }
inline Complex clog(const Complex&z){ return Complex(std::log(abs(z)), arg(z)); }
inline Complex cpow(const Complex&z,const Complex&w){ Complex L=clog(z); return cexp( Complex(w.r*L.r - w.i*L.i, w.r*L.i + w.i*L.r) ); }
inline Complex csqrt(const Complex&z){ DBL m=abs(z); DBL s=std::sqrt((m+fabs(z.r))/2); DBL d=(s==0)?0:(z.i/(2*s)); return (z.r>=0)?Complex(s,d):Complex(d,(z.i>=0?s:-s)); }

inline Complex csin(const Complex&z){ return Complex(std::sin(z.r)*std::cosh(z.i), std::cos(z.r)*std::sinh(z.i)); }
inline Complex ccos(const Complex&z){ return Complex(std::cos(z.r)*std::cosh(z.i),-std::sin(z.r)*std::sinh(z.i)); }
inline Complex ctan(const Complex&z){ return csin(z)/ccos(z); }

inline Complex cis(DBL th){ return Complex(std::cos(th), std::sin(th)); }
inline Complex make_complex(DBL re, DBL im){ return Complex(re,im);} 

static const Complex I(0.0, 1.0);

} // namespace pov

#endif
