// Persistence of Vision Ray Tracer Include File
// File: param.inc
// Vers: 3.5
// Desc: An include file to generate a mesh2 of smooth_triangles 
//       with uv_vectors from a set of parametric functions. 
// Date: 2001-09-12
// Auth: Ingo Janssen

/*****************************************************************************************************
   Use :
   
   #include "param.inc" to your scene file.
   
   Set up a set of parametric functions, Fx, Fy and Fz. This can be done using
   the function(){} statement or using #macro().

   A unit sphere may look like this:
                                       #declare Fx = function(U,V){sin(U)*cos(V)}
                                       #declare Fy = function(U,V){cos(U)}
                                       #declare Fz = function(U,V){sin(U)*sin(V)}
   or using macros like this:
                                       #macro Fx(U,V) (sin(U)*cos(V)) #end
                                       #macro Fy(U,V) (cos(U))        #end
                                       #macro Fz(U,V) (sin(U)*sin(V)) #end 

   Using function is much faster. Using macros makes it possible to use float- and vector functions
   that are not supported in function(){}, for example trace().
   
   Next the boundaries of u and v have to be specified. #declare Umin, Umax, Vmin and Vmax.
   Umin=a; Umax=b specifies the range a <= u <= b. If you need a range a < u < b use
   #declare Umin=FromU(a); Umax=ToU(b);  FromV() and ToV() for the range of v.
   
   The resolution of the mesh is set with Iter_U and Iter_V, the amount of steps by which
   the u and v ranges are divided. The total amount of triangles calculated will be 
   Iter_U*Iter_V*2.
   
   The above mentioned u, v ranges are also used for the uv_mapping. The area of the
   texture from the x-y-plane that is mapped to the object is defined by the rectangle
   <Umin,Vmin>, <Umax,Vmax>.
   
   If you want to save the resulting mesh to a file, #declare WriteReadMesh("filename"). On the
   next runs the saved mesh will be used until the file is deleted, or the 'WriteReadMesh' 
   line is commented out.
   
   Finaly add Parametric(), to generate the mesh. The mesh is #declared as Surface and
   can be used in the scene using object{Surface ...}. Additionaly the vectors SurfMin and 
   SurfMax are #declared. They give the coordinates of the box that encloses the Surface.
   
   Trouble shooting:
   
   ** Parse Error: Normal vector in mesh smooth_triangle cannot be zero.
    Check if the u and v boundaries are specified the right way? Should all values be included,
   or should FromU() ToV() etc. be used?
   ** Parse Error: Expected 'numeric expression', undeclared identifier 'IND' found instead
    Actually the same error as above, but it occurred while reading a mesh from file, written
   while using erroneous uv boundaries. Delete the file.
   ** Parse Error: No matching } in 'smooth_triangle', camera found instead
    This happens while reading an unfinished mesh. Was parsing stopped while writing the mesh?
   Delete the file.
************************************************************************************************************/

#version 3.5;

#declare EPS=(1e-12);
#declare EPSNorm=(1e-14);

#declare FU=0; #declare TU=0;
#declare FV=0; #declare TV=0;

#macro FromU(N)#local N=(N+EPS); #declare FU=1; (N) #end
#macro ToU(N)  #local N=(N-EPS); #declare TU=1; (N) #end
#macro FromV(N)#local N=(N+EPS); #declare FV=1; (N) #end
#macro ToV(N)  #local N=(N-EPS); #declare TV=1; (N) #end

#declare RangeMM=function(Val,Rmin,Rmax,Min,Max){
   (((Val-Rmin)/(Rmax-Rmin))*(Max-Min)+Min)
}

#macro Parametric()
   #ifdef(WriteReadMesh) 
      #if (file_exists(WriteReadMesh))
         #debug concat("\n Parsing '",WriteReadMesh,"' from file\n")
         #include WriteReadMesh
         #declare CalculateSurface=0;
      #else
         #declare CalculateSurface=1;
         #declare ToFile=1;   
      #end
   #else 
      #declare CalculateSurface=1;
      #declare ToFile=0;
   #end
   #if(CalculateSurface>0)
      CalcSurf()
   #end
   #declare SurfMin=min_extent(Surface);
   #declare SurfMax=max_extent(Surface);
   #debug concat("\n SurfMin: <",vstr(3,SurfMin,", ",0,-1),">\n")
   #debug concat(" SurfMax: <",vstr(3,SurfMax,", ",0,-1),">\n")   
#end

#macro CalcNormal(P)                 //calculate the normal of a vertex,
   #local Un=U+(iU/10);              //based on two nearby points.
   #local Vn=V+(iV/10);
   #if(TU&Un>=Umax)
      #local Un=U+EPSNorm;
   #end
   #if(TV&Vn>=Vmax)         
      #local Vn=V+EPSNorm;
   #end
   #local N2=<(Fx(Un,V)),(Fy(Un,V)),(Fz(Un,V))>;
   #local N3=<(Fx(U,Vn)),(Fy(U,Vn)),(Fz(U,Vn))>;
   #local A=(N2-P);
   #local B=(N3-P);      
   #declare Norm=vnormalize(vcross(A,B));   
   (Norm)
#end

#macro CalcSurf()
   #ifndef(Iter_U) #declare Iter_U=20; #end
   #ifndef(Iter_V) #declare Iter_V=20; #end
   
   #declare iU=(Umax-Umin)/Iter_U;
   #declare iV=(Vmax-Vmin)/Iter_V;
   
   #declare NumVertices=(Iter_U+1)*(Iter_V+1);
   #declare NumFaces=Iter_U*Iter_V*2;
   #debug concat("\n Calculating ",str(NumVertices,0,0)," vertices for ", str(NumFaces,0,0)," triangles\n\n")
   #debug "\n"
   
   #local V_vec_Arr=array[NumVertices] 
   #local N_vec_Arr=array[NumVertices] 
   #local UV_vec_Arr=array[NumVertices]

   #local Ind=0;   
   #local I=0;  
   #while (I<Iter_V+1)
      #local V=RangeMM(I,0,Iter_V,Vmin,Vmax);
      #local J=0;
      #while (J<Iter_U+1)
         #local U=RangeMM(J,0,Iter_U,Umin,Umax);
         #local V_vec_Arr[Ind]=<(Fx(U,V)),(Fy(U,V)),(Fz(U,V))>;  
         #local N_vec_Arr[Ind]=CalcNormal(V_vec_Arr[Ind]);  
         #local UV_vec_Arr[Ind]=<U,V>;
         #local Ind=Ind+1;
         #local J=J+1;            
      #end
      #debug concat("\r Done ", str(Ind,0,0)," vertices : ",str(100*Ind/NumVertices,0,2)," %")
      #local I=I+1;
   #end

   #if (ToFile>0)
      WriteMesh2()
   #end

   // Spheres at each corner//
   #declare Wertices = union {
   #local I=0;
   #while (I<NumVertices)
     sphere{V_vec_Arr[I], 0.2}
     #local I=I+1; 
   #end
   }

   // Links?
   #declare Lynks = union {
   #local I=0;
   #while (I<Iter_V)
      #local J=0;
      #while (J<Iter_U)
         #local Ind=(I*Iter_U)+I+J;
         cylinder {V_vec_Arr[Ind], V_vec_Arr[Ind+1], 0.02}
         //cylinder {V_vec_Arr[Ind], V_vec_Arr[Ind+Iter_U+2], 0.01}
         cylinder {V_vec_Arr[Ind+1], V_vec_Arr[Ind+Iter_U+2], 0.02}
         //cylinder {V_vec_Arr[Ind+1], V_vec_Arr[Ind+Iter_U+1], 0.01}
         #local J=J+1;
      #end
      #local I=I+1;
   #end
   }

   #debug concat("\n\n Building mesh2 \n   vertex_vectors\n")   
   #declare Surface=mesh2 {
      #local I=0;
      vertex_vectors {
         NumVertices
         #while (I<NumVertices)
            V_vec_Arr[I]
            #local I=I+1; 
         #end
      }
      #debug concat("   normal_vectors\n")   
      #local I=0;
      normal_vectors {
         NumVertices
         #while (I<NumVertices)
            N_vec_Arr[I]
            #local I=I+1;
         #end
      }
      #debug concat("   uv_vectors\n")   
      #local I=0;
      uv_vectors {
         NumVertices
         #while (I<NumVertices)
            UV_vec_Arr[I]
            #local I=I+1;
         #end
      }
      #debug concat("   face_indices\n")   
      #local I=0;
      face_indices {
         NumFaces
         #local I=0;
         #while (I<Iter_V)
            #local J=0;
            #while (J<Iter_U)
               #local Ind=(I*Iter_U)+I+J;
               <Ind, Ind+1, Ind+Iter_U+2>, <Ind, Ind+Iter_U+1, Ind+Iter_U+2>
               #local J=J+1;
            #end
            #local I=I+1;
         #end
      }
   }
   
   #declare FU=0;   #declare TU=0;
   #declare FV=0;   #declare TV=0;
   #ifdef(WriteReadMesh)
      #undef WriteReadMesh
   #end
#end   

#macro WriteMesh2()
   #debug concat("\n\n Writing : '",WriteReadMesh,"'\n   vertex_vectors\n")
   #fopen MeshFile WriteReadMesh write
   #write(MeshFile, "#debug concat(\"\\n uv mapping area: <",str(Umin,0,-1),", ",str(Vmin,0,-1),">, <",str(Umax,0,-1),", ",str(Vmax,0,-1),">\\n\")\n\n")
   #write(
      MeshFile,
      "#declare Surface = mesh2 {\n",
      "  vertex_vectors {\n",
      "    ", NumVertices,"\n    "
   )
   #local I=0;
   #while (I<NumVertices)
      #write(MeshFile, V_vec_Arr[I])
      #local I=I+1;
      #if(mod(I,5)=0)
         #write(MeshFile,"\n    ")
      #end 
   #end 
   #write(MeshFile,"\n  }\n")
   #write(
      MeshFile,
      "  normal_vectors {\n",
      "    ", NumVertices,"\n    "
   )
   #debug concat("   normal_vectors\n")   
   #local I=0;
   #while (I<NumVertices)
      #write(MeshFile N_vec_Arr[I])
      #local I=I+1; 
      #if(mod(I,5)=0)
         #write(MeshFile,"\n    ")
      #end 
   #end
   #write(MeshFile,"\n  }\n")
   #write(
      MeshFile, 
      "  uv_vectors {\n",
      "    ", NumVertices,"\n    "
   )
   #debug concat("   uv_vectors\n")   
   #local I=0;
   #while (I<NumVertices)
      #write(MeshFile UV_vec_Arr[I])
      #local I=I+1; 
      #if(mod(I,5)=0)
         #write(MeshFile,"\n    ")
      #end 
   #end
   #write(MeshFile,"\n  }\n")
   #write(
      MeshFile,
      "  face_indices {\n"
      "    ",NumFaces,"\n    "
   )
   #debug concat("   face_indices\n")   
   #local I=0;
   #local H=0;
   #while (I<Iter_V)
      #local J=0;
      #while (J<Iter_U)
         #local Ind=(I*Iter_U)+I+J;
         #write(MeshFile,<Ind, Ind+1, Ind+Iter_U+2>, <Ind, Ind+Iter_U+1, Ind+Iter_U+2>)
         #local J=J+1;
         #local H=H+1;
         #if(mod(H,5)=0)
            #write(MeshFile,"\n    ")
         #end 
      #end
      #local I=I+1;
   #end
   #write(MeshFile, "\n  }\n}")
   #fclose MeshFile
   #debug concat(" Done writing\n")   
#end