// Persistence of Vision Ray Tracer Include File
// File: SweepSpline.inc
// For POV Version: 3.5
// Desc: Macro sweeping one spline along another
// SweepSpline Version: 3
// Date: 1-Mar-2004
// Auth: Mike Williams
// Based on an idea by Greg M. Johnson


// Requires makemesh.inc by Ingo Janssen <http://members.home.nl/seedseven/>
//
// Parameters
// Track        A spline that specifies the path along the object
//              The section of the spline between control points 0 and 1 will be used
// Shape        A spline that describes the cross section 
//              The section of the spline between control points 0 and 1 will be used
// Waist        A spline with x coordinates that specify the radius of the spheresweep
//              The section of the spline between control points 0 and 1 will be used
// U            The number of vertices along the path
// V            The number of vertices around the circumpherence
// FileName     The name of the file to whitch the mesh will be written. If is an
//              empty string (""), no file will be written. If a file name is given,
//              the macro will first check if it already exists. If that is so, it
//              will expect a mesh2 with the name "Surface" and try to parse the
//              existing file.
  
#include "makemesh.inc"

#macro FindPoint(su,sv)

     // Spline point and radius 
     #local P = Track(su);
     #local W = vlength(Waist(su).x);

     // Vertex position
     // To prevent flipping, calculate from the previous DRY vector 
     #local DRX = W*vnormalize(vcross(DR,DRY));
     #local DRY = W*vnormalize(vcross(DRX,DR));
     P + (Shape(sv).x)*DRX + (Shape(sv).y)*DRY;

#end


#macro SweepSpline(Track,Shape,Waist,U,V,Filename)
#if(strlen(Filename)>0)
  #if(file_exists(Filename))
    #debug concat("\n Parsing mesh2 from file: ", Filename, "\n")
    #local Build=0;
    #include Filename
     object{Surface}
  #else
    #local Build=1;
  #end
#else
  #local Build=1;
#end

#if (Build)
#local Verts = array[U*(V+1)]
#local Norms = array[U*(V+1)]
#local UV    = array[U*(V+1)]
                           
// Calculate the Vertexes, Normals and UV arrays
#local DRY = y; // Arbitrary initial vector that X will be perpendicular to
#local uu=0;
#while (uu<U)
  #local vv=0;
  #while (vv<=V)
     // UV information
     #local su = uu/U;
     #local sv = vv/V;
     #local UV[uu*(V+1)+vv] = <su,sv>;

     // Direction the spline is pointing
     #local DR = Track(su+0.001)-Track(su-0.001); 
     
     // Find some points
     #local P = FindPoint(su,sv);
     #local Verts[uu*(V+1)+vv] = P;
     
     #local Pu0=P; 
     #local Pu1 = FindPoint(su+0.001,sv);
     #if (vlength(Pu1-Pu0)=0)
       #local Pu1 = Pu0;
       #local Pu0 = FindPoint(su-0.001,sv);
     #end
     
     #local Pv0=P; 
     #local Pv1 = FindPoint(su,sv+0.001);
     #if (vlength(Pv1-Pv0)=0)
       #local Pv1 = Pv0;
       #local Pv0 = FindPoint(su,sv-0.001);
     #end
                        
     // calculate the normal                        
     #local Norms[uu*(V+1)+vv] = vcross(Pu1-Pu0,Pv1-Pv0);

     #local vv=vv+1;
  #end
  #local uu=uu+1;
#end

  BuildWriteMesh2(Verts, Norms, UV, V, U-1, Filename)

#end
#end
