
/**********************************************************************************
 Persistence of Vision Ray Tracer Scene Description File
 File name   : pylon.inc
 Version     : 3.6.1c / MegaPOV 1.21
 Description : 
               
 Date        : 2000
 Author      : Lance Purple
 E-mail      : http://objects.povworld.org/cat/Engineering/
 
 Copyright (C) 2000. All rights reserved.
 Use and/or modification of this scene is free, except for commercial purposes.
 Commercial distribution is not allowed without written permission from the author.
**********************************************************************************/

#declare INCHES = 1.0;
#declare FEET   = 12*INCHES;
#declare MILES  = 5280*FEET;

// =============================================================
// Steel Lattice 345 kv Transmission Tower
// =============================================================

// -------------------------------------------------------------
// Some basic constants
// -------------------------------------------------------------

#declare BASE_WIDTH  =  40*FEET;

#declare UPPER_HEIGHT =  90*FEET; // height where taper ends
#declare UPPER_WIDTH  =  10*FEET; // width at UPPER_HEIGHT
#declare TAPER_HEIGHT = 120*FEET; // height where tapered width --> 0

#declare BEAM_THICKNESS = 2*INCHES;

#declare CROSSARM_LENGTH = 20*FEET;
#declare CROSSARM_HEIGHT =  8*FEET;

#declare N_INSULATORS = 18;
#declare INSULATOR_HEIGHT = 8*INCHES;
#declare INSULATOR_STRING_LEN = (N_INSULATORS*INSULATOR_HEIGHT);
#declare INSULATOR_CABLE_THICKNESS = 1.0*INCHES;

#declare CONDUCTOR_THICKNESS = 1*FEET;

#declare FOOTING_HEIGHT = 3*FEET;
#declare FOOTING_RADIUS = 3*FEET;

// -------------------------------------------------------------
// Misc. textures
// -------------------------------------------------------------
#declare DullAluminum = texture {
   pigment { rgb < 0.85, 0.85, 0.85 > }
   finish  { phong .3 phong_size 10 metallic}
}

#declare DullConcrete = texture {
    pigment {
        spotted
        color_map {
            [ 0.0 color red 0.9843   green 0.9843   blue 0.96875  ]
            [ 0.8 color red 0.953125 green 0.953125 blue 0.921875 ]
            [ 0.8 color red 0.953125 green 0.953125 blue 0.921875 ]
            [ 1.0 color red 0.90625  green 0.90625  blue 0.90625  ]
        }
        turbulence 0.3
        scale <24, 24, 24>*INCHES
    }
}

#declare BlackCeramic = texture {
    pigment { color Black }
}

#declare RedCeramic = texture {
    pigment { color Red }
}

// -------------------------------------------------------------
// macro for looking up tower lattice coordinates
// -------------------------------------------------------------

#declare SW_CORNER=0;
#declare SE_CORNER=1;
#declare NW_CORNER=2;
#declare NE_CORNER=3;

#declare S_QUARTER_W=4;
#declare S_QUARTER_E=5;

#declare N_QUARTER_W=6;
#declare N_QUARTER_E=7;

#declare W_QUARTER_S=8;
#declare W_QUARTER_N=9;

#declare E_QUARTER_S=10;
#declare E_QUARTER_N=11;

#declare S_CENTER=12;
#declare N_CENTER=13;
#declare W_CENTER=14;
#declare E_CENTER=15;

#declare CENTER=16;

#declare quadrantOffsets = array[17]
{
    < -1.0,  0.0, -1.0 > // SW_CORNER
    <  1.0,  0.0, -1.0 > // SE_CORNER
    < -1.0,  0.0,  1.0 > // NW_CORNER
    <  1.0,  0.0,  1.0 > // NE_CORNER

    < -0.5,  0.0, -1.0 > // S_QUARTER_W
    <  0.5,  0.0, -1.0 > // S_QUARTER_E
    
    < -0.5,  0.0,  1.0 > // N_QUARTER_W
    <  0.5,  0.0,  1.0 > // N_QUARTER_E
    
    < -1.0,  0.0, -0.5 > // W_QUARTER_S
    < -1.0,  0.0,  0.5 > // W_QUARTER_N
    
    <  1.0,  0.0, -0.5 > // E_QUARTER_S
    <  1.0,  0.0,  0.5 > // E_QUARTER_N

    <  0.0,  0.0, -1.0 > // S_CENTER
    <  0.0,  0.0,  1.0 > // N_CENTER
    < -1.0,  0.0,  0.0 > // W_CENTER
    <  1.0,  0.0,  0.0 > // E_CENTER
    
    <  0.0,  0.0,  0.0 > // CENTER
}

#macro TC(height, qOffset)

    #if (height < UPPER_HEIGHT)
        #local halfWidth = (1.0-(height/TAPER_HEIGHT))*(BASE_WIDTH/2); 
    #else
        #local halfWidth = (UPPER_WIDTH/2);
    #end

    (quadrantOffsets[qOffset]*halfWidth) + <0,height,0>

#end

// -------------------------------------------------------------
// Macro for square cross-section box passing through two points
// -------------------------------------------------------------
#macro BoxThroughPts(p0, p1)

    #local bLen=vlength(p1-p0);
    #local bThick=BEAM_THICKNESS;
    #local vX=vnormalize(p1-p0);
    #local vY=vnormalize(vcross(y,vX));
    #local vZ1=vnormalize(vcross(y,vY));
    #local vZ2=vnormalize(vcross(vX,vY));
    box {
        < -bThick, 0, -bThick >  <  bThick, bLen, bThick >
        matrix <    0,vY.x,vZ1.x,     1,vY.y,vZ1.y,     0,vY.z, vZ1.z, 0,0,0 >
        matrix < vX.x,vX.y, vX.z,  vY.x,vY.y, vY.z, vZ2.x,vZ2.y,vZ2.z, 0,0,0 >
        translate p0
    }
#end

// -------------------------------------------------------------
// alternate form of above macro
// -------------------------------------------------------------
#macro BoxThrough(h0, qOff0, h1, qOff1)

    #local p0=TC(h0,qOff0);
    #local p1=TC(h1,qOff1);
    
    BoxThroughPts(p0, p1)
#end

// -------------------------------------------------------------
// Macro for cross-brace of diagonal beams
// -------------------------------------------------------------
#macro CrossBrace(h0, h1)

    BoxThrough(h0, SW_CORNER, h1, SE_CORNER)
    BoxThrough(h0, SE_CORNER, h1, SW_CORNER)
    BoxThrough(h0, NW_CORNER, h1, NE_CORNER)
    BoxThrough(h0, NE_CORNER, h1, NW_CORNER)
    BoxThrough(h0, SW_CORNER, h1, NW_CORNER)
    BoxThrough(h0, NW_CORNER, h1, SW_CORNER)
    BoxThrough(h0, SE_CORNER, h1, NE_CORNER)
    BoxThrough(h0, NE_CORNER, h1, SE_CORNER)

#end

// -------------------------------------------------------------
// Macro for ledge of horizontal beams
// -------------------------------------------------------------
#declare EWBeamStart = < 0, -BEAM_THICKNESS, -BEAM_THICKNESS >;
#declare EWBeamEnd   = < 0,  BEAM_THICKNESS,  BEAM_THICKNESS >;
#declare NSBeamStart = < -BEAM_THICKNESS, -BEAM_THICKNESS, 0 >;
#declare NSBeamEnd   = <  BEAM_THICKNESS,  BEAM_THICKNESS, 0 >;
#declare VBeamStart  = < -BEAM_THICKNESS, 0, -BEAM_THICKNESS >;
#declare VBeamEnd    = <  BEAM_THICKNESS, 0,  BEAM_THICKNESS >;

#macro Ledge(height)

    box { TC(height, SW_CORNER)+EWBeamStart
          TC(height, SE_CORNER)+EWBeamEnd }
    box { TC(height, NW_CORNER)+EWBeamStart
          TC(height, NE_CORNER)+EWBeamEnd }
    box { TC(height, SW_CORNER)+NSBeamStart
          TC(height, NW_CORNER)+NSBeamEnd }
    box { TC(height, SE_CORNER)+NSBeamStart
          TC(height, NE_CORNER)+NSBeamEnd }
#end

// -------------------------------------------------------------
// Macro for crossarm height at given horizontal coordinate
// -------------------------------------------------------------

#declare CA_SWU=0;
#declare CA_SWL=1;
#declare CA_SEU=2;
#declare CA_SEL=3;
#declare CA_NWU=4;
#declare CA_NWL=5;
#declare CA_NEU=6;
#declare CA_NEL=7;
#declare CA_WT =8;
#declare CA_ET =9;

#declare CAOffsets = array[10]
{
    < -1.0, 1.0, -1.0 > // CA_SWU
    < -1.0, 0.0, -1.0 > // CA_SWL
    <  1.0, 1.0, -1.0 > // CA_SEU
    <  1.0, 0.0, -1.0 > // CA_SEL
    < -1.0, 1.0,  1.0 > // CA_NWU
    < -1.0, 0.0,  1.0 > // CA_NWL
    <  1.0, 1.0,  1.0 > // CA_NEU
    <  1.0, 0.0,  1.0 > // CA_NEL
    < -1.0, 0.0,  0.0 > // CA_WT
    <  1.0, 0.0,  0.0 > // CA_ET
}

#macro CXP(hDist, qOffset)

    #local pWidth  = (1.0-(hDist/CROSSARM_LENGTH))*(UPPER_WIDTH/2);
    #local pHeight = (1.0-(hDist/CROSSARM_LENGTH))*(CROSSARM_HEIGHT);
    #local op      = CAOffsets[qOffset];
    < (op.x)*(hDist+(UPPER_WIDTH/2)), (op.y)*pHeight, (op.z)*pWidth >
#end

// -------------------------------------------------------------
// insulator object
// -------------------------------------------------------------
#declare Insulator = union
{
    sphere { <0,0,0>  1  scale <0.6, 0.2, 0.6>  translate <0, 0.2, 0>
             texture { BlackCeramic } }
    sphere { <0,0,0>  1  scale <0.2, 0.4, 0.2>  translate <0, 0.1, 0>
             texture { RedCeramic } }
    scale INSULATOR_HEIGHT
}

// -------------------------------------------------------------
// insulator strings
// -------------------------------------------------------------
#declare InsulatorString = union
{
    cylinder { <0,0,0>  <0, -INSULATOR_STRING_LEN, 0>
               INSULATOR_CABLE_THICKNESS/2
               texture { DullAluminum } }

    #declare i=0;
    #while (i < N_INSULATORS)
        object { Insulator  translate -i*INSULATOR_HEIGHT*y }
        #declare i=i+1;
    #end
}

// -------------------------------------------------------------
// cross-arms
// -------------------------------------------------------------
#declare Crossarm = union
{
    BoxThroughPts(CXP(CROSSARM_LENGTH, CA_WT), CXP(0*FEET, CA_SWU))
    BoxThroughPts(CXP(CROSSARM_LENGTH, CA_WT), CXP(0*FEET, CA_SWL))
    BoxThroughPts(CXP(CROSSARM_LENGTH, CA_WT), CXP(0*FEET, CA_NWU))
    BoxThroughPts(CXP(CROSSARM_LENGTH, CA_WT), CXP(0*FEET, CA_NWL))

    BoxThroughPts(CXP(CROSSARM_LENGTH, CA_ET), CXP(0*FEET, CA_SEU))
    BoxThroughPts(CXP(CROSSARM_LENGTH, CA_ET), CXP(0*FEET, CA_SEL))
    BoxThroughPts(CXP(CROSSARM_LENGTH, CA_ET), CXP(0*FEET, CA_NEU))
    BoxThroughPts(CXP(CROSSARM_LENGTH, CA_ET), CXP(0*FEET, CA_NEL))

    object { InsulatorString  translate CXP(CROSSARM_LENGTH, CA_WT)}
    object { InsulatorString  translate CXP(CROSSARM_LENGTH, CA_ET)}
             
    box {CXP(11*FEET, CA_SWU)+VBeamStart, CXP(11*FEET, CA_SWL)+VBeamEnd }
    box {CXP(11*FEET, CA_SWL)+VBeamStart, CXP(11*FEET, CA_NWL)+VBeamEnd }
    box {CXP(11*FEET, CA_NWL)+VBeamStart, CXP(11*FEET, CA_NWU)+VBeamEnd }
    box {CXP(11*FEET, CA_NWU)+VBeamStart, CXP(11*FEET, CA_SWU)+VBeamEnd }

    box {CXP(11*FEET, CA_SEU)+VBeamStart, CXP(11*FEET, CA_SEL)+VBeamEnd }
    box {CXP(11*FEET, CA_SEL)+VBeamStart, CXP(11*FEET, CA_NEL)+VBeamEnd }
    box {CXP(11*FEET, CA_NEL)+VBeamStart, CXP(11*FEET, CA_NEU)+VBeamEnd }
    box {CXP(11*FEET, CA_NEU)+VBeamStart, CXP(11*FEET, CA_SEU)+VBeamEnd }

    BoxThroughPts(CXP(11*FEET, CA_SWU), CXP(4*FEET, CA_SWL))
    BoxThroughPts(CXP(11*FEET, CA_NWU), CXP(4*FEET, CA_NWL))
    BoxThroughPts(CXP(11*FEET, CA_SEU), CXP(4*FEET, CA_SEL))
    BoxThroughPts(CXP(11*FEET, CA_NEU), CXP(4*FEET, CA_NEL))

    box {CXP(4*FEET, CA_SWU)+VBeamStart, CXP(4*FEET, CA_SWL)+VBeamEnd }
    box {CXP(4*FEET, CA_SWL)+VBeamStart, CXP(4*FEET, CA_NWL)+VBeamEnd }
    box {CXP(4*FEET, CA_NWL)+VBeamStart, CXP(4*FEET, CA_NWU)+VBeamEnd }
    box {CXP(4*FEET, CA_NWU)+VBeamStart, CXP(4*FEET, CA_SWU)+VBeamEnd }

    box {CXP(4*FEET, CA_SEU)+VBeamStart, CXP(4*FEET, CA_SEL)+VBeamEnd }
    box {CXP(4*FEET, CA_SEL)+VBeamStart, CXP(4*FEET, CA_NEL)+VBeamEnd }
    box {CXP(4*FEET, CA_NEL)+VBeamStart, CXP(4*FEET, CA_NEU)+VBeamEnd }
    box {CXP(4*FEET, CA_NEU)+VBeamStart, CXP(4*FEET, CA_SEU)+VBeamEnd }
    
    BoxThroughPts(CXP(4*FEET, CA_SWU), CXP(0*FEET, CA_SWL))
    BoxThroughPts(CXP(4*FEET, CA_NWU), CXP(0*FEET, CA_NWL))
    BoxThroughPts(CXP(4*FEET, CA_SEU), CXP(0*FEET, CA_SEL))
    BoxThroughPts(CXP(4*FEET, CA_NEU), CXP(0*FEET, CA_NEL))
}

// -------------------------------------------------------------
// warning sign
// -------------------------------------------------------------
#declare DangerSign = union
{
    box { < -3*FEET, -1.5*FEET, 0*INCHES >
          <  3*FEET,  1.5*FEET, 1*INCHES >
          texture { DullAluminum }
    }
    box { < -0.5, -0.5, -1 > < 0.5, 0.5, 0 >
        pigment {
            image_map {
                png "Third Party\Pylon\danger.png"
                map_type 0
                interpolate 2
            }
            translate < -0.5, -0.5, -1 >
        }
        finish { phong .6 }       
        scale < 6*FEET, 3*FEET, 1*INCHES>
    }
    rotate 9.46*x
    translate < 0, 0, -2 >*INCHES
}

// -------------------------------------------------------------
// anti-collision light
// -------------------------------------------------------------
#declare AntiCollisionLight = sphere
{
    <0,0,0>  6*INCHES   pigment { color Red }
}

// -------------------------------------------------------------
// tower lattice
// -------------------------------------------------------------

#declare TowerLattice = union
{
    BoxThrough(0*FEET, SW_CORNER, 90*FEET, SW_CORNER)
    BoxThrough(0*FEET, SE_CORNER, 90*FEET, SE_CORNER)
    BoxThrough(0*FEET, NW_CORNER, 90*FEET, NW_CORNER)
    BoxThrough(0*FEET, NE_CORNER, 90*FEET, NE_CORNER)

    BoxThrough(8*FEET, S_QUARTER_W, 12*FEET, SW_CORNER)
    BoxThrough(8*FEET, W_QUARTER_S, 12*FEET, SW_CORNER)
    BoxThrough(8*FEET, S_QUARTER_E, 12*FEET, SE_CORNER)
    BoxThrough(8*FEET, E_QUARTER_S, 12*FEET, SE_CORNER)
    BoxThrough(8*FEET, N_QUARTER_W, 12*FEET, NW_CORNER)
    BoxThrough(8*FEET, W_QUARTER_N, 12*FEET, NW_CORNER)
    BoxThrough(8*FEET, N_QUARTER_E, 12*FEET, NE_CORNER)
    BoxThrough(8*FEET, E_QUARTER_N, 12*FEET, NE_CORNER)
        
    BoxThrough(0*FEET, SW_CORNER, 15*FEET, S_CENTER)
    BoxThrough(0*FEET, SW_CORNER, 15*FEET, W_CENTER)
    BoxThrough(0*FEET, SE_CORNER, 15*FEET, S_CENTER)
    BoxThrough(0*FEET, SE_CORNER, 15*FEET, E_CENTER)
    BoxThrough(0*FEET, NW_CORNER, 15*FEET, N_CENTER)
    BoxThrough(0*FEET, NW_CORNER, 15*FEET, W_CENTER)
    BoxThrough(0*FEET, NE_CORNER, 15*FEET, N_CENTER)
    BoxThrough(0*FEET, NE_CORNER, 15*FEET, E_CENTER)

    BoxThrough(12*FEET, SW_CORNER, 15*FEET, S_CENTER)
    BoxThrough(12*FEET, SW_CORNER, 15*FEET, W_CENTER)
    BoxThrough(12*FEET, SE_CORNER, 15*FEET, S_CENTER)
    BoxThrough(12*FEET, SE_CORNER, 15*FEET, E_CENTER)
    BoxThrough(12*FEET, NW_CORNER, 15*FEET, N_CENTER)
    BoxThrough(12*FEET, NW_CORNER, 15*FEET, W_CENTER)
    BoxThrough(12*FEET, NE_CORNER, 15*FEET, N_CENTER)
    BoxThrough(12*FEET, NE_CORNER, 15*FEET, E_CENTER)

    BoxThrough(15*FEET, S_CENTER, 20*FEET, SW_CORNER)
    BoxThrough(15*FEET, S_CENTER, 20*FEET, SE_CORNER)
    BoxThrough(15*FEET, N_CENTER, 20*FEET, NW_CORNER)
    BoxThrough(15*FEET, N_CENTER, 20*FEET, NE_CORNER)
    BoxThrough(15*FEET, W_CENTER, 20*FEET, SW_CORNER)
    BoxThrough(15*FEET, W_CENTER, 20*FEET, NW_CORNER)
    BoxThrough(15*FEET, E_CENTER, 20*FEET, SE_CORNER)
    BoxThrough(15*FEET, E_CENTER, 20*FEET, NE_CORNER)
  
    BoxThrough(15*FEET, S_CENTER, 20*FEET, S_CENTER)
    BoxThrough(15*FEET, N_CENTER, 20*FEET, N_CENTER)
    BoxThrough(15*FEET, W_CENTER, 20*FEET, W_CENTER)
    BoxThrough(15*FEET, E_CENTER, 20*FEET, E_CENTER)
    
    object { DangerSign               translate TC(15*FEET, S_CENTER) }
    object { DangerSign rotate  90*y  translate TC(15*FEET, W_CENTER) }
    object { DangerSign rotate 180*y  translate TC(15*FEET, N_CENTER) }
    object { DangerSign rotate 270*y  translate TC(15*FEET, E_CENTER) }

    Ledge(20*FEET)

    BoxThrough(20*FEET, S_QUARTER_W, 29*FEET, S_QUARTER_W)
    BoxThrough(20*FEET, S_QUARTER_E, 29*FEET, S_QUARTER_E)
    BoxThrough(20*FEET, N_QUARTER_W, 29*FEET, N_QUARTER_W)
    BoxThrough(20*FEET, N_QUARTER_E, 29*FEET, N_QUARTER_E)
    BoxThrough(20*FEET, W_QUARTER_S, 29*FEET, W_QUARTER_S)
    BoxThrough(20*FEET, W_QUARTER_N, 29*FEET, W_QUARTER_N)
    BoxThrough(20*FEET, E_QUARTER_S, 29*FEET, E_QUARTER_S)
    BoxThrough(20*FEET, E_QUARTER_N, 29*FEET, E_QUARTER_N)

    BoxThrough(20*FEET, S_CENTER, 36*FEET, SW_CORNER)
    BoxThrough(20*FEET, S_CENTER, 36*FEET, SE_CORNER)
    BoxThrough(20*FEET, N_CENTER, 36*FEET, NW_CORNER)
    BoxThrough(20*FEET, N_CENTER, 36*FEET, NE_CORNER)
    BoxThrough(20*FEET, W_CENTER, 36*FEET, SW_CORNER)
    BoxThrough(20*FEET, W_CENTER, 36*FEET, NW_CORNER)
    BoxThrough(20*FEET, E_CENTER, 36*FEET, SE_CORNER)
    BoxThrough(20*FEET, E_CENTER, 36*FEET, NE_CORNER)
    
    CrossBrace(20*FEET, 48*FEET)
  
    Ledge(36*FEET)    

    CrossBrace(48*FEET, 66*FEET)

    Ledge(48*FEET)    

    BoxThrough(48*FEET, S_CENTER, 58*FEET, S_CENTER)
    BoxThrough(48*FEET, N_CENTER, 58*FEET, N_CENTER)
    BoxThrough(48*FEET, W_CENTER, 58*FEET, W_CENTER)
    BoxThrough(48*FEET, E_CENTER, 58*FEET, E_CENTER)
    
    Ledge(58*FEET)    

    CrossBrace(66*FEET, 80*FEET)
    
    Ledge(72*FEET)
    
    CrossBrace(80*FEET, 90*FEET)

    // -----------
    // Upper Tower
    // -----------  

    box { TC( 90*FEET, SW_CORNER)+VBeamStart
          TC(130*FEET, SW_CORNER)+VBeamEnd }
    box { TC( 90*FEET, SE_CORNER)+VBeamStart
          TC(130*FEET, SE_CORNER)+VBeamEnd }
    box { TC( 90*FEET, NW_CORNER)+VBeamStart
          TC(130*FEET, NW_CORNER)+VBeamEnd }
    box { TC( 90*FEET, NE_CORNER)+VBeamStart
          TC(130*FEET, NE_CORNER)+VBeamEnd }
    
    Ledge(90*FEET)
    CrossBrace(90*FEET, 98*FEET)
    Ledge(98*FEET)    
    CrossBrace(98*FEET, 104*FEET)
    Ledge(104*FEET)
    CrossBrace(104*FEET, 110*FEET)
    Ledge(110*FEET)
    CrossBrace(110*FEET, 118*FEET)
    Ledge(118*FEET)
    CrossBrace(118*FEET, 124*FEET)
    Ledge(124*FEET)
    CrossBrace(124*FEET, 130*FEET)
    Ledge(130*FEET)
   
    BoxThrough(130*FEET, SE_CORNER, 136*FEET, CENTER)
    BoxThrough(130*FEET, SW_CORNER, 136*FEET, CENTER)
    BoxThrough(130*FEET, NE_CORNER, 136*FEET, CENTER)
    BoxThrough(130*FEET, NW_CORNER, 136*FEET, CENTER)

    object { AntiCollisionLight  translate TC(136*FEET, CENTER) }
    
    // Cross-arms
    // ----------
    object { Crossarm translate  90*FEET*y }
    object { Crossarm translate 110*FEET*y }
    
    texture { DullAluminum }  
}

// -------------------------------------------------------------
// tower set upon footings
// -------------------------------------------------------------

#declare Footing = cylinder {
    <0,0,0> <0,FOOTING_HEIGHT,0> FOOTING_RADIUS
    texture { DullConcrete }
}

#declare TransmissionTower = union {

    object { TowerLattice  translate FOOTING_HEIGHT*y }
    object { Footing       translate TC(0.0, SE_CORNER) }
    object { Footing       translate TC(0.0, SW_CORNER) }
    object { Footing       translate TC(0.0, NE_CORNER) }
    object { Footing       translate TC(0.0, NW_CORNER) }
       
}

// -------------------------------------------------------------
// absolute coordinates of conductor attachment points
// -------------------------------------------------------------
#declare ConductorAttachments = array[4]
{
    (CXP(CROSSARM_LENGTH, CA_WT)+
        <0,( 90*FEET-INSULATOR_STRING_LEN+FOOTING_HEIGHT), 0>),
    (CXP(CROSSARM_LENGTH, CA_ET)+
        <0,( 90*FEET-INSULATOR_STRING_LEN+FOOTING_HEIGHT), 0>),
    (CXP(CROSSARM_LENGTH, CA_WT)+
        <0,(110*FEET-INSULATOR_STRING_LEN+FOOTING_HEIGHT), 0>),
    (CXP(CROSSARM_LENGTH, CA_ET)+
        <0,(110*FEET-INSULATOR_STRING_LEN+FOOTING_HEIGHT), 0>)
}

// -------------------------------------------------------------
// macro to string powerline between p0 and p1.
// uses parabolic arc instead of exact catenary
// -------------------------------------------------------------
#macro PowerLine(p0,p1,R,sag,n,pTexture)

    #if (p0.y != p1.y)
        #error "PowerLine start and end must have same y"
    #end
    
    #local k=1;
    #local pLast = p1;
    
    #while (k <= n)
        #local mu = k/n;
        #local dy = sag*4*mu*(1-mu);
        #local pk =
            < ((p0.x*mu) + (p1.x*(1-mu))),
                  (p0.y-dy), ((p0.z*mu) + (p1.z*(1-mu))) >;

            cylinder { pLast, pk, R  texture { pTexture } } 
        #local pLast=pk;
        #local k=k+1;
    #end // while
    
#end // macro PowerLine

// -------------------------------------------------------------
// macro to string all powerlines between a pair of towers
// -------------------------------------------------------------
#macro LinesConnecting(tower1Ctr, tower2Ctr, sag)

    PowerLine((ConductorAttachments[0]+tower1Ctr),
              (ConductorAttachments[0]+tower2Ctr),
               1*INCHES, sag, 32, DullAluminum)
    PowerLine((ConductorAttachments[1]+tower1Ctr),
              (ConductorAttachments[1]+tower2Ctr),
               1*INCHES, sag, 32, DullAluminum)
    PowerLine((ConductorAttachments[2]+tower1Ctr),
              (ConductorAttachments[2]+tower2Ctr),
               1*INCHES, sag, 32, DullAluminum)
    PowerLine((ConductorAttachments[3]+tower1Ctr),
              (ConductorAttachments[3]+tower2Ctr),
               1*INCHES, sag, 32, DullAluminum)

#end // macro LinesBetween