
// ============================================================================
// Photon Flowchart (POV-Ray SDL) — Customizable
// Version: 1.0 (2025-12-11)
// Author: M365 Copilot
// ----------------------------------------------------------------------------
// Purpose:
// 	Render an editable 2D flowchart that explains photon behavior in POV‑Ray.
// 	You can customize node positions, sizes, colors, and text labels with SDL keywords.
//
// Behavior reflected here (source‑verified):
// 	• global_settings { photons {...} } must be enabled to shoot photons.
// 	• Lights aim photons at objects with photons { target }, and merged flags decide
// 	  reflection/refraction effectiveness (ON & not OFF on both light and object).
// 	• Pre‑target pass_through panes allow photons to continue and tint them (PT_FILTER_BEFORE_TARGET);
// 	  non‑pass_through pre‑target hits block photons before the target.
// 	• Surface storage occurs unless collect off or ignore_photons; media storage requires
// 	  global_settings { photons { media steps[, factor] } } and container pass_through.
//
// References:
// 	• photons.cpp pre‑target filtering (PT_FILTER_BEFORE_TARGET): GitHub POV‑Ray source.
// 	• photonestimationtask.cpp merged flags & bail‑out: GitHub POV‑Ray source.
// 	• docs 3.6/3.7: Using Photon Mapping; Media & photons.
// ----------------------------------------------------------------------------
// Formatting aims to match user preferences:
// 	- Tabs for indentation
// 	- Capitalized identifiers with type prefixes S_/V_/A1D_
// 	- Spaces around operators
// 	- #for loops 0-based if used
// 	- Camera on -z axis, looking toward +z, orthographic, left-handed coords
// 	- Specular highlights (not phong) in materials
// 	- End-of-code marker
// ============================================================================

// ---------------------------
// Scene framing & style
// ---------------------------
#declare S_CANVAS_W = 28; // logical width units
#declare S_CANVAS_H = 44; // logical height units
#declare S_THICK = 0.06;  // z thickness for 2D shapes
#declare S_FONT = "cyrvetic"; // change to a local .ttf if preferred

camera {
	orthographic
	location < 0, 0, -10 >
	look_at  < 0, 0,  0 >
	right    x * S_CANVAS_W
	up       y * S_CANVAS_H
}

light_source {
	< -S_CANVAS_W, S_CANVAS_H, -20 >
	rgb < 1, 1, 1 >
}

#declare V_NODE_COLOR_BOX     = < 0.98, 0.98, 0.98 >;
#declare V_NODE_COLOR_DIAMOND = < 1.00, 0.95, 0.88 >;
#declare V_NODE_COLOR_NOTE    = < 0.95, 0.98, 1.00 >;
#declare V_EDGE_COLOR         = < 0.20, 0.20, 0.20 >;
#declare V_TEXT_COLOR         = < 0.00, 0.00, 0.00 >;
#declare S_TEXT_SCALE         = 0.45;  // adjust for your font
#declare S_TITLE_SCALE        = 0.55;
#declare S_ARROW_R            = 0.10;
#declare S_ARROW_HEAD_R       = 0.20;
#declare S_ARROW_HEAD_H       = 0.60;

#declare MAT_NODE = texture {
	pigment { rgb V_NODE_COLOR_BOX }
	finish { specular 0.2 roughness 0.02 }
}
#declare MAT_DIAMOND = texture {
	pigment { rgb V_NODE_COLOR_DIAMOND }
	finish { specular 0.2 roughness 0.02 }
}
#declare MAT_NOTE = texture {
	pigment { rgb V_NODE_COLOR_NOTE }
	finish { specular 0.2 roughness 0.02 }
}
#declare MAT_EDGE = texture {
	pigment { rgb V_EDGE_COLOR }
	finish { specular 0.1 roughness 0.02 }
}

// ---------------------------
// Helpers (nodes, diamonds, text, arrows)
// ---------------------------
#macro Flow_Text( S_STR, V_POS, S_SCALE )
	text { ttf S_FONT S_STR S_SCALE, 0
		pigment { rgb V_TEXT_COLOR }
		translate < V_POS.x, V_POS.y, 0 >
	}
#end

#macro Flow_BoxNode( V_POS, V_SIZE, S_TITLE, S_TEXT, MAT )
	// V_POS = center <x,y>, V_SIZE = <w,h>
	union {
		box { < -V_SIZE.x/2, -V_SIZE.y/2, -S_THICK/2 >, < V_SIZE.x/2, V_SIZE.y/2, S_THICK/2 >
			texture { MAT }
		}
		// Title (top line)
		Flow_Text( S_TITLE, < V_POS.x - V_SIZE.x/2 + 0.6, V_POS.y + 0.2, 0 >, S_TITLE_SCALE )
		// Body text (left aligned, wrap manually with \n)
		Flow_Text( S_TEXT, < V_POS.x - V_SIZE.x/2 + 0.6, V_POS.y - 1.0, 0 >, S_TEXT_SCALE )
		translate < V_POS.x, V_POS.y, 0 >
	}
#end

#macro Flow_DiamondNode( V_POS, V_SIZE, S_TITLE, S_TEXT, MAT )
	// rhombus via prism (2D diamond extruded to thickness)
	#local V_HALF = < V_SIZE.x/2, V_SIZE.y/2, 0 >;
	prism {
		linear_spline
		- S_THICK/2, + S_THICK/2,
		5,
		< 0,  V_HALF.y >,
		<  V_HALF.x, 0 >,
		< 0, -V_HALF.y >,
		< -V_HALF.x, 0 >,
		< 0,  V_HALF.y >
		texture { MAT }
		translate < V_POS.x, V_POS.y, 0 >
	}
	// Text
	Flow_Text( S_TITLE, < V_POS.x - V_SIZE.x/2 + 0.6, V_POS.y + 0.3, 0 >, S_TITLE_SCALE )
	Flow_Text( S_TEXT,  < V_POS.x - V_SIZE.x/2 + 0.6, V_POS.y - 0.9, 0 >, S_TEXT_SCALE )
#end

#macro Flow_Arrow( V_A, V_B, S_LABEL )
	#local V_DIR = V_B - V_A;
	#local S_LEN = sqrt( V_DIR.x*V_DIR.x + V_DIR.y*V_DIR.y );
	#local V_U   = < V_DIR.x/S_LEN, V_DIR.y/S_LEN, 0 >;
	// Shaft
	cylinder { < V_A.x, V_A.y, 0 >, < V_B.x - V_U.x*S_ARROW_HEAD_H, V_B.y - V_U.y*S_ARROW_HEAD_H, 0 >, S_ARROW_R
		texture { MAT_EDGE }
	}
	// Arrow head
	cone { < V_B.x - V_U.x*S_ARROW_HEAD_H, V_B.y - V_U.y*S_ARROW_HEAD_H, 0 >, S_ARROW_HEAD_R,
		   < V_B.x, V_B.y, 0 >, 0
		texture { MAT_EDGE }
	}
	// Label near middle
	#local V_MID = < (V_A.x + V_B.x)/2, (V_A.y + V_B.y)/2, 0 >;
	Flow_Text( S_LABEL, < V_MID.x - 0.8, V_MID.y + 0.8, 0 >, 0.40 )
#end

// ---------------------------
// Layout (you can edit positions & sizes)
// ---------------------------
#declare V_ROW_START      =  < 14, 40, 0 >;
#declare V_ROW_DECISION1  =  < 14, 36, 0 >;
#declare V_ROW_LIGHTS     =  < 18, 32, 0 >;
#declare V_ROW_OBJECTS    =  < 18, 28, 0 >;
#declare V_ROW_PRE        =  < 18, 24, 0 >;
#declare V_ROW_PT         =  < 10, 20, 0 >;
#declare V_ROW_BLOCK      =  < 26, 20, 0 >;
#declare V_ROW_TARGET     =  < 12, 16, 0 >;
#declare V_ROW_SHOOT      =  < 22, 12, 0 >;
#declare V_ROW_STORE      =  < 12, 8,  0 >;
#declare V_ROW_POST       =  < 14, 4,  0 >;

// Start
Flow_BoxNode( V_ROW_START, < 6.0, 3.0, 0 >,
	"Start — Photon Pass",
	"Scene pre‑process",
	MAT_NODE )

// Decision: global enable
Flow_DiamondNode( V_ROW_DECISION1, < 12.0, 4.5, 0 >,
	"Enable photons",
	"Is `global_settings { photons {...} }` present?",
	MAT_DIAMOND )
Flow_Arrow( < 14, 38.5, 0 >, < 14, 37.0, 0 >, "" )

// NO branch
Flow_BoxNode( < 6, 32.5, 0 >, < 9.5, 3.8, 0 >,
	"NO",
	"No photon pass; render without photon map",
	MAT_NODE )
Flow_Arrow( < 10, 35.0, 0 >, < 8, 33.6, 0 >, "NO" )

// YES branch → lights
Flow_BoxNode( V_ROW_LIGHTS, < 10.5, 4.5, 0 >,
	"YES → For each Light",
	"`photons { reflection on|off, refraction on|off }`\n(light) + types (e.g., `parallel`)",
	MAT_NODE )
Flow_Arrow( < 16, 35.5, 0 >, < 18, 33.0, 0 >, "YES" )

// Objects/target
Flow_BoxNode( V_ROW_OBJECTS, < 10.5, 4.5, 0 >,
	"For each Object",
	"Is `photons { target }` set?\nMay also have `collect off`, `ignore_photons`",
	MAT_NODE )
Flow_Arrow( < 18, 31.0, 0 >, < 18, 29.4, 0 >, "" )

// Non‑target note
Flow_BoxNode( < 6, 26.6, 0 >, < 12.0, 5.4, 0 >,
	"NO (not target)",
	"Light does not aim DIRECT photons here.\nObject can still SHOW caustics unless `collect off`.\nIt may pass/reflect/refract INDIRECT photons if\n`photons { reflection/refraction on }` are set.",
	MAT_NOTE )
Flow_Arrow( < 14, 29.4, 0 >, < 10.5, 27.6, 0 >, "NO" )

// Pre‑target decision
Flow_DiamondNode( V_ROW_PRE, < 13.0, 5.0, 0 >,
	"On path to target",
	"Hit PRE‑TARGET object? Is it `photons { pass_through }`?",
	MAT_DIAMOND )
Flow_Arrow( < 18, 27.0, 0 >, < 18, 25.4, 0 >, "YES" )

// Pass_through branch
Flow_BoxNode( V_ROW_PT, < 12.0, 5.0, 0 >,
	"pass_through",
	"CONTINUE tracing; photon may be TINTED\n(PT_FILTER_BEFORE_TARGET)",
	MAT_NODE )
Flow_Arrow( < 16, 24.0, 0 >, < 12, 21.0, 0 >, "pass_through" )

// Block branch
Flow_BoxNode( V_ROW_BLOCK, < 12.0, 5.0, 0 >,
	"not pass_through",
	"STOP tracing; photon path is BLOCKED",
	MAT_NODE )
Flow_Arrow( < 20, 24.0, 0 >, < 26, 21.0, 0 >, "not pass_through" )

// At target: merged flags
Flow_BoxNode( V_ROW_TARGET, < 14.5, 6.5, 0 >,
	"AT TARGET (light + object)",
	"Effective Reflection ⇔ both `reflection on` and NOT `reflection off`\nEffective Refraction ⇔ both `refraction on` and NOT `refraction off`",
	MAT_NODE )
Flow_Arrow( < 12, 21.0, 0 >, < 12, 17.4, 0 >, "" )
Flow_Arrow( < 26, 21.0, 0 >, < 16, 17.2, 0 >, "" )

// Shoot decision
Flow_DiamondNode( V_ROW_SHOOT, < 10.5, 4.6, 0 >,
	"Photon SHOOT?",
	"(Reflection OR Refraction effective)",
	MAT_DIAMOND )
Flow_Arrow( < 17, 15.0, 0 >, < 22, 13.0, 0 >, "" )

// No‑shoot
Flow_BoxNode( < 26, 10.0, 0 >, < 10.5, 4.0, 0 >,
	"NO",
	"No photons shot for this light → target pair",
	MAT_NODE )
Flow_Arrow( < 24, 12.0, 0 >, < 26, 11.2, 0 >, "NO" )

// Yes‑shoot → storage
Flow_BoxNode( V_ROW_STORE, < 16.0, 6.0, 0 >,
	"YES → On hit: STORE",
	"SURFACE: store unless `collect off` or `ignore_photons`\nMEDIA: requires `global_settings { photons { media steps[, factor] } }`\nAND container `photons { pass_through }`",
	MAT_NODE )
Flow_Arrow( < 20, 11.0, 0 >, < 14, 9.0, 0 >, "YES" )

// Post‑target
Flow_BoxNode( V_ROW_POST, < 20.0, 5.0, 0 >,
	"POST‑TARGET propagation",
	"Reflect if reflection effective; refract if refraction effective.\nIndirect photons may continue through non‑targets with\n`photons { reflection/refraction on }`.",
	MAT_NOTE )
Flow_Arrow( < 14, 8.0, 0 >, < 16, 5.6, 0 >, "" )

// ***end of code***
