
// ============================================================================
// POV-Ray Photon Behavior Analyzer (SDL)
// Version: 1.0 (2025-12-11)
// Author: M365 Copilot
// ----------------------------------------------------------------------------
// Purpose:
// 	Provide macros to register user light sources and objects, then compute
// 	and report the effective photon behavior (shoot/no-shoot, reflection/refraction,
// 	collect, pass_through, ignore_photons, target) to the #debug stream.
//
// How it works:
// 	POV-Ray SDL cannot introspect built-in photon flags from arbitrary scene items.
// 	Therefore, you register each light and object with the analyzer, mirroring the
// 	flags you use in your scene's photons{...} blocks. The analyzer then emulates
// 	the merged-flag logic implemented in POV-Ray's source modules and prints plain
// 	text diagnostics that describe the expected behavior.
//
// References for behavior (consulted while designing the logic):
// 	- Core photon tracing & pre-target pass_through handling:
// 		 source/core/lighting/photons.cpp (PT_FILTER_BEFORE_TARGET)
// 	- Merged flags and photon estimation bails:
// 		 source/backend/lighting/photonestimationtask.cpp (LightTargetCombo)
// 	- Official docs on Using Photon Mapping & Media:
// 		 https://www.povray.org/documentation/view/8.2.0/425/
// 		 https://www.povray.org/documentation/view/3.60/101/
// ----------------------------------------------------------------------------
// Formatting style honors user preferences:
// 	- Tabs for indentation
// 	- Identifiers capitalized, type prefixes S_/V_/A1D_
// 	- Spaces around operators
// 	- #for loops with 0-based bounds
// 	- Specular (not phong) if finish appears in examples
// 	- End-of-code marker
// ============================================================================

// ---------------------------
// Guards & core state
// ---------------------------
#declare S_LIGHT_COUNT = 0;
#declare S_OBJECT_COUNT = 0;
#declare A1D_LIGHTS = array;
#declare A1D_OBJECTS = array;

// Global photon settings (user-provided mirror of global_settings photons)
#declare S_GLOBAL_PHOTONS_ENABLED = 1; // 1:on, 0:off
#declare S_GLOBAL_SURFACE_SEPARATION = 0.01; // analyzer hint only
#declare S_GLOBAL_MEDIA_STEPS = 0; // 0 disables media photons
#declare S_GLOBAL_MEDIA_FACTOR = 0.00;

// ---------------------------
// Registration macros
// ---------------------------
// Light flags mirror photons{...} on lights:
// 	S_RFL_ON, S_RFL_OFF, S_RFR_ON, S_RFR_OFF -> 0 or 1
// 	S_PARALLEL -> 0 or 1 (for direction model hints)
#macro PH_Register_Light( S_ID, S_NAME, S_RFL_ON, S_RFL_OFF, S_RFR_ON, S_RFR_OFF, S_PARALLEL )
	#local A_LIGHT = array[7];
	#local A_LIGHT[0] = S_ID;
	#local A_LIGHT[1] = S_NAME;
	#local A_LIGHT[2] = S_RFL_ON;
	#local A_LIGHT[3] = S_RFL_OFF;
	#local A_LIGHT[4] = S_RFR_ON;
	#local A_LIGHT[5] = S_RFR_OFF;
	#local A_LIGHT[6] = S_PARALLEL;
	#declare A1D_LIGHTS[S_LIGHT_COUNT] = A_LIGHT;
	#declare S_LIGHT_COUNT = S_LIGHT_COUNT + 1;
#end

// Object flags mirror photons{...} on objects:
// 	S_TARGET, S_RFL_ON, S_RFL_OFF, S_RFR_ON, S_RFR_OFF, S_COLLECT_OFF,
// 	S_PASS_THROUGH, S_IGNORE_PHOTONS -> 0 or 1
// 	S_PH_DENSITY -> density scalar used by analyzer for spacing hints (optional)
#macro PH_Register_Object( S_ID, S_NAME, S_TARGET, S_RFL_ON, S_RFL_OFF, S_RFR_ON, S_RFR_OFF, S_COLLECT_OFF, S_PASS_THROUGH, S_IGNORE_PHOTONS, S_PH_DENSITY )
	#local A_OBJ = array[11];
	#local A_OBJ[0] = S_ID;
	#local A_OBJ[1] = S_NAME;
	#local A_OBJ[2] = S_TARGET;
	#local A_OBJ[3] = S_RFL_ON;
	#local A_OBJ[4] = S_RFL_OFF;
	#local A_OBJ[5] = S_RFR_ON;
	#local A_OBJ[6] = S_RFR_OFF;
	#local A_OBJ[7] = S_COLLECT_OFF;
	#local A_OBJ[8] = S_PASS_THROUGH;
	#local A_OBJ[9] = S_IGNORE_PHOTONS;
	#local A_OBJ[10] = S_PH_DENSITY;
	#declare A1D_OBJECTS[S_OBJECT_COUNT] = A_OBJ;
	#declare S_OBJECT_COUNT = S_OBJECT_COUNT + 1;
#end

// Mirror global_settings photons for analyzer reporting
#macro PH_GlobalPhotons( S_PHOTONS_ENABLED, S_SURFACE_SEPARATION, S_MEDIA_STEPS, S_MEDIA_FACTOR )
	#declare S_GLOBAL_PHOTONS_ENABLED = S_PHOTONS_ENABLED;
	#declare S_GLOBAL_SURFACE_SEPARATION = S_SURFACE_SEPARATION;
	#declare S_GLOBAL_MEDIA_STEPS = S_MEDIA_STEPS;
	#declare S_GLOBAL_MEDIA_FACTOR = S_MEDIA_FACTOR;
#end

// ---------------------------
// Internal helpers
// ---------------------------
#macro PH_Effective_On( S_ON, S_OFF )
	// Returns 1 if ON and not OFF, else 0
	#local S_RES = ( S_ON & ( 1 - S_OFF ) );
	S_RES
#end

#macro PH_BoolStr( S_VAL )
	#local S_OUT = ( S_VAL ? "ON" : "OFF" );
	S_OUT
#end

#macro PH_YesNo( S_VAL )
	#local S_OUT = ( S_VAL ? "YES" : "NO" );
	S_OUT
#end

#macro PH_MergedReflection( A_LIGHT, A_OBJ )
	// Effective reflection requires both light and object to be ON (and not OFF)
	#local S_L_ON = PH_Effective_On( A_LIGHT[2], A_LIGHT[3] );
	#local S_O_ON = PH_Effective_On( A_OBJ[3], A_OBJ[4] );
	#local S_RES = ( S_L_ON & S_O_ON );
	S_RES
#end

#macro PH_MergedRefraction( A_LIGHT, A_OBJ )
	// Effective refraction requires both light and object to be ON (and not OFF)
	#local S_L_ON = PH_Effective_On( A_LIGHT[4], A_LIGHT[5] );
	#local S_O_ON = PH_Effective_On( A_OBJ[5], A_OBJ[6] );
	#local S_RES = ( S_L_ON & S_O_ON );
	S_RES
#end

#macro PH_ShouldShoot( A_LIGHT, A_OBJ )
	// Mirrors photonestimation early bail: if neither reflection nor refraction is effective, do not shoot
	#local S_RFL = PH_MergedReflection( A_LIGHT, A_OBJ );
	#local S_RFR = PH_MergedRefraction( A_LIGHT, A_OBJ );
	#local S_RES = ( S_RFL | S_RFR );
	S_RES
#end

#macro PH_SpacingHint( A_LIGHT, A_OBJ )
	// Rough hint similar to estimation logic (not physically exact):
	#local S_DENS = max( A_OBJ[10], 0.001 );
	#local S_SEP = S_GLOBAL_SURFACE_SEPARATION;
	#local S_RAD = 1.00; // unit radius heuristic
	#local S_X = ( S_RAD / ( S_DENS * S_SEP ) );
	#local S_X = S_X * S_X * 3.14159265;
	S_X
#end

// ---------------------------
// Analysis & reporting
// ---------------------------
#macro PH_Analyze()
	#debug "\n=== PHOTON ANALYSIS (Source-Emulated) ===\n"
	#debug concat( "Photons enabled: ", PH_YesNo( S_GLOBAL_PHOTONS_ENABLED ), "\n" )
	#debug concat( "Surface separation (hint): ", str( S_GLOBAL_SURFACE_SEPARATION, 0, 6 ), "\n" )
	#debug concat( "Media photons: ", ( S_GLOBAL_MEDIA_STEPS > 0 ? concat( "ON (steps=", str( S_GLOBAL_MEDIA_STEPS, 0, 0 ), ", factor=", str( S_GLOBAL_MEDIA_FACTOR, 0, 3 ), ")" ) : "OFF" ), "\n" )
	#debug concat( "Registered lights: ", str( S_LIGHT_COUNT, 0, 0 ), "; objects: ", str( S_OBJECT_COUNT, 0, 0 ), "\n" )
	#debug "\n-- Pairwise light → object results --\n"
	#for ( S_I_LIGHT, 0, S_LIGHT_COUNT - 1 )
		#for ( S_I_OBJ, 0, S_OBJECT_COUNT - 1 )
			#local A_L = A1D_LIGHTS[S_I_LIGHT];
			#local A_O = A1D_OBJECTS[S_I_OBJ];
			#local S_RFL = PH_MergedReflection( A_L, A_O );
			#local S_RFR = PH_MergedRefraction( A_L, A_O );
			#local S_SHOOT = PH_ShouldShoot( A_L, A_O );
			#local S_SPREAD_HINT = PH_SpacingHint( A_L, A_O );
			#debug concat( 
				"Light[", str( A_L[0], 0, 0 ), "] ", A_L[1], " → Object[", str( A_O[0], 0, 0 ), "] ", A_O[1], ": ",
				"shoot=", PH_YesNo( S_SHOOT ), ", ",
				"reflection=", PH_BoolStr( S_RFL ), ", ",
				"refraction=", PH_BoolStr( S_RFR ), ", ",
				"target=", PH_YesNo( A_O[2] ), ", ",
				"collect=", ( A_O[7] ? "OFF" : "ON" ), ", ",
				"pass_through=", PH_BoolStr( A_O[8] ), ", ",
				"ignore_photons=", PH_BoolStr( A_O[9] ), ", ",
				"spread_hint≈", str( S_SPREAD_HINT, 0, 3 ),
				"\n" )
			// Pre-target notes (general rule-of-thumb based on source):
			#debug concat( "\tRule: non-pass_through objects encountered before the target block photons; pass_through before target tints photons (PT_FILTER_BEFORE_TARGET).\n" )
			#debug concat( "\tMedia: To visualize beams, use global_settings{ photons{ media steps[, factor] } } and mark volume containers pass_through.\n\n" )
		#end
	#end
	#debug "=== END ANALYSIS ===\n\n"
#end

// ---------------------------
// Example usage (commented)
// ---------------------------
/*
// Mirror your global_settings photons (for reporting):
PH_GlobalPhotons( 1, 0.01, 200, 0.50 )

// Register lights (IDs, names, flags):
PH_Register_Light( 1, "KeyLight", 1, 0, 1, 0, 0 )
PH_Register_Light( 2, "Laser",    1, 0, 0, 0, 1 )

// Register objects (IDs, names, flags, density):
PH_Register_Object( 10, "GlassSphere", 1, 0, 0, 1, 0, 1, 0, 0, 1.00 )
PH_Register_Object( 11, "MirrorPanel", 1, 1, 0, 0, 0, 1, 0, 0, 1.00 )
PH_Register_Object( 12, "MediaBox",    0, 0, 0, 0, 0, 0, 1, 1, 1.00 )

// Run analysis:
PH_Analyze()
*/

// ***end of code***
