
// ============================================================================
// Photon Permutations — Full Render Scene (POV-Ray SDL)
// Version: 1.0 (2025-12-11)
// Author: M365 Copilot
// ----------------------------------------------------------------------------
// Goal:
// 	Render a 2x(16x16) grid implementing EVERY permutation of light/object
// 	photon flags, so that photons are actually set and aimed per cell.
// 	Left grid: TARGET = NO (objects present but not photon targets)
// 	Right grid: TARGET = YES (objects marked photon targets)
//
// 	Light states per column: Reflection (L_R) × Refraction (L_F) ∈ { —, on, off, on+off }
// 	Object states per row:   Reflection (O_R) × Refraction (O_F) ∈ { —, on, off, on+off }
//
// 	Effective logic mirrors POV-Ray source:
// 	- A channel (reflection/refraction) is effective ON only if ON and not OFF
// 	  on BOTH the light and the object.
// 	- Photons are shot only if TARGET=YES and at least one channel is effective.
//
// Scene notes & performance:
// 	- Each cell contains its OWN light and object, placed far apart to minimize
// 	  cross-talk. Counts/spacings are kept modest for demo purposes.
// 	- Use the subset controls to render fewer cells while testing.
// ----------------------------------------------------------------------------
// Formatting:
// 	- Tabs for indentation
// 	- Capitalized identifiers with type prefixes S_/V_/A1D_
// 	- Directives separated; #for loops 0-based
// 	- Camera on -z axis toward +z; left-handed coords
// 	- Specular highlights (not phong)
// 	- End-of-code marker at bottom
// ============================================================================

// ---------------------------
// Global canvas & styling
// ---------------------------
#declare S_CELL_W = 3.50;     // per-cell width
#declare S_CELL_H = 3.50;     // per-cell height
#declare S_GAP    = 0.50;     // gap between cells
#declare S_LABEL  = 0.40;     // text scale
#declare S_FONT   = "cyrvetic"; // set to a local TTF if needed (e.g., "arial.ttf")

#declare S_GRID_ORIGIN_NO   = <  0,  0, 0 >;
#declare S_GRID_ORIGIN_YES  = <  0, 68, 0 >; // second grid above the first

#declare S_SUBSET_ENABLE = 0;      // 1 to render a subset
#declare S_COL_START     = 0;      // column start (0..15)
#declare S_COL_END       = 15;     // column end   (0..15)
#declare S_ROW_START     = 0;      // row start    (0..15)
#declare S_ROW_END       = 15;     // row end      (0..15)

// ---------------------------
// Camera & lighting (global)
// ---------------------------
#declare S_CANVAS_W = ( 16 * S_CELL_W ) + ( 15 * S_GAP ) + ( 4 * S_GAP );
#declare S_CANVAS_H = ( 32 * S_CELL_H ) + ( 31 * S_GAP ) + ( 6 * S_GAP );

camera {
	orthographic
	location < 0, 0, -120 >
	look_at  < 0, 0,   0 >
	right x * S_CANVAS_W
	up    y * S_CANVAS_H
}

light_source {
	< -S_CANVAS_W, S_CANVAS_H, -300 >
	rgb 1
}

background { color rgb < 1, 1, 1 > }

// ---------------------------
// Materials
// ---------------------------
#declare TEX_PLANE = texture { pigment { rgb < 1, 1, 1 > } finish { specular 0.10 roughness 0.02 } };
#declare TEX_GLASS = texture { pigment { rgbt < 1, 1, 1, 0.90 > } finish { specular 0.35 roughness 0.005 reflection 0.15 } };
#declare TEX_TEXT  = pigment { rgb < 0, 0, 0 > };

// ---------------------------
// Global photons settings (modest for demo)
// ---------------------------
#declare S_PHOTON_SPACING = 0.02;
#declare S_PHOTON_AUTOSTOP = 0.0; // keep shooting
#declare S_PHOTON_GATHER_MIN = 20;
#declare S_PHOTON_GATHER_MAX = 60;
#declare S_MEDIA_STEPS = 0;   // set >0 to enable media photons globally

global_settings {
	assumed_gamma 1
	max_trace_level 25
	photons {
		spacing S_PHOTON_SPACING
		autostop S_PHOTON_AUTOSTOP
		gather S_PHOTON_GATHER_MIN, S_PHOTON_GATHER_MAX
		#if ( S_MEDIA_STEPS > 0 )
			media S_MEDIA_STEPS, 0.50
		#end
	}
}

// ---------------------------
// Helpers: state mapping & effective logic
// States: 0: — ; 1: on ; 2: off ; 3: on+off (OFF wins)
// ---------------------------
#macro S_IsOn( S_STATE )
	#local S_RES = ( S_STATE = 1 ? 1 : ( S_STATE = 3 ? 1 : 0 ) );
	S_RES
#end

#macro S_IsOff( S_STATE )
	#local S_RES = ( S_STATE = 2 ? 1 : ( S_STATE = 3 ? 1 : 0 ) );
	S_RES
#end

#macro S_Effective( S_STATE )
	#local S_RES = ( S_IsOn( S_STATE ) & ( 1 - S_IsOff( S_STATE ) ) );
	S_RES
#end

#macro S_Name( S_STATE )
	#local S_STR = ( S_STATE = 0 ? "—" : ( S_STATE = 1 ? "on" : ( S_STATE = 2 ? "off" : "on+off" ) ) );
	S_STR
#end

// ---------------------------
// Per-cell builder
// ---------------------------
#macro Build_Cell( V_ORIGIN, S_TARGET, S_LR, S_LF, S_OR, S_OF )
	// Compute effective channels
	#local S_EFF_R = S_Effective( S_LR ) & S_Effective( S_OR );
	#local S_EFF_F = S_Effective( S_LF ) & S_Effective( S_OF );
	#local S_SHOOT = S_TARGET & ( S_EFF_R | S_EFF_F );

	// Local plane (receiver) and object (glass sphere) per cell
	#local V_C = V_ORIGIN + < S_CELL_W/2, S_CELL_H/2, 0 >;
	#local V_PL_BOTTOM = < V_C.x - ( S_CELL_W/2 - 0.3 ), V_C.y - ( S_CELL_H/2 - 0.3 ), 0 >;
	#local V_PL_TOP    = < V_C.x + ( S_CELL_W/2 - 0.3 ), V_C.y + ( S_CELL_H/2 - 0.3 ), 0 >;

	// Receiver plane (slightly inset to avoid touching neighbors)
	box { < V_PL_BOTTOM.x, V_PL_BOTTOM.y, -0.05 >, < V_PL_TOP.x, V_PL_TOP.y, 0.00 > texture { TEX_PLANE } }

	// Glass sphere (acts as both reflector and refractor)
	sphere {
		< V_C.x, V_C.y + 0.65, 0.30 >, 0.45
		texture { TEX_GLASS }
		interior { ior 1.5 }
		photons {
			#if ( S_TARGET ) target #end
			#if ( S_IsOn( S_OR ) ) reflection on #end
			#if ( S_IsOff( S_OR ) ) reflection off #end
			#if ( S_IsOn( S_OF ) ) refraction on #end
			#if ( S_IsOff( S_OF ) ) refraction off #end
		}
	}

	// Local light for the cell
	light_source {
		< V_C.x - 0.80, V_C.y + 1.40, 0.80 >
		rgb < 1, 1, 1 >
		fade_distance 1.20
		fade_power 2
		photons {
			#if ( S_IsOn( S_LR ) ) reflection on #end
			#if ( S_IsOff( S_LR ) ) reflection off #end
			#if ( S_IsOn( S_LF ) ) refraction on #end
			#if ( S_IsOff( S_LF ) ) refraction off #end
		}
	}

	// Labels: show states and computed effective/shoot
	text { ttf S_FONT concat( "L R:", S_Name( S_LR ), "  L F:", S_Name( S_LF ) ) S_LABEL, 0
		pigment { TEX_TEXT }
		translate < V_C.x - ( S_CELL_W/2 - 0.25 ), V_C.y + ( S_CELL_H/2 - 0.30 ), 0 >
	}
	text { ttf S_FONT concat( "O R:", S_Name( S_OR ), "  O F:", S_Name( S_OF ) ) S_LABEL, 0
		pigment { TEX_TEXT }
		translate < V_C.x - ( S_CELL_W/2 - 0.25 ), V_C.y + ( S_CELL_H/2 - 0.75 ), 0 >
	}
	text { ttf S_FONT concat( "R:", ( S_EFF_R ? "✓" : "×" ), "  F:", ( S_EFF_F ? "✓" : "×" ), "  S:", ( S_SHOOT ? "✓" : "×" ) ) S_LABEL, 0
		pigment { TEX_TEXT }
		translate < V_C.x - ( S_CELL_W/2 - 0.25 ), V_C.y - ( S_CELL_H/2 - 0.35 ), 0 >
	}
#end

// ---------------------------
// Build grids
// ---------------------------
#macro Build_Grid( V_GRID_ORIGIN, S_TARGET )
	#for ( S_COL, S_COL_START, S_COL_END )
		#for ( S_ROW, S_ROW_START, S_ROW_END )
			#local S_LR = int( S_COL / 4 );
			#local S_LF = mod( S_COL, 4 );
			#local S_OR = int( S_ROW / 4 );
			#local S_OF = mod( S_ROW, 4 );
			#local V_POS = V_GRID_ORIGIN + < S_COL * ( S_CELL_W + S_GAP ), S_ROW * ( S_CELL_H + S_GAP ), 0 >;
			Build_Cell( V_POS, S_TARGET, S_LR, S_LF, S_OR, S_OF )
		#end
	#end
#end

// Titles
text { ttf S_FONT "Target = NO" 0.60, 0 pigment { TEX_TEXT } translate < 1.5, -2.0, 0 > }
text { ttf S_FONT "Target = YES" 0.60, 0 pigment { TEX_TEXT } translate < 1.5, 66.0, 0 > }

// Build both grids (optionally subset)
#if ( S_SUBSET_ENABLE )
	Build_Grid( S_GRID_ORIGIN_NO, 0 )
	Build_Grid( S_GRID_ORIGIN_YES, 1 )
#else
	Build_Grid( S_GRID_ORIGIN_NO, 0 )
	Build_Grid( S_GRID_ORIGIN_YES, 1 )
#end

// ***end of code***
