/* This include file to the POV raytracer defines two macros to construct geodesic spheres.

                                       version 2.0

   -------------------------------------------------------------------------------------------        

   #macro geodesic(A, freq, mode)      usage: #declare vertices = geodesic(A, freq, mode)
        
        returnes an array 'vertices[i][j]' holding the coordinates of the 
        corner points of a geodesic sphere.
        
        <A>    describes the basic polyhedron on which the geodesic shape is constructed
        
                 A = 3   tetrahedron   (four triangular faces)
                 A = 4   oktahedron    (eight triangluar faces)
                 A = 5   icosahedron   (twenty triangular faces)
              
               values for A outside the interval [3,5] result in an error
        
        <freq> gives the number of subdivision for each primary triangle.
               
                 freq = 1   the basic polyhedron shape is created
                 freq > 1   the faces of the polyhedron are split into freq*freq triangles each
        
        <mode> determines the mode of triangle subdivision   *NEW* in version 2.0
        
                 mode = 1   The subdivision is performed on the faces of the triangle.
                            This is the compatibility mode to the first release of "geodesic.inc"
                 mode = 2   The subdivision takes place on the surface of the unit sphere
                            Especially with the tetrahedron as the basic shape, this mode 
                            results in a somewhat 'smoother' surface consiting of less divergent
                            triangles. For the icosahedron the difference between both modes is 
                            nearly neglectable.
               
        The array 'vertices[i][j]' contains the coordinates of these corner points.
        See the macro 'geodesic_vertices' for the decoding of this array structure
        depending on the type (A=3,4 or 5) of basic polyheder.     
        

   #macro geodesic_vertices(A, freq, vertices)      (internal use)
   
        decodes the structure of the vector array vertices calculated by 'geodesic(A, freq)'.
        I split the whole procedure into two macros for the following reasons:
        
          a) better readability of the source code (minor reason)
          b) reusability of the calculated vector array for different objects in one scene
             (thus you only have to calculate the vertices once)

        As POV does not support RECORD like structures (yet?) you have to take care yourself
        of the correct connection between the parameters 'A', 'freq' and 'vertices' if
        you decide to re-use the data structure in several objects.
        This routine executes the macros 'do_corner', 'do_edge' or 'do_face' at
        the location of the vertices. If one of these macros should be
        undefined it will result in a fatal error.


   #macro geodesic_sphere(A, freq, vertices)
     
        ?calls the macro 'geodesic_vertices'? and returns a CSG union containing 
        the executed macros 'do_corner','do_edge' and 'do_face' at the 
        appropiate positions given by the calculated vertex field. 
        If either one of the macros 'do_corner', 'do_edge' or 'do_face' is undefined 
        while calling this macro, a dummy macro is used to render the corner, edge or face
        object respectively. If you don't want to have one or more of these parts of
        the figure unrendered you have to define an empty macro otherwise you 
        should provide your own macros to draw these objects:
        
           #macro do_corner( position )                  
             // draw the corner object at vector 'position'
           #end

           #macro do_edge  ( corner_1, corner_2 )
             // draw the edge object between vector 'corner_1' and 'corner_2'
           #end

           #macro do_face  ( corner_1, corner_2, corner_3 )
             // draw the triangular face object with the vectors 'corner_1', 'corner_2' and 'corner_3'
           #end
                                                            

   #macro geodesic_mesh(A, freq, vertices)
     
        calls the macro 'geodesic_vertices' and returns a mesh object with the 
        appropiate positions given by the calculated vertex field. 
        This routine replaces the contents of the macros 'do_corner' and 'do_edge'
        and internally calls the same drawing routine as the 'geodesic_sphere' macro.
        The contents of the macro 'do_face' should contain the definition for the
        mesh triangles

           #macro do_face  ( corner_1, corner_2, corner_3 )
             // draw the triangular face object with the vectors 'corner_1', 'corner_2' and 'corner_3'
           #end
                                                                  
                                                                  
see the file geodesic.pov for some examples
           
           
Disclaimer:    
    
                  This macro is free for use together with the POVray raytracer.
              It should be distributed with the files geodesic.txt and geodesic.pov.

                                                     Uwe Zimmermann, Stockholm, Sweden 1999.
                                                                  
   -------------------------------------------------------------------------------------------     */


//---------------------------------------------------------------------------------------------------
//
//   macro definition for "divide_triangle" 
//
//             - This macro is meant for internal purposes only.
//             - It takes care of the subdivision of the face triangles
//               by returning the vertex position of the [i,j] vertex in a 
//               triangle divided with a frequency given in the parameter 
//               'freq'. The parameter mode determines if the triangle
//               subdivision is done planar (mode=1) or spherical (mode=2).
//
//---------------------------------------------------------------------------------------------------
                                                     
#macro divide_triangle(corner_1, corner_2, corner_3, i, j, freq, mode)                                                     
  #switch (mode)
    #case (1) // linear interpolation on polyheder
      #local v1=corner_1+i*(corner_2-corner_1)/(freq);   // right side
      #local v2=corner_1+i*(corner_3-corner_1)/(freq);   // left side
      #local vertex=vnormalize(v1+j/i*(v2-v1));          // get vertex and shift to unit sphere
    #break  
    #case (2) // linear interpolation on unit sphere using the triangular sine law                                                                      
      #local corner_1 = vnormalize(corner_1);
      #local corner_2 = vnormalize(corner_2);
      #local corner_3 = vnormalize(corner_3);
      #local alpha_v1=acos(vdot(corner_1,corner_2));       // right side angle
      #local beta_v1 =i*alpha_v1/(freq);                   // fraction of the angle
      #local alpha_v2=acos(vdot(corner_1,corner_3));       // left side angle
      #local beta_v2 =i*alpha_v2/(freq);                   // fraction of the angle
      #local v1=vnormalize(corner_1+vnormalize(corner_2-corner_1)*sin(beta_v1)/(sin(pi/2+alpha_v1/2-beta_v1)));
      #local v2=vnormalize(corner_1+vnormalize(corner_3-corner_1)*sin(beta_v2)/(sin(pi/2+alpha_v2/2-beta_v2)));       
      #local alpha_v3=acos(vdot(v1, v2));                                
      #local beta_v3 =j*alpha_v3/i;
      #local vertex=vnormalize(v1+vnormalize(v2-v1)*sin(beta_v3)/(sin(pi/2+alpha_v3/2-beta_v3)));  // get vertex
    #break                                                                 
    #else 
      #error "Error: Illegal 'mode' selected\n"
  #end  
  vertex
#end

//---------------------------------------------------------------------------------------------------
//
//   macro definition for "geodesic" 
//
//             - this macro calculates the positions of the vertices and returns them in
//               an array structure, readable by the macro "geodesic_vertices"
//
//---------------------------------------------------------------------------------------------------
                                                     
#macro geodesic(A, freq, mode) 
  #if (freq<1)
    #error "Error: The frequency of division must be a positive non-zero interger\n"   
  #end  
  #switch(A)
    #case (3)        
      #local zc=-1/3;
      #local r=2/3*sqrt(2);
      #local corners=array[4] 
      #local vertices=array[2*freq+2][3*freq] 
      #local Lv=(2*freq+2)*3*freq;
      #local corners[0]=<0,0,1>;
      #local i=0;
      #while (i<3)
        #local corners[i+1]= <r*cos(i*2*pi/3), r*sin(i*2*pi/3), zc>;
        #local i=i+1;
      #end  
    #break
    #case (4)
      #local vertices=array[2*freq+1][4*freq]
      #local Lv=(2*freq+1)*4*freq;
      #local corners=array[6] {<0,0,1>,
                               <1,0,0>,
                               <0,1,0>,
                               <-1,0,0>,
                               <0,-1,0>,
                               <0,0,-1>}
    #break
    #case (5)
      #local corners=array[12] 
      #local vertices=array[3*freq+1][5*freq]
      //#local Lv=(3*freq+1)*5*freq;
      #local corners[0]=<0,0,1>;
      #local a=4/sqrt(2*(5+sqrt(5)));
      #local zc=1-a*a/2;
      #local r=sqrt(1-zc*zc);
      #local i=0;
      #while (i<5)   
        #local corners[i+1]=<r*cos(i*2*pi/5),r*sin(i*2*pi/5),zc>*1;
        #local corners[i+6]=<r*cos((i-0.5)*2*pi/5),r*sin((i-0.5)*2*pi/5),-zc>;
        #local i=i+1;
      #end  
      #local corners[11]=<0,0,-1>;
    #break
  #else
    #error "Error: The base symmetry A has to be 3, 4 or 5\n"
  #end                             
  //#debug concat("vertices = ",str(Lv,0,0),"\n")
  // prevent user from error "uninitialized array element"
  // only vertex coordinates with vlength=1 are valid!
  /*  disabled for debugging use
  #local i=0;
  #while (i<dimension_size(vertices,1))
    local j=0;
    #while (j<dimension_size(vertices,2))
      #local vertices[i][j]=<0,0,0>;
      #local j=j+1;
    #end;
    #local i=i+1;
  #end;
  */
            
  #local i=0;
  #local vertices[0][0]=corners[0];
  
  // main loop calculates northern (z>0) "hemi"sphere      
  #local Fp=0;
  #local i=1;
  #while (i<=freq)
    #local j=0;
    #while (j<i)
      #local k=0;                                         
      #while (k<A)           
        #local vertices[i][j+k*i]=divide_triangle(corners[0], 
                                         corners[mod(k,A)+1], 
                                       corners[mod(k+1,A)+1], 
                                           i, j, freq, mode);
        #local Fp=Fp+1;                                                     
        #local k=k+1;
      #end  
      #local j=j+1;
    #end  
    #local i=i+1;
  #end                                             
   #debug concat("Fp =",str(Fp,0,0),"\n")                            
  // going deeper into the different polyhedra                           
  #switch (A)
    #case (3)   //tetrahedron                 
      // for the tetrahedron the dissection of the groundplane remains
      #local i=0;
      #while (i<=freq)
        #local j=0;
        #while (j<=i)
          #if (i>0)
            #local vertices[i+freq+1][j]=
              divide_triangle(corners[1],corners[2],corners[3],i, j, freq, mode);
          #else
            #local vertices[i+freq+1][j]=corners[1];                             // get vertex
          #end  
          //#local vertices[i+freq+1][j]=vnormalize(vertices[i+freq+1][j]);  //mode 1 died here! put it on unit sphere
          #local j=j+1;
        #end  
        #local i=i+1;
      #end  
    #break  

    #case (4)   //octahedron      
      // for the octahedron it's only a matter of symmetry
      #local i=2*freq;
      #local j=0;
      #local vertices[i][j]=vertices[0][0]*<1,1,-1>;         // south pole
      #local i=1;
      #while (i<freq)
        #local j=0;
        #while (j<A*i)
          #local vertices[2*freq-i][j]=vertices[i][j]*<1,1,-1>;          // just mirror the vertices
          #local j=j+1;
        #end  
        #local i=i+1;
      #end                                             
    #break  

    #case (5)   //icosahedron
      // for the icosahedron it's a little bit more...
      // first the equatorial row of 2x5 triangles
      #local i=1;
      #while (i<freq)
        #local j=0;       // southward opened
        #while (j<i)
          #local k=0;
          #while (k<A)            
            #local vertices[i+freq][j+k*freq]=divide_triangle(corners[k+1], 
                                                       corners[mod(k,A)+6], 
                                                     corners[mod(k+1,A)+6], 
                                                         i, j, freq, mode);
            #local k=k+1;
          #end  
          #local j=j+1;
        #end
        #local j=0;       // northward opened
        #while (j<=(freq-1-i))
          #local k=0;
          #while (k<A)                                               
            #local vertices[i+freq][j+k*freq+i]=divide_triangle(corners[mod(k+1,A)+6], 
                                                                  corners[mod(k,A)+1], 
                                                                corners[mod(k+1,A)+1], 
                                                             (freq-i), j, freq, mode);
            #local k=k+1;
          #end  
          #local j=j+1;
        #end  
        #local i=i+1;
      #end                                             
      //
      // now apply the symmetry
      //                      
      #local i=3*freq;
      #local j=0;
      #local vertices[i][j]=vertices[0][0]*<1,1,-1>;         // south pole
      #local i=1;
      #while (i<=freq)
        #local j=0;
        #while (j<A*i)
          #local vertices[3*freq-i][j]=vaxis_rotate(vertices[i][j]*<1,1,-1>, z, -360/10) ;          // just mirror the vertices
          #local j=j+1;
        #end  
        #local i=i+1;
      #end                                             
    #break                           
  #end      
  vertices
#end                
         
//---------------------------------------------------------------------------------------------------
//
//   macro definition for "geodesic_vertices" 
//
//             - this macro is for internal purpose only, it decodes the structure of the vertex array
//               and executes the macros "do_corner", "do_edge" and "do_face"
//
//---------------------------------------------------------------------------------------------------
         
#macro geodesic_vertices(A, freq, vertices)
  // now we should have all vertices in the array.... we JUST have to sort out, which one is where
 
  // the northern "hemi"sphere again is similar to all shapes  
  #local i=0;
  #while (i<freq)
    #local j=0;
    #while ((j<i)|((j=0)&(i=0)))
      #local k=0;
      #while (k<A)
        #if ((i>0)|(k=0))
          do_corner(vertices[i][k*i+j])
        #end  
        do_edge(vertices[i][k*i+j], vertices[i+1][k*(i+1)+j])
        do_face(vertices[i][k*i+j], vertices[i+1][k*(i+1)+j], vertices[i+1][mod(k*(i+1)+j+1,(i+1)*A)])
        #if (i>0)
          do_edge(vertices[i][k*i+j], vertices[i+1][k*(i+1)+j+1])
          do_edge(vertices[i][k*i+j], vertices[i][mod(k*i+j+1,A*i)])
          do_face(vertices[i][k*i+j], vertices[i+1][mod(k*(i+1)+j+1,(i+1)*A)], vertices[i][mod(k*i+j+1,i*A)])
        #end  
        #if ((j=0)&(i>0))
          do_edge(vertices[i][k*i+j], vertices[i+1][mod(k+A-1,A)*(i+1)+i])
          do_face(vertices[i][k*i+j], vertices[i+1][mod(k+A-1,A)*(i+1)+i], vertices[i+1][k*(i+1)+j] )
        #end
            
        // southern "hemi"sphere for A=4 and A=5 is symmetric    
        #if (A>3)
          #if ((i>0)|(k=0))
            do_corner(vertices[(A-2)*freq-i][k*i+j])
          #end  
          do_edge(vertices[(A-2)*freq-i][k*i+j], vertices[(A-2)*freq-(i+1)][k*(i+1)+j])
          do_face(vertices[(A-2)*freq-i][k*i+j], vertices[(A-2)*freq-(i+1)][k*(i+1)+j], vertices[(A-2)*freq-(i+1)][mod(k*(i+1)+j+1,(i+1)*A)])
          #if (i>0)
            do_edge(vertices[(A-2)*freq-i][k*i+j], vertices[(A-2)*freq-(i+1)][k*(i+1)+j+1])
            do_edge(vertices[(A-2)*freq-i][k*i+j], vertices[(A-2)*freq-i][mod(k*i+j+1,A*i)])
            do_face(vertices[(A-2)*freq-i][k*i+j], vertices[(A-2)*freq-(i+1)][mod(k*(i+1)+j+1,(i+1)*A)], vertices[(A-2)*freq-i][mod(k*i+j+1,i*A)])
          #end  
          #if ((j=0)&(i>0))
            do_edge(vertices[(A-2)*freq-i][k*i+j], vertices[(A-2)*freq-(i+1)][mod(k+A-1,A)*(i+1)+i])
            do_face(vertices[(A-2)*freq-i][k*i+j], vertices[(A-2)*freq-(i+1)][mod(k+A-1,A)*(i+1)+i], vertices[(A-2)*freq-(i+1)][k*(i+1)+j] )
          #end
        #end
        #local k=k+1;
      #end
      #local j=j+1;  
    #end  
    #local i=i+1;
  #end

  // now add the missing corners and links at the open end of the "hemi"sphere(S)
  #local i=freq; 
  #local j=0;
  #while (j<i)
    #local k=0;
    #while (k<A)   
      do_corner(vertices[i][k*i+j])
      do_edge(vertices[i][k*i+j], vertices[i][mod(k*i+j+1,i*A)])
      #if (A=5)  // do both rings for the icosahedron
        do_corner(vertices[3*freq-i][k*i+j])
        do_edge(vertices[3*freq-i][k*i+j], vertices[3*freq-i][mod(k*i+j+1,i*A)])
      #end
      #local k=k+1;
    #end
    #local j=j+1;
  #end  

  #switch(A)
    #case (3)  // the tetrahedron  - the simplest shape, the most problems
      #local i=0;
      #while (i<freq)
        #local j=0;
        #while (j<=i)
          #if ((i>0)&(j>0)&(i<freq)&(j<i))
            do_corner(vertices[freq+i+1][j])
          #end
          #if ((i>0)&(j<i))
            do_edge(vertices[freq+i+1][j], vertices[freq+i+1][j+1])
            do_edge(vertices[freq+i+1][j], vertices[freq+i+2][j+1])
            do_face(vertices[freq+i+1][j], vertices[freq+i+2][j+1], vertices[freq+i+1][j+1])
          #end  
          #if (j>0)
            do_edge(vertices[freq+i+1][j],vertices[freq+i+2][j])
          #end                                                             
          do_face(vertices[freq+i+1][j], vertices[freq+i+2][j], vertices[freq+i+2][j+1])
          #local j=j+1;
        #end  
        #local i=i+1;
      #end  
    #break     
    
    #case (4) // the octahedron
      // I didn't believe it, but it's already complete at this point!
    #break
    
    #case (5) // the icosahedron  
      #local i=0;
      #while (i<freq)
        #local j=0;
        #while (j<freq*A)
          #if (i>0)
            do_corner(vertices[freq+i][j])
            do_edge(vertices[freq+i][j],vertices[freq+i][mod(j+1,A*freq)])
          #end
          do_edge(vertices[freq+i][j],vertices[freq+i+1][j])
          do_edge(vertices[freq+i][j],vertices[freq+i+1][mod(j+1,A*freq)])
          do_face(vertices[freq+i][j],vertices[freq+i+1][j],vertices[freq+i+1][mod(j+1,A*freq)])
          do_face(vertices[freq+i][j],vertices[freq+i+1][mod(j+1,A*freq)],vertices[freq+i][mod(j+1,A*freq)])
          #local j=j+1;
        #end
        #local i=i+1;
      #end      
    #break
  #end  
#end

//---------------------------------------------------------------------------------------------------
//
//   macro definition for "geodesic_sphere" 
//
//             - the result of this macro is an object containing corner, link and face objects
//
//---------------------------------------------------------------------------------------------------
         
#macro geodesic_sphere(A, freq, vertices)
  // test if the do_corner and do_edge macros are defined
  // otherwise we would run into trouble later - so do a dummy definition
  #ifndef (do_corner)
    #macro do_corner( position )
      sphere { position, 0.05 texture { pigment {color rgb <1,0,0> } } }
    #end
  #end  
  #ifndef (do_edge)
    #macro do_edge( corner_1, corner_2 )
      cylinder { corner_1, corner_2, 0.02 texture { pigment {color rgb <0,0,1> } } }
    #end
  #end  
  #ifndef (do_face)
    #macro do_face( corner_1, corner_2, corner_3 )
      triangle { corner_1, corner_2, corner_3 texture { pigment {color rgbf <0.4,0.9,0.4,0.8>} } }
    #end
  #end  
                                                                         
  union{ geodesic_vertices(A, freq, vertices) } 

#end  

//---------------------------------------------------------------------------------------------------
//
//   macro definition for "geodesic_mesh" 
//
//             - the result of this macro is a mesh object 
//
//---------------------------------------------------------------------------------------------------
         
#macro geodesic_mesh(A, freq, vertices)
  // test if the do_corner and do_edge macros are defined
  // otherwise we would run into trouble later - so do a dummy definition
  #macro do_corner( position )
  #end

  #macro do_edge( corner_1, corner_2 )
  #end           

  #local my_dummy_geomesh_texture = texture { pigment {color rgbf<0.4,0.9,0.4,0.8>} }
  #ifndef (do_face)
    #macro do_face( corner_1, corner_2, corner_3 )
      triangle { corner_1, corner_2, corner_3 texture { my_dummy_geomesh_texture }}
    #end
  #end  

  mesh{ geodesic_vertices(A, freq, vertices) }       
  
#end  
                                        
//---------------------------------------------------------------------------------------------------

