// the macros for subdividing the mesh

#ifndef(C2S3_DIV_INC)
#declare C2S3_DIV_INC = true;

// c2s3_DivideFaces() macro:
// Incoming preconditions:

// c2s3cV == the number of vertices
// c2s3aV[] == an array containing the location of each vertex
// built from qaVV, qaVF, and qaVE

// c2s3cU == the number of defined texture mapping (u,v) points
// c2s3aU[] == an array containing the location of each (u,v) mapping point
// built from qaUU, qaUF, and qaUM

// c2s3cE == the number of geometry edges
// c2s3aEA[] == array containing the starting vertex of each edge
// built from qaEA
// c2s3aEB[] == array containing the ending vertex of each edge
// built from qaEB
// c2s3aEL[] == array containing the index to the face on the left side of each edge
// built from qaEL (not done)
// c2s3aER[] == array containing the index to the face on the right side of each edge
// built from qaER (not done)
// c2s3aES[] == array containing the sharpness level of each edge
// built from qaES

// c2s3cM == the number of map edges
// c2s3aMA[] == array containing the starting uv point of each edge
// built from qaMA
// c2s3aMB[] == array containing the ending uv point of each edge
// built from qaMB
// c2s3aML[] == array containing the index to the face on the left side of each edge
// built from qaML (not done, can be built from the qaFE array)
// c2s3aMR[] == array containing the index to the face on the right side of each edge
// built from qaMR (not done, can be built from the qaFE array)
// no sharpness array--texture map edges are either smooth or borders

// c2s3cF == the number of faces
// c2s3aFV[] == an array containing vertex records for each face
// built from qaFV
// c2s3aFE[] == an array containing edge records for each face
// built from qaFE (not done)
// c2s3aFT[] == an array containing the index of the texture used by each face
// built from qaFT (not done)
// c2s3aFU[] == an array containing uv-mapping records for each face
// built from qaFU (not done)

// each array replaced with subdivided versions

// new vertices:
// first c2s3cV	vertex points -- one for each old vertex
// next	c2s3cE	edge points -- one for each old edge
// next c2s3cF	face points -- one for each old face

// new edges:
// first c2s3cE	A end of old edges -- one for each old edge
// next c2s3cE B end of old edges -- one for each old edge
// remainder	edges within subdivided faces -- one for each side of each face

// new faces:
// one face for each vertex of each face -- one for each corner of each face

// new texture mapping points:
// first c2s3cU -- old points -- one for each old mapping point
// next c2s3cM -- edge uv points -- one for each mapping edge
// next c2s3cF -- face uv points -- one for each mapped face

// subdivision steps
// 1. calc face geometry points
// 2. calc edge geometry points
// 3. calc vertex geometry points
// 4. calc face uv points
// 5. calc edge uv points
// 6. calc vertex uv points
// 7. build replacement edges
// 8. build new interior edges
// 9. build new faces

#macro c2s3_Subdivide()
// this is here because it's best done before the first subdivision,
// and subdivision leaves face windings intact
  #if(c2s3fO=false) c2s3_OrientFaces() #end

 #ifndef(c2s3cE) c2s3_BuildEdges() #end

// delete stuff that isn't needed for subidvision and would become invalid anyway
  #while(defined(c2s3cN)) #undef c2s3cN #end
  #while(defined(c2s3aN)) #undef c2s3aN #end
  #while(defined(c2s3aFN)) #undef c2s3aFN #end
  #while(defined(c2s3aFI)) #undef c2s3aFI #end
  #while(defined(c2s3aFJ)) #undef c2s3aFJ #end

  #debug "Subdividing mesh.\n"

// 1. calc face geometry points
// Simple:  Each face point is always the average of its face's vertices
  #local qaVF=array[c2s3cF];
  #local iR=0;
  #local iF=0; #while(iF<c2s3cF)
    #local qaVF[iF]=<0,0,0>;
    #local cV=c2s3aFV[iR]; // get number of vertices in this face
    #local iR=iR+1; // advance past vertex count
    #local iJ=0;#while(iJ<cV)
      #local qaVF[iF]=qaVF[iF]+c2s3aV[c2s3aFV[iR]];
      #local iR=iR+1;
    #local iJ=iJ+1;#end
    #local qaVF[iF]=qaVF[iF]/cV;
  #local iF=iF+1;#end

// 2. calc edge geometry points
// Almost as simple:  Each smooth edge point is the average of the ends and the bordering face points
// and the sharp edge point is the mid-point of the edge
  #local qaVE=array[c2s3cE];
  #local iE=0; #while(iE<c2s3cE)
    #local pQA=(c2s3aV[c2s3aEA[iE]]+c2s3aV[c2s3aEB[iE]])*.5; // fully sharp midpoint
    #if(c2s3aEL[iE]>-1 & c2s3aER[iE]>-1) // smooth interior edge
      // the midpoint is weighted with the mid points of the bordering faces
      #local pQB=pQA*.5+(qaVF[c2s3aEL[iE]]+qaVF[c2s3aER[iE]])*.25; // fully smooth midpoint
    #else
      #local pQB=pQA;
    #end
    #if(c2s3aES[iE]>=1 | c2s3aES[iE]<0)
      #local qaVE[iE]=pQA;	// use the fully sharp point
    #else
      #local qaVE[iE]=pQB+c2s3aES[iE]*(pQA-pQB);	// interpolate between full sharp and full smooth
    #end
  #local iE=iE+1;#end

// 3.  Calculate the vertex geometry points
  // now type the vertices
  // b= number of border edges
  // h = number of hard sharp edges (sharpness >1)
  // s = number of semi-sharp edges (sharpness >0 but <1)
  // i = interior smooth edges (sharpness =0)
  // b=2, s=0, t=2 -> corner vertex; these do not move (this rule was not in the NSSS)
  // (b+s)>2 -> corner vertex
  // b=2, s=0, t>2 -> crease vertex
  // b=0, s=2, t=DC -> crease vertex
  // b=0, s<2, t=DC -> smooth vertex

  // need to know:
  // b=number of border edges
  // h=number of edges >=1 in sharpness
  // s=number of edges >0, <1 in sharpness
  // t=total edges
  // a1=the largest sharpness of semi-smooth edges
  // a2=the average sharpness of all sharp edges

  #local qaVQ=array[c2s3cV];	// the vertex typing
  #local qaV1=array[c2s3cV];	// the sharpnesses maxes
  #local qaV2=array[c2s3cV];	// the sharpnesses averages
  #local iV=0;#while(iV<c2s3cV)
    #local qaVQ[iV]=<0,0,0,0>;
    #local qaV1[iV]=0;
    #local qaV2[iV]=0;
  #local iV=iV+1;#end

  #local iE=0; #while(iE<c2s3cE)
    #local iA=c2s3aEA[iE];
    #local iB=c2s3aEB[iE];
    #local sE=c2s3aES[iE];
    #if(sE<0) #local sE=255; #end
    #if(c2s3aEL[iE]=-1 | c2s3aER[iE]=-1) // this is a border edge
      #local qaVQ[iA]=qaVQ[iA]+x;
      #local qaVQ[iB]=qaVQ[iB]+x;
    #else
      #if(sE>0)
        #if(sE<1)
          #local qaVQ[iA]=qaVQ[iA]+z;
          #local qaVQ[iB]=qaVQ[iB]+z;
          #local qaV1[iA]=max(sE,qaV1[iA]);
          #local qaV1[iB]=max(sE,qaV1[iB]);
        #else
          #local qaVQ[iA]=qaVQ[iA]+y;
          #local qaVQ[iB]=qaVQ[iB]+y;
        #end
        #local qaV2[iA]=qaV2[iA]+c2s3aES[iE]*.5;
        #local qaV2[iB]=qaV2[iB]+c2s3aES[iE]*.5;
      #end
    #end
    #local qaVQ[iA]=qaVQ[iA]+t;
    #local qaVQ[iB]=qaVQ[iB]+t;
  #local iE=iE+1;#end

  // b   h   s   t
  //* [>2     ]   x   0: corner vertex
  //* 2   0   0   0   0: corner vertex
  //x [ 2 ]   >0  x   1: interpolate between crease and corner using a1
  //x 0   1   >1  x   2: interpolate between smooth and corner using a1
  //x 0   0   >2  x   2: interpolate between smooth and corner using a1

  //* [ 2 ]   0   >2  3: crease vertex
  //* 0   [ 2 ]   x   3: a2>=1: crease
  //*                 4: a2<1: interpolate between smooth and crease using a2
  //* 0   [<2 ]   x   5: smooth

  #local qaVC=array[c2s3cV]; // crease vertices
  #local qaVS=array[c2s3cV]; // smooth vertices
  #local qaVT=array[c2s3cV]; // types
  #local iV=0;#while(iV<c2s3cV)
    #local qaVC[iV]=<0,0,0>; // initialize
    #local qaVS[iV]=<0,0,0>; // initialize
// set smooth as default
    #local qaVT[iV]=5;
// if border + sharp is greater than three, vertex is a corner
    #if((qaVQ[iV].x + qaVQ[iV].y + qaVQ[iV].z)>2)
      #local qaVT[iV]=0;
    #end
// if border=2 and total=2, vertex is a corner
    #if(qaVQ[iV].x=2 & qaVQ[iV].t=2) // other values cannot be non-zero
      #local qaVT[iV]=0;
    #end

// if border+hard=2 and semi>0, interpolate between crease and corner
//    #if((qaVQ[iV].x + qaVQ[iV].y =2) & qaVQ[iV].z>0)
//      #local qaVT[iV]=1;
//    #end

// if one hard and more than one semi, it's type 2
//    #if(qaVQ[iV].x=0 & qaVQ[iV].y=1 & qaVQ[iV].z>1)
//      #local qaVT[iV]=2;
//    #end

// if no hard and more than two semi, it's type 2
//    #if(qaVQ[iV].x=0 & qaVQ[iV].y=0 & qaVQ[iV].z>2)
//      #local qaVT[iV]=2;
//    #end

// if border+hard=2, semi=0, total>2, it's type 3
    #if((qaVQ[iV].x + qaVQ[iV].y =2) & qaVQ[iV].z=0 & qaVQ[iV].t>2)
      #local qaVT[iV]=3;
    #end

// if border=0, hard+semi=2, and a2>=1 it's type 3
    #if(qaVQ[iV].x=0 & (qaVQ[iV].y + qaVQ[iV].z=2) & qaV2[iV]>=1)
      #local qaVT[iV]=3;
    #end

// if border=0, hard+semi=2, and a2>=1 it's type 3
    #if(qaVQ[iV].x=0 & (qaVQ[iV].y + qaVQ[iV].z=2) & qaV2[iV]<1)
      #local qaVT[iV]=4;
    #end
  #local iV=iV+1;#end

  // set up vertex point array, initialize with weighting of original vertex
  #local qaVC=array[c2s3cV]; // sum for crease
  #local qaVS=array[c2s3cV]; // sum for smooth
  #local iV=0;#while(iV<c2s3cV)
    #local qaVC[iV]=c2s3aV[iV]*.75; // always weighted at 75%
    #local sW=1;
    #if(qaVQ[iV].t > 0)
      #local sW=1-2/qaVQ[iV].t;
    #end
    #local qaVS[iV]=c2s3aV[iV]*sW;
  #local iV=iV+1;#end

  // the qaVT array now means:
  // 0 = corner
  // 1 = interpolate between crease and corner using qaV1
  // 2 = interpolate between smooth and corner using qaV1
  // 3 = crease vertex
  // 4 = interpolate between smooth and crease using qaV2
  // 5 = smooth vertex

  // weight in the face points for the smooth points
  #local iR=0;
  #local iF=0;#while(iF<c2s3cF)  
    #local cS=c2s3aFV[iR];
    #local iR=iR+1;
    #local iJ=0;#while(iJ<cS)
      #local iV=c2s3aFV[iR];
      #local cV=qaVQ[iV].t;
      #local iR=iR+1;
      #local qaVS[iV]=qaVS[iV]+qaVF[iF]/cV/cV;
    #local iJ=iJ+1;#end
  #local iF=iF+1; #end

  //weight in the edge points
  #local iE=0;#while(iE<c2s3cE)
    #local iA=c2s3aEA[iE];
    #local iB=c2s3aEB[iE];
  // first, weight in the values for the smooth points
    #local cV=qaVQ[iA].t;
    #local qaVS[iA]=qaVS[iA]+c2s3aV[iB]/cV/cV;
    #local cV=qaVQ[iB].t;
    #local qaVS[iB]=qaVS[iB]+c2s3aV[iA]/cV/cV;
  // then weight in the values for the crease points
    #switch(qaVT[iA])
      #case(1) // weight in for border and hard edges
        #if(c2s3aES[iE]>=1 | c2s3aES[iE]<0 | c2s3aEL[iE]=-1 | c2s3aER[iE]=-1)
          #local qaVC[iA]=qaVC[iA]+.125*c2s3aV[iB];
        #end
      #break
      #case(3)
      #case(4) // weight in for border, hard, or semi-hard
        #if(c2s3aES[iE]!=0 | c2s3aEL[iE]=-1 | c2s3aER[iE]=-1)
          #local qaVC[iA]=qaVC[iA]+.125*c2s3aV[iB];
        #end
      #break
    #end

    #switch(qaVT[iB])
      #case(1) // weight in for border and hard edges
        #if(c2s3aES[iE]>=1 | c2s3aES[iE]<0 | c2s3aEL[iE]=-1 | c2s3aER[iE]=-1)
          #local qaVC[iB]=qaVC[iB]+.125*c2s3aV[iA];
        #end
      #break
      #case(3)
      #case(4) // weight in for border, hard, or semi-hard
        #if(c2s3aES[iE]!=0 | c2s3aEL[iE]=-1 | c2s3aER[iE]=-1)
          #local qaVC[iB]=qaVC[iB]+.125*c2s3aV[iA];
        #end
      #break
    #end
  #local iE=iE+1;#end

  #local qaVV=array[c2s3cV];

// and now to calculate the final vertex points
  #local iV=0;#while(iV<c2s3cV)
    #switch(qaVT[iV])
      #case(0)
        #local qaVV[iV]=c2s3aV[iV];
      #break
      #case(1)
        #local qaVV[iV]=qaVC[iV]+(c2s3aV[iV]-qaVC[iV])*qaV1[iV];
      #break
      #case(2)
        #local qaVV[iV]=qaVS[iV]+(c2s3aV[iV]-qaVS[iV])*qaV1[iV];
      #break
      #case(3)
        #local qaVV[iV]=qaVC[iV];
      #break
      #case(4)
        #local qaVV[iV]=qaVS[iV]+(qaVC[iV]-qaVS[iV])*qaV2[iV];
      #break
      #case(5)
        #local qaVV[iV]=qaVS[iV];
      #break
    #end
  #local iV=iV+1;#end

  #undef qaVC
  #undef qaVS
  #undef qaVQ
  #undef qaVT
  #undef qaV1
  #undef qaV2

// 4. calc face mapping points
// Simple:  Each face point is always the average of its face's vertices
#ifdef(c2s3cU)
  #local qaUF=array[c2s3cF];
  #local iS=0;
  #local iF=0; #while(iF<c2s3cF)
    #local qaUF[iF]=<0,0>;
    #local cU=c2s3aFU[iS];
    #local iS=iS+1; // advance past record size
    #local iJ=0;#while(iJ<cU)
      #local qaUF[iF]=qaUF[iF]+c2s3aU[c2s3aFU[iS]]/cU;
      #local iS=iS+1;
    #local iJ=iJ+1;#end
  #local iF=iF+1;#end

// 5. calc edge mapping points
// Amost as simple:  Each smooth edge point is the average of the ends and the bordering face points
// and the border edge point is the mid-point of the edge
  #local qaUM=array[c2s3cM];
  #local iM=0; #while(iM<c2s3cM)
    #local qaUM[iM]=(c2s3aU[c2s3aMA[iM]]+c2s3aU[c2s3aMB[iM]])*.5; // fully sharp midpoint
    #if(c2s3aML[iM]>-1 & c2s3aMR[iM]>-1) // smooth interior edge
      // the midpoint is weighted with the mid points of the bordering faces
      #local qaUM[iM]=qaUM[iM]*.5+(qaUF[c2s3aML[iM]]+qaUF[c2s3aMR[iM]])*.25; // fully smooth midpoint
    #end
  #local iM=iM+1;#end

// 6.  Calculate the vertex mapping points
  // now type the vertices
  // b= number of border edges
  // s = number of sharp interior edges (always 0 for texture mapping)
  // t = total number of edges
  // types are as follows:
  // b>=2, s=0, t=2 -> corner vertex; these do not move (this rule was not in the NSSS)
  // b=2, s=0, t>2 -> crease vertex
  // b=0, s<2, t=DC -> smooth vertex

  #local qaUQ=array[c2s3cU];	// the typing
  #local iU=0;#while(iU<c2s3cU)
    #local qaUQ[iU]=<0,0,0>;
  #local iU=iU+1;#end

  #local iM=0; #while(iM<c2s3cM)
    #local iA=c2s3aMA[iM];
    #local iB=c2s3aMB[iM];
    #if(c2s3aML[iM]=-1 | c2s3aMR[iM]=-1) // this is a border edge
      #local qaUQ[iA]=qaUQ[iA]+x;
      #local qaUQ[iB]=qaUQ[iB]+x;
    #end
    #local qaUQ[iA]=qaUQ[iA]+z;
    #local qaUQ[iB]=qaUQ[iB]+z;
  #local iM=iM+1;#end

  #local qaUT=array[c2s3cU];
  #local iU=0;#while(iU<c2s3cU)
// set smooth as default
    #local qaUT[iU]=qaUQ[iU].z;
// if border + sharp is greater than three, vertex is a corner
    #if(vdot(<1,0,0>,qaUQ[iU])>2)
      #local qaUT[iU]=0;
    #end
// if border + sharp = 2, vertex is a crease
    #if(vdot(<1,0,0>,qaUQ[iU])=2)
      #local qaUT[iU]=-1;
    #end
// if border =2, sharp=0, and total=2, vertex is a corner
// this will override previous test; this is desired
    #if(qaUQ[iU].x=2 & qaUQ[iU].z=2)
      #local qaUT[iU]=0;
    #end
  #local iU=iU+1;#end

  #undef qaUQ // no longer need these
  // the qaUT array now means:
  // -1 = crease vertex
  // 0 = corner
  // 1+ = smooth, with the value equaling the number of edges meeting there

  // set up mapping vertex array, initialize with weighting of original vertex
  #local qaUU=array[c2s3cU];
  #local iU=0;#while(iU<c2s3cU)
    #switch(qaUT[iU])
    #case(0)
      #local sW=1;
    #break
    #case(-1)
      #local sW=.75;
    #break
    #else
      #local sW=1-2/qaUT[iU];
    #end
    #local qaUU[iU]=c2s3aU[iU]*sW;
  #local iU=iU+1;#end

  // weight in the face points for the smooth points
  #local iR=0;
  #local iF=0;#while(iF<c2s3cF)  
    #local cS=c2s3aFU[iR];
    #local iR=iR+1;
    #local iJ=0;#while(iJ<cS)
      #local iU=c2s3aFU[iR];
      #local iR=iR+1;
      #if(qaUT[iU]>0)
        #local qaUU[iU]=qaUU[iU]+qaUF[iF]/qaUT[iU]/qaUT[iU];
      #end
    #local iJ=iJ+1;#end
  #local iF=iF+1; #end

  //weight in the edge points
  #local iM=0;#while(iM<c2s3cM)
    #local iA=c2s3aMA[iM];
    #local iB=c2s3aMB[iM];
  // first, weight in the value of the end to the start vertex point
    #switch(qaUT[iA])
      #case(0)
        #local sW=0;
      #break
      #case(-1)
        #if(c2s3aML[iM]=-1 | c2s3aMR[iM]=-1)
          #local sW=.125;
        #else
          #local sW=0;
        #end
      #break
    #else
      #local sW=1/qaUT[iA]/qaUT[iA];
    #end
    #if(sW>0)
      #local qaUU[iA]=qaUU[iA]+c2s3aU[iB]*sW;
    #end
  // then weight in the value of the start to the end vertex point
    #switch(qaUT[iB])
      #case(0)
        #local sW=0;
      #break
      #case(-1)
        #if(c2s3aML[iM]=-1 | c2s3aMR[iM]=-1)
          #local sW=.125;
        #else
          #local sW=0;
        #end
      #break
    #else
      #local sW=1/qaUT[iB]/qaUT[iB];
    #end
    #if(sW>0)
      #local qaUU[iB]=qaUU[iB]+c2s3aU[iA]*sW;
    #end
  #local iM=iM+1;#end

// we're now done with the typing array
  #undef qaUT
#end // end of stuff for texture-mapping

// 7. build replacement geometry edges

// new geometry edge count = oGEC*2 + 1 per side of each old face
// the last value is the size of the face-vertex array, minus the number of faces

#local qcE = c2s3cE*2 + dimension_size(c2s3aFV,1)-c2s3cF;

#local qaEA=array[qcE];
#local qaEB=array[qcE];
#local qaES=array[qcE];

#local iE=0;#while(iE<c2s3cE)
  #local qaEA[iE]=c2s3aEA[iE];
  #local qaEB[iE]=c2s3cV+iE;
  #local qaEB[iE+c2s3cE]=c2s3cV+iE;
  #local qaEA[iE+c2s3cE]=c2s3aEB[iE];
  #local qS=c2s3aES[iE]-1;
  #if(qS<0) #local qS=floor(qS)+1; #end
  #local qaES[iE]=qS;
  #local qaES[iE+c2s3cE]=qS;
#local iE=iE+1;#end

#local iR=0;
#local iE=c2s3cE*2;
#local iF=0;#while(iF<c2s3cF)
  #local cV=c2s3aFV[iR];
  #local iR=iR+1;
  #local iI=0;#while(iI<cV)
    #local qaEA[iE]=c2s3cV+c2s3cE+iF;
    #local qaEB[iE]=c2s3cV+c2s3aFE[iR+iI];
    #local qaES[iE]=0;
    #local iE=iE+1;
  #local iI=iI+1;#end
  #local iR=iR+cV;
#local iF=iF+1;#end

// 8. build replacement mapping edges

#ifdef(c2s3cU)
// new mapping edge count = oGEC*2 + 1 per side of each old face that has mapping
#local qcM=c2s3cM*2 ;
#local iF=0;#local iR=0;#while(iF<c2s3cF)
  #local cU=c2s3aFU[iR];
  #local iR=iR+cU+1;
  #local qcM=qcM+cU;
#local iF=iF+1;#end

#local qaMA=array[qcM];
#local qaMB=array[qcM];

#local iM=0;#while(iM<c2s3cM)
  #local qaMA[iM]=c2s3aMA[iM];
  #local qaMB[iM]=c2s3cU+iM;
  #local qaMB[iM+c2s3cM]=c2s3cU+iM;
  #local qaMA[iM+c2s3cM]=c2s3aMB[iM];
#local iM=iM+1;#end

#local iR=0;
#local iM=c2s3cM*2;
#local iMF=0;
#local iF=0;#while(iF<c2s3cF)
  #local cU=c2s3aFU[iR];
  #local iR=iR+1;
  #local iI=0;#while(iI<cU)
    #local qaMA[iM]=c2s3cU+c2s3cM+iMF;
    #local qaMB[iM]=c2s3cU+c2s3aFM[iR+iI];
    #local iM=iM+1;
  #local iI=iI+1;#end
  #if(cU>0) #local iMF=iMF+1; #end
  #local iR=iR+cU;
#local iF=iF+1;#end
#end // end of #ifdef(c2s3cU) block

// now it is time to build the new geometry faces

// new face count = 1 per side of each old face

#local qcF = dimension_size(c2s3aFV,1) - c2s3cF;

// build the face-vertex and face-edge arrays
// all new faces have four sides, so all vertex and edge records have four entries

#local qaFV=array[qcF*5];
#local qaFE=array[qcF*5];
#local qaFT=array[qcF];

#local iR=0;
#local iRN=0;
#local cVP=0;
#local iF=0;#while(iF<c2s3cF)
  #local cV=c2s3aFV[iR];
  #local iR=iR+1;
  #local iI=0;#while(iI<cV)
    #local iP=iI-1;
    #if(iI=0) #local iP=cV-1;#end
    #local qaFV[iRN]=4;
    #local qaFE[iRN]=4;
    #local iRN=iRN+1;
    #local qaFV[iRN+0]=c2s3aFV[iR+iI];
    #local qaFV[iRN+1]=c2s3cV+c2s3aFE[iR+iI];
    #local qaFV[iRN+2]=c2s3cV+c2s3cE+iF;
    #local qaFV[iRN+3]=c2s3cV+c2s3aFE[iR+iP];
    #local iE=c2s3aFE[iR+iI];
    #if(qaEA[iE]!=c2s3aFV[iR+iI]) #local iE=iE+c2s3cE; #end
    #local qaFE[iRN+0]=iE;
    #local qaFE[iRN+1]=c2s3cE*2+cVP+iI;
    #local qaFE[iRN+2]=c2s3cE*2+cVP+iP;
    #local iE=c2s3aFE[iR+iP];
    #if(qaEA[iE]!=c2s3aFV[iR+iI]) #local iE=iE+c2s3cE; #end
    #local qaFE[iRN+3]=iE;
    #local iRN=iRN+4;
  #local iI=iI+1;#end
  #local iR=iR+cV;
  #local cVP=cVP+cV;
#local iF=iF+1;#end

// and build the face-texture array
#ifdef(c2s3aFT)
#local iR=0;
#local iF=0;#while(iF<c2s3cF)
  #local cV=c2s3aFV[iR];
  #local iI=0;#while(iI<cV)
    #local qaFT[iR-iF]=c2s3aFT[iF];
    #local iR=iR+1;
  #local iI=iI+1;#end
  #local iR=iR+1;
#local iF=iF+1;#end
#end

// and build the edge-face arrays

// create and initialize the arrays
#local qaEL=array[qcE];
#local qaER=array[qcE];
#local iE=0;#while(iE<qcE)
  #local qaEL[iE]=-1;
  #local qaER[iE]=-1;
#local iE=iE+1;#end

// now build their required content
#local iR=0;
#local iF=0;#while(iF<qcF)
  #local cV=qaFE[iR];
  #local iR=iR+1;
  #local iI=0;#while(iI<cV)
    #local iE=qaFE[iR];
    #local iR=iR+1;
    #if(qaEL[iE]=-1)
      #local qaEL[iE]=iF;
    #else
      #if(qaER[iE]=-1)
        #local qaER[iE]=iF;
      #else
        #error "More than two faces border on a given edge.\n"
      #end
    #end
  #local iI=iI+1;#end
#local iF=iF+1;#end

// new geometry vertex count = oGVC + oEC + oFC

#local qcV=c2s3cV + c2s3cE + c2s3cF;

#ifdef(c2s3cU)

// and for the mapping:
// new mapping vertex count = old map vertex count + old map edge count + old mapped face count

#local qcU=c2s3cU + c2s3cM ;
#local iF=0;#local iR=0;#while(iF<c2s3cF)
  #local cU=c2s3aFU[iR];
  #local iR=iR+1;
  #if(cU)
    #local qcU=qcU+1;
    #local iR=iR+cU;
  #end
#local iF=iF+1;#end

// copy over mapping vertex points
#local qaU=array[qcU];
#local iU=0;#while(iU<c2s3cU)
  #local qaU[iU]=qaUU[iU];
#local iU=iU+1;#end
#undef qaUU

// and the mapping edge points
#local iM=0;#while(iM<c2s3cM)
  #local qaU[iU]=qaUM[iM];
  #local iU=iU+1;
#local iM=iM+1;#end
#undef qaUM

// and the mapping face points
#local iR=0;
#local iF=0; #while(iF<c2s3cF)
  #local cU=c2s3aFU[iR];
  #if(cU>0)
    #local qaU[iU]=qaUF[iF];
    #local iU=iU+1;
  #end
  #local iR=iR+cU+1;
#local iF=iF+1;#end
#undef qaUF

// mapping point array contains vert points, then edge points, then map points

#local qcX=0;
#local iRV=0;
#local iRU=0;
#local iF=0;#while(iF<c2s3cF)
  // one new record for every new face
  #local cV=c2s3aFV[iRV]; #local iRV=iRV+1;
  #local cU=c2s3aFU[iRU]; #local iRU=iRU+1;
  // one entry for every vertex of each face
  #local qcX=qcX+cV;
  // plus four entries for every vertex of faces with mapping data
  #local qcX=qcX+4*cU;
  #local iRU=iRU+cU;
  #local iRV=iRV+cV;
#local iF=iF+1;#end

#local qaFU=array[qcX];
#local qaFM=array[qcX];

// now build the new face-vertex and face-edge arrays for mapping

#local iRV=0;
#local iRU=0;
#local iRN=0;
#local iFU=0;
#local iRM=c2s3cM*2;
#local iF=0;#while(iF<c2s3cF)
  #local cV=c2s3aFV[iRV];
  #local iRV=iRV+1;
  #local cU=c2s3aFU[iRU];
  #local iRU=iRU+1;
  #if(cU=0)
    #local iI=0;#while(iI<cV)
      #local qaFU[iRN]=0;
      #local qaFM[iRN]=0;
      #local iRN=iRN+1;
    #local iI=iI+1;#end
  #else
    #local iI=0;#while(iI<cV)
      #local iU=c2s3aFU[iRU+iI];
      #local iJ=((iI=0)?(cV-1):(iI-1));
      #local qaFU[iRN]=4;
      #local qaFM[iRN]=4;
      #local qaFU[iRN+1]=iU;
      #local iM=c2s3aFM[iRU+iI];
      #local qaFU[iRN+2]=c2s3cU+iM;
      #if(c2s3aMA[iM]=iU)
        #local qaFM[iRN+1]=iM;
      #else
        #local qaFM[iRN+1]=c2s3cM+iM;
      #end
      #local qaFM[iRN+2]=iI+iRM;
      #local qaFM[iRN+3]=iJ+iRM;
      #local qaFU[iRN+3]=c2s3cU+c2s3cM+iFU;
      #local iM=c2s3aFM[iRU+iJ];
      #local qaFU[iRN+4]=c2s3cU+iM;
      #if(c2s3aMA[iM]=iU)
        #local qaFM[iRN+4]=iM;
      #else
        #local qaFM[iRN+4]=c2s3cM+iM;
      #end
      #local iRN=iRN+5;
    #local iI=iI+1;#end
    #local iFU=iFU+1;
  #end
  #local iRV=iRV+cV;
  #local iRU=iRU+cU;
  #local iRM=iRM+cU;
#local iF=iF+1;#end

#local qaML=array[qcM];
#local qaMR=array[qcM];
#local iM=0;#while(iM<qcM)
  #local qaML[iM]=-1;
  #local qaMR[iM]=-1;
#local iM=iM+1;#end

#local iR=0;
#local iF=0;#while(iF<qcF)
  #local cU=qaFU[iR];
  #local iR=iR+1;
  #if(cU>0)
    #local iI=0;#while(iI<cU)
      #local iM=qaFM[iR+iI];
      #if(qaML[iM]=-1)
        #local qaML[iM]=iF;
      #else
        #local qaMR[iM]=iF;
      #end
    #local iI=iI+1;#end
    #local iR=iR+cU;
  #end
#local iF=iF+1;#end

#end // end of #ifdef(c2s3cU) block

// now that the temporaries for everything are built, we copy them over

// copy over the geometry vertex data first
#undef c2s3aV
#declare c2s3aV=array[qcV];

#local iV=0;#while(iV<c2s3cV)
  #declare c2s3aV[iV]=qaVV[iV];
#local iV=iV+1;#end

#local iE=0;#while(iE<c2s3cE)
  #declare c2s3aV[iV]=qaVE[iE];
  #local iV=iV+1;
#local iE=iE+1;#end

#local iF=0;#while(iF<c2s3cF)
  #declare c2s3aV[iV]=qaVF[iF];
  #local iV=iV+1;
#local iF=iF+1;#end

#declare c2s3cV=qcV;

// copy over the geometry edges

#undef c2s3aEA
#undef c2s3aEB
#undef c2s3aEL
#undef c2s3aER
#undef c2s3aES

#declare c2s3cE=qcE;
#declare c2s3aEA=array[c2s3cE];
#declare c2s3aEB=array[c2s3cE];
#declare c2s3aEL=array[c2s3cE];
#declare c2s3aER=array[c2s3cE];
#declare c2s3aES=array[c2s3cE];
#local iE=0;#while(iE<c2s3cE)
  #declare c2s3aEA[iE]=qaEA[iE];
  #declare c2s3aEB[iE]=qaEB[iE];
  #declare c2s3aEL[iE]=qaEL[iE];
  #declare c2s3aER[iE]=qaER[iE];
  #declare c2s3aES[iE]=qaES[iE];
#local iE=iE+1;#end

// copy over the faces

#undef c2s3aFV
#undef c2s3aFE

#declare c2s3cF=qcF;
#declare c2s3aFV=array[c2s3cF*5];
#declare c2s3aFE=array[c2s3cF*5];
#local iR=0;#while(iR<c2s3cF*5)
  #declare c2s3aFV[iR]=qaFV[iR];
  #declare c2s3aFE[iR]=qaFE[iR];
#local iR=iR+1;#end

#ifdef(c2s3aFT)
  #undef c2s3aFT

  #declare c2s3aFT=array[c2s3cF];
  #local iF=0;#while(iF<c2s3cF)
    #declare c2s3aFT[iF]=qaFT[iF];
  #local iF=iF+1;#end
#end

// copy over the texture-mapping information
#ifdef(c2s3cU)

// first the mapping vertices
  #undef c2s3aU #declare c2s3aU=array[qcU];

  #local iU=0;#while(iU<qcU)
    #declare c2s3aU[iU]=qaU[iU];
  #local iU=iU+1;#end

  #undef qaU
  #declare c2s3cU=qcU;

// then the mapping edges
  #undef c2s3aMA #declare c2s3aMA=array[qcM];
  #undef c2s3aMB #declare c2s3aMB=array[qcM];
  #undef c2s3aML #declare c2s3aML=array[qcM];
  #undef c2s3aMR #declare c2s3aMR=array[qcM];

  #local iM=0;#while(iM<qcM)
    #declare c2s3aMA[iM]=qaMA[iM];
    #declare c2s3aMB[iM]=qaMB[iM];
    #declare c2s3aML[iM]=qaML[iM];
    #declare c2s3aMR[iM]=qaMR[iM];
  #local iM=iM+1;#end
  #undef qaMA
  #undef qaMB
  #undef qaML
  #undef qaMR
  #declare c2s3cM=qcM;

// and finally the mapping faces
  #undef c2s3aFU #declare c2s3aFU=array[qcX];
  #undef c2s3aFM #declare c2s3aFM=array[qcX];

  #local iX=0;#while(iX<qcX)
    #declare c2s3aFU[iX]=qaFU[iX];
    #declare c2s3aFM[iX]=qaFM[iX];
  #local iX=iX+1;#end

  #undef qaFU
  #undef qaFM
  
#end // of #ifdef(c2s3cU) check

#end // of #macro c2s3_Subdivide()

#end // #ifdef(C2S3_DIV_INC)

