# POV Primitive Mosaic, Linen Print
# Program for conversion of image into a set of shapes, simulating fabric,
# colored according to source image pixels, a-la textile printing
# (c) Ilya Razmanov (mailto:ilyarazmanov@gmail.com)
#
# Input: PNG
# Output: POVRay
#
# History:
# 2007  General idea for Kris Z.
# 2024  Complete internal rewriting for POVray. Versions from now on:
#
# 01.000    linen.py Initial release
# 01.001    Per knit normal added
# 01.002    Normal randomization
# 01.003    Node thickness randomisation to break monotonous shading pattern
#           Minimizing import
#
#       Project mirrors:
#       https://github.com/Dnyarri/POVmosaic
#       https://gitflic.ru/project/dnyarri/povmosaic
#

from tkinter.filedialog import askopenfilename
from tkinter.filedialog import asksaveasfile
from png import Reader  # I/O with PyPNG from: https://gitlab.com/drj11/pypng
from time import time
from time import ctime
from random import random

# Open source image
sourcefilename = askopenfilename(title='Open source PNG file', filetypes=[('PNG','.png')], defaultextension = ('PNG','.png'))
if (sourcefilename == ''):
    quit()

source = Reader(filename = sourcefilename)  # starting PyPNG

X,Y,pixels,info = source.asRGBA() # Opening image, iDAT comes to "pixels" as bytearray, to be tuple'd later

totalpixels = X*Y               # Total number OF pixels, not PIXEL NUMBER
rowlength = X*(info['planes'])  # Row length
Z = (info['planes'])            # Maximum CHANNEL NUMBER
imagedata = tuple((pixels))     # Attempt to fix all bytearrays as tuple

if (info['bitdepth'] == 8):
    maxcolors = 255             # Maximal value for 8-bit channel
if (info['bitdepth'] == 16):
    maxcolors = 65535           # Maximal value for 16-bit channel

# Open export file
resultfile = asksaveasfile(mode='w', title='Save resulting POV file', filetypes = 
            [
			('POV-Ray scene file', '*.pov'),
            ('All Files', '*.*'),],
    defaultextension = ('POV-Ray scene file','.pov'))
if (resultfile == ''):
    quit()
# Both files opened

# src a-la FM style src(x,y,z)
# Image should be opened as "imagedata" by main program before
# Note that X, Y, Z are not determined in function, you have to determine it in main program

def src(x, y, z):  # Analog src from FM, force repeate edge instead of out of range

    cx = x; cy = y
    cx = max(0,cx); cx = min((X-1),cx)
    cy = max(0,cy); cy = min((Y-1),cy)

    position = (cx*Z) + z   # Here is the main magic of turning two x, z into one array position
    channelvalue = int(((imagedata[cy])[position]))
    
    return channelvalue
# end of src function

def srcY(x, y):  # Converting to greyscale, returns Y, force repeate edge instead of out of range

    cx = x; cy = y
    cx = max(0,cx); cx = min((X-1),cx)
    cy = max(0,cy); cy = min((Y-1),cy)

    if (info['planes'] < 3):    # supposedly L and LA
        Yntensity = src(x, y, 0)
    else:                       # supposedly RGB and RGBA
        Yntensity = int(0.2989*src(x, y, 0) + 0.587*src(x, y, 1) + 0.114*src(x, y, 2))
    
    return Yntensity
# end of srcY function

#	WRITING POV FILE

# ------------
#  POV header
# ------------

resultfile.write('// Persistence of Vision Ray Tracer Scene Description File\n')
resultfile.write('// Vers: 3.7\n')
resultfile.write('// Description: Mosaic picture simulating woven fabric mesh (linen) with textile printing\n')
resultfile.write('// Auth: Automatically generated by linen.py program\n')
resultfile.write('// developed by Ilya Razmanov\n// mailto:ilyarazmanov@gmail.com\n\n')
resultfile.write('// Project mirrors:\n')
resultfile.write('// https://github.com/Dnyarri/POVmosaic\n')
resultfile.write('// https://gitflic.ru/project/dnyarri/povmosaic\n\n')
resultfile.write(f'// Converted from: {sourcefilename} ')
seconds = time()
localtime = ctime(seconds)
resultfile.write(f'at: {localtime}\n')
resultfile.write(f'// Source info: {info}\n\n')

#  Statements

resultfile.write('#version 3.7;\n\n')
resultfile.write('global_settings\n')
resultfile.write('{\n')
resultfile.write('  max_trace_level 3\n')
resultfile.write('  adc_bailout 0.01\n')
resultfile.write('  ambient_light <0.5, 0.5, 0.5>\n')
resultfile.write('  assumed_gamma 1.0\n')
resultfile.write('}\n\n')

# Standard includes
resultfile.write('#include "finish.inc"\n#include "golds.inc"\n#include "metals.inc"\n\n')

# Main element
resultfile.write('\n// Main torus, to be edited\n\n')
resultfile.write('#declare bublik = torus {0.5, 0.25 scale <1.0, 1.0, 1.0>}     // Default round thread\n')
resultfile.write('// #declare bublik = torus {0.5, 0.1 scale <1.0, 4.0, 1.0>}     // Makes flat ribbon weaving\n')
resultfile.write('// #declare bublik = torus {0.5, 0.4 scale <1.0, 0.2, 1.0>}     // Makes threads unrealistic yet funny\n')
resultfile.write('#declare thickscale = 0.5;    // Random thikness variation, 0.0 gives regular structure, > 2.0 not recommended\n')

resultfile.write('#declare thingie_finish = finish{ambient 0.1 diffuse 0.7 specular 0.8 roughness 0.001}\n')
resultfile.write('#declare color_factor = 1.0;  // Color multiplier for all channels\n\n')

resultfile.write('\n// Node properties - normal\n')
resultfile.write('#declare normalheight = 0.5;     // Normal intensity. Default 0.5\n')
resultfile.write('#declare normalangle = 15;       // Normal is rotated at this angle around z, degrees\n')
resultfile.write('#declare normalanglerange = 15;  // Normal rotation angle randomy varies within this range, degrees\n\n')
resultfile.write(f'#declare normalrand = seed({int(seconds*10000000)});  // Seeding random\n')

scalestring = (' scale <1.0+thickscale*(rand(normalrand)-0.5), 1.0, 1.0>')  # string for random scale
normalstring = (' normal {spiral1 5, normalheight scallop_wave scale <0.1, 0.1, 1.0> rotate x*90 rotate z*(normalangle+normalanglerange*(rand(normalrand)-0.5))}    // Normal for each fibre\n')    # string containing all normal text

# Object "thething" made of thingies

resultfile.write('\n// Object thething made out of thingies (nodes)\n')
resultfile.write('#declare thething = union {\n')  # Opening object "thething"

# Now going to cycle through image and build object

for y in range(0, Y, 1):

    resultfile.write(f'\n\n // Row {y}\n')

    for x in range(0, X, 1):

        r = float(src(x,y,0))/maxcolors; g = float(src(x,y,1))/maxcolors; b = float(src(x,y,2))/maxcolors    # Normalize colors to 0..1.0
        a = float(src(x,y,3))/maxcolors     # a = 0 - transparent, a = 1.0 - opaque
        tobeornottobe = random()     # to be used for alpha dithering
        yarkost = float(0.2989*r)+float(0.587*g)+float(0.114*b) # brightness, not used by default

        if (a > tobeornottobe):             # whether to draw thingie in place of partially transparent pixel or not

            if ((y+1)%2) == ((x+1)%2):           # chessboard pattern

                # krasnij
                resultfile.write('    object {bublik clipped_by{plane{x,0}}')
                resultfile.write(normalstring); resultfile.write(scalestring)
                resultfile.write(' pigment {')                  # open pigment
                resultfile.write(f'rgb <color_factor*{r}, color_factor*{g}, color_factor*{b}>')
                resultfile.write('} finish {thingie_finish} ')  # close pigment
                resultfile.write(f'translate <0, {y}, {x}>')    # placing half node
                resultfile.write('}\n')         # Closing half node

                # sinij
                resultfile.write('    object {bublik clipped_by{plane{-x,0}}')
                resultfile.write(normalstring); resultfile.write(scalestring)
                resultfile.write(' rotate x*90 pigment {')      # open pigment
                resultfile.write(f'rgb <color_factor*{r}, color_factor*{g}, color_factor*{b}>')
                resultfile.write('} finish {thingie_finish} ')  # close pigment
                resultfile.write(f'translate <0, {y}, {x}>')    # placing half node
                resultfile.write('}\n')         # Closing half node

            else:                                           # chessboard pattern 

                # sinij
                resultfile.write('    object {bublik clipped_by{plane{-x,0}}')
                resultfile.write(normalstring); resultfile.write(scalestring)
                resultfile.write(' pigment {')                   # open pigment
                resultfile.write(f'rgb <color_factor*{r}, color_factor*{g}, color_factor*{b}>')
                resultfile.write('} finish {thingie_finish} ')   # close pigment
                resultfile.write(f' translate <0, {y}, {x}>')    # placing half node
                resultfile.write('}\n')         # Closing half node

                # krasnij
                resultfile.write('    object {bublik clipped_by{plane{x,0}}')
                resultfile.write(normalstring); resultfile.write(scalestring)
                resultfile.write(' rotate x*90 pigment {')      # open pigment
                resultfile.write(f'rgb <color_factor*{r}, color_factor*{g}, color_factor*{b}>')
                resultfile.write('} finish {thingie_finish} ')  # close pigment
                resultfile.write(f' translate <0, {y}, {x}>')   # placing half node
                resultfile.write('}\n')         # Closing half node
                


# Transform object to fit 1, 1, 1 cube at 0, 0, 0 coordinates
resultfile.write('\n// Object transforms to fit 1, 1, 1 cube at 0, 0, 0 coordinates\n')
resultfile.write('translate <0.0, 0.5, 0.5>\n')   # compensate for -0.5 extra
resultfile.write(f'translate <0.0, -0.5*{Y}, -0.5*{X}>\n')  # translate to center object center at 0, 0, 0
resultfile.write(f'scale <-1.0/{max(X,Y)}, -1.0/{max(X,Y)}, 1.0/{max(X,Y)}>\n')  # rescale, mirroring POV coordinates

resultfile.write('} // thething closed\n')   # Closing object "thething"

# Insert object into scene
resultfile.write('object {thething}\n')

# Camera
proportions = max(X,Y)/X
resultfile.write('#declare camera_height = 2.0;\n\n')
resultfile.write('camera {\n   // orthographic\n    location <camera_height, 0.0, 0.0>\n    right x*image_width/image_height\n    up y\n    direction <0,0,1>\n    angle 2.0*(degrees(atan2(')
resultfile.write(f'{0.5 * proportions}')
resultfile.write(f', camera_height-(1.0/{max(X,Y)})))) // Supposed to fit object \n    look_at <0.0, 0.0, 0.0>')
resultfile.write('\n}\n\n')

# Light 1
resultfile.write('light_source {0*x\n   color rgb <1.1,1,1>\n   translate <4, 2, 3>\n}\n\n')
# Light 2
resultfile.write('light_source {0*x\n   color rgb <0.9,1,1>\n   translate <-2, 6, 7>\n}\n\n')
resultfile.write('\n/*\n\nhappy rendering\n\n  0~0\n (---)\n(.>|<.)\n-------\n\n*/')
# Close output
resultfile.close()