#local SplineTmpVersion = version;
#version 3.1;

// Una spline (spl) e' una curva in R^3, definita per tratti di cubiche.
// E' implementata come un array di array di 4 elementi, ognuno dei quali
// e' un vettore tridimensionale e rappresenta un parametro della cubica.

// Una slope (slp) e' una funzione definita a tratti di cubiche.
// E' implementata come un array di array di 6 elementi.
// I primi 4 elementi degli array minori sono i coefficienti di un tratto di cubica
// gli elementi [4] e [5] degli array indicano l'inizio e la fine di un tratto.

// Una splinelist (spll) e' un elenco di splines, ad ognuna delle quali e' 
// associato un valore crescente real.

/******************************************************************************
** Matrici di conversione da basi particolari alla base canonica polinomiale **
******************************************************************************/
/* Come leggere la matrice "M"
	|M00 M10 M20 M30|
	|M01 M11 M21 M31|
	|M02 M12 M22 M32|
	|M03 M13 M23 M33|
*/
#declare m4BesselMatrix = array[4][4] {
	{-1/2,   1,-1/2, 0}
	{ 3/2,-5/2,   0, 1}
	{-3/2,   2, 1/2, 0}
	{ 1/2,-1/2,   0, 0}
}

#declare m4BezierMatrix = array[4][4] {
	{-1, 3,-3, 1}
	{ 3,-6, 3, 0}
	{-3, 3, 0, 0}
	{ 1, 0, 0, 0}
}

#declare m4HermiteMatrix = array[4][4] {
	{ 2,-3, 0, 1}
	{-2, 3, 0, 0}
	{ 1,-2, 1, 0}
	{ 1,-1, 0, 0}
}

/******************************************************************************
**                   Funzioni generiche di vettor e numeri                   **
******************************************************************************/
// Restituisce i vettori di minimi e massimi.
#macro vMin(V1, V2)
	<min(V1.x, V2.x), min(V1.y, V2.y), min(V1.z, V2.z)>
#end
#macro vMax(V1, V2)
	<max(V1.x, V2.x), max(V1.y, V2.y), max(V1.z, V2.z)>
#end

// Restituisce la parte decimale di un numero
#macro Float(R) mod(R,1) #end

/******************************************************************************
**                        Funzioni di gestione array                         **
******************************************************************************/

// Restituisce l'ultimo elemento di un array
#macro LastElement(A) A[dimension_size(A,1)-1] #end

// Restituisce la dimensione di un array unidimensionale
#macro UBound(A) dimension_size(A,1) #end
 
// Limita la dimensione di un array
#macro aCrop(A, Size)
	#local SizeA = dimension_size(A,1);
	#if (SizeA > Size)
		#local New = array[Size]
		#local I = 0;
		#while (I < Size)
			#version -1.0;
			#local New[I] = A[I]
			#version 3.1;
			#local I=I+1;
		#end
		New
	#else
		A
	#end
#end

// Unisce due array ordinati in un unico array ordinato. Non sono ammesse ripetizioni.
#macro aMergeSorted(A1, A2)
	#local Size1 = dimension_size(A1,1);
	#local Size2 = dimension_size(A2,1);
	#local A = array[Size1+Size2]
	#local I=0;
	#local J=0;
	#local K=0;
	#while (I < Size1 & J < Size2)
		#if (A1[I] < A2[J])
			#local A[K] = A1[I];
			#local I = I + 1;
		#else
			#local A[K] = A2[J];
			#if (A1[I] = A2[J])
				#local I = I + 1;
			#end
			#local J = J + 1;
		#end
		#local K=K+1;
	#end
	#if (I >= Size1)
		#while (J < Size2)
			#local A[K] = A2[J];
			#local J = J + 1;
			#local K = K + 1;
		#end
	#else
		#while (I < Size1)
			#local A[K] = A1[I];
			#local I = I + 1;
			#local K = K + 1;
		#end
	#end
	aCrop(A, K)
#end

// Inserisce un nuovo elemento in un array
#macro aInsertElement(A, NewElem, Pos)
	#local Size = dimension_size(A, 1);
	#if (Pos > Size)
		#local Pos = Size;
	#end
	#local NewArray = array[Size+1]
	#local I = 0;
	#local J = 0;
	#while (I < Size)
		#if (I=Pos)
			#local NewArray[J] = NewElem;
			#local J=J+1;
		#end
		#local NewArray[J] = A[I];
		#local I=I+1;
		#local J=J+1;
	#end
	#if (I=Pos)
		#local NewArray[J] = NewElem;
	#end
	NewArray
#end

// Inserisce un nuovo elemento in un array di array
#macro InsertArrayElement(A, NewElem, Pos)
	#local Size = dimension_size(A, 1);
	#if (Pos > Size)
		#local Pos = Size;
	#end
	#local NewArray = array[Size+1]
	#local I = 0;
	#local J = 0;
	#while (I < Size)
		#if (I=Pos)
			#local NewArray[J] = NewElem
			#local J=J+1;
		#end
		#local NewArray[J] = A[I]
		#local I=I+1;
		#local J=J+1;
	#end
	#if (I=Pos)
		#local NewArray[J] = NewElem
	#end
	NewArray
#end

/******************************************************************************
**                     Funzioni polinomiali e matriciali                     **
******************************************************************************/
// Valuta un polinomio di terzo grado in un punto
#macro EvalCubic(V,X) (V[3] + X*(V[2] + X*(V[1] + X*V[0]))) #end

// Valuta la derivata di un polinomio di terzo grado in un punto
#macro EvalCubicDer(V,X) (V[2] + X*(2*V[1] + 3*X*V[0])) #end

// Valuta un polinomio di terzo grado in un punto come se fosse uno di secondo
// Utile per le derivate delle cubiche in cui V[0] = 0 sempre.
#macro EvalQuadric(V,X) (V[3] + X*(V[2] + X*V[1])) #end

// Moltiplica una matrice 4x4 per un vettore di lunghezza 4.
#macro v4MultiplyMatrix(M,V)
	array[4] {
		M[0][0]*V[0] + M[1][0]*V[1] + M[2][0]*V[2] + M[3][0]*V[3],
		M[0][1]*V[0] + M[1][1]*V[1] + M[2][1]*V[2] + M[3][1]*V[3],
		M[0][2]*V[0] + M[1][2]*V[1] + M[2][2]*V[2] + M[3][2]*V[3],
		M[0][3]*V[0] + M[1][3]*V[1] + M[2][3]*V[2] + M[3][3]*V[3]
	}
#end // v4MultiplyMatrix

// Moltiplica un vettore per una base. 
#macro vMatrix(B, V)
	<B[0].x * V.x + B[1].x * V.y + B[2].x * V.z,
	 B[0].y * V.x + B[1].y * V.y + B[2].y * V.z,
	 B[0].z * V.x + B[1].z * V.y + B[2].z * V.z>
#end

// Moltiplica tra loro due matrici di trasformazione 
#macro bTransformMatrix(M1, M2)
	array[4] {
		(vMatrix(M1, M2[0])), (vMatrix(M1, M2[1])), 
		(vMatrix(M1, M2[2])), (vMatrix(M1, M2[3]) + M1[3])
	}
#end

/******************************************************************************
**                        Creazione e gestione spline                        **
******************************************************************************/
// Crea una spline. Prende come argomento un vettore contenente l'elenco dei parametri
// dei segmenti nella base canonica
/*#macro splSpline(V)
	#local nDati = dimension_size(V,1);
	#if (mod(nDati,4) != 0)
	  #error "Per definire una spline il numero dei dati dev'essere un multiplo di 4\n"
	#end

	#local Spline = array[nDati/4]
	#local I=0;
	#while (I < nDati)
		#local Spline[int(I/4)] = array[4] {V[I], V[I+1], V[I+2], V[I+3]}
		#local I = I+4;
	#end
	Spline
#end // splSpline
*/

// Crea una spline. Prende come argomento un vettore contenente l'elenco dei parametri
// dei segmenti nella base di bezier
#macro splBezierSpline(V)
	#local nDati = dimension_size(V,1);
	#if (mod(nDati,4) != 0)
	  #error "Per definire una spline il numero dei dati dev'essere un multiplo di 4\n"
	#end

	// Conta il numero di buchi nella spline
	#local Holes = 0;
	/*#local I = 4;
	#while (I < nDati)
		#if (vlength(V[I]-V[I-1]) > 0)
			#local Holes=Holes+1;
		#end
		#local I = I+4;
	#end
   */
	#local Spline = array[nDati/4 + Holes]

	#local v4Segment = array[4] {V[0], V[1], V[2], V[3]}
	#local Spline[0] = v4MultiplyMatrix(m4BezierMatrix, v4Segment)
	#local I=4;
	#local J=1;
	#while (I < nDati)
		/*#if (vlength(V[I]-V[I-1]) > 0)
			// Qui c'e' il tappo di un buco!
			#local v4Segment = array[4] {V[I-1], 2/3*V[I-1] + 1/3*V[I],
			                             1/3*V[I-1] + 2/3*V[I], V[I]}
			#local Spline[J] = v4MultiplyMatrix(m4BezierMatrix, v4Segment)
			#local J=J+1;
		#end*/
		#local v4Segment = array[4] {V[I], V[I+1], V[I+2], V[I+3]}
		#local Spline[J] = v4MultiplyMatrix(m4BezierMatrix, v4Segment)
		#local I=I+4;
		#local J=J+1;
	#end
	Spline
#end // splBezierSpline

// Crea una spline. Prende come argomento un vettore contenente l'elenco dei punti
// di passaggio. Il primo e l'ultimo punto sono nodi di controllo.
#macro splCubicSpline(V)
	#local nDati = dimension_size(V,1);
	#local Spline = array[nDati-3]
	#local I=0;
	#while (I < nDati-3)
		#local v4Segment = array[4] {V[I], V[I+1], V[I+2], V[I+3]}
		#local Spline[int(I)] = v4MultiplyMatrix(m4BesselMatrix, v4Segment)
		#local I = I+1;
	#end
	Spline
#end

// Crea una curva spezzata. Prende come argomento un vettore contenente l'elenco dei vertici
#macro splLinearSpline(V)
   #local Dim = dimension_size(V,1)-1;
	#local Spline = array[Dim]
	#local I = 0;
	#while (I < Dim)
		#local Spline[I] = array[4] { <0, 0, 0>, <0, 0, 0>, V[I+1] - V[I], V[I] }
		#local I=I+1;
	#end
	Spline
#end 

// Restituisce i coefficienti di un dato segmento componente una spline. 
#macro aGetSegment(Spl,I) Spl[I] #end

// Restituisce il numero di componenti di una spline. Il valore coincide anche con
// il massimo del dominio utile
#macro GetSegmentsNumber(Spl) dimension_size(Spl,1) #end

// Valuta una spline in un punto. Il punto dovrebbe essere compreso 
// tra 0 e GetSegmentsNumber, ma la funzione accetta anche valori oltre l'intervallo utile:
// restituisce un punto rispettivamente sulla prima o sull'ultima componente, 
// mantenendo quindi le caratteristche di derivabilita' e di continuita'.
#macro EvalSpline(Spl,T)
	#local nSgm = GetSegmentsNumber(Spl);
	#if ((T>=0) & (T<nSgm))
		#local Offset = int(T);
		#local Pos1 = Float(T);
	#else
		#if (T < 0)
			#local Offset = 0;
			#local Pos1 = T;
		#else
			#local Offset = nSgm-1;
			#local Pos1 = T-nSgm+1;
		#end
	#end
	EvalCubic(Spl[Offset],Pos1)
#end // EvalSpline

#macro EvalDerSpline(Spl,T)
	#local nSgm = GetSegmentsNumber(Spl);
	#if ((T>=0) & (T<nSgm))
		#local Offset = int(T);
		#local Pos1 = Float(T);
	#else
		#if (T < 0)
			#local Offset = 0;
			#local Pos1 = T;
		#else
			#local Offset = nSgm-1;
			#local Pos1 = T-nSgm+1;
		#end
	#end
	EvalCubicDer(Spl[Offset],Pos1)
#end // EvalDerSpline

// Calcola gli estremi di una spline
#macro GetSplineBox(Spline, vMinBox, vMaxBox)
	#local N = GetSegmentsNumber(Spline);
	#local vMinBox = EvalSpline(Spline, N);
	#local vMaxBox = vMinBox;
	#local I = 0;
	#while (I < N)
		#local vMinBox = vMin(vMinBox, Spline[I][3]);
		#local vMaxBox = vMax(vMaxBox, Spline[I][3]);
		#local I=I+1;
	#end
#end

// Restituisce un vettore contenente il numero di punti richiesto sulla curva.
// Il primo e l'ultimo punto coincidono con le estremita' del supporto della curva.
#macro SampleSpline(Spl,Samples)
	#local nSgm = GetSegmentsNumber(Spl);
	#local Vector = array[Samples]
	#local Step = nSgm / (Samples-1);
	#local I=0;
	#while(I<Samples-1)
		#local Offset = int(I*Step);
		#local Pos = Float(I*Step);
		#local Vector[I] = EvalCubic(Spl[Offset],Pos);
		#local I=I+1;
	#end
	#local Vector[Samples-1] = EvalCubic(Spl[nSgm-1],1);
	Vector
#end // SampleSpline

// Restituisce un vettore contenente il numero di punti richiesto sulla curva.
// La curva e' una spline formata da un solo segmento.
#macro SampleSegment(Spl,Samples,Dest)
	#local Step = 1/(Samples-1);
	#local I=0;
	#while(I<Samples)
		#local Dest[I] = EvalCubic(Spl[0],I*Step);
		#local I=I+1;
	#end
#end // SampleSegment

// Restituisce un vettore contenente un campionamento della spline, in modo
// che la spezzata ottenibile collegando i punti abbia un'approssimazione che
// non ecceda il valore richiesto.
// La spline non avra' comunque meno di Div Elementi e l'algoritmo di bisezione
// giungera' ad una profondita' massima pari a Depth.
#macro SampleSplineError(Spl, Error, Div, Depth)
	// Ehm... bella prova!
	// No, lo so fare, ma non mi serve :)
#end // SampleSplineError

// Restituisce una spline unione di due splines
#macro splConcatSplines(Spl1, Spl2)
	#local I=0;
	#while (I < GetSegmentsNumber(Spl2))
		#local Spl1 = InsertArrayElement(Spl1, Spl2[I], GetSegmentsNumber(Spl1))
		#local I=I+1;
	#end
	Spl1
#end

// Restituisce la somma pesata di due splines che hanno lo stesso numero di componenti.
#macro splWeightedSumSplines(Spl1, W1, Spl2, W2)
	#local Segm = GetSegmentsNumber(Spl1);
	#if (Segm != GetSegmentsNumber(Spl2))
		#error "The splines must have the same number of knots\n"
	#end
	#local Sum = array[Segm]
	#local I=0;
	#while (I<Segm)
		#local Sum[I] = array[4] {
			W1 * Spl1[I][0] + W2 * Spl2[I][0],
			W1 * Spl1[I][1] + W2 * Spl2[I][1],
			W1 * Spl1[I][2] + W2 * Spl2[I][2],
			W1 * Spl1[I][3] + W2 * Spl2[I][3]
		}
		#local I=I+1;
	#end
	Sum
#end // splAverageSplines

#macro splSumSplines(Spl1, Spl2)
	splWeightedSumSplines(Spl1, 1, Spl2, 1)
#end

// Restituisce la somma di due spline

// Restituisce la media pesata di due splines che hanno lo stesso numero di componenti.
// C=0 -> restituisce Spl1
// C=1 -> restituisce Spl2
#macro splAverageSplines(Spl1, Spl2, W)
	//splWeightedSumSplines(Spl1, (1-W), Spl2, W)
	#local Segm = GetSegmentsNumber(Spl1);
	#local Sum = array[Segm]
	#local I=0;
	#while (I<Segm)
		#local Sum[I] = array[4] {
			Spl1[I][0] + W * (Spl2[I][0]-Spl1[I][0]),
			Spl1[I][1] + W * (Spl2[I][1]-Spl1[I][1]),
			Spl1[I][2] + W * (Spl2[I][2]-Spl1[I][2]),
			Spl1[I][3] + W * (Spl2[I][3]-Spl1[I][3])
		}
		#local I=I+1;
	#end
	Sum
#end

// restituisce la differenza tra due spline.
#macro splSubtractSplines(Spl1, Spl2)
	#local Segm = GetSegmentsNumber(Spl1);
	#local Diff = array[Segm]
	#local I=0;
	#while (I<Segm)
		#local Diff[I] = array[4] {
			Spl1[I][0]-Spl2[I][0],
			Spl1[I][1]-Spl2[I][1],
			Spl1[I][2]-Spl2[I][2],
			Spl1[I][3]-Spl2[I][3]
		}
		#local I=I+1;
	#end
	Diff
#end

// Restituisce una spline che e' la derivata della spline usata per parametro.
#macro splDifferentiateSpline(Spl)
	#local nSgm = GetSegmentsNumber(Spl);
	#local Spline = array[nSgm]
	#local I=0;
	#while(I<nSgm)
		#local Spline[I] = array[4] {<0, 0, 0>, 3*Spl[I][0], 2*Spl[I][1], Spl[I][2]}
		#local I=I+1;
	#end
	Spline
#end // DifferentiateSpline

// Restituisce una spline trasformata secondo una matrice.
// La matrice ha dimensioni [4] e contiene i tre vettori della base piu' un vettore
// di traslazione
#macro splMatrixSpline(Spl, B)
	#local nSgm = GetSegmentsNumber(Spl);
	#local Spline = array[nSgm]
	#local I=0;
	#while(I<nSgm)
		#local Spline[I] = array[4] {
			(vMatrix(B, Spl[I][0])), 
			(vMatrix(B, Spl[I][1])), 
			(vMatrix(B, Spl[I][2])), 
			(vMatrix(B, Spl[I][3]) + B[3])
		}
		#local I=I+1;
	#end
	Spline
#end

#macro splTranslateSpline(Spl, V)
	#local B = array[4] {<1, 0, 0>, <0, 1, 0>, <0, 0, 1>, V}
	#local Spline = splMatrixSpline(Spl, B)
	Spline
#end

#macro splScaleSpline(Spl, V)
	#local B = array[4] {<V.x, 0, 0>, <0, V.y, 0>, <0, 0, V.z>, <0, 0, 0>}
	#local Spline = splMatrixSpline(Spl, B)
	Spline
#end

#macro splRotateSpline(Spl, V)
	#local Spl1 = Spl
	#if (V.x != 0)
		#local B = array[4] {<1, 0, 0>, <0, cos(radians(V.x)), sin(radians(V.x))>,
		                     <0, -sin(radians(V.x)), cos(radians(V.x))>, <0, 0, 0>}
		#local Spl1 = splMatrixSpline(Spl1, B)
		#undef B
	#end
	#if (V.y != 0)
		#local B = array[4] {<cos(radians(V.y)), 0, sin(radians(V.y))>, <0, 1, 0>,
		                     <-sin(radians(V.y)), 0, cos(radians(V.y))>, <0, 0, 0>}
		#local Spl1 = splMatrixSpline(Spl1, B)
		#undef B
	#end
	#if (V.z != 0)
		#local B = array[4] {<cos(radians(V.z)), sin(radians(V.z)), 0>,
		                     <-sin(radians(V.z)), cos(radians(V.z)), 0>, 
		                     <0, 0, 1>, <0, 0, 0>}
		#local Spl1 = splMatrixSpline(Spl1, B)
		#undef B
	#end
	Spl1
#end

// Unisce due spline. Non necessariamente crea una spline continua.
#macro splCombineSplines(S1, S2)
	#local D1 = dimension_size(S1,1);
	#local D2 = dimension_size(S2,1);
	#local S = array[D1+D2]
	#local I = 0;
	#while (I < D1)
		#local S[I] = S1[I]
		#local I=I+1;
	#end
	#local I = 0;
	#while (I < D2)
		#local S[I+D1] = S2[I]
		#local I=I+1;
	#end
	S
#end

// Unisce due spline. La seconda viene traslata in modo che il primo vertice
// coincida con l'ultimo vertice della prima.
#macro splJoinSplines(S1, S2)
	#local D1 = GetSegmentsNumber(S1);
	#local Spost = EvalSpline(S1, D1) - EvalSpline(S2, 0);
	#local S3 = splTranslateSpline(S2, Spost)
	splCombineSplines(S1, S3)
#end

// Restituisce 1 se la curva e' chiusa
#macro IsClosedSpline(Spline)
	(vlength(EvalCubic(Spline[0],0) - 
	         EvalCubic(Spline[GetSegmentsNumber(Spline)-1],1)) = 0)
#end

// Restituisce 1 se la spline e' discontinua nel punto indicato.
#macro IsBrokenSpline(Spline, T)
	#if(Float(T))
		#local Res = 0;
	#else
	   #if ((T = 0) | (T = GetSegmentsNumber(Spline)))
	   	#local Sgm1 = GetSegmentsNumber(Spline)-1; #local Sgm2 = 0;
	   #else
	   	#local Sgm1 = T-1; #local Sgm2 = T;
	   #end
	   #local D = vlength(EvalCubic(Spline[Sgm1],1) - EvalCubic(Spline[Sgm2],0));
	   #local Res = (D > 1e-6);
	#end
	Res
#end

// Restituisce 0 se la spline e' discontinua oppure ha una cuspide di 180 gradi.
// Altrimenti, restituisce 1 se la spline ha una cuspide nel punto indicato.
#macro IsCuspSpline(Spline, T)
	#if (Float(T) | IsBrokenSpline(Spline, T))
		#local Res = 0;
	#else
	   #if ((T = 0) | (T = GetSegmentsNumber(Spline)))
	   	#local Sgm1 = GetSegmentsNumber(Spline)-1; #local Sgm2 = 0;
	   #else
	   	#local Sgm1 = T-1; #local Sgm2 = T;
	   #end
	   #local Res = (vlength(vcross(EvalCubicDer(Spline[Sgm1],1), 
	                                EvalCubicDer(Spline[Sgm2],0)))!=0);
	#end
	Res
#end

// Restituisce 1 se la spline ha una cuspide di 180 gradi nel punto indicato.
#macro Is180CuspSpline(Spline, T)
	#if (Float(T) | IsBrokenSpline(Spline, T))
		#local Res = 0;
	#else
	   #if ((T = 0) | (T = GetSegmentsNumber(Spline)))
	   	#local Sgm1 = GetSegmentsNumber(Spline)-1; #local Sgm2 = 0;
	   #else
	   	#local Sgm1 = T-1; #local Sgm2 = T;
	   #end
	   #local V1 = EvalCubicDer(Spline[Sgm1],1);
	   #local V2 = EvalCubicDer(Spline[Sgm2],0);
	   #local Res = (vlength(vcross(V1,V2)) = 0 & vdot(V1,V2) < 0);
	#end
	Res
#end

// Restituisce la derivata sinistra di una spline in un punto
#macro EvalLeftDerSpline(Spline, T)
	#if (Float(T))
		#local Res = EvalDerSpline(Spline, T);
	#else
	   #if ((T = 0) | (T = GetSegmentsNumber(Spline)))
	   	#local Sgm1 = GetSegmentsNumber(Spline)-1;
	   #else
	   	#local Sgm1 = T-1;
	   #end
	   #local Res = EvalCubicDer(Spline[Sgm1], 1);
	#end
	Res
#end

// Restituisce la derivata destra di una spline in un punto
#macro EvalRightDerSpline(Spline, T)
	#if (Float(T))
		#local Res = EvalDerSpline(Spline, T);
	#else
	   #if ((T = 0) | (T = GetSegmentsNumber(Spline)))
	   	#local Sgm2 = 0;
	   #else
	   	#local Sgm2 = T;
	   #end
	   #local Res = EvalCubicDer(Spline[Sgm2], 0);
	#end
	Res
#end

// Taglia una spline in modo da eliminare eventuali discontinuita'
// nei punti T0 e T1. La nuova spline non sara' ben definita
// in tutto l'intervallo ma solo tra T0 e T1.
// Saranno sufficientemente corrette (direi al primo ordine) anche
// le letture poco prima o poco dopo i tempi di taglio .
#macro splCropSpline(Spline, T0, T1)
	#local splNew = Spline
	#if ((T0 >= 1) & 
		  (IsCuspSpline(Spline, T0) | Is180CuspSpline(Spline, T0) | IsBrokenSpline(Spline, T0)))
		#local splNew[T0-1][0] = <0, 0, 0>;
		#local splNew[T0-1][1] = <0, 0, 0>;
		#local splNew[T0-1][2] = EvalCubicDer(Spline[T0], 0);
		#local splNew[T0-1][3] = EvalCubic(Spline[T0], 0) - splNew[T0-1][2];
	#end
	#if ((T1 <= GetSegmentsNumber(Spline) - 1) & 
	     (IsCuspSpline(Spline, T1) | Is180CuspSpline(Spline, T1) | IsBrokenSpline(Spline, T1)))
		#local splNew[T1][0] = <0, 0, 0>;
		#local splNew[T1][1] = <0, 0, 0>;
		#local splNew[T1][2] = EvalCubicDer(Spline[T1-1], 1);
		#local splNew[T1][3] = EvalCubic(Spline[T1-1], 1);
	#end
	splNew
#end

/******************************************************************************
**              Conversione tra ascisse temporali e curvilinee               **
******************************************************************************/
// Numero di passi in cui viene diviso un componente della spline per le
// approssimazioni sulla lunghezza
#declare CalcLengthSteps = 10;
#declare CalcLengthStep = 1/CalcLengthSteps;

// Ottimizza una spline per il calcolo delle lunghezze
#macro sltOptimizeSpline(Spline)
	
	#local nSgm = GetSegmentsNumber(Spline);
	#local nSteps = nSgm*CalcLengthSteps+1; 
	#local Vector = SampleSpline(Spline, nSteps)
	#local Length = 0;
	#local Table = array[nSteps]
	#local Table[0] = 0;
	
	#local I = 1;
	#while (I < nSteps)
		#local Table[I] = Table[I-1] + vlength(Vector[I]-Vector[I-1]);
		#local I=I+1;
	#end
	Table
#end // OptimizeSpline

// Calcola il parametro di lunghezza di corda corrispondente al parametro temporale T
#macro CalcChordAbscissa(Slot, T)
   #local fStep = T / CalcLengthStep;
   #local N = dimension_size(Slot, 1) - 1;
   #local Step = int(fStep);
	#if ((Step>=0) & (Step < N))
		#local Return = Slot[Step] + Float(fStep)*(Slot[Step+1]-Slot[Step]);
	#else
		#if (T<0)
			#local Return = ((T/CalcLengthStep)*Slot[1]);
		#else
			#local Return = Slot[N-1] + (T/CalcLengthStep-N+1)*(Slot[N]-Slot[N-1]);
		#end
	#end
	Return
#end

// Calcola il parametro temporale corrispondente al parametro di lunghezza di corda L
#macro CalcTimeAbscissa(Slot,L)
	#local P1 = 0;
	#local P2 = dimension_size(Slot, 1) - 1;
	#if ((L >= 0) & (L < Slot[P2]))
		#while (P2 > P1 + 1)
			#local P3 = int((P1+P2)/2);
			#if (L > Slot[P3])
				#local P1 = P3;
				#if (L < Slot[P3+1]) #local P2 = P3 + 1; #end
			#else
				#local P2 = P3;
				#if (L > Slot[P3-1]) #local P1 = P3 - 1; #end
			#end
		#end
		#local Res = ((P1 + (L-Slot[P1])/(Slot[P2]-Slot[P1])) * CalcLengthStep);
	#else
		#if (L < 0)
			#local Res = (L / Slot[1]) * CalcLengthStep;
		#else
			#local P1 = P2 - 1;
			#local Res = ((P1 + (L-Slot[P1])/(Slot[P2]-Slot[P1])) * CalcLengthStep);
		#end
	#end
	Res
#end

#macro CalcSplineLength(Slot)
	Slot[dimension_size(Slot, 1) - 1]
#end

/******************************************************************************
**                        Creazione e gestione slopes                        **
******************************************************************************/
#macro v4BuildReParamMatrix(P1,L)
	#local L1 = 1/L; #local L2 = L1*L1; #local L3 = L1*L2;
	#local P2 = P1*P1; #local P3 = P1*P2;
	array[4][4] {
		{L3, -3*P1*L3,  3*P2*L3, -P3*L3}
		{0,        L2, -2*P1*L2,  P2*L2}
		{0,   0,             L1, -P1*L1}
		{0,   0,        0,            1}
	}
#end

#macro slpSlope(V)
	#local nDati = dimension_size(V,1);
	#local Slope = array[nDati - 1]
	#local I = 0; // Conta sul vettore di input
	#local J = 0; // conta sul vettore slope
	#while (I < nDati-1)
		#local Length = (V[I+1]-V[I]).x;
		#if (Length)
			#local v4Segment = array[4] {V[I].y, V[I+1].y, V[I].z*Length, V[I+1].z*Length}
			#local v4Segment = v4MultiplyMatrix(m4HermiteMatrix, v4Segment)
			#local v4ReParam = v4BuildReParamMatrix(V[I].x,Length)
			#local v4Segment = v4MultiplyMatrix(v4ReParam, v4Segment)
			#local Slope[J] = array[6] {
				v4Segment[0], v4Segment[1], v4Segment[2], v4Segment[3], V[I].x, V[I+1].x
			}
			#local J = J + 1;
		#end
		#local I=I+1;
	#end
	aCrop(Slope,J)
#end

#macro slpLinearSlope(V)
	#local nDati = dimension_size(V,1);
	#local Slope = array[nDati - 1]
	#local I = 0; // Conta sul vettore di input
	#local J = 0; // conta sul vettore slope
	#while (I < nDati-1)
		#local Length = V[I+1].u - V[I].u;
		#if (Length)
			#local V1 = V[I].v;
			#local V2 = V[I+1].v;
			#local v4Segment = v4MultiplyMatrix(m4HermiteMatrix, array[4] {V1, V2, V2-V1, V2-V1})
			#local v4ReParam = v4BuildReParamMatrix(V[I].u,Length)
			#local v4Segment = v4MultiplyMatrix(v4ReParam, v4Segment)
			#local Slope[J] = array[6] {
				v4Segment[0], v4Segment[1], v4Segment[2], v4Segment[3], V[I].u, V[I+1].u
			}
			#local J = J + 1;
		#end
		#local I=I+1;
	#end
	aCrop(Slope,J)
#end

#macro slpLinearSlopeBug(V)
	#local nDati = dimension_size(V,1);
	#local Slope = array[nDati - 1]
	#local I = 0; // Conta sul vettore di input
	#local J = 0; // conta sul vettore slope
	#while (I < nDati-1)
		#local Length = (V[I+1]-V[I]).x;
		#if (Length)
			#local V1 = V[I].y;
			#local V2 = V[I+1].y;
			#local v4Segment = v4MultiplyMatrix(m4HermiteMatrix, array[4] {V1, V2, V2-V1, V2-V1})
			#local v4ReParam = v4BuildReParamMatrix(V[I].x,Length)
			#local v4Segment = v4MultiplyMatrix(v4ReParam, v4Segment)
			#local Slope[J] = array[6] {
				v4Segment[0], v4Segment[1], v4Segment[2], v4Segment[3], V[I].x, V[I+1].x
			}
			#local J = J + 1;
		#end
		#local I=I+1;
	#end
	aCrop(Slope,J)
#end

// Restituisce il numero di componenti di una slope.
#macro GetSlopeSegmentsNumber(Slp) dimension_size(Slp,1) #end

// Restituisce un segmento di slope
#macro GetSlopeSegment(Slope,I) Slope[I] #end

// Restituisce l'inizio di un segmento di slope
#macro GetSlopeSegmentStart(Sgm) Sgm[4] #end

// Restituisce la lunghezza di un segmento di slope
#macro GetSlopeSegmentLength(Sgm) (Sgm[5]-Sgm[4]) #end

// Restituisce la fine di un segmento di slope
#macro GetSlopeSegmentEnd(Sgm) Sgm[5] #end

// Restituisce l'inizio di una slope
#macro GetSlopeStart(Slope) Slope[0][4] #end

// Restituisce la fine di una slope
#macro GetSlopeEnd(Slope) 
	Slope[dimension_size(Slope,1)-1][5]
#end

// Calcola il valore di una slope in un punto
#macro EvalSlope(Slope, X)
	#local nSgm = GetSlopeSegmentsNumber(Slope);
	#if ((X > Slope[0][5]) & (X < Slope[nSgm-1][4]))
		#local I=1; #local J=nSgm-2; #local M = div(I+J,2);
		#while ((X < Slope[M][4]) | (X > Slope[M][5]))
			#if (X < Slope[M][4])
				#local J = M-1;
			#else
				#local I = M+1;
			#end 
			#local M = div(I+J,2);
		#end
	#else
		#if (X <= Slope[0][5])
			#local M = 0;
		#else
			#local M = nSgm-1;
		#end
	#end
	EvalCubic(Slope[M],X)
#end // EvalSlope

#macro EvalSlopeSegment(Segment, X)
	EvalCubic(Segment,X)
#end

#macro EvalSlopeSegmentDer(Segment, X)
	EvalCubicDer(Segment,X)
#end

// Indica se c'e' una discontinuita' in un punto della slope o in una sua derivata. 
#macro IsDiscontinualSlope(Slope, X)
	#local nSgm = GetSlopeSegmentsNumber(Slope);
	#if ((X >= GetSlopeSegmentStart(Slope[0])) & (X <= GetSlopeSegmentEnd(Slope[nSgm-2])))
		#local I = 0;
		#while (X > GetSlopeSegmentEnd(Slope[I]))
			#local I=I+1;
		#end
		#if (X < GetSlopeSegmentEnd(Slope[I]))
			#local Res = 0;
		#else
			#local Res = ((EvalCubic(Slope[I],X) != EvalCubic(Slope[I+1],X)) |
			              (EvalCubicDer(Slope[I],X) != EvalCubicDer(Slope[I+1],X)));
		#end
	#else
		#local Res = 0;
	#end
	Res
#end

// Taglia una slope all'interno di un intervallo.
#macro slpCropSlope(Slope, Start, End)
	#if (End <= GetSlopeStart(Slope))
		#local Return = array[1] {Slope[0]}
	#else
		#if (Start >= GetSlopeEnd(Slope))
			#local Return = array[1] {Slope[dimension_size(Slope,1)-1]}
		#else
			#local First=0;
			#while (Start >= GetSlopeSegmentEnd(Slope[First]))
				#local First=First+1;
			#end
			#local Last = First;
			#local Exit = 0;
			#while (!Exit)
				#if (Last < dimension_size(Slope,1))
					#if (End > GetSlopeSegmentEnd(Slope[Last]))
						#local Last=Last+1;
					#else
						#local Exit = 1;
					#end
				#else
					#local Last=Last-1;
					#local Exit = 1;
				#end
			#end
			#local Return = array[Last-First+1]
			#local I=First;
			#while (I <= Last)
				#local Return[I-First] = Slope[I]
				#local I=I+1;
			#end
		#end
	#end
	Return
#end

/******************************************************************************
**                      Creazione e gestione spline list                     **
******************************************************************************/
#macro sllCreateSplineList(Spl, Pos)
	array[2] {array[1] {Pos}, Spl}
#end

// Aggiunge una spline ad una spline list.
#macro sllAddSpline(SplList, Spl, Pos)
	#if (Pos > LastElement(SplList[0]))
		#local I = dimension_size(SplList[0],1);
	#else
		#local I = 0;
		#while (Pos >= SplList[0][I])
			#local I=I+1;
		#end
	#end
	#local SplList[0] = aInsertElement(SplList[0], Pos, I)
	#local SplList = InsertArrayElement(SplList, Spl, I+1)
	SplList
#end // spllAddSpline

// Restituisce la posizione di una spline da una lista
#macro GetSplinePosition(List, I)
	List[0][I]
#end

// Restituisce una spline da una lista
#macro splGetSpline(List, I)
	#if (I+1 >= dimension_size(List,1)) 
		List[dimension_size(List,1)-1]
	#else
		List[I+1]
	#end
#end

// Restituisce il numero di splines in una lista
#macro GetSplinesNumber(List)
	dimension_size(List[0],1)
#end

// Restituisce un valore real che indica le spline precedenti e sucessive ad una 
// posizione X. La parte intera del numero e' la spline precedente; la parte 
// decimale e' un coefficiente (da 0 a 1) che indica la posizione precisa nel tratto
// tra la spline indicata e quella successiva.
#macro GetWeightedIndex(List, X)
	#if ((X > List[0][0]) & (X < LastElement(List[0])))
		#local I=0;
		#while (List[0][I+1] < X)
			#local I=I+1;
		#end
	#else
		#if (X <= List[0][0])
			#local I=0;
		#else
			#local I=dimension_size(List[0],1)-2;
		#end
	#end
	(I+(X-List[0][I])/(List[0][I+1]-List[0][I]))
#end

// Restituisce un valore real che indica la spline precedente una posizione X.
#macro GetSplineIndex(List, X)
	#if ((X > List[0][0]) & (X < LastElement(List[0])))
		#local I=0;
		#while (List[0][I+1] < X)
			#local I=I+1;
		#end
	#else
		#if (X <= List[0][0])
			#local I=0;
		#else
			#local I=dimension_size(List[0],1)-2;
		#end
	#end
	I
#end

#version SplineTmpVersion;
