/*-----------------------------------------------
p_surf.inc
An include file for generating parameterized
"uv" surfaces with POV-Ray 3.1.

The following requires that a vector valued macro
named 'XYZ' and variables uusteps, vvsteps (mesh
resolution), uumin, uumax, vvmin, and vvmax (limits
on the uu and vv variables)  have been previously
defined.

If an additional macro 'mytexture1(integer,integer)'
been defined, it will be used to texture individual
triangles.
	
A particular caveat for making smooth triangles
is that even when uumin, uumax,vvmin, and vvmax have
been carefully set to avoid trouble (square roots of
negative numbers, etc), in fact the routine for calculating 
normal vectors explores areas of the u-v plane
outside of the uumin->uumax X vvmin->vvmax rectangle.
This can lead to errors if XYZ(uu,vv) happens to explore a 
'sore spot'. A way to avoid these problems is to have 
XYZ(uu,vv) return a 'good' value if <uu,vv> are in a bad place.


-----------------------------------------------*/
#if (version<3.1)
	#error "This file requires features of POV-Ray version 3.1 (or later?)"
#end


#if(MakeTriangles) //MakeTriangles set to 1 by the calling file
	#ifdef(WriteTrianglesTo)	// WriteTrianglesTo is name for 
						// triangle data file to write to
		#fopen DataFileHandle WriteTrianglesTo write
		#if ( !defined(DataFileHandle))
			#warning concat("\nUnable to open ",WriteTrianglesTo)
			#undef WriteTrianglesTo
		#end
	#end
     
	// To speed up triangle and normals calculations,
	// we will compute the surface points and store
	// them in XYZ_Array
	#local XYZ_Array=array[uusteps+3][vvsteps+3]
     
	// If we want smooth triangles, we will pre-compute
	// and store the normals too.
	#if(use_smooth_triangles)
		#local Normal_Array=array[uusteps+3][vvsteps+3]
	#end

	// uu,vv lattice interval sizes
	#local duu= (uumax-uumin)/uusteps;//u-increment per step
	#local dvv= (vvmax-vvmin)/vvsteps;//v-increment per step

	#debug "Calculating surface points"
	#local vv= vvmin-dvv;  
	#local vvcount = 0;
	#while (vvcount<vvsteps+3) // vv loop
		#local uucount = 0;
		#local uu = uumin-duu;
		#while (uucount<uusteps+3) // uu loop
			#local XYZ_Array[uucount][vvcount]=XYZ(uu,vv);
			#local uucount=uucount+1;
			#local uu = uu+duu;
		#end  //end uu loop  		

		#local vvcount = vvcount + 1;
		#local vv = vv+dvv;
	#end	//end vv loop
     
	#if(use_smooth_triangles)
		#debug "\nCalculating normals"
		#local vv= vvmin;  
		#local vvcount = 1;

		#while (vvcount<=vvsteps+1) // vv loop
			#local uucount=1;
			#local uu = uumin;
			#while (uucount<=uusteps+1) // uu loop			
				#local p1=XYZ_Array[uucount][vvcount];
				#local p2=XYZ_Array[uucount+1][vvcount];
				#local p4=XYZ_Array[uucount][vvcount+1];
				#local p1a=XYZ_Array[uucount-1][vvcount];
				#local p1b=XYZ_Array[uucount][vvcount-1];
/*  
                  p4
                  | 
                  | 
                  | 
         p1a-----p1------p2
                  |
                  |         
     /|\          |       
      v          p1b     
       u->    
*/
				// Calculate normal at p1:
				#local n1=vcross(p1-p1b,p1-p1a);
				#local n2=vcross(p1-p2,p1-p1b);
				#local n3=vcross(p1-p4,p1-p2);
				#local n4=vcross(p1-p1a,p1-p4);
				#local Normal_Array[uucount][vvcount]=n1+n2+n3+n4;
     
				#local uucount=uucount+1;
				#local uu = uu+duu;
			#end  //end uu loop  		
     		
			#local vvcount = vvcount + 1;
			#local vv = vv+dvv;
		#end	//end vv loop
	#end //end if(use_smooth_triangles)
          
	#debug "\nMaking "
	#if (defined(DataFileHandle))
		#debug "and writing "
	#end
	#debug "triangles"

	#declare NumberOfTriangles=0;
	#local vvcount = 1;
	#declare p_surface =
	mesh{
	#while (vvcount<=vvsteps) // vv loop
		//#debug concat(str(vvcount,0,0)," / ",str(vvsteps,0,0),"\r")
                             
		#local uucount=1;
     
		#while (uucount<=uusteps) // uu loop			
                             
			// get triangle vertices p1,p2,p3,p4
			// Each uuXvv rectangle gives 2 triangles
			#local p1=XYZ_Array[uucount][vvcount];
			#local p2=XYZ_Array[uucount+1][vvcount];
			#local p3=XYZ_Array[uucount+1][vvcount+1];
			#local p4=XYZ_Array[uucount][vvcount+1];
/*
                 p4a     p3b
                  .       .
                  .       .
         p4b.... p4------p3.... p3a
                  | tr2.  |
                  |  .    |
                  |.  tr1 |
         p1a.... p1------p2.... p2b
                  .       .  
     /|\          .       .
      v          p1b     p2a   
       u->
         
*/             
			// if a texture macro named mytexture1
			// has been declared, use it for
			// the triangles
			#ifdef(mytexture1)
				#local T1=mytexture1(uucount,vvcount)
			#end
                 
			#if(use_smooth_triangles)
				// get normals:
				#local n1=Normal_Array[uucount][vvcount];
				#local n2=Normal_Array[uucount+1][vvcount];
				#local n3=Normal_Array[uucount+1][vvcount+1];
				#local n4=Normal_Array[uucount][vvcount+1];
				//make smooth triangles
                         
				smooth_triangle  {
					p1, n1,
					p2, n2,
					p3, n3
					#ifdef(T1)
						texture{T1}
					#end
				}
     
				smooth_triangle  {
					p1, n1,
					p3, n3,
					p4, n4
					#ifdef(T1)
						texture{T1}
					#end
				}
				#if(defined(DataFileHandle))
     					
					#write (DataFileHandle,p1,",",n1,",",p2,",",n2,",",p3,",",n3,",","\n")
					#write (DataFileHandle,p1,",",n1,",",p3,",",n3,",",p4,",",n4,",","\n")
     
				#end
			#else // 'flat' triangles
     				
				triangle  {
					p1,p2,p3
					#ifdef(mytexture1)
						texture{T1}
					#end
				}
				triangle  {
					p1,p3,p4
					#ifdef(mytexture1)
						texture{T1}
					#end
				}  
				#if(defined(DataFileHandle))
     					
					#write (DataFileHandle,p1,",",p2,",",p3,",","\n")
					#write (DataFileHandle,p1,",",p3,",",p4,",","\n")
     
				#end             
			#end
			#declare NumberOfTriangles=NumberOfTriangles+2;
			#local uucount=uucount+1;
     		
		#end  //end uu loop  		
     		
		#local vvcount = vvcount + 1;
     		
	#end	//end vv loop
	} // end of mesh
	#ifdef(WriteTrianglesTo)
		#fclose DataFileHandle
		#debug concat("\nWrote ",str(NumberOfTriangles,0,0)," triangles to \"",WriteTrianglesTo,"\"\n")
	#end

#else // if not making triangles, we read them

	#fopen DataFileHandle ReadTrianglesFrom read
	#debug concat("Reading triangles from \"",ReadTrianglesFrom,"\" ... ")

	#declare p_surface=
	mesh{
		#declare NumberOfTriangles=0;
		#while(defined(DataFileHandle))
			#if(use_smooth_triangles)
				#read(DataFileHandle,p1,n1,p2,n2,p3,n3)
				smooth_triangle{
					p1,n1,p2,n2,p3,n3
				}
			#else
				#read(DataFileHandle,p1,p2,p3)
				triangle{
					p1,p2,p3
				}

			#end
			#declare NumberOfTriangles=NumberOfTriangles+1;
		#end
		#debug concat("(",str(NumberOfTriangles-1,0,0)," read)\n")
	}


#end


#declare MINUU=1;
#declare MAXUU=2;
#declare MINVV=4;
#declare MAXVV=8;
#macro p_edge( whichEdge,edgeDivisions,edgeRadius)

  #switch (whichEdge)
          #case (MINUU)
                #local edgemin=vvmin;
                #local edgemax=vvmax;
                #local edgeu=uumin;
                #local edgev=vvmin;
          #break
          #case (MAXUU)
                #local edgemin=vvmin;
                #local edgemax=vvmax;
                #local edgeu=uumax;
                #local edgev=vvmin;
          #break 
        
          #case (MINVV)
                #local edgemin=uumin;
                #local edgemax=uumax;
                #local edgeu=uumin;
                #local edgev=vvmin;
          #break 
          #case (MAXVV)
                #local edgemin=uumin;
                #local edgemax=uumax;
                #local edgeu=uumin;
                #local edgev=vvmax
          #break 
  #end
  #local uu=edgeu;
  #local vv=edgev;
  #local de=(edgemax-edgemin)/edgeDivisions;
  #local edgeN=0;
  union{
        #local lastvertex=XYZ(uu,vv);
        #while(edgeN<=edgeDivisions)
               #local thisvertex=XYZ(uu,vv);
               #if(lastvertex.x!=thisvertex.x | lastvertex.y!=thisvertex.y | lastvertex.z!=thisvertex.z)
                         cylinder{lastvertex,thisvertex,edgeRadius }
               #end
               sphere{thisvertex,edgeRadius}
               #local lastvertex=thisvertex;
               #switch(whichEdge)
                   #case (MINUU)
                        #local vv=vv+de;
                   #break
                   #case (MAXUU)
                        #local vv=vv+de;
                   #break
                   #case (MINVV)
                        #local uu=uu+de;
                   #break
                   #case (MAXUU)
                        #local uu=uu+de;
                   #break
               #end
               #local edgeN=edgeN+1;
        #end
  }
#end

