/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* *                                                                                                                 * */
/* * [BORDERED CHARACTERS VERSION 1.8]                                                                               * */
/* *                                                                                                                 * */
/* * INSTRUCTIONS FOR USE:                                                                                           * */
/* *                                                                                                                 * */
/* * This file contains some useful macros to help you manage the character objects as lines of text. The fact that  * */
/* * some characters fit "inside" other characters (for example, in the sequence "P." the top of the letter "P"      * */
/* * extends beyond the right edge of the period) greatly complicates the task of applying the kerning values. These * */
/* * macros are not perfect, however, the character combinations that will cause them to fail are not commonly used  * */
/* * in normal writing.                                                                                              * */
/* *                                                                                                                 * */
/* * The macros in this file work together to create an object called "Paragraph." The Paragraph object consists of  * */
/* * one or more lines of characters arranged vertically. Each line can consist of one or more characters. It's what * */
/* * an actual written paragraph is.                                                                                 * */
/* *                                                                                                                 * */
/* * Under most circumstances, the only macros you will need to call are SetFontStyle (), InitializeVariables (),    * */
/* * and CreateParagraph (). The rest of the macros are called internally by these macros.                           * */
/* *                                                                                                                 * */
/* * SetFontStyle () Macro:                                                                                          * */
/* * ----------------------                                                                                          * */
/* *                                                                                                                 * */
/* * The term "Font Style" refers to indexed records of settings that partially control the appearance of the        * */
/* * characters. The "Font Style Index" identifies a specific "Font Style" record. There can be 100 "Font Style"     * */
/* * records within any scene, numbered 0 - 99.                                                                      * */
/* *                                                                                                                 * */
/* * "Font Styles" are applied to the characters within the Paragraph object by embedding formatting codes within    * */
/* * the text of your message. The formatting code consists of a "Font Style Index" number surrounded by TILDES      * */
/* * ("~"). The index number indicates which pre-defined "Font Style" to apply to the characters following the       * */
/* * formatting code. For example:                                                                                   * */
/* *                                                                                                                 * */
/* *    #declare LineText [0] = "~0~This text uses Font Style #0. ~1~This text uses Font Style #1.";                 * */
/* *                                                                                                                 * */
/* * The SetFontStyle () macro has the following parameters:                                                         * */
/* *                                                                                                                 * */
/* *    1: [FontStyleIndex] Indicates which "Font Style" record is to be set using the remaining parameters. Valid   * */
/* *    values are 0 - 99.                                                                                           * */
/* *                                                                                                                 * */
/* *    2: [FontVariationNumber] This variable refers to the font variations defined in the file                     * */
/* *    "BorderChars_FontVariations.inc."                                                                            * */
/* *                                                                                                                 * */
/* *    3: [P0], [P1],... [P8] Variation-specific variables. Copy and paste from "Demo_FontVariations.pov." [P0] is  * */
/* *    generally used for the depth of the characters along the z axis.                                             * */
/* *                                                                                                                 * */
/* *    4: [TextColor_Outer] This texture that will be applied to the "outside" of the characters.                   * */
/* *                                                                                                                 * */
/* *    5: [TextColor_Inner] This texture that will be applied to the "inside" of the characters.                    * */
/* *                                                                                                                 * */
/* *    6: [TextColor_Back] This texture that will be applied to the back of the characters.                         * */
/* *                                                                                                                 * */
/* * In addition to the settings contained in the "Font Style" records, the appearance of the characters is affected * */
/* * by three additional factors:                                                                                    * */
/* *                                                                                                                 * */
/* *    1: Any additional font variation-specific elements defined as global variables. (See font variation number 4 * */
/* *    for an example.)                                                                                             * */
/* *                                                                                                                 * */
/* *    2: Italicization. Text bracketed by CURLY BRACKETS ("{" and "}") will be italicized. If the end of a line    * */
/* *    containing italicized text is reached and no closing CURLY BRACKET is located, the next line will be reset   * */
/* *    to non-italicized.                                                                                           * */
/* *                                                                                                                 * */
/* *    3: Scale. See [TextScale] and [MaxMicroJustificationScale] below.                                            * */
/* *                                                                                                                 * */
/* * InitializeVariables () Macro:                                                                                   * */
/* * -----------------------------                                                                                   * */
/* *                                                                                                                 * */
/* * This is a "housekeeping" macro that initializes several arrays that are used during the creation of the         * */
/* * Paragraph object. Its only parameter is the number of lines of text in your paragraph.                          * */
/* *                                                                                                                 * */
/* * CreateParagraph () Macro:                                                                                       * */
/* * -------------------------                                                                                       * */
/* *                                                                                                                 * */
/* * This macro creates an object called "Paragraph" which is a graphical representation of the lines of text you    * */
/* * specified in the LineText [] array elements, using all the formatting and styling tools available.              * */
/* *                                                                                                                 * */
/* * The appearance of the Paragraph object is controlled by a number of things including the parameters of the      * */
/* * CreateParagraph () macro, some global variables that should be set with #declare statements prior to calling    * */
/* * the CreateParagraph () macro, the selected "Font Style(s)," and any formatting codes within the text.           * */
/* * Technically, the global variables should all be macro parameters but I compromised good programming practice.   * */
/* *                                                                                                                 * */
/* * The CreateParagraph () macro has the following parameters:                                                      * */
/* *                                                                                                                 * */
/* *    1: [NLines] The number of lines of text that the Paragraph object will consist of.                           * */
/* *                                                                                                                 * */
/* *    2: [TextScale] The Paragraph object will be scaled along the x and y axes by [TextScale].                    * */
/* *                                                                                                                 * */
/* *    3: [UseActualParagraphHeight] This is a boolean variable that determines the final vertical location of the  * */
/* *    Paragraph object, as well as the meaning of the [ParagraphHeight] variable (see below).                      * */
/* *                                                                                                                 * */
/* * The CreateParagraph () macro utilizes the following global Variables which you can set:                         * */
/* *                                                                                                                 * */
/* *    [GAP_Width] This is the horizontal distance between characters. The default value is .06.                    * */
/* *                                                                                                                 * */
/* *    [SPACE_Width] This is the horizontal distance between words. The default value is .26.                       * */
/* *                                                                                                                 * */
/* *    [LineSpacing] This is the vertical distance between lines. The default value is .5. Note that portions of    * */
/* *    any characters which ascend above the normal upper character boundary (the ASTERISK) or descend below the    * */
/* *    normal character boundary will intrude into this space. Note also that vertical spacing and horizontal       * */
/* *    spacing operate differently in that horizontal spacing will change according to [TextScale] while vertical   * */
/* *    spacing will remain constant. This is required for consistency when using the "scaled" micro-justification   * */
/* *    method. If you need vertical spacing to change with [TextScale], simply multiply your desired [LineSpacing]  * */
/* *    value by [TextScale]. Similarly, if you need horizontal spacing to remain constant then divide your          * */
/* *    desired [SPACE_Width] and [GAP_Width] by [TextScale].                                                        * */
/* *                                                                                                                 * */
/* *    [LineAlignment] Paragraphs can be left aligned, right aligned, center aligned, or justified. The             * */
/* *    corresponding values are LEFT, RIGHT, CENTERED, or JUSTIFIED. The default value is LEFT. Micro-justification * */
/* *    is activated when you set [LineAlignment] = JUSTIFIED. It can be controlled very precisely to give different * */
/* *    results.                                                                                                     * */
/* *                                                                                                                 * */
/* *    [MaxMicroJustificationScale] There are two different methods available to accomplish the task of equalizing  * */
/* *    the widths of the lines of the Paragraph object (micro-justification). The first method is to scale the      * */
/* *    entire lines, both horizontally and vertically, so that they are all the same width. This will produce lines * */
/* *    of varying heights, creating an interesting visual effect. This method can be invoked by setting the         * */
/* *    variable [MaxMicroJustificationScale] = UNLIMITED. What this means is that each line will be scaled as much  * */
/* *    as necessary to achieve the correct width. If one or more lines are extremely narrow compared to the widest  * */
/* *    line, however, you will end up with very tall lines as a result. In this case you may want to limit the      * */
/* *    amount that the lines can be scaled. Set [MaxMicroJustificationScale] to some value greater than 1 in order  * */
/* *    to do this. For best results when using larger values, especially UNLIMITED, use all capital letters.        * */
/* *                                                                                                                 * */
/* *    The second method to widen lines is to increase the distance between characters and/or words. Set            * */
/* *    [MaxMicroJustificationScale] = 1 to invoke this method. Note that if you limit the amount that lines can be  * */
/* *    scaled to some value less than the required amount (by setting [MaxMicroJustificationScale] greater than 1   * */
/* *    but less than the required amount), then the second method will be utilized in combination with the first    * */
/* *    method to make up the difference. The default value is 1. Values less than 1 are undefined.                  * */
/* *                                                                                                                 * */
/* *    [MaxAdditionalGAP_WidthPercentage] The value of this variable represents a maximum percentage that           * */
/* *    [GAP_Width] can be increased to justify a line. As described above, the second method of micro-justification * */
/* *    works by expanding [GAP_Width] and [SPACE_Width] to widen any lines that are shorter than the longest line   * */
/* *    of the paragraph.                                                                                            * */ 
/* *                                                                                                                 * */
/* *    Set [MaxAdditionalGAP_WidthPercentage] to 0 to retain constant spacing between characters and widen lines    * */
/* *    when necessary by only increasing [SPACE_Width].                                                             * */
/* *                                                                                                                 * */
/* *    Setting [MaxAdditionalGAP_WidthPercentage] to something less than UNLIMITED means that as [GAP_Width] and    * */
/* *    [SPACE_Width] increase, [GAP_Width] will be limited to [GAP_Width] plus some percentage of [GAP_Width]. Any  * */
/* *    additional space that must be added to the line beyond that limited amount will be distributed among the     * */
/* *    spaces between words. If no spaces are present in the line then the value of the variable                    * */
/* *    [MaxAdditionalGAP_WidthPercentage] will be over-ridden.                                                      * */
/* *                                                                                                                 * */
/* *    Setting the value to UNLIMITED means that there will be no limit to the amount that [GAP_Width] can be       * */
/* *    increased in order to justify a line. This will allow the micro-justification feature to balance the values  * */
/* *    of [GAP_Width] and [SPACE_Width] automatically, giving an emphasis (controlled by [SPACE_ExpansionRate]) to  * */
/* *    [SPACE_Width].                                                                                               * */
/* *                                                                                                                 * */
/* *    The default value is UNLIMITED. This variable is only meaningful when [LineAlignment] = JUSTIFIED and        * */
/* *    [MaxMicroJustificationScale] is less than UNLIMITED.                                                         * */
/* *                                                                                                                 * */
/* *    [SPACE_ExpansionRate] Spaces can be expanded at a rate greater than the gap expansion rate. A value of 1     * */
/* *    will cause [SPACE_Width] to increase at the same rate as [GAP_Width] in order to justify a line. Values      * */
/* *    greater than 1 will cause [SPACE_Width] to increase more than [GAP_Width]. The default value is 1.5, meaning * */
/* *    that [SPACE_Width] will increase 1.5 times more than [GAP_Width]. This variable is only meaningful when      * */
/* *    [LineAlignment] = JUSTIFIED and [MaxMicroJustificationScale] is set to something less than UNLIMITED.        * */
/* *    Setting [SPACE_ExpansionRate] greater than 1 is most effective when [MaxAdditionalGAP_WidthPercentage] =     * */
/* *    UNLIMITED.                                                                                                   * */
/* *                                                                                                                 * */
/* *    [JustifyLastLinePercentage] This variable determines whether or not the last line of a multi-line paragraph  * */
/* *    will be justified when [LineAlignment] is set to JUSTIFIED and [MaxMicroJustificationScale] is set to 1.     * */
/* *    If [MaxMicroJustificationScale] is set to something greater than 1 then all the lines will always be         * */
/* *    justified. The value represents a percentage of the width of the last line to the width of the widest line   * */
/* *    of the Paragraph object. If the percentage is greater than or equal to the value of                          * */
/* *    [JustifyLastLinePercentage], the last line will be justified. Otherwise, it won't be. Therefore, a value of  * */
/* *    0 will cause the last line to always be justified and a value of 100 will cause the last line to never be    * */
/* *    justified. The default value is 95.                                                                          * */
/* *                                                                                                                 * */
/* *    [MinJustifiedParagraphWidth] This variable specifies the minimum width of a justified paragraph. If          * */
/* *    [LineAlignment] is not set to JUSTIFIED then this variable will have no effect. The default value is 0. You  * */
/* *    can use this feature to match the width of a Paragraph object to some other scene element, such as another   * */
/* *    Paragraph object.                                                                                            * */
/* *                                                                                                                 * */
/* *    [LineText] This is an array that will be created for you by the InitializeVariables () macro. Set each array * */
/* *    element equal to a line of characters. You must provide the InitializeVariables () macro with the number of  * */
/* *    lines that your paragraph will consist of, and you must assign values to a corresponding number of           * */
/* *    [LineText] array elements prior to calling the CreateParagraph () macro. Note that POV-Ray has an            * */
/* *    unfortunate string variable length limit of 255 characters.                                                  * */
/* *                                                                                                                 * */
/* *    [ApplyTextures] There may be occasions where untextured character objects need to be integrated with other   * */
/* *    objects and then have textures applied afterward. The [ApplyTextures] setting provides the ability to create * */
/* *    untextured characters. Set this variable to "false" to leave characters untextured. The default setting is   * */
/* *    "true." Note that currently only Font Variation #0 supports this feature, but it's there for anyone who      * */
/* *    needs to use it.                                                                                             * */
/* *                                                                                                                 * */
/* * Additionally, there are some useful variables that will be set for you by the CreateParagraph () macro:         * */
/* *                                                                                                                 * */
/* *    [ParagraphWidth] This variable represents the total width of the Paragraph object (the object will extend    * */
/* *    from x = 0 to x = [ParagraphWidth]). This value is useful when positioning the Paragraph object.             * */
/* *                                                                                                                 * */
/* *    [ParagraphHeight] This variable behaves differently based on the setting of the variable                     * */
/* *    [UseActualParagraphHeight] (see above). If [UseActualParagraphHeight] is set to "yes" or "true,"             * */
/* *    [ParagraphHeight] represents the total height of the Paragraph object adjusted for any characters that       * */
/* *    ascend above, descend below, or otherwise do not conform to the normal character boundaries of y = 0 to y =  * */
/* *    1. In this case, the vertical location of the Paragraph object will be adjusted so that the top of the       * */
/* *    tallest character on the first line will be at y = 0. If [UseActualParagraphHeight] is set to "no" or        * */
/* *    "false," [ParagraphHeight] represents the height of the Paragraph object ignoring any characters that do not * */
/* *    conform to the normal character boundaries of y = 0 to y = 1. Run the sample file                            * */
/* *    "Demo_ParagraphHeight.pov" for a visual demonstration of this variable. Note that [ParagraphHeight] is a     * */
/* *    positive value, but the top of the Paragraph object is at y = 0 and the bottom of the object is at y =       * */
/* *    -[ParagraphHeight]. This value is useful when positioning the Paragraph object.                              * */
/* *                                                                                                                 * */
/* *    [ParagraphHistory_Width] This array will contain the widths of all the Paragraph objects created up to this  * */
/* *    point in your scene. These values are useful for aligning or sizing multiple Paragraph objects. Note that    * */
/* *    one element is added to the array each time the CreateParagraph () macro is called. Therefore, if you create * */
/* *    a Paragraph object just to measure its width, the re-create it with a computed [TextScale] value, this will  * */
/* *    add two entries to the [ParagraphHistory_Width] array. The first value in the array is numbered "0."         * */
/* *                                                                                                                 * */
/* *    [ParagraphHistory_Height] This array works like the [ParagraphHistory_Width] array, but contains the heights * */
/* *    of all the Paragraph objects created up to this point in your scene.                                         * */
/* *                                                                                                                 * */
/* *    [CurrentParagraphNumber] This variable indicates the current Paragraph number. Chances are, if you want to   * */
/* *    match the width or height of a previous Paragraph it's probably the last one created. For example:           * */
/* *                                                                                                                 * */
/* *       #declare MinJustifiedParagraphWidth = ParagraphHistory_Width [CurrentParagraphNumber - 1];                * */
/* *                                                                                                                 * */
/* *    As long as LineAlignment = JUSTIFIED, this will create a Paragraph object that is at least as wide as the    * */
/* *    previous Paragraph object.                                                                                   * */
/* *                                                                                                                 * */
/* *    [LineScale] This array will contain the scale value of each line of the Paragraph object. These values will  * */
/* *    always be 1 unless [MaxMicroJustificationScale] is set to greater than 1. The first value is numbered "0."   * */
/* *                                                                                                                 * */
/* *    [SumOfAllScales] The sum of all the [LineScale] array elements. Only useful when                             * */
/* *    [MaxMicroJustificationScale] is set to a value greater than 1.                                               * */
/* *                                                                                                                 * */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* *                                                                                                                 * */
/* *                                                                                                                 * */
/* * SET CONSTANTS AND DEFAULT VARIABLE VALUES                                                                       * */
/* *                                                                                                                 * */
/* *                                                                                                                 * */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#declare LEFT = 0;
#declare RIGHT = 1;
#declare CENTERED = 2;
#declare JUSTIFIED = 3;

#declare UNLIMITED = -1;

#declare GAP_Width = .06;
#declare SPACE_Width = .26;
#declare LineSpacing = .5;

#declare LineAlignment = LEFT;
#declare MaxMicroJustificationScale = 1;
#declare MaxAdditionalGAP_WidthPercentage = UNLIMITED;
#declare SPACE_ExpansionRate = 1.5;
#declare JustifyLastLinePercentage = 95;
#declare MinJustifiedParagraphWidth = 0;

#declare CurFontStyleIndex = 0;
#declare LastFontStyleIndex = -1;

#declare FontStyleRecord_Valid = array [100]
#declare FontStyleRecord_FontVariationNumber = array [100]
#declare FontStyleRecord_P0 = array [100]
#declare FontStyleRecord_P1 = array [100]
#declare FontStyleRecord_P2 = array [100]
#declare FontStyleRecord_P3 = array [100]
#declare FontStyleRecord_P4 = array [100]
#declare FontStyleRecord_P5 = array [100]
#declare FontStyleRecord_P6 = array [100]
#declare FontStyleRecord_P7 = array [100]
#declare FontStyleRecord_P8 = array [100]
#declare FontStyleRecord_TextColor_Outer = array [100]
#declare FontStyleRecord_TextColor_Inner = array [100]
#declare FontStyleRecord_TextColor_Back = array [100]

#for (I, 0, 99)
   #declare FontStyleRecord_Valid [I] = true;
   #declare FontStyleRecord_FontVariationNumber [I] = 0;
   #declare FontStyleRecord_P0 [I] = .08;
   #declare FontStyleRecord_P1 [I] = 0;
   #declare FontStyleRecord_P2 [I] = 999;
   #declare FontStyleRecord_P3 [I] = 999;
   #declare FontStyleRecord_P4 [I] = 999;
   #declare FontStyleRecord_P5 [I] = 999;
   #declare FontStyleRecord_P6 [I] = 999;
   #declare FontStyleRecord_P7 [I] = 999;
   #declare FontStyleRecord_P8 [I] = 999;
   #declare FontStyleRecord_TextColor_Outer [I] = texture {pigment {color <1, 1, 1>}}
   #declare FontStyleRecord_TextColor_Inner [I] = texture {pigment {color <.620, .694, .071>}}
   #declare FontStyleRecord_TextColor_Back [I] = texture {pigment {color <1, 1, 1>}}
#end //#for

#declare CurrentParagraphNumber = 0;

#declare ParagraphHistory_Width = array [1000]
#declare ParagraphHistory_Height = array [1000]

#declare ApplyTextures = true;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* *                                                                                                                 * */
/* *                                                                                                                 * */
/* * MACROS                                                                                                          * */
/* *                                                                                                                 * */
/* *                                                                                                                 * */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#macro InitializeVariables (NLines)

   #declare LineText = array [NLines]
   #declare LineTextObject = array [NLines]
   #declare LineNCharacters = array [NLines]
   #declare LineNSpaces = array [NLines]
   #declare LineNVisibleCharacters = array [NLines]
   #declare LineScale = array [NLines]
   #declare LineWidth = array [NLines]
   #declare ItalicOffset_Beginning = array [NLines]
   #declare LineCharacters = array [NLines] [256]
   #declare LineKerning = array [NLines] [255]
   #declare CharacterVisibilityCondition = array [NLines] [256]
   #declare CharacterItalicStatus = array [NLines] [256]
   #declare CharacterFontStyleIndex = array [NLines] [256]
   #declare CharacterInUse = array [NDefinedCharacters];

#end //#macro InitializeVariables

#macro InitializeLine (I)

   #local TempLineCharacters = array [256]
   #local TempCharacterFontStyleIndex = array [256]

   #local LineText_Copy = LineText [I];

   #declare LineScale [I] = 1;

   #declare LineNCharacters [I] = strlen (LineText_Copy);

   #for (J, 0, LineNCharacters [I] - 1)
      #declare TempCharacterFontStyleIndex [J] = CurFontStyleIndex;
   #end //#for

   #local NeedToCheck = yes;

   #while (NeedToCheck)
      #local NeedToCheck = no;
      #for (J, 1, LineNCharacters [I])
         #local CurChar = asc (substr (LineText_Copy, J, 1));
         #if (CurChar = asc ("~"))
            #local NeedToCheck = yes;
            #local EndBracketFound = false;
            #for (K, J + 2, LineNCharacters [I])
               #local CurChar = asc (substr (LineText_Copy, K, 1));
               #if (CurChar = asc ("~"))
                  #local EndBracketFound = true;
                  #break //#for
               #end //#if
            #end //#for
            #if (EndBracketFound)
               #for (L, J + 1, K - 1)
                  #local CurChar = asc (substr (LineText_Copy, L, 1));
                  #if (CurChar < asc ("0") | CurChar > asc ("9"))
                     #error concat ("* * * Line #", str (I, 0, 0), " has a mis-formated style index. (Proper format is \"~nn~\" where \"nn\" is an integer.)")
                  #end //#if
               #end //#for
            #else
               #error concat ("* * * Line #", str (I, 0, 0), " has a mis-formated style index. (Proper format is \"~nn~\" where \"nn\" is an integer.)")
            #end //#if
            #local FontStyleIndexString = substr (LineText_Copy, J + 1, K - J - 1)
            #declare CurFontStyleIndex = val (FontStyleIndexString);
            #local StringLength = K - J - 1 + 2;
            #declare LineText_Left = substr (LineText_Copy, 1, J - 1);
            #declare LineText_Right = substr (LineText_Copy, K + 1, LineNCharacters [I] - K);
            #declare LineText_Copy = concat (LineText_Left, LineText_Right);
            #declare LineNCharacters [I] = LineNCharacters [I] - StringLength;
            #for (L, J, LineNCharacters [I])
               #declare TempCharacterFontStyleIndex [L - 1] = CurFontStyleIndex;
            #end //#for
            #break //#for
         #end //#if
      #end //#for
   #end //#while

   #for (J, 1, LineNCharacters [I] - 1)
      #local CurChar = asc (substr (LineText_Copy, J, 1));
      #local NextChar = asc (substr (LineText_Copy, J + 1, 1));
      #if (CurChar = asc ("{") & NextChar = asc (" "))
         #declare LineText_Copy = concat (substr (LineText_Copy, 1, J - 1), " {",
            substr (LineText_Copy, J + 2, LineNCharacters [I] - J - 1));
      #elseif (CurChar = asc ("}") & NextChar = asc (" "))
         #declare LineText_Copy = concat (substr (LineText_Copy, 1, J - 1), " }",
            substr (LineText_Copy, J + 2, LineNCharacters [I] - J - 1));
      #end //#if
   #end //#for

   #local CurItalicCondition = false;
   #local TempLineNCharacters = 0;

   #for (J, 0, LineNCharacters [I] - 1)
      #local CurChar = asc (substr (LineText_Copy, J + 1, 1));
      #if (CurChar = asc ("{") | CurChar = asc ("}"))
         #if (CurChar = asc ("{"))
            #local CurItalicCondition = true;
         #else
            #local CurItalicCondition = false;
         #end //#if
      #else
         #local TempLineNCharacters = TempLineNCharacters + 1;
         #local TempLineCharacters [TempLineNCharacters - 1] = CurChar;
         #declare CharacterItalicStatus [I] [TempLineNCharacters - 1] = CurItalicCondition;
         #declare CharacterFontStyleIndex [I] [TempLineNCharacters - 1] = TempCharacterFontStyleIndex [J];
      #end //#if
   #end //#for

   #declare LineNCharacters [I] = TempLineNCharacters;

   #if (LineNCharacters [I] = 0)
      #error concat ("* * * Line #", str (I, 0, 0), " evaluates to a blank line.")
   #end //#if

   #for (J, 0, LineNCharacters [I] - 1)
      #declare LineCharacters [I] [J] = ASCIIToIndex (TempLineCharacters [J]);
      #declare CharacterVisibilityCondition [I] [J] = 0;
   #end //#for

   #while (LineCharacters [I] [0] = -1)
      #local N = LineNCharacters [I];
      #if (N = 1)
         #error concat ("* * * Line #", str (I, 0, 0), " evaluates to a blank line.")
      #else
         #for (J, 0, N - 2)
            #declare LineCharacters [I] [J] = LineCharacters [I] [J + 1];
            #declare CharacterItalicStatus [I] [J] = CharacterItalicStatus [I] [J + 1];
            #declare CharacterFontStyleIndex [I] [J] = CharacterFontStyleIndex [I] [J + 1];
         #end //#for
         #declare LineNCharacters [I] = LineNCharacters [I] - 1;
      #end //#if
   #end //#while

   #while (LineCharacters [I] [LineNCharacters [I] - 1] = -1)
      #declare LineNCharacters [I] = LineNCharacters [I] - 1;
   #end //#while

   #local NeedToCheck = yes;

   #while (NeedToCheck)
      #local NeedToCheck = no;
      #if (LineNCharacters [I] > 3)
         #local N = LineNCharacters [I];
         #for (J, 1, N - 3)
            #if (LineCharacters [I] [J] = -1 & LineCharacters [I] [J + 1] = -1)
               #for (K, J + 1, N - 2)
                  #declare LineCharacters [I] [K] = LineCharacters [I] [K + 1];
                  #declare CharacterItalicStatus [I] [K] = CharacterItalicStatus [I] [K + 1];
                  #declare CharacterFontStyleIndex [I] [K] = CharacterFontStyleIndex [I] [K + 1];
               #end //#for
               #declare LineNCharacters [I] = LineNCharacters [I] - 1;
               #local NeedToCheck = yes;
               #break //#for
            #end //#if
         #end //#for
      #end //#if
   #end //#while

#end //#macro InitializeLine

#macro MarkInvisibleCharacters (I)

   #for (J, 2, LineNCharacters [I] - 1)
      #local CurChar = LineCharacters [I] [J];
      #if (CurChar != -1)
         #local PrevChar1 = LineCharacters [I] [J - 1];
         #local PrevChar2 = LineCharacters [I] [J - 2];
         #local Condition = 1;
         #if (PrevChar1 = -1)
            #local Condition = 0;
         #end //#if
         #if (J > 2)
            #local PrevChar3 = LineCharacters [I] [J - 3];
            #if (PrevChar3 != -1)
               #if (PrevChar1 = -1)
                  #local Condition = 2;
               #elseif (PrevChar2 = -1)
                  #local Condition = 3;
               #end //#if
            #end //#if
         #end //#if
         #switch (Condition)
            #case (1) //Example: "P.T"
               #local X1 = Kerning [PrevChar2] [PrevChar1] + GAP_Width + BorderCharacterWidth [PrevChar1] +
                  Kerning [PrevChar1] [CurChar];
               #local X2 = Kerning [PrevChar2] [CurChar];
               #if (X2 > X1)
                  //#debug concat ("\nCondition 1:\nLine #", str (I, 0, 0), ": \n")
                  //#debug concat ("Invisible character: ", str (PrevChar1, 0, 0), "\n")
                  #declare CharacterVisibilityCondition [I] [J - 1] = 1;
               #end //#if
            #break //#case
            #case (2) //Example: "P. T"
               #local X1 = Kerning [PrevChar3] [PrevChar2] + GAP_Width + BorderCharacterWidth [PrevChar2] +
                  Kerning [PrevChar2] [CurChar];
               #local X2 = Kerning [PrevChar3] [CurChar];
               #if (X2 > X1)
                  //#debug concat ("\nCondition 2:\nLine #", str (I, 0, 0), ": \n")
                  //#debug concat ("Invisible character: ", str (PrevChar2, 0, 0), "\n")
                  #declare CharacterVisibilityCondition [I] [J - 2] = 2;
               #end //#if
            #break //#case
            #case (3) //Example: "I .T"
               #local X1 = Kerning [PrevChar3] [PrevChar1] + BorderCharacterWidth [PrevChar1] + GAP_Width +
                  Kerning [PrevChar1] [CurChar];
               #local X2 = Kerning [PrevChar3] [CurChar];
               #if (X2 > X1)
                  //#debug concat ("\nCondition 3:\nLine #", str (I, 0, 0), ": \n")
                  //#debug concat ("Invisible character: ", str (PrevChar1, 0, 0), "\n")
                  #declare CharacterVisibilityCondition [I] [J - 1] = 3;
               #end //#if
            #break //#case
         #end //#switch
      #end //#if
   #end //#for

   #if (LineNCharacters [I] > 1) //See if the first character is invisible
      #local Char1 = LineCharacters [I] [0];
      #local Char2 = LineCharacters [I] [1];
      #if (Char2 != -1)
         #local X1 = BorderCharacterWidth [Char1] + GAP_Width + Kerning [Char1] [Char2];
         #if (X1 <= 0)
            //#debug concat ("\nLine #", str (I, 0, 0), ": First character is invisible\n")
            #declare CharacterVisibilityCondition [I] [0] = 3;
         #end //#if
      #end //#if
   #end //#if

   #if (LineNCharacters [I] > 1) //See if the last character is invisible
      #local Char1 = LineCharacters [I] [LineNCharacters [I] - 2];
      #local Char2 = LineCharacters [I] [LineNCharacters [I] - 1];
      #if (Char1 != -1)
         #local X1 = Kerning [Char1] [Char2] + GAP_Width + BorderCharacterWidth [Char2];
         #if (X1 <= 0)
            //#debug concat ("\nLine #", str (I, 0, 0), ": Last character is invisible\n")
            #declare CharacterVisibilityCondition [I] [LineNCharacters [I] - 1] = 1;
         #end //#if
      #end //#if
   #end //#if

#end //#macro MarkInvisibleCharacters

#macro ComputeLineWidth (I)

   #declare LineNVisibleCharacters [I] = 0;
   #declare LineNSpaces [I] = 0;
   #declare LineWidth [I] = 0;

   #local LastItalicCondition = CharacterItalicStatus [I] [0];

   #for (J, 0, LineNCharacters [I] - 1)
      #local CurChar = LineCharacters [I] [J];
      #if (CharacterVisibilityCondition [I] [J] = 0)
         #declare LineNVisibleCharacters [I] = LineNVisibleCharacters [I] + 1;
         #if (CurChar = -1)
            #declare LineKerning [I] [J] = SPACE_Width;
            #declare LineWidth [I] = LineWidth [I] + SPACE_Width + GAP_Width;
            #declare LineNSpaces [I] = LineNSpaces [I] + 1;
         #else
            #declare LineWidth [I] = LineWidth [I] + BorderCharacterWidth [CurChar];
            #if (J < LineNCharacters [I] - 2) //There ARE more remaining VISIBLE characters
               #declare LineWidth [I] = LineWidth [I] + GAP_Width;
            #end //#if
            #if (J = LineNCharacters [I] - 2) //There MIGHT BE more remaining VISIBLE characters
               #if (CharacterVisibilityCondition [I] [J + 1] = 0)
                  #declare LineWidth [I] = LineWidth [I] + GAP_Width;
               #end //#if
            #end //#if
            #if (J <= LineNCharacters [I] - 2)
               #local NextCharFound = false;
               #for (K, J + 1, LineNCharacters [I] - 1)
                  #local CompChar = LineCharacters [I] [K];
                  #if (CharacterVisibilityCondition [I] [K] = 0 & CompChar != -1)
                     #local NextCharFound = true;
                     #local NextChar = LineCharacters [I] [K];
                     #break //#for
                  #end //#if
               #end //#for
               #if (NextCharFound)
                  #declare LineKerning [I] [J] = Kerning [CurChar] [NextChar];
                  #declare LineWidth [I] = LineWidth [I] + LineKerning [I] [J];
               #end //#if
            #end //#if
         #end //#if
      #end //#if
      #local CurItalicCondition = CharacterItalicStatus [I] [J];
      #if (CurItalicCondition != LastItalicCondition)
         #if (CurItalicCondition = true) //Transitioning from normal to italic
            #local PrevCharFound = false;
            #for (K, J - 1, 0, -1)
               #if (CharacterVisibilityCondition [I] [K] = 0 & LineCharacters [I] [K] != -1)
                  #local PrevCharFound = true;
                  #local PrevChar = LineCharacters [I] [K];
                  #break //#for
               #end //#if
            #end //#for
            #if (PrevCharFound)
               #declare LineKerning [I] [K] = LineKerning [I] [K] + ItalicOffset [CurChar * 2] - Kerning [PrevChar] [CurChar] + Kerning_NormalToItalic [PrevChar] [CurChar];
               #declare LineWidth [I] = LineWidth [I] + ItalicOffset [CurChar * 2] - Kerning [PrevChar] [CurChar] + Kerning_NormalToItalic [PrevChar] [CurChar];
            #end //#if
         #else //Transitioning from italic to normal
            #local PrevCharFound = false;
            #for (K, J - 1, 0, -1)
               #if (CharacterVisibilityCondition [I] [K] = 0 & LineCharacters [I] [K] != -1)
                  #local PrevCharFound = true;
                  #local PrevChar = LineCharacters [I] [K];
                  #break //#for
               #end //#if
            #end //#for
            #if (PrevCharFound)
               #declare LineKerning [I] [K] = LineKerning [I] [K] + ItalicOffset [PrevChar * 2 + 1] - Kerning [PrevChar] [CurChar] + Kerning_ItalicToNormal [PrevChar] [CurChar];
               #declare LineWidth [I] = LineWidth [I] + ItalicOffset [PrevChar * 2 + 1] - Kerning [PrevChar] [CurChar] + Kerning_ItalicToNormal [PrevChar] [CurChar];
            #end //#if
         #end //#if
         #local LastItalicCondition = CurItalicCondition;
      #end //#if
   #end //#for

   #declare ItalicOffset_Beginning [I] = 0;

   #declare ItalicOffset_End = 0;

   #if (CharacterItalicStatus [I] [0])
      #local FirstChar = LineCharacters [I] [0];
      #declare ItalicOffset_Beginning [I] = ItalicOffset [FirstChar * 2];
      #declare LineWidth [I] = LineWidth [I] + ItalicOffset_Beginning [I];
   #end //#if

   #if (CharacterItalicStatus [I] [LineNCharacters [I] - 1])
      #if (CharacterVisibilityCondition [I] [LineNCharacters [I] - 1] = 0)
         #local LastChar = LineCharacters [I] [LineNCharacters [I] - 1];
      #else
         #local LastChar = LineCharacters [I] [LineNCharacters [I] - 2];
      #end //#if
      #local ItalicOffset_End = ItalicOffset [LastChar * 2 + 1];
      #declare LineWidth [I] = LineWidth [I] + ItalicOffset_End;
   #end //#if

#end //#macro ComputeLineWidth

#macro CreateLine (I, MaxLineWidth, TextScale)

   #if (MaxAdditionalGAP_WidthPercentage != UNLIMITED)
      #local MaxGapPadding = GAP_Width * MaxAdditionalGAP_WidthPercentage / 100; //Padding is in addition to GAP_Width
   #end //#if

   #local GapPadding = 0;
   #local SpacePadding = 0;

   #if (LineAlignment = JUSTIFIED)
      #if (MaxMicroJustificationScale < 1 & MaxMicroJustificationScale != -1)
         #declare MaxMicroJustificationScale = 1;
      #end //#if
      #declare LineScale [I] = ParagraphWidth / LineWidth [I];
      #if (MaxMicroJustificationScale != UNLIMITED)
         #if (LineScale [I] > MaxMicroJustificationScale)
             #declare LineScale [I] = MaxMicroJustificationScale;
         #end //#if
      #end //#if
      #if (MaxMicroJustificationScale != 1)
         #declare LastFontStyleIndex = -1;
      #end //#if
   #end //#if

   #if (LineAlignment = JUSTIFIED & LineNVisibleCharacters [I] > 1)
      #if (I < NLines - 1 | NLines = 1 | MaxMicroJustificationScale != 1 | (LineWidth [I] * LineScale [I] / MaxLineWidth * 100 >= JustifyLastLinePercentage))
         #local ExtraWidth = MaxLineWidth - LineWidth [I] * LineScale [I];
         #local P1 = GAP_Width * LineScale [I] * (LineNVisibleCharacters [I] - 1);
         #local P2 = SPACE_Width * LineScale [I] * SPACE_ExpansionRate * LineNSpaces [I];
         #if (P1 + P2 > 0)
            #local WidthGrowthFactor = ExtraWidth / (P1 + P2);
         #else
            #local WidthGrowthFactor = 0;
         #end //#if
         #local GapPadding = GAP_Width * WidthGrowthFactor;
         #local SpacePadding = SPACE_Width * WidthGrowthFactor * SPACE_ExpansionRate;
         #if (MaxAdditionalGAP_WidthPercentage != UNLIMITED)
            #if (GapPadding > MaxGapPadding & LineNSpaces [I] > 0)
               #local GapPadding = MaxGapPadding;
               #local SpacePadding = (ExtraWidth / LineScale [I] - (LineNVisibleCharacters [I] - 1) * GapPadding) / LineNSpaces [I];
            #end //#if
         #end //#if
      #end //#if
   #end //#if

   #local CurGap = GAP_Width + GapPadding;
   #local CurSpace = SPACE_Width + SpacePadding;

   #declare LineTextObject [I] = object {
      #local CurX = ItalicOffset_Beginning [I];
      #if (LineNCharacters [I] > 1)
         union {
            #for (J, 0, LineNCharacters [I] - 1)
               #if (J = 0 & CharacterVisibilityCondition [I] [0] != 0) //First character is invisible
                  #local CurChar = LineCharacters [I] [0];
                  #declare CurFontStyleIndex = CharacterFontStyleIndex [I] [0];
                  #if (CurFontStyleIndex != LastFontStyleIndex)
                     Stylize (CurFontStyleIndex, TextScale * LineScale [I])
                     #declare LastFontStyleIndex = CurFontStyleIndex;
                  #end //#if
                  object {
                     BorderCharacter [CurChar]
                     translate CurX * x
                     #if (CharacterItalicStatus [I] [0])
                        Italicize ()
                     #end //#if
                  } //object
                  #local NextChar = LineCharacters [I] [1];
                  #local CurX = CurX + BorderCharacterWidth [CurChar] + Kerning [CurChar] [NextChar] + CurGap;
                  #if (CurX < ItalicOffset_Beginning [I])
                     #local CurX = ItalicOffset_Beginning [I];
                  #end //#if
               #end //#if
               #if (CharacterVisibilityCondition [I] [J] = 0)
                  #local CurChar = LineCharacters [I] [J];
                  #local LastX = CurX;
                  #if (CurChar = -1)
                     #local CurX = CurX + LineKerning [I] [J] + CurGap + SpacePadding;
                  #else
                     #declare CurFontStyleIndex = CharacterFontStyleIndex [I] [J];
                     #if (CurFontStyleIndex != LastFontStyleIndex)
                        Stylize (CurFontStyleIndex, TextScale * LineScale [I])
                        #declare LastFontStyleIndex = CurFontStyleIndex;
                     #end //#if
                     object {
                        BorderCharacter [CurChar]
                        translate CurX * x
                        #if (CharacterItalicStatus [I] [J])
                           Italicize ()
                        #end //#if
                     } //object
                     #if (J <= LineNCharacters [I] - 2)
                        #if (J < LineNCharacters [I] - 2 | CharacterVisibilityCondition [I] [LineNCharacters [I] - 1] = 0)
                           #local CurX = CurX + BorderCharacterWidth [CurChar] + LineKerning [I] [J] + CurGap;
                        #end //#if
                     #end //#if
                  #end //#if
                  #if (J < LineNCharacters [I] - 1)
                     #local NextLVC = CharacterVisibilityCondition [I] [J + 1];
                     #if (NextLVC != 0)
                        #local NextChar = LineCharacters [I] [J + 1];
                        #if (CurChar = -1)
                           #local TempX = LastX + CurSpace + CurGap;
                        #else
                           #local TempX = LastX + BorderCharacterWidth [CurChar] + Kerning [CurChar] [NextChar] + CurGap;
                        #end //#if
                        #declare CurFontStyleIndex = CharacterFontStyleIndex [I] [J + 1];
                        #if (CurFontStyleIndex != LastFontStyleIndex)
                           Stylize (CurFontStyleIndex, TextScale * LineScale [I])
                           #declare LastFontStyleIndex = CurFontStyleIndex;
                        #end //#if
                        object {
                           BorderCharacter [NextChar]
                           translate TempX * x
                           #if (CharacterItalicStatus [I] [J + 1])
                              Italicize ()
                           #end //#if
                        } //object
                     #end //#if
                  #end //#if
               #end //#if
            #end //#for
         } //union
      #else
         #local CurChar = LineCharacters [I] [0];
         #declare CurFontStyleIndex = CharacterFontStyleIndex [I] [0];
         #if (CurFontStyleIndex != LastFontStyleIndex)
            Stylize (CurFontStyleIndex, TextScale * LineScale [I])
            #declare LastFontStyleIndex = CurFontStyleIndex;
         #end //#if
         object {
            BorderCharacter [CurChar]
            translate CurX * x
            #if (CharacterItalicStatus [I] [0])
               Italicize ()
            #end //#if
         } //object
      #end //#if
   } //object

#end //#macro CreateLine

#macro CreateParagraph (NLines, TextScale, UseActualParagraphHeight)

   #declare ParagraphWidth = 0;
   #declare ParagraphHeight = 0;

   #for (I, 0, NDefinedCharacters - 1)
      #declare CharacterInUse [I] = false;
   #end //#for

   #for (I, 0, NLines - 1)
      InitializeLine (I)
      #for (J, 0, LineNCharacters [I] - 1)
         #if (LineCharacters [I] [J] != -1)
            #declare CharacterInUse [LineCharacters [I] [J]] = true;
         #end //#if
      #end //#for
   #end //#for

   #declare CurFontStyleIndex = CharacterFontStyleIndex [0] [0];

   Stylize (CurFontStyleIndex, TextScale)

   #declare LastFontStyleIndex = CurFontStyleIndex;

   #for (I, 0, NLines - 1)
      MarkInvisibleCharacters (I)
      ComputeLineWidth (I)
      #declare ParagraphWidth = max (ParagraphWidth, LineWidth [I]);
   #end //#for

   #if (LineAlignment = JUSTIFIED)
      #declare ParagraphWidth = max (ParagraphWidth, MinJustifiedParagraphWidth / TextScale);
   #end //#if

   #for (I, 0, NLines - 1)
      CreateLine (I, ParagraphWidth, TextScale)
   #end //#for

   #declare Paragraph = object {
      #local CurY = 0;
      #if (NLines > 1)
         union {
            #for (I, NLines - 1, 0, -1)
               #switch (LineAlignment)
                  #case (LEFT)
                     #local XOffset = 0;
                  #break //#case
                  #case (RIGHT)
                     #local XOffset = ParagraphWidth * TextScale - LineWidth [I] * TextScale;
                  #break //#case
                  #case (CENTERED)
                     #local XOffset = ParagraphWidth / 2 * TextScale - LineWidth [I] / 2 * TextScale;
                  #break //#case
                  #case (JUSTIFIED)
                     #local XOffset = 0;
                  #break //#case
               #end //#switch
               object {
                  LineTextObject [I]
                  scale <TextScale * LineScale [I], TextScale * LineScale [I], 1>
                  translate <XOffset, CurY, 0>
               } //object
               #if (I > 0)
                  #local CurY = CurY + Height_Upper * TextScale * LineScale [I] + LineSpacing;
               #end //#if
            #end //#for
         } //union
      #else
         LineTextObject [0]
         scale <TextScale * LineScale [0], TextScale * LineScale [0], 1>
      #end //#if
      #declare ParagraphWidth = ParagraphWidth * TextScale;
      #declare ParagraphHeight = CurY + Height_Upper * TextScale * LineScale [0];
      translate -ParagraphHeight * y
   } //object

   #if (UseActualParagraphHeight)
      #declare FirstLineHeight_Top = -999;
      #for (I, 0, LineNCharacters [0] - 1)
         #local CurChar = LineCharacters [0] [I];
         #if (CurChar != -1)
            #local CurHeight = BorderCharacterHeight_Top [CurChar];
            #if (CurHeight > FirstLineHeight_Top)
               #declare FirstLineHeight_Top = CurHeight;
            #end //#if
         #end //#if
      #end //#for
      #declare ParagraphYOffset = (Height_Upper - FirstLineHeight_Top) * TextScale * LineScale [0];
      #declare ParagraphHeight = ParagraphHeight - ParagraphYOffset;
      #declare LastLineHeight_Bottom = 999;
      #for (I, 0, LineNCharacters [NLines - 1] - 1)
         #local CurChar = LineCharacters [NLines - 1] [I];
         #if (CurChar != -1)
            #local CurHeight = BorderCharacterHeight_Bottom [CurChar];
            #if (CurHeight < LastLineHeight_Bottom)
               #declare LastLineHeight_Bottom = CurHeight;
            #end //#if
         #end //#if
      #end //#for
      #declare ParagraphHeight = ParagraphHeight - LastLineHeight_Bottom * TextScale * LineScale [NLines - 1];
      #declare Paragraph = object {Paragraph translate ParagraphYOffset * y}
   #end //#if

   #declare ParagraphHistory_Width [CurrentParagraphNumber] = ParagraphWidth;
   #declare ParagraphHistory_Height [CurrentParagraphNumber] = ParagraphHeight;

   #declare CurrentParagraphNumber = CurrentParagraphNumber + 1;

   #if (LineAlignment = JUSTIFIED & MaxMicroJustificationScale != 1)
      #declare FontStyleRecord_Valid [CurFontStyleIndex] = false;
   #end //#if

   #declare SumOfAllScales = 0;

   #for (I, 0, NLines - 1)
      #declare SumOfAllScales = SumOfAllScales + LineScale [I];
   #end //#for

#end //#macro CreateParagraph

#macro SetFontStyle (I, FontVariationNumber, P0, P1, P2, P3, P4, P5, P6, P7, P8, TextColor_Outer, TextColor_Inner, TextColor_Back)

   #declare FontStyleRecord_Valid [I] = false;
   #declare FontStyleRecord_FontVariationNumber [I] = FontVariationNumber;
   #declare FontStyleRecord_P0 [I] = P0;
   #declare FontStyleRecord_P1 [I] = P1;
   #declare FontStyleRecord_P2 [I] = P2;
   #declare FontStyleRecord_P3 [I] = P3;
   #declare FontStyleRecord_P4 [I] = P4;
   #declare FontStyleRecord_P5 [I] = P5;
   #declare FontStyleRecord_P6 [I] = P6;
   #declare FontStyleRecord_P7 [I] = P7;
   #declare FontStyleRecord_P8 [I] = P8;
   #declare FontStyleRecord_TextColor_Outer [I] = texture {TextColor_Outer}
   #declare FontStyleRecord_TextColor_Inner [I] = texture {TextColor_Inner}
   #declare FontStyleRecord_TextColor_Back [I] = texture {TextColor_Back}

#end //#macro SetFontStyle
