/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* *                                                                                                                 * */
/* * [VERSION 1.2]                                                                                                   * */
/* *                                                                                                                 * */
/* * INSTRUCTIONS FOR USE:                                                                                           * */
/* *                                                                                                                 * */
/* * This file contains some useful macros to help you manage the character objects as parts of words and lines of   * */
/* * text. The fact that some characters fit "inside" other characters (for example, the top of the letter "P"       * */
/* * extends beyond the edge of the period) greatly complicates the task of applying the kerning values. These       * */
/* * macros are not perfect, however, the character combinations that will cause them to fail are not commonly used  * */
/* * in normal writing. Therefore, you may find them useful. See the sample .pov files for examples of how to use    * */
/* * these macros.                                                                                                   * */
/* *                                                                                                                 * */
/* * The macros in this file work together to create an object called "Paragraph." The Paragraph object consists of  * */
/* * one or more lines of characters arranged vertically. Each line can consist of one or more characters. It's      * */
/* * exactly what an actual written paragraph is.                                                                    * */
/* *                                                                                                                 * */
/* * There are a number of variables that you can set to control the appearance of the Paragraph object. These       * */
/* * variables should be set with #declare statetements prior to creating the Paragraph object.                      * */
/* *                                                                                                                 * */
/* *    [TextScale] The Paragraph object will be scaled along the x and y axes by [TextScale]. Also, many of the     * */
/* *    sample character styles included in the file BorderChars_Styles.inc use this variable to maintain consistent * */
/* *    border widths. The default value is 1.                                                                       * */
/* *                                                                                                                 * */
/* *    [GAP_Width] This is the distance between characters. The default value is .06.                               * */
/* *                                                                                                                 * */
/* *    [SPACE_Width] This is the distance between words. The default value is .2.                                   * */
/* *                                                                                                                 * */
/* *    [LineSpacing] This is the vertical distance between lines. The default value is 1.5.                         * */
/* *                                                                                                                 * */
/* *    [LineAlignment] Paragraphs can be left-aligned, right-aligned, center-aligned, or fully justified. The       * */
/* *    corresponding values are LEFT, RIGHT, CENTERED, or JUSTIFIED. The default value is LEFT.                     * */
/* *                                                                                                                 * */
/* *    [AlignLastLine] This boolean variable determines whether the last line of a multi-line paragraph will be     * */
/* *    justified when [LineAlignment] is set to JUSTIFIED. The default value is no/false.                           * */
/* *                                                                                                                 * */
/* *    [AdditionalGAP_WidthPercentage] This variable controls the micro-justification feature. The value represents * */
/* *    a maximum percentage that [GAP_Width] can be increased to justify a line. If [GAP_Width] is increased by the * */
/* *    maximum amount allowed by [AdditionalGAP_WidthPercentage] and a line is still not wide enough, then          * */
/* *    [SPACE_Width] will be increased to make up the difference. If no spaces are present in the line then         * */
/* *    [AdditionalGAP_WidthPercentage] will be over-ridden. The default value is 60.                                * */
/* *                                                                                                                 * */
/* *    Setting [AdditionalGAP_WidthPercentage] to -1 will cause the micro-justification macro to balance the values * */
/* *    of [GAP_Width] and [SPACE_Width] automatically, giving a slight emphasis to [SPACE_Width].                   * */
/* *                                                                                                                 * */
/* *    In summary: Set [AdditionalGAP_WidthPercentage] to 0 to retain constant spacing between characters and widen * */
/* *    lines when necessary by increasing [SPACE_Width]. Set the value to some positive number to increase          * */
/* *    [GAP_Width] up to that percentage, and increase [SPACE_Width] as necessary to make up any additional         * */
/* *    required width. Set the value to -1 to have the macro balance [GAP_Width] and [SPACE_Width] automatically.   * */
/* *                                                                                                                 * */
/* *    [LineText] This is an array that will be created for you by the InitializeVariables macro. Set each array    * */
/* *    element equal to a line of characters. Note that you must provide the InitializeVariables macro with the     * */
/* *    number of lines that your paragraph will consist of, and you must assign values to a corresponding number of * */
/* *    [LineText] array elements prior to calling the CreateParagraphTextObject macro.                              * */
/* *                                                                                                                 * */
/* * Additionally, there is one important variable to be aware of:                                                   * */
/* *                                                                                                                 * */
/* *    [MaxLineWidth] This variable will be set for you by the CreateParagraphTextObject macro. It represents the   * */
/* *    total width of the Paragraph object (the object will extend from x = 0 to x = [MaxLineWidth]). This value is * */
/* *    very useful when positioning the Paragraph object.                                                           * */
/* *                                                                                                                 * */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#declare LEFT = 0;
#declare RIGHT = 1;
#declare CENTERED = 2;
#declare JUSTIFIED = 3;

#declare TextScale = 1;
#declare GAP_Width = .06;
#declare SPACE_Width = .2;
#declare LineSpacing = 1.5;
#declare LineAlignment = LEFT;
#declare AlignLastLine = no;
#declare AdditionalGAP_WidthPercentage = 60;
   
#macro InitializeVariables (NLines)

   #if (NLines < 1)
      #error "\nMust specify at least one line.\n"
   #end //#ifndef

   #declare LineText = array [NLines]
   #declare LineTextObject = array [NLines]
   #declare LineNCharacters = array [NLines]
   #declare LineNSpaces = array [NLines]
   #declare LineNVisibleCharacters = array [NLines]
   #declare LineWidth = array [NLines]
   #declare ItalicOffset_Beginning = array [NLines]
   #declare LineCharacters = array [NLines] [1000]
   #declare LineKerning = array [NLines] [999]
   #declare LineVisibilityCondition = array [NLines] [1000]
   #declare LineItalicCondition = array [NLines] [1000]

#end //#macro InitializeVariables

#macro InitializeLine (I)

   #local TempLineCharacters = array [2000]

   #declare LineNCharacters [I] = strlen (LineText [I]);

   #for (J, 1, LineNCharacters [I] - 1)
      #local CurChar = asc (substr (LineText [I], J, 1));
      #local NextChar = asc (substr (LineText [I], J + 1, 1));
      #if (CurChar = asc ("<") & NextChar = asc (" "))
         #declare LineText [I] = concat (substr (LineText [I], 1, J - 1), " <",
            substr (LineText [I], J + 2, LineNCharacters [I] - J - 1));
      #elseif (CurChar = asc (" ") & NextChar = asc (">"))
         #declare LineText [I] = concat (substr (LineText [I], 1, J - 1), "> ",
            substr (LineText [I], J + 2, LineNCharacters [I] - J - 1));
      #end //#if
   #end //#for

   #local CurItalicCondition = false;
   #local TempLineNCharacters = 0;
   #for (J, 0, LineNCharacters [I] - 1)
      #local CurChar = asc (substr (LineText [I], J + 1, 1));
      #if (CurChar = asc ("<") | CurChar = asc (">"))
         #if (CurChar = asc ("<"))
            #local CurItalicCondition = true;
         #else
            #local CurItalicCondition = false;
         #end //#if
      #else
         #local TempLineNCharacters = TempLineNCharacters + 1;
         #local TempLineCharacters [TempLineNCharacters - 1] = CurChar;
         #declare LineItalicCondition [I] [TempLineNCharacters - 1] = CurItalicCondition;
      #end //#if
   #end //#for

   #declare LineNCharacters [I] = TempLineNCharacters;
   
   #if (LineNCharacters [I] = 0)
      #error concat ("* * * Line #", str (I, 0, 0), " evaluates to a blank line.")
   #end //#if

   #for (J, 0, LineNCharacters [I] - 1)
      #declare LineCharacters [I] [J] = ASCIIToIndex (TempLineCharacters [J]);
      #declare LineVisibilityCondition [I] [J] = 0;
   #end //#for

   #while (LineCharacters [I] [0] = -1)
      #local N = LineNCharacters [I];
      #if (N = 1)
         #error concat ("* * * Line #", str (I, 0, 0), " evaluates to a blank line.")
      #else
         #for (K, 0, N - 2)
            #declare LineCharacters [I] [K] = LineCharacters [I] [K + 1];
            #declare LineItalicCondition [I] [K] = LineItalicCondition [I] [K + 1];
         #end //#for
         #declare LineNCharacters [I] = LineNCharacters [I] - 1;
      #end //#if
   #end //#while

   #while (LineCharacters [I] [LineNCharacters [I] - 1] = -1)
      #declare LineNCharacters [I] = LineNCharacters [I] - 1;
   #end //#while

   #local NeedToCheck = yes;
   #while (NeedToCheck)
      #local NeedToCheck = no;
      #if (LineNCharacters [I] > 3)
         #local N = LineNCharacters [I];
         #for (K, 1, N - 3)
            #if (LineCharacters [I] [K] = -1 & LineCharacters [I] [K + 1] = -1)
               #for (L, K + 1, N - 2)
                  #declare LineCharacters [I] [L] = LineCharacters [I] [L + 1];
                  #declare LineItalicCondition [I] [L] = LineItalicCondition [I] [L + 1];
               #end //#for
               #declare LineNCharacters [I] = LineNCharacters [I] - 1;
               #local NeedToCheck = yes;
               #break
            #end //#if
         #end //#for
      #end //#if
   #end //#while

#end //#macro InitializeLine

#macro MarkInvisibleCharacters (I)

   #for (J, 2, LineNCharacters [I] - 1)
      #local CurChar = LineCharacters [I] [J];
      #if (CurChar != -1)
         #local PrevChar1 = LineCharacters [I] [J - 1];
         #local PrevChar2 = LineCharacters [I] [J - 2];
         #local Condition = 1;
         #if (PrevChar1 = -1)
            #local Condition = 0;
         #end //#if
         #if (J > 2)
            #local PrevChar3 = LineCharacters [I] [J - 3];
            #if (PrevChar3 != -1)
               #if (PrevChar1 = -1)
                  #local Condition = 2;
               #elseif (PrevChar2 = -1)
                  #local Condition = 3;
               #end //#if
            #end //#if
         #end //#if
         #switch (Condition)
            #case (1) //Example: "P.T"
               #local X1 = Kerning [PrevChar2] [PrevChar1] + GAP_Width + BorderCharacterWidth [PrevChar1] +
                  Kerning [PrevChar1] [CurChar];
               #local X2 = Kerning [PrevChar2] [CurChar];
               #if (X2 > X1)
                  //#debug concat ("\nCondition 1:\nLine #", str (I, 0, 0), ": \n")
                  //#debug concat ("Invisible character: ", str (PrevChar1, 0, 0), "\n")
                  #declare LineVisibilityCondition [I] [J - 1] = 1;
               #end //#if
            #break
            #case (2) //Example: "P. T"
               #local X1 = Kerning [PrevChar3] [PrevChar2] + GAP_Width + BorderCharacterWidth [PrevChar2] +
                  Kerning [PrevChar2] [CurChar];
               #local X2 = Kerning [PrevChar3] [CurChar];
               #if (X2 > X1)
                  //#debug concat ("\nCondition 2:\nLine #", str (I, 0, 0), ": \n")
                  //#debug concat ("Invisible character: ", str (PrevChar2, 0, 0), "\n")
                  #declare LineVisibilityCondition [I] [J - 2] = 2;
               #end //#if
            #break
            #case (3) //Example: "I .T"
               #local X1 = Kerning [PrevChar3] [PrevChar1] + BorderCharacterWidth [PrevChar1] + GAP_Width +
                  Kerning [PrevChar1] [CurChar];
               #local X2 = Kerning [PrevChar3] [CurChar];
               #if (X2 > X1)
                  //#debug concat ("\nCondition 3:\nLine #", str (I, 0, 0), ": \n")
                  //#debug concat ("Invisible character: ", str (PrevChar1, 0, 0), "\n")
                  #declare LineVisibilityCondition [I] [J - 1] = 3;
               #end //#if
            #break
         #end //#switch
      #end //#if
   #end //#for

   #if (LineNCharacters [I] > 1) //See if the first character is invisible
      #local Char1 = LineCharacters [I] [0];
      #local Char2 = LineCharacters [I] [1];
      #if (Char2 != -1)
         #local X1 = BorderCharacterWidth [Char1] + GAP_Width + Kerning [Char1] [Char2];
         #if (X1 <= 0)
            //#debug concat ("\nLine #", str (I, 0, 0), ": First character is invisible\n")
            #declare LineVisibilityCondition [I] [0] = 3;
         #end //#if
      #end //#if
   #end //#if

   #if (LineNCharacters [I] > 1) //See if the last character is invisible
      #local Char1 = LineCharacters [I] [LineNCharacters [I] - 2];
      #local Char2 = LineCharacters [I] [LineNCharacters [I] - 1];
      #if (Char1 != -1)
         #local X1 = Kerning [Char1] [Char2] + GAP_Width + BorderCharacterWidth [Char2];
         #if (X1 <= 0)
            //#debug concat ("\nLine #", str (I, 0, 0), ": Last character is invisible\n")
            #declare LineVisibilityCondition [I] [LineNCharacters [I] - 1] = 1;
         #end //#if
      #end //#if
   #end //#if

#end //#macro MarkInvisibleCharacters

#macro ComputeLineWidth (I)

   #local LastItalicCondition = LineItalicCondition [I] [0];
   #declare LineNVisibleCharacters [I] = 0;
   #declare LineNSpaces [I] = 0;
   #declare LineWidth [I] = 0;

   #for (J, 0, LineNCharacters [I] - 1)
      #local CurChar = LineCharacters [I] [J];
      #if (LineVisibilityCondition [I] [J] = 0)
         #declare LineNVisibleCharacters [I] = LineNVisibleCharacters [I] + 1;
         #if (CurChar = -1)
            #declare LineKerning [I] [J] = SPACE_Width;
            #declare LineWidth [I] = LineWidth [I] + SPACE_Width + GAP_Width;
            #declare LineNSpaces [I] = LineNSpaces [I] + 1;
         #else
            #declare LineWidth [I] = LineWidth [I] + BorderCharacterWidth [CurChar];
            #if (J < LineNCharacters [I] - 2) //There ARE more remaining VISIBLE characters
               #declare LineWidth [I] = LineWidth [I] + GAP_Width;
            #end //#if
            #if (J = LineNCharacters [I] - 2) //There MIGHT BE more remaining VISIBLE characters
               #if (LineVisibilityCondition [I] [J + 1] = 0)
                  #declare LineWidth [I] = LineWidth [I] + GAP_Width;
               #end //#if
            #end //#if
            #if (J <= LineNCharacters [I] - 2)
               #local NextCharFound = false;
               #for (K, J + 1, LineNCharacters [I] - 1)
                  #local CompChar = LineCharacters [I] [K];
                  #if (LineVisibilityCondition [I] [K] = 0 & CompChar != -1)
                     #local NextCharFound = true;
                     #local NextChar = LineCharacters [I] [K];
                     #break
                  #end //#if
               #end //#for
               #if (NextCharFound)
                  #declare LineKerning [I] [J] = Kerning [CurChar] [NextChar];
                  #declare LineWidth [I] = LineWidth [I] + LineKerning [I] [J];
               #end //#if
            #end //#if
         #end //#if
      #end //#if
      #local CurItalicCondition = LineItalicCondition [I] [J];
      #if (CurItalicCondition != LastItalicCondition)
         #if (CurItalicCondition = true) //Transitioning from normal to italic
            #local PrevCharFound = false;
            #for (K, J - 1, 0, -1)
               #if (LineVisibilityCondition [I] [K] = 0 & LineCharacters [I] [K] != -1)
                  #local PrevCharFound = true;
                  #local PrevChar = LineCharacters [I] [K];
                  #break
               #end //#if
            #end //#for
            #if (PrevCharFound)
               #declare LineKerning [I] [K] = LineKerning [I] [K] + ItalicOffset [CurChar * 2];
               #declare LineWidth [I] = LineWidth [I] + ItalicOffset [CurChar * 2];
            #end //#if
         #else //Transitioning from italic to normal
            #local PrevCharFound = false;
            #for (K, J - 1, 0, -1)
               #if (LineVisibilityCondition [I] [K] = 0 & LineCharacters [I] [K] != -1)
                  #local PrevCharFound = true;
                  #local PrevChar = LineCharacters [I] [K];
                  #break
               #end //#if
            #end //#for
            #if (PrevCharFound)
               #declare LineKerning [I] [K] = LineKerning [I] [K] + ItalicOffset [PrevChar * 2 + 1];
               #declare LineWidth [I] = LineWidth [I] + ItalicOffset [PrevChar * 2 + 1];
            #end //#if
         #end //#if
         #local LastItalicCondition = CurItalicCondition;
      #end //#if
   #end //#for

   #declare ItalicOffset_Beginning [I] = 0;

   #declare ItalicOffset_End = 0;

   #if (LineItalicCondition [I] [0])
      #local FirstChar = LineCharacters [I] [0];
      #declare ItalicOffset_Beginning [I] = ItalicOffset [FirstChar * 2];
      #declare LineWidth [I] = LineWidth [I] + ItalicOffset_Beginning [I];
   #end //#if

   #if (LineItalicCondition [I] [LineNCharacters [I] - 1])
      #if (LineVisibilityCondition [I] [LineNCharacters [I] - 1] = 0)
         #local LastChar = LineCharacters [I] [LineNCharacters [I] - 1];
      #else
         #local LastChar = LineCharacters [I] [LineNCharacters [I] - 2];
      #end //#if
      #local ItalicOffset_End = ItalicOffset [LastChar * 2 + 1];
      #declare LineWidth [I] = LineWidth [I] + ItalicOffset_End;
   #end //#if

   //#debug concat ("\nLine ", str (I, 0, 0), " width: ", str (LineWidth [I], 0, 3), "\n")

#end //#macro ComputeLineWidth

#macro CreateLineTextObject (I, LineAlignment, MaxLineWidth)

   #ifndef (AdditionalGAP_WidthPercentage)
      #declare AdditionalGAP_WidthPercentage = 60;
   #end //#ifndef

   #local MaxGapPadding = GAP_Width * AdditionalGAP_WidthPercentage / 100; //Padding is in addition to GAP_Width
   #local GapPadding = 0;
   #local SpacePadding = 0;

   #if (LineAlignment = JUSTIFIED & LineNVisibleCharacters [I] > 1)
      #if (I < NLines - 1 | AlignLastLine)
         #local ExtraWidth = MaxLineWidth - LineWidth [I];
         #if (AdditionalGAP_WidthPercentage < 0)
            #local SpaceGrowthRate = 1.5; //Values > 1 will cause SPACE width to increase faster than GAP width
            #local P1 = GAP_Width * (LineNVisibleCharacters [I] - 1);
            #local P2 = SPACE_Width * SpaceGrowthRate * LineNSpaces [I];
            #if (P1 + P2 > 0)
               #local WidthGrowthFactor = ExtraWidth / (P1 + P2);
            #else
               #local WidthGrowthFactor = 0;
            #end //#if
            #local GapPadding = GAP_Width * WidthGrowthFactor;
            #local SpacePadding = SPACE_Width * WidthGrowthFactor * SpaceGrowthRate;
         #else
            #local GapPadding = ExtraWidth / (LineNVisibleCharacters [I] - 1);
            #if (GapPadding > MaxGapPadding & LineNSpaces [I] > 0)
               #local GapPadding = MaxGapPadding;
               #local SpacePadding = (ExtraWidth - (LineNVisibleCharacters [I] - 1) * GapPadding) / LineNSpaces [I];
            #end //#if
         #end //#if
      #end //#if
   #end //#if

   #local CurGap = GAP_Width + GapPadding;
   #local CurSpace = SPACE_Width + SpacePadding;

   #declare LineTextObject [I] = object {
      #local CurX = ItalicOffset_Beginning [I];
      #if (LineNCharacters [I] > 1)
         union {
            #for (J, 0, LineNCharacters [I] - 1)
               #if (J = 0 & LineVisibilityCondition [I] [0] != 0) //First character is invisible
                  #local CurChar = LineCharacters [I] [0];
                  object {
                     BorderCharacter [CurChar]
                     translate CurX * x
                     #if (LineItalicCondition [I] [0])
                        Italicize ()
                     #end //#if
                  } //object
                  #local NextChar = LineCharacters [I] [1];
                  #local CurX = CurX + BorderCharacterWidth [CurChar] + Kerning [CurChar] [NextChar] + CurGap;
                  #if (CurX < ItalicOffset_Beginning [I])
                     #local CurX = ItalicOffset_Beginning [I];
                  #end //#if
               #end //#if
               #if (LineVisibilityCondition [I] [J] = 0)
                  #local CurChar = LineCharacters [I] [J];
                  #local LastX = CurX;
                  #if (CurChar = -1)
                     #local CurX = CurX + LineKerning [I] [J] + CurGap + SpacePadding;
                  #else
                     object {
                        BorderCharacter [CurChar]
                        translate CurX * x
                        #if (LineItalicCondition [I] [J])
                           Italicize ()
                        #end //#if
                     } //object
                     #if (J <= LineNCharacters [I] - 2)
                        #if (J < LineNCharacters [I] - 2 | LineVisibilityCondition [I] [LineNCharacters [I] - 1] = 0)
                           #local CurX = CurX + BorderCharacterWidth [CurChar] + LineKerning [I] [J] + CurGap;
                        #end //#if
                     #end //#if
                  #end //#if
                  #if (J < LineNCharacters [I] - 1)
                     #local NextLVC = LineVisibilityCondition [I] [J + 1];
                     #if (NextLVC != 0)
                        #local NextChar = LineCharacters [I] [J + 1];
                        #if (CurChar = -1)
                           #local TempX = LastX + CurSpace + CurGap;
                        #else
                           #local TempX = LastX + BorderCharacterWidth [CurChar] + Kerning [CurChar] [NextChar] + CurGap;
                        #end //#if
                        object {
                           BorderCharacter [NextChar]
                           translate TempX * x
                           #if (LineItalicCondition [I] [J + 1])
                              Italicize ()
                           #end //#if
                        } //object
                     #end //#if
                  #end //#if
               #end //#if
            #end //#for
         } //union
      #else
         #local CurChar = LineCharacters [I] [0];
         object {
            BorderCharacter [CurChar]
            translate CurX * x
            #if (LineItalicCondition [I] [0])
               Italicize ()
            #end //#if
         } //object
      #end //#if
   } //object

#end //#macro CreateLineTextObject

#macro CreateParagraphTextObject (NLines, LineAlignment, AlignLastLine)

   #declare MaxLineWidth = 0;

   #for (I, 0, NLines - 1)
      InitializeLine (I)
      MarkInvisibleCharacters (I)
      ComputeLineWidth (I)
      #declare MaxLineWidth = max (MaxLineWidth, LineWidth [I]);
   #end //#for

   #for (I, 0, NLines - 1)
      CreateLineTextObject (I, LineAlignment, MaxLineWidth)
   #end //#for

   #declare Paragraph = object {
      #if (NLines > 1)
         union {
            #for (I, 0, NLines - 1)
               #switch (LineAlignment)
                  #case (LEFT)
                     #local XOffset = 0;
                  #break
                  #case (RIGHT)
                     #local XOffset = MaxLineWidth - LineWidth [I];
                  #break
                  #case (CENTERED)
                     #local XOffset = MaxLineWidth / 2 - LineWidth [I] / 2;
                  #break
                  #case (JUSTIFIED)
                     #local XOffset = 0;
                  #break
               #end //#switch
               object {LineTextObject [I] translate <XOffset, -LineSpacing * I, 0>}
            #end //#for
         } //union
      #else
         LineTextObject [0]
      #end //#if
      scale <TextScale, TextScale, 1>
   } //object

   #declare MaxLineWidth = MaxLineWidth * TextScale;

#end //#macro CreateParagraphTextObject
